////////////////////////////////////////////////////////////////////////////////
//                       Spherical harmonics computation.                     //   
// LAST EDIT: Fri Aug  5 08:55:04 1994 by ekki(@prakinf.tu-ilmenau.de)
////////////////////////////////////////////////////////////////////////////////
//  This file belongs to the YART implementation. Copying, distribution and   //
//  legal info is in the file COPYRIGHT which should be distributed with this //
//  file. If COPYRIGHT is not available or for more info please contact:      //
//                                                                            //
//              yart@prakinf.tu-ilmenau.de                                    //
//                                                                            //
// (C) Copyright 1994 YART team                                               //
////////////////////////////////////////////////////////////////////////////////

#include <math.h>
#include <values.h>
#include <stdlib.h>
#include "rs_io.h"
#include "rs_sphr.h"
#include "../global.h"
#include "../error.h"

/////////////////  THE RT_RS_NormConstants MEMBER FUNCTIONS /////////////////////////

// compute the normalizing constants Nlm
// n > l >= m >= -l
float RT_RS_NormConstants::comp_NormConst(int l, int m, boolean Init)
{
  int lmm, lpm;
  double n;

  if(Init)
    for(int i=0;i<RS_MAXl;i++)
      for(int j=0;j<=i;j++)
        (*this)(i,j)=comp_NormConst(i,j,FALSE);

  lmm=l-abs(m);
  lpm=l+abs(m);
  n=1.0;
  while((lmm<lpm) && (n < BIGFLOAT))
    n*=++lmm;
  if (n >= BIGFLOAT)
    return .0;
  return sqrt((2*l+1)/(M_2PI*n));
}

/////////////  THE RT_RS_spherical_harmonics MEMBER FUNCTIONS //////////////

RT_RS_NormConstants RT_RS_spherical_harmonics::N(RS_MAXl);
RT_RS_lmMatrix RT_RS_spherical_harmonics::P(RS_MAXl);
RT_RS_lmMatrix RT_RS_spherical_harmonics::X(RS_MAXl, BIGFLOAT);

// compute the Legendre functions Plm(x)
// RS_MAXl > l >= m >= -l
// -1 >= x <= 1
float RT_RS_spherical_harmonics::comp_LegendreFunc(int l, int m, float x)
{
  float lminusm, res;

  if(fabs(x-X(l,m))<epsilon)
    return P(l,m);
  if(!l && !m)
    res=1.0;
  else
    if(l==m)
      res=(1-2*m)*sqrt(1-x*x)*comp_LegendreFunc(m-1,m-1,x);
    else
      if(l==(m+1))
        res=x*(2*m+1)*comp_LegendreFunc(m,m,x);
      else
      {
        lminusm=l-m;
        res=x*((2*l-1)/lminusm)*comp_LegendreFunc(l-1,m,x)-
            ((l+m-1)/lminusm)*comp_LegendreFunc(l-2,m,x);
      }
  X(l,m)=x;
  P(l,m)=res;
  return res;
}

// compute the normalized spherical harmonics Ylm(theta,phi)
// RS_MAXl > l >= m >= 0
// 0 <= theta <= Pi
// 0 <= phi <= 2*Pi
float RT_RS_spherical_harmonics::comp_SpherHarms(int l, int m, float theta, float phi)
{
#ifdef RS_DEBUG
  if (l >= RS_MAXl || l < 0 || m > l || m < -l)
    rt_Output->errorVar(get_class(), ":Bad arguments in comp_SpherHarms()", NULL);
#endif
  if(m>0)
    return get_NormConst(l,m)*comp_LegendreFunc(l,m,cos(theta))*cos(m*phi);
  if(m<0)
    return get_NormConst(l,m)*comp_LegendreFunc(l,abs(m),cos(theta))*sin(abs(m)*phi);
  return get_NormConst(l,0)*comp_LegendreFunc(l,0,cos(theta))/M_SQRT2; // if m==0
}

// compute the integral of y from a to b using Simpson's rule
// nelm odd!
float RT_RS_spherical_harmonics::comp_Integr(int nelm, RT_RS_Vector& y, float a, float b)
{
  int n=nelm-1;  // number of intervalls
  int nm3=n*3;
  float f,g;
  int i;

#ifdef RS_DEBUG
  if (n & 1)
    rt_Output->errorVar(get_class(), ":Number of elements is even in comp_Integr()", NULL);
#endif
  f=0;
  for(i=2;i<(n-1);i+=2)
    f+=y(i);
  g=2*f;
  f=0;
  for(i=1;i<n;i+=2)
    f+=y(i);
  g+=4*f+y(0)+y(n);
  return g*(b-a)/((float)nm3);
}

// compute the coefficients Clm
// RS_MAXl > l >= m >= 0
float RT_RS_spherical_harmonics::comp_Coeff(int l, int m, RT_RS_Matrix& f)
{
  int fnr = f.get_rows();
  int fnc   = f.get_columns();
  float phi, theta;
  int p,t;
  RT_RS_Vector iphi(RS_IPHI);
  RT_RS_Vector itheta(RS_ITHETA);

  for(p=0;p<fnc;p++)
  {
    phi=p/((float)(fnc-1))*M_2PI;
    for(t=0;t<fnr;t++)
    {
      theta=t/((float)(fnr-1))*M_PI;
      itheta(t)=f(t,p)*comp_SpherHarms(l,m,theta,phi)*sin(theta);
    }
    iphi(p)=comp_Integr(fnr, itheta, .0, (float)M_PI);
  }
  return comp_Integr(fnc, iphi, .0, (float)M_2PI);
}
