/* bdm-icd.c - routines to talk to CPU16 or CPU32 target
 * via ICD interface hardware on IBM PC parallel printer port
 * Copyright (C) 1992 by Scott Howard, all rights reserved
 * Permission is hereby granted to freely copy and use this code or derivations thereof
 * as long as no charge is made to anyone for its use
 *
 * Ported to Linux by D.Jeff Dionne, April 25, 1995
 *
 * NOTE: ICD is a product of P&E Microsystems Inc.
 */

#include	"bdm.h"
#include	"bdmcalls.h"
#include	"bdmerror.h"
#include	"trgtstat.h"

#define	bdm_ctl         0               /* offset of control port from base    */
#define	dsi             1               /* data shift input - PC->MCU          */
#define	dsclk           2               /* data shift clock/breakpoint pin     */
#define	step_out        4               /* set low to force breakpoint	       */
#define	rst_out         8               /* set low to force reset on MCU       */
#define	oe		0x10		/* set to a 1 to enable DSI 	       */
#define	force_berr	0x40		/* set to a 1 to force BERR on target  */

#define	bdm_stat        1               /* offset of status port from base     */
#define	freeze		0x40		/* FREEZE asserted when MCU stopped    */
#define	dso             0x80            /* data shift output - MCU->PC         */

#define	waitcnt         0xffff          /* no of loops to wait for response    */

extern void bdm_delay (unsigned);
extern void bdm_error (int);

static unsigned bdm_speed, bdm_port;
extern unsigned go_cmd, CommandBitCount;
unsigned StatMask = TARGETSTOPPED;

void bdm_init (int port, int baud)
{
	RegsValid = 0;
	bdm_port = port;
	bdm_speed = baud;
	outportb (bdm_port+bdm_ctl,step_out | dsclk | rst_out);
}

void bdm_deinit (void)
{}

/* StopChip stops MCU in preparation for sending command	 */

int StopChip (void)
{
	unsigned ctr;
	char frozen = 0;

	RegsValid = 0;
	if (inportb (bdm_port + bdm_stat) & freeze) return frozen;
	frozen = 1;
	outportb (bdm_port + bdm_ctl, dsclk + rst_out);
	for (ctr = waitcnt; ctr; ctr--)
	{
		if (inportb (bdm_port + bdm_stat) & freeze) break;
		bdm_delay (1);
	}
	if (!ctr)
	{
		outportb (bdm_port + bdm_ctl, dsclk | rst_out | force_berr);
		bdm_delay (waitcnt);
		for (ctr = waitcnt; ctr; ctr--)
		{
			if (inportb (bdm_port + bdm_stat) & freeze) break;
			bdm_delay (1);
		}
	}
	outportb (bdm_port + bdm_ctl, dsclk | rst_out | step_out);
	if (!ctr) bdm_error (BDM_FAULT_RESPONSE);
	return frozen;
}

/* RestartChip resets target MCU, then stops it on first instruction fetch */

void RestartChip (void)
{
	unsigned LoopCount;

	outportb (bdm_port + bdm_ctl, dsclk);
	bdm_delay (waitcnt);
	StopChip ();
}

/* ResetChip applies hardware reset to the target MCU */

void ResetChip (void)
{
	RegsValid = 0;
	outportb (bdm_port + bdm_ctl, dsclk | step_out);
	bdm_delay (waitcnt);
	outportb (bdm_port + bdm_ctl, dsclk | rst_out | step_out);
}

/* bdm_clk sends <value> to MCU for <parameter> bits, returns MCU response */

LONG bdm_clk (WORD value, int count)
{
	LONG ShiftRegister = ((LONG) value) << (32 - count);

	unsigned char DataOut;

	while (count--)
	{
		DataOut = (ShiftRegister & 0x80000000) ? dsi : 0;
		ShiftRegister <<= 1;
		if (!(inportb (bdm_port + bdm_stat) & dso))
			ShiftRegister |= 1;
		outportb (bdm_port + bdm_ctl, DataOut | rst_out | oe | step_out);
		bdm_delay (bdm_speed + 1);
		outportb (bdm_port + bdm_ctl, DataOut | rst_out | oe | step_out | dsclk);
		bdm_delay ((bdm_speed >> 1) + 1);
	}

	outportb (bdm_port + bdm_ctl, dsclk | step_out | rst_out);
	return ShiftRegister;
}

/* StepChip sends GO command word, then triggers breakpoint on first fetch        */

void StepChip (void)
{
#define	DataOut	(go_cmd & 1 ? dsi : 0)

	bdm_clk (go_cmd >> 1, CommandBitCount - 1);
	outportb (bdm_port + bdm_ctl, oe | step_out | DataOut | rst_out);
	bdm_delay (bdm_speed + 1);
	disable ();
	outportb (bdm_port + bdm_ctl, oe | DataOut | dsclk | rst_out);
	outportb (bdm_port + bdm_ctl, dsclk | rst_out);
	enable ();
	StopChip ();
}

/* GetStatus is called by user code to get current status of CPU signals */

unsigned GetStatus (void)
{
	BYTE temp = inportb (bdm_port + bdm_stat);

	return 	(temp & freeze ? TARGETSTOPPED: 0);
}

/* GetStatusMask returns mask showing which stat bits are valid */

unsigned GetStatusMask (void)
{
	return TARGETSTOPPED;
}
