/*
    brs_extract : Browsing an Analysis Tool for C-source code;
	          Extracts definition line and leading comment 
                  from source file

    Copyright (C) 1994  Eckehard Stolz

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation (version 2 of the License).

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

%{



   #include <stdio.h>
   #include <stdlib.h>
   #include <string.h>
   #include <time.h>
   #include <sys/types.h>
   #include <unistd.h>

   #define TYPE_FUNCTION    1
   #define TYPE_OTHER       2
   
   static long   target_line;           /* lineno, where function defined */
   static long   block_depth;           /* level ov'f actual block        */
   static long   comment_start = -1;    /* start of a comment             */
   static long   comment_end = -1;      /* end of a comment               */
   static long   cmd_start   = -1;      /* start of a comment             */
   static long   cmd_end     = -1;      /* end of a comment               */
   static long   block_start = -1;      /* start of a comment             */
   static long   block_end = -1;        /* end of a comment               */
   static long   definition_start = -1; /* start of a comment             */
   static long   definition_end = -1;   /* end of a comment               */
   static long   item_start = -1;       /* start of a comment             */
   
   static char   target_id[1024];
   static char   source_file[1024];
   
   static short  target_type = TYPE_FUNCTION;
   
   static short  target_found = 0; 
   
   /* set to 1 if we should include leading comment 
    */
   static short  flag_comment = 0; 
   
   /* set to 1 if we should include the body 
    */
   static short  flag_body = 0;    

   /* if set to 1, we will wait for end of the definition
      ( ';' in the same block level as the definition )
    */ 
   static short wait_for_body_flag = -1;

   static long   lineno = 1;
   static long   file_pos = 0;

%}

DIGIT   [0-9]
ID      [a-zA-Z_][a-zA-Z_0-9]*

INT     {DIGIT}*
FILEN   [^\"<>]*
WHITE   [\t ]*

%x toplev in_comment in_string in_quote definition

%%

          
<toplev>\n      {
	lineno++;
        if( file_pos == item_start )
          item_start += yyleng;
   
	file_pos += yyleng;
	}

<toplev>"#define".*\\\n	 {     /* #define more than one line */
        BEGIN( definition );
        lineno++;
        if( file_pos == item_start )
          item_start += yyleng;
        file_pos += yyleng;
        }

<toplev>"#define".*\n	 {     /* #define exactly one line */
        lineno++;
        /* cmd_start = item_start; */
        if( file_pos == item_start )
          item_start += yyleng;
        file_pos += yyleng;
        /* cmd_end   = file_pos; */
        comment_start = -1;
        }

<definition>.*\\\n	{  /* continuing definition */
        lineno++;
        if( file_pos == item_start )
          item_start += yyleng;
        file_pos += yyleng;
        }

<definition>.*\n	{  /* end of a continued definition */
        BEGIN( toplev );
        lineno++;
        /* cmd_start = item_start; */
        if( file_pos == item_start )
          item_start += yyleng;
        file_pos += yyleng;
        /* cmd_end   = file_pos; */
        comment_start = -1;
        }

<in_comment,in_quote,in_string>\n      {
	lineno++;
	file_pos += yyleng;
	}


<toplev>"/*"          { /* comment */
   
        if( lineno > target_line && wait_for_body_flag == -1 )
          return; /* not found */
   
	BEGIN( in_comment );
   
	if( wait_for_body_flag == -1 ) 
          { if(  comment_start == -1 ||       /* first comment found */
             cmd_start > comment_start || /* something between this and the last c. */
             definition_start > comment_start  )
              { /* If there is something between two comments,
                   mark this comments as new comment-start
                 */
                comment_start = file_pos;
               }
           }
   
	file_pos += yyleng;
	}

<in_comment>"*/"	{ /* end of comment */
        BEGIN( toplev );
   
	if( wait_for_body_flag == -1 )
           comment_end = file_pos + yyleng;
   
	file_pos += yyleng;
        item_start = file_pos;
        }

<toplev>{WHITE}		{ /* whitespaces but no CR/LF */
        if( file_pos == item_start )
          item_start += yyleng;
   
	file_pos += yyleng;
        }

<toplev>";"|","|"="	{ /* end of a command or variable def. */
   
        if( !target_found && lineno > target_line && wait_for_body_flag == -1 )
          return; /* not found */
   
        /* if( !block_depth ) */
        if( target_found )
         { /* target found in this command !
              if we are looking for a type or a variable
              and the line matches, extract this command 
            */
           if( wait_for_body_flag != -1 )
             { /* we wait for the end of the body !
                  if the block_depth is matching and we
                  found the ';' , terminate with the body-
                  end set properly
                */
               if( block_depth == wait_for_body_flag &&
                   yytext[0] == ';' )
                 { block_end = file_pos + 1;
                   return;
                  }
               }
           else if( target_type != TYPE_FUNCTION &&
         	    wait_for_body_flag == -1 )
             { /* we are looking for a variable or a
                  typedef (and are not waiting for
                  the end of the body)
                */
               if( lineno >= target_line )
                 { /* we are processing the target-command
                      or definition
                    */
                   definition_end = file_pos + 1;
                   definition_start = item_start;
            
                   if( cmd_start > comment_start )
                     { /* command between comment and target-item 
                        */   
                       comment_start = -1; /* command between comment and function */   
                      }
            
                   if( !flag_body || yytext[0] == ';' )
                     return;
                   else
                     { /* also extract body of the definition: */
                       block_start = file_pos;
                       wait_for_body_flag = block_depth;
                      }
                  }
         
              }
           else if( target_type == TYPE_FUNCTION )
             { /* we are looking for a function: do nothing until
                  we reach the beginning of the block
                */
               ;
              }
          }
        else
         { /* target not found in this command !
              (if ';', we are at the end of an command)
            */
           if( yytext[0] == ';' )
            { target_found = 0;
              cmd_start = item_start;
              cmd_end   = file_pos + yyleng;
              item_start = file_pos + yyleng;
             }
          } 
   
	file_pos += yyleng;
        }
   
<toplev>{ID}	{ /* some id */
           
        if( !strcmp( yytext, target_id )  )
          target_found = 1;
   
        if( wait_for_body_flag != -1 )
          { /* we wait for the end of the body !
               if the block_end is set, we can terminate immediatly
             */
            if( block_end != -1 )
              { return;
               }
            }
   
	file_pos += yyleng;
        }

<toplev>"{"	{ /* begin of block */
      
        block_depth++;
      
        if( block_depth == 1 )
          { /* start of a toplevel-block 
             */
            block_start = file_pos;
         
           if( wait_for_body_flag != -1 )
             { /* we wait for the end of the body !
                  ignore beginning of this block
                */
                ;
               }
            else if( target_found && target_line <= lineno )
             { /* we found the target-id and the actual linenumber
                  is higher than the target-line ! So we found the
                  definition position of this function
                */
               definition_end = file_pos;
               definition_start = item_start;
            
               if( cmd_start > comment_start )
                 { /* command between comment and target-item 
                    */   
                   comment_start = -1; /* command between comment and function */   
                  }
            
               if( !flag_body )
                 return;
               else
                 { /* also extract body of the definition: */
                   block_start = file_pos;
                   wait_for_body_flag = block_depth - 1;
                  }
              }
               
           }
        if( file_pos == item_start )
          item_start += yyleng;
         
        target_found = 0;
	file_pos += yyleng;
        item_start = file_pos;
        }
   
<toplev>"}"	{ /* end of block */
        block_depth--;
      
        if( wait_for_body_flag == block_depth )
          { /* we waited for the end of the body !
               we set the appropriate block-end, but we
               do continue, since there might be a trailing
               ';' ! But the appropriate set block_end will
               cause termination at the end of the next 
               command without causing any harm, because
               the wait_for_body_flag will still be set
             */
            block_end = file_pos + 1;
            }

	file_pos += yyleng;
        item_start = file_pos;
        comment_start = -1;
        }

<toplev>"\""	{ /* start of a string */
        BEGIN( in_string );
	file_pos += yyleng;
        }
   
<in_string>"\\\""	{ /* '\"' in string: stay in this state */
	file_pos += yyleng;
        }

<in_string>"\""	{ /* end of a string */
        BEGIN( toplev );
   
	file_pos += yyleng;
        }

<toplev>"'"	{ /* Quoting */
        BEGIN( in_quote );
   
	file_pos += yyleng;
        }

<in_quote>"'"	{ /* End of quoting */
        BEGIN( toplev );
   
	file_pos += yyleng;
        }

<in_quote>"\""	{ /* '\"' in quotes */
   
	file_pos += yyleng;
        }

<toplev>"#include".*\n	 {     /* #include  */
        lineno++;
        cmd_start = item_start;
        if( file_pos == item_start )
          item_start += yyleng;
        file_pos += yyleng;
        cmd_end   = file_pos;
        }

<toplev,in_comment,in_string,in_quote>.               { /* Nothing to do else */
	file_pos += yyleng;
	}


.	{ /* initial rule , if file-positioning works, this rule is obsolete
             and just consist of an unput and a BEGIN(toplev)
           */
         unput( yytext[0] ); 
         BEGIN( toplev );
	}

%%



#ifdef STAND_ALONE
main( int argc, char *argv[] )
{ short        i, flag;

  char         *ptr;
      
  long         cnt;

   if( argc <= 1 || argv[1][1] == '?' )
     { fprintf(stderr, "Usage: brs_extract [Options] <source-file> <item> <lineno>\n" );
       fprintf(stderr, "Browser-extract: tries to cut the definition line and a leading \n");
       fprintf(stderr, "comment out of the source code \n\n");
       fprintf(stderr, "<source-file>    ...  filename, where the item is defined\n");
       fprintf(stderr, "<item>           ...  identifier of the item to be extracted\n");
       fprintf(stderr, "<lineno>         ...  linenumber, where this item is defined\n");
       fprintf(stderr, "                      (because of the browser-bug with the linenumber \n");
       fprintf(stderr, "                       set to the first token AFTER the definition \n");
       fprintf(stderr, "                       this function does accept linenumber which are\n");
       fprintf(stderr, "                       within a range of the definition position !\n");
       fprintf(stderr, "                       BUT CAUTION: maybe it works, maybe not !)\n");
       fprintf(stderr, "\nOptions:\n");
       fprintf(stderr, "-c    extract comment\n");
       fprintf(stderr, "-b    extract body\n");
       fprintf(stderr, "-n    item is NOT a function\n");
       exit(1);
      }

   yyin = stdin;
   
   source_file[0] = 0;
   target_id[0] = 0;

   for( i = 1; i < argc; i++ )
     { /* ordinary option given: */
       if( argv[i][0] != '-' )
         { if( source_file[0] == 0 )
             { strcpy( source_file, argv[i] ); 
               yyin = fopen( argv[i], "r" );
	       if( !yyin )
		 { fprintf(stderr,
		     "Could not open Source-file \"%s\" !\n",
		     argv[i]);
		   exit(1);
		  }
               }
            else if( target_id[0] == 0 )
              { strcpy( target_id, argv[i] );
               }
            else
              target_line = atoi( argv[i] );
          }
       else
        { switch( argv[i][1] )
             { case 'n': target_type = TYPE_OTHER;
                         break;
               case 'c': flag_comment = 1;
                         break;
               case 'b': flag_body = 1;
                         break;
               default:
	          fprintf(stderr, "Unknown Option \"%s\" !\n", argv[i] );
	          fprintf(stderr, "Type \"brs_extract -?\" for help !\n" );
	          exit(1);
              }
	  }
      }

   block_depth = 0;
   
   /* yy_flex_debug = 0; */
   yylex();
   
   if( definition_start == -1 || definition_end == -1 )
     { fprintf( stderr, "Function definition for %s (line %ld) not found !\n",
           target_id, target_line);
       exit(1);
      }

   cnt = definition_end - definition_start;
   ptr = malloc( cnt + 10);
   if( !ptr ) 
     { fprintf( stderr, "Not enough memory !\n" );
       exit(1);
      }
   
   if( lseek( fileno(yyin), definition_start, SEEK_SET ) == -1 )
     { fprintf( stderr, "lseek failed !\n" );
       exit(1);
      }
   
   if( read (fileno(yyin), ptr, cnt) < cnt )
     { fprintf( stderr, "read failed !\n" );
       exit(1);
      }
   
   printf("******************************************************\n");
   for( i = 0; i < cnt; i++ )
     { putchar( ptr[i] );
      }
   printf("******************************************************\n");

   free( ptr );
      
   if( flag_comment && comment_start != -1 && comment_end != -1 )
     {  cnt = comment_end - comment_start + 1;
        ptr = malloc( cnt + 10);
        if( !ptr ) 
          { fprintf( stderr, "Not enough memory !\n" );
            exit(1);
           }

        if( lseek( fileno(yyin), comment_start, SEEK_SET ) == -1 )
          { fprintf( stderr, "lseek failed !\n" );
            exit(1);
           }
   
        if( read (fileno(yyin), ptr, cnt) < cnt )
          { fprintf( stderr, "read failed !\n" );
            exit(1);
           }
      
        printf("******************************************************\n");
        for( i = 0; i < cnt; i++ )
          { putchar( ptr[i] );
           }
        printf("******************************************************\n");
      }

   if( flag_body && block_start != -1 && block_end != -1 )
     {  cnt = block_end - block_start + 1;
        ptr = malloc( cnt + 10);
        if( !ptr ) 
          { fprintf( stderr, "Not enough memory !\n" );
            exit(1);
           }

        if( lseek( fileno(yyin), block_start, SEEK_SET ) == -1 )
          { fprintf( stderr, "lseek failed !\n" );
            exit(1);
           }
   
        if( read (fileno(yyin), ptr, cnt) < cnt )
          { fprintf( stderr, "read failed !\n" );
            exit(1);
           }
      
        printf("******************************************************\n");
        for( i = 0; i < cnt; i++ )
          { putchar( ptr[i] );
           }
        printf("******************************************************\n");
      }
}

#else

/* cut's the definition line of a function and a possible leading
   comment.

   type         ...  0 item is a function, 1 item is something else

   source_file  ...  file where the function is defined
   target_id    ...  identifier of the function
   target_line  ...  line, where function is defined (may be set
                     between the identifier and the beginning of
                     the function-body
   comment      ...  pointer to a char-pointer; if NULL: do not
                     extract comment
   body         ...  pointer to a char-pointer; if NULL: do not
                     extract body
   

   Return-values:
   def_line     ...  pointer to a malloced memory-area with the
                     definition line ( '\0'-terminated )
                     (or NULL if definition line could not found)
   
   comment      ...  pointer to the leading comment (malloc used)
                     (or NULL if comment could not found)
   
   body         ...  pointer to the function body (malloc used)
                     (or NULL if body could not found)
   
   return-value      0 ... O.K.
                     1 ... item not found
                    -1 ... not enough memory
                    -2 ... file read or positioning error
 */
short brs_cut(short type, char *s_file, char *t_id, long t_line, 
              char **def_line, char ** comment, char**body)
{ short        i, flag;

  char         *ptr;
      
  long         cnt;
   
   comment_start = -1;
   comment_end = -1;
   cmd_start   = -1;
   cmd_end     = -1;
   block_start = -1;
   block_end = -1;
   definition_start = -1;
   definition_end = -1;
   item_start = -1;
   target_found = 0; 
   wait_for_body_flag = -1;
   lineno = 1;
   file_pos = 0;
   block_depth = 0;
   
   /* allways set debugging flag to 0, so debugging is only
      available via brs_extract
   yy_flex_debug = 0; 
    */

   if( !def_line )
     return(-1);
   
   *def_line = NULL;
   
   if( comment )
     flag_comment = 1;
   
   if( body )
     flag_body = 1;
      
   strcpy( source_file, s_file );
   strcpy( target_id, t_id );
   target_line = t_line;
   
   
   yyin = fopen( source_file, "r" );
   if( !yyin )
     { return( -2 );
      }
   
   /* yy_flex_debug = 0; */
   yylex();
   
   if( definition_start == -1 || definition_end == -1 )
     { yyrestart(yyin);
       fclose( yyin );
       return( 1);
      }

   cnt = definition_end - definition_start;
   ptr = malloc( cnt + 10);
   if( !ptr ) 
     { yyrestart(yyin);
       fclose( yyin );
       return(-1);
      }
   
   if( lseek( fileno(yyin), definition_start, SEEK_SET ) == -1 )
     { yyrestart(yyin);
       fclose( yyin );
       return( -2);
      }
   
   if( read (fileno(yyin), ptr, cnt) < cnt )
     { yyrestart(yyin);
       fclose( yyin );
       return( -2);
      }
   ptr[cnt] = 0;
      
   *def_line = ptr;

   if( flag_comment )      
      *comment = NULL;
   
   if( flag_comment && comment_start != -1 && comment_end != -1 )
     {  cnt = comment_end - comment_start + 1;
      
        ptr = malloc( cnt + 1);
        if( !ptr ) 
          { yyrestart(yyin);
            fclose( yyin );
            return(-1);
           }

        if( lseek( fileno(yyin), comment_start, SEEK_SET ) == -1 )
          { yyrestart(yyin);
            fclose( yyin );
            return(-2);
           }
   
        if( read (fileno(yyin), ptr, cnt) < cnt )
          { yyrestart(yyin);
            fclose( yyin );
            return(-2);
           }
      
        ptr[cnt] = 0;
        *comment = ptr;
      }

   if( flag_body )
     *body = NULL;
   
   if( flag_body && block_start != -1 && block_end != -1 )
     {  cnt = block_end - block_start + 1;

        ptr = malloc( cnt + 1);
        if( !ptr ) 
          { yyrestart(yyin);
            fclose( yyin );
            return(-1);
           }

        if( lseek( fileno(yyin), block_start, SEEK_SET ) == -1 )
          { yyrestart(yyin);
            fclose( yyin );
            return(-2);
           }
   
        if( read (fileno(yyin), ptr, cnt) < cnt )
          { yyrestart(yyin);
            fclose( yyin );
            return(-2);
           }
      
        ptr[cnt] = 0;
        *body = ptr;
      }
   
  yyrestart(yyin);
  fclose( yyin );
  return( 0 );
}
   
#endif
