/*
 * static char *rcsid_xio_c =
 *   "$Id: x11.c,v 1.4 1995/04/15 03:27:03 master Exp master $";
 *
 * This file handles all the windowing stuff.  The idea is
 * that all of it is in one file, so to port to different systems
 * or toolkits, only this file needs to be updated.  All windowing
 * variables (display, gc's, windows, etc), should be stored in
 * this file as statics.
 *
 * This file is largely a combination of the common/xutil.c and server/xio.c
 * file.  While I don't think this was a particulary great interface, the code
 * was there, and I figured it was probably easier to re-use that
 * code instead of writing new code, plus the old code worked.
 *
 */

/*
    CrossFire, A Multiplayer game for X-windows

    Copyright (C) 1994 Mark Wedel
    Copyright (C) 1992 Frank Tore Johansen

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    The author can be reached via e-mail to master@rahul.net
*/

/* Most functions in this file are private.  Here is a list of
 * the global functions:
 *
 * draw_color_info(int color, char*buf) - draws text in specified color
 * draw_info - draw info in the info window
 * end_windows - used when exiting
 * init_windows - called when starting up
 * load_images - creates the bitmaps and pixmaps (if being used)
 * create_pixmap - creates a pixmap from given file and assigns to
 *		the given face
 * create_xpm - as create_pixmap, but does an XPM image
 * load_additional_images - loads images that have been downloaded
 *	from the server in prior sessions
 *
 * draw_stats(int) - draws the stat window.  Pass 1 to redraw all
 *	stats, not only those that changed
 *
 * draw_message_window(int) - draws the message window.  Pass 1 to redraw
 *	all the bars, not only those that changed.
 *
 * draw_look, draw_inv:  Update the look and inventory windows.
 *
 * NOTE: create_pixmap and create_xpm can be empty functions if the
 * client will always use fonts - in that case, it should never
 * request Bitmap or Pixmap data, and thus not need the create
 * functions above
 *
 * Only functions in this file should be calling functions like
 * draw_stats and draw_message_window with redraw set - functions
 * in other files should always pass 0, because they will never have
 * the information of whether a redraw is needed.
 */


#include <client.h>
#include <clientbmap.h>
#include <item.h>

#ifdef Xpm_Pix
#include <X11/xpm.h>
#endif
#include <X11/Xlib.h>
#include <X11/Xutil.h>

#define MAX_BUF 256

/* All the following are static because these variables should
 * be local only to this file.  Since the idea is to have only
 * this file be replaced for different windowing systems, use of
 * any of these variables anyplace else would not be portable.
 */

/*
 *  This is similar obwin, but totally redone for client
 */
typedef struct {
    item *env;		  /* Environment shown in window */
    char title[MAX_BUF];  /* title of item list */
    char old_title[MAX_BUF];  /* previos title (avoid redrawns) */

    Window win;		  /* for X-windows */
    GC gc_text;
    GC gc_icon;
    GC gc_status;

    uint8 show_icon:1;	  /* show status icons */
    uint8 show_weight:1;  /* show item's weight */

    char format_nw[16];	  /* sprintf-format for text (name and weight) */
    char format_n[16];	  /* sprintf-format for text (only name) */
    sint16 text_len;	  /* How wide the text-field is */

    sint16 width;	  /* How wide the window is in pixels */
    sint16 height;	  /* How height the window is in pixels */

    sint16 item_pos;	  /* The sequence number of the first drawn item */
    sint16 item_used;	  /* How many items actually drawn. (0 - size) */

    sint16 size;	  /* How many items there is room to display */
    sint16 *faces;	  /* [size] */
    sint8 *icon1;	  /* status icon : locked */
    sint8 *icon2;	  /* status icon : applied / unpaid */
    sint8 *icon3;	  /* status icon : magic */
    sint8 *icon4;	  /* status icon : damned / cursed */
    char **names;	  /* [size][NAME_LEN] */

    /* The scrollbar */
    sint16 bar_length; 	  /* the length of scrollbar in pixels */
    sint16 bar_width;	  /* the width of scrollbar in pixels (not used yet) */
    sint16 bar_size;	  /* the current size of scrollbar in pixels */
    sint16 bar_pos;	  /* the starting position of scrollbar in pixels */
} itemlist;

static char font_text_stats[]="8x13", font_text_info[]="8x13",
	font_inv_text[]="8x13",
	font_graphic[]=FONTNAME, /* Defined in global.h */
	**gargv;

#define INFOCHARS 50
#define INFOLINES 36
#define FONTWIDTH 8
#define FONTHEIGHT 13
#define MAX_INFO_WIDTH 80
#define MAXNAMELENGTH 50
#define WINUPPER (-5)
#define WINLOWER 5
#define WINLEFT (-5)
#define WINRIGHT 5

static int gargc,maxinfolines=INFOLINES;

Display_Mode display_mode = Xpm_Display;
static Display *display;
static long screen_num;
static unsigned long foreground,background;
static Window win_root, win_game,win_stats,win_info,win_message;
static Colormap colormap;
static XColor discolor[16];
static Font font; /* I don't see why we should try to support fonts,
		bitmaps are more convinient, and won't require
		downloading entire .pcf files, etc. 
		- eanders@cs.berkeley.edu */
static XEvent event;

/* These are local to this file because depending on the toolkit, the
 * scrollbars may be handled automatically, and these values would then
 * note be needed.
 */
static uint16 info_chars,
	infopos=0,		/* Where in the info array to put the data */
	infoline=0;		/* Where on the window to draw the line */

static uint8	scroll_info_window=TRUE, info_full=FALSE,
	split_windows=FALSE,
	iscolor = TRUE;


static char **info;

#define MAXFACES 100
#define MAXPIXMAPNUM 10000
struct MapCell {
  short faces[MAXFACES];
  int count;
};

struct Map {
  struct MapCell cells[11][11];
};

struct PixmapInfo {
  Pixmap pixmap,mask;
  Pixmap bitmap;
  long fg,bg;
};

static struct Map the_map;
static char stats_buff[7][600];
static struct PixmapInfo pixmaps[MAXPIXMAPNUM];
#define INFOLINELEN 500
#define XPMGCS 100

enum {
    locked_icon = 1, applied_icon, unpaid_icon,
    damned_icon, cursed_icon, magic_icon, close_icon, max_icons
};
static struct PixmapInfo icons[max_icons];

static Pixmap icon,xpm_pixmap,xpm_masks[XPMGCS]; 
static GC gc_root,gc_game,gc_stats,gc_info,gc_message,
	gc_floor,gc_xpm_object,gc_clear_xpm,gc_xpm[XPMGCS],
	gc_blank; 

/*
 * These are used for inventory and look window
 */
static itemlist look_list, inv_list;


static Window win_root, win_game,win_stats,win_info,win_message;
/* info win */
#define INFOCHARS 50
#define FONTWIDTH 8

#include <xutil.c>

int misses=0,total=0;

static void gen_draw_face(Drawable where,int face,int x,int y)
{
  if (display_mode == Pix_Display) {
    XSetForeground(display,gc_game,discolor[pixmaps[face].fg].pixel);
    XSetBackground(display,gc_game,discolor[pixmaps[face].bg].pixel);
    XCopyPlane(display,pixmaps[face].bitmap,where,gc_game,
	     0,0,24,24,x,y,1);
  } else if (display_mode == Xpm_Display && pixmaps[face].mask == None) {
	XCopyArea(display, pixmaps[face].pixmap,
	      where,gc_floor,
	      0,0,24,24,x,y);
  } else if (display_mode == Xpm_Display) {
    int gcnum,i;
    Pixmap mask;
    GC gc;
    total++;
    mask = pixmaps[face].mask;
    for(gcnum=0;gcnum<XPMGCS;gcnum++) {
	if (xpm_masks[gcnum] == mask)
	break;
    }
    if (gcnum == XPMGCS) {
	misses++;
	gcnum--;
	gc = gc_xpm[gcnum];
	XSetClipMask(display,gc,mask);
    } 
    gc = gc_xpm[gcnum];
    for(i=gcnum-1;i>=0;i--) {
	xpm_masks[i+1] = xpm_masks[i];
	gc_xpm[i+1] = gc_xpm[i];
    }
    xpm_masks[0] = mask;
    gc_xpm[0] = gc;
    XCopyArea(display, pixmaps[face].pixmap,
	    where,gc_xpm[0],
	    0,0,24,24,x,y);
  } else {
    fprintf(stderr,"Unknown display mode %d\n",display_mode);
    abort();
  }
}

void end_windows()
{
    XFreeGC(display, gc_root);
    XFreeGC(display, gc_game);
    XFreeGC(display, gc_stats);
    XFreeGC(display, gc_info);
    XFreeGC(display, inv_list.gc_text);
    XFreeGC(display, inv_list.gc_icon);
    XFreeGC(display, inv_list.gc_status);
    XFreeGC(display, look_list.gc_text);
    XFreeGC(display, look_list.gc_icon);
    XFreeGC(display, look_list.gc_status);
    XFreeGC(display, gc_message);
    if (display_mode==Xpm_Display) {
	XFreeGC(display, gc_xpm_object);
    }
    XDestroyWindow(display,win_game);
    XCloseDisplay(display);
}

/* get_root_display: 
 * this sets up the root window (or none, if in split
 * windows mode, and also scans for any Xdefaults.  Right now, only
 * splitwindow and image are used.  image is the display
 * mechanism to use.  I thought having one type that is set
 * to font, xpm, or pixmap was better than using xpm and pixmap
 * resources with on/off values (which gets pretty weird
 * if one of this is set to off.
 */

/* Error handlers removed.  Right now, there is nothing for
 * the client to do if it gets a fatal error - it doesn't have
 * any information to save.  And we might as well let the standard
 * X11 error handler handle non fatal errors.
 */
int sync_display = 0;
static int get_root_display(char *display_name) {
    char *cp;
    XSizeHints roothint;
    static char errmsg[MAX_BUF];

    display=XOpenDisplay(display_name);


    if (!display) {
	sprintf(errmsg, "Can't open display %s.", display_name);
	return 1;
    }
    if ((getenv("ERIC_SYNC")!= NULL) || sync_display)
	XSynchronize(display,True);

    /* For both split_windows and display mode, check to make sure that
     * the command line has not set the value.  Command line settings
     * should always have precedence over settings in the Xdefaults file.
     *
     * Also, if you add new defaults, make sure that the same rules for
     * the command line defaults are used.  The client assumes that certain
     * values can only be set if they have meaning (save_data will not
     * be set unless the data can actually be saved, for example.)  If
     * this is not followed for then the client might very well crash.
     */
    if (!split_windows && 
      (cp=XGetDefault(display,X_PROG_NAME,"splitwindow")) != NULL) {
	if (!strcmp("on",cp) || !strcmp("yes",cp))
	    split_windows = TRUE;
	else if (!strcmp("off",cp) || !strcmp("no",cp))
	    split_windows = FALSE;
    }
    if (!cpl.echo_bindings &&
      (cp=XGetDefault(display,X_PROG_NAME,"echo")) != NULL) {
	if (!strcmp("on",cp) || !strcmp("yes",cp))
	    cpl.echo_bindings = TRUE;
	else if (!strcmp("off",cp) || !strcmp("no",cp))
	    cpl.echo_bindings = FALSE;
    }
    if (display_mode==Font_Display &&
      (cp=XGetDefault(display,X_PROG_NAME, "image")) != NULL) {
	if (!strcmp("pixmap",cp))
	    display_mode = Pix_Display;
	else if (!strcmp("xpm",cp))
	    display_mode = Xpm_Display;
	else if (!strcmp("font",cp))
	    display_mode = Font_Display;
	else
	    fprintf(stderr,"Warning: Unknown image option: %s\n", cp);
    }

    screen_num=DefaultScreen(display);
    background=WhitePixel(display,screen_num);
    foreground=BlackPixel(display,screen_num);
    roothint.x=0;
    roothint.y=0;
    roothint.width=582+6+INFOCHARS*FONTWIDTH;
    roothint.height=482;
    roothint.max_width=roothint.min_width=roothint.width;
    roothint.max_height=roothint.min_height=roothint.height;
    roothint.flags=PPosition | PSize;

    if(!split_windows) {
	win_root=XCreateSimpleWindow(display,DefaultRootWindow(display),
	    roothint.x,roothint.y,roothint.width,roothint.height,2,
	    background,foreground);

	iscolor=allocate_colors(display, win_root, screen_num,
	    &colormap, discolor);
	if (iscolor){
	    foreground=discolor[0].pixel;
	    background=discolor[8].pixel;
	}
	icon=XCreateBitmapFromData(display,win_root,
	    (_Xconst char *) crossfire_bits,
	    (unsigned int) crossfire_width, (unsigned int)crossfire_height);
	XSetStandardProperties(display,win_root,X_PROG_NAME,X_PROG_NAME,
	    icon,gargv,gargc,&(roothint));
	gc_root=XCreateGC(display,win_root,0,0);
	XSetForeground(display,gc_root,foreground);
	XSetBackground(display,gc_root,background);
    }
    else win_root = DefaultRootWindow(display);

    if(display_mode==Font_Display)
	(void) fixfontpath(display,"client");

    if(!split_windows) {
	if(display_mode==Font_Display) {
	    font=XLoadFont(display,font_graphic);
	    XSetFont(display,gc_root,font);
	}
	XSelectInput(display,win_root,KeyPressMask|
	     KeyReleaseMask|ExposureMask|StructureNotifyMask);
	XMapRaised(display,win_root);
	XNextEvent(display,&event);
    }
    return 0;
}


static int get_game_display() {
    XSizeHints gamehint;
   
    gamehint.x=305;
    gamehint.y=104;
    gamehint.width=269;
    gamehint.height=269;
    gamehint.max_width=gamehint.min_width=gamehint.width;
    gamehint.max_height=gamehint.min_height=gamehint.height;
    gamehint.flags=PPosition | PSize;
    win_game=XCreateSimpleWindow(display,win_root,
	gamehint.x,gamehint.y,gamehint.width,gamehint.height,2,
	background,foreground);
    icon=XCreateBitmapFromData(display,win_game,
	(_Xconst char *) crossfire_bits,
	(unsigned int) crossfire_width, (unsigned int)crossfire_height);
    if (split_windows) {
	iscolor=allocate_colors(display, win_root, screen_num,
	    &colormap, discolor);
	if (iscolor){
	    foreground=discolor[0].pixel;
	    background=discolor[8].pixel;
	}
    } else
	XSetWindowColormap(display, win_game, colormap);

    XSetStandardProperties(display,win_game,X_PROG_NAME, X_PROG_NAME,
	icon,gargv,gargc, &(gamehint));

    gc_game=XCreateGC(display,win_game,0,0);
    if (iscolor){
	XSetForeground(display,gc_game,discolor[0].pixel);
	XSetBackground(display,gc_game,discolor[12].pixel);
    }
    else {
	XSetForeground(display,gc_game,foreground);
	XSetBackground(display,gc_game,background);
    }
    XSetGraphicsExposures(display, gc_game, False);
    if(display_mode==Font_Display) {
	font=XLoadFont(display,font_graphic);
	XSetFont(display,gc_game,font);
    }
    gc_floor = XCreateGC(display,win_game,0,0);
    XSetGraphicsExposures(display, gc_floor, False);
    gc_blank = XCreateGC(display,win_game,0,0);
    XSetForeground(display,gc_blank,discolor[12].pixel);
    XSetGraphicsExposures(display,gc_blank,False);
    if (display_mode==Xpm_Display) {
	int i;
	for (i=0; i<XPMGCS; i++) {
	    gc_xpm[i] = XCreateGC(display, win_game, 0,0);
	    XSetClipOrigin(display, gc_xpm[i], 0, 0);
	    XSetGraphicsExposures(display, gc_xpm[i], False);
	}
	gc_xpm_object = XCreateGC(display,win_game,0,0);
	XSetGraphicsExposures(display, gc_xpm_object, False);
	XSetClipOrigin(display, gc_xpm_object,0, 0);
	xpm_pixmap = XCreatePixmap(display, RootWindow(display,
		DefaultScreen(display)), 24, 24, DefaultDepth(display,
		DefaultScreen(display)));
	gc_clear_xpm = XCreateGC(display,xpm_pixmap,0,0);
	XSetGraphicsExposures(display,gc_clear_xpm,False);
	XSetForeground(display,gc_clear_xpm,BlackPixel(display,screen_num));
    }

    XSelectInput(display,win_game,
	ButtonPressMask|KeyPressMask|KeyReleaseMask|ExposureMask);
    XMapRaised(display,win_game);
    return 0;
}


#if 0

/* This function loads the images onto the display.
 * Various notes:  If the display mode is Font_Display, then (under X11),
 * the font can not be changed.  Thus, for porting to other systems,
 * if you want to be able to download images, do not set display_mode
 * to Font_Display
 *
 * While the calls to ReadBitmaps and ReadPixmaps could easily be
 * made from client.c or elsewhere, creating the images might vary
 * system to system.  Thus, by putting the code here, different function
 * calls can be used to load the images instead of ReadBitmaps or ReadPixmaps.
 * 
 */
void load_images()
{

    if (display_mode==Font_Display) return;
    if (display_mode==Pix_Display)
	pixmaps = ReadBitmaps(display);		/* common/xutil.c */
    else if (display_mode==Xpm_Display)
	ReadPixmaps(display, &pixmaps, &masks); /* common/xutil.c */
    allocated_pixmaps = nrofpixmaps;
}

/* This creates a temporary image that can be accessed until the
 * real image arrives.
 */

void create_temporary_image(int number)
{
	if (number>=allocated_pixmaps) {
		pixmaps = realloc(pixmaps, sizeof(Pixmap) * (number+1));
		allocated_pixmaps = number;
	}
	pixmaps[number] = pixmaps[stipple1_face.number];
	if (display_mode==Xpm_Display)
		masks[number] = masks[stipple1_face.number];
}

/* This creates a pixmap from file 'name', in array position
 * number.  Other functions should have set up the xbm and xbm_names
 * arrays appropriately (note that xbm_names[number] = image_name,
 * and xbm[FindFace(image_name)].number = number.
 *
 * return 1 on error, 0 on success.  Errors (missing files) should
 * only occur from calls in load additional images.  This is because
 * there are 2 image types - bitmap and XPM.  It is possible that one
 * has been downloaded, but not the other.  Thus, the local bitmap name
 * file is updated when we only have one of the images
 */

int create_pixmap(char *name, int number)
{
	int error,tmp,w,h,face;

	if (number>=allocated_pixmaps) {
		pixmaps = realloc(pixmaps, sizeof(Pixmap) * (number+1));
		allocated_pixmaps = number;
	}
	face = FindFace(xbm_names[number],-1);
	/* Perhaps we have a duplicate. */
	if (face!=-1 && face!=number) {
		pixmaps[number] = pixmaps[face];
		return 0;
	}
	if ((error=XReadBitmapFile(display, win_root,name, &w,&h,
	    &pixmaps[number], &tmp, &tmp))!=0) {
		fprintf(stderr,"XReadBitmapFile returned %d (file=%s)\n",
			error, name);
		pixmaps[number] = pixmaps[blank_face.number];
		return 1;
	    }
	return 0;
}

/* This operates pretty much the same way as create_pixmap above */

int create_xpm(char *name, int number)
{
#ifdef Xpm_Pix
	int error,face;
	char buf[MAX_BUF];

	if (number>=allocated_pixmaps) {
		pixmaps = realloc(pixmaps, sizeof(Pixmap *) * (number+1));
		masks = realloc(pixmaps, sizeof(Pixmap *) * (number+1));
		allocated_pixmaps = number;
	}
	face = FindFace(xbm_names[number],-1);
	/* Perhaps we have a duplicate. */
	if (face!=-1 && face!=number) {
		pixmaps[number] = pixmaps[face];
		masks[number] = masks[face];
		return 0;
	}
	strcpy(buf,name);
	strcat(buf,".xpm");
	if ((error=XpmReadFileToPixmap(display, win_root,buf, 
	    &pixmaps[number], &masks[number], NULL))!=0) {
		fprintf(stderr,"XpmReadFileToPixmap returned %d (file=%s)\n",
			error, name);
		pixmaps[number] = pixmaps[blank_face.number];
		masks[number] = masks[blank_face.number];
		return 1;
	    }
	return 0;
#endif
}

/* This loads the additional images that have been downloaded or
 * are not part of the standard image set.
 * Should only be called if the images might exist (client_libdir
 * actually exists)
 */
void load_additional_images()
{
	int	count,error;
	char	fname[MAX_BUF];

	if (display_mode==Font_Display) {
	    fprintf(stderr,"Can not make additional images - in font mode\n");
	    return;
	}
	for (count=basenrofpixmaps; count<nrofpixmaps; count++) {
	    sprintf("%s/images/%s", client_libdir, xbm_names[count]);
	    if (display_mode==Xpm_Display)
		error=create_xpm(fname,count);
	    else 
		error=create_pixmap(fname,count);
	    /* missing image name */
	    if (error)
		add_missing_image_name(xbm_names[count]);
	}
}

#endif

/******************************************************************************
 *
 * The functions dealing with the info window follow
 *
 *****************************************************************************/

static int get_info_display() {
    XSizeHints infohint;
    int i;

    infohint.x=579;
    infohint.y=0;
    infohint.width=6+INFOCHARS*FONTWIDTH;
    infohint.height=11+maxinfolines*13;
    infohint.min_width=100;
    infohint.min_height=30;
    infohint.flags=PPosition | PSize;
    win_info=XCreateSimpleWindow(display, win_root,
	infohint.x,infohint.y,infohint.width,infohint.height,2,
	foreground,background);
    XSetWindowColormap(display, win_info, colormap);
    icon=XCreateBitmapFromData(display,win_info,
	(_Xconst char *) crossfire_bits,
	(unsigned int) crossfire_width, (unsigned int)crossfire_height);
    XSetStandardProperties(display,win_info,"Crossfire - text","Crosstext",
	icon,gargv,gargc,&(infohint));
    gc_info=XCreateGC(display,win_info,0,0);
    XSetForeground(display,gc_info,foreground);
    XSetBackground(display,gc_info,background);
    font=XLoadFont(display,font_text_info);
    XSetFont(display,gc_info,font);
    XSelectInput(display,win_info,
	ButtonPressMask|KeyPressMask|KeyReleaseMask|ExposureMask|
	StructureNotifyMask);
    XMapRaised(display,win_info);
    info_chars = (infohint.width/FONTWIDTH)-1;
    info=(char **) malloc(sizeof(char *) * maxinfolines);
    for (i=0; i<maxinfolines; i++) {
	info[i] = malloc(sizeof(char)* (info_chars+1));
	info[i][0]='\0';
    }

    return 0;
}

void delete_ch() {
    if(strlen(cpl.input_text)==0)
	return;
    cpl.input_text[strlen(cpl.input_text)-1] = '\0';

    info[infopos][strlen(info[infopos])-1]='\0';
    XDrawImageString(display,win_info,gc_info,
	(strlen(info[infopos])+1)*FONTWIDTH,(infoline+1)*FONTHEIGHT," ",1);
}

/* Writes one character to the screen.  Used when player is typing
 * stuff we that we want to appear, or used to give prompts.
 */

void write_ch(char key)
{
  char c2[2];

    if ((key < 32 || (unsigned char) key > 127) && key != 8)
	return;
    c2[0] = key;
    c2[1] ='\0';

    if(key==8||key==127) {
	/* By backspacking enough, let them get out of command mode */
	if (cpl.input_text[0]=='\0' && cpl.input_state==Command_Mode) {
	    cpl.input_state=Playing;
	    /* Erase the prompt */
	    XDrawImageString(display,win_info,gc_info,
		FONTWIDTH,(infoline+1)*FONTHEIGHT," ",1);
	}
	delete_ch();
	return;
    }
    if(strlen(cpl.input_text)>=info_chars)
	return;

    strcat(cpl.input_text,c2);
    strcat(info[infopos],(cpl.no_echo? "?": c2));

    XDrawImageString(display,win_info,gc_info,
	FONTWIDTH,(infoline+1)*FONTHEIGHT,info[infopos],strlen(info[infopos]));
}



/* This is similar to draw_info below, but doesn't advance to a new
 * line.  Generally, queries use this function to draw the prompt for
 * the name, password, etc.
 */

void draw_prompt(const char *str)
{
  strncpy(info[infopos],str,info_chars);
  info[infopos][info_chars] = '\0';
  XDrawImageString(display,win_info,
    gc_info,FONTWIDTH,(infoline+1)*FONTHEIGHT,
    info[infopos], strlen(info[infopos]));
}

void draw_info(const char *str) {
  static char blanks[]="                                                      ";

  char *cp;

  if(str == (char *) NULL) {
    draw_info("[NULL]");
    return;
  }

  /* Don't have to worry about newlines in the string - there can't
   * be any, as that would be an end of packet mark
   *
   * Not true with the eutl package - there can be new lines, so
   * we better handle them.
   */

  if((cp=strchr(str,'\n'))!=NULL) {
    /* 4096 is probably way overkill, but 1024 could very well be too small.
     * And I don't see the need to malloc and then free this either -
     * this is a single user program.
     */
    char obuf[4096],*buf = obuf;

    strcpy(buf,str);
    do {
      if ((cp = strchr(buf, '\n'))) {
	*cp='\0';
	draw_info(buf);
	buf = cp +1;
      } else
	draw_info(buf);
    } while (cp!=NULL);
    return;
  }

  /* Lets do the word wrap for messages - MSW (master@rahul.net) */
  if ((int)strlen(str) >= info_chars) {
	int i=info_chars-1;

	/* i=last space (or ')' for armor.  Wrap armor, because
	otherwise, the two sets of ()() can be about half the line */
	while ((str[--i]!=' ') && (str[i]!=')') && (i!=0)) ;
	/* if i==0, string has no space.  Just let it be truncated */
	if (i!=0) {
	    char *buf = (char *)malloc(sizeof(char)*(i+2));
	    int j;

	    i++;	/* want to keep the ')'.  This also keeps
			the space, but that really doesn't matter */
	    strncpy(buf, str, i);
	    buf[i]='\0';
	    draw_info(buf);
	    free(buf);

	    for (j=i; j < (int)strlen(str); j++) /* if the wrap portion is */
		if (str[j]!=' ') break;		/* only space, don't wrap it*/
	    if ((((strlen(str)-i)!=1) || (str[i]!='.')) && (j!=strlen(str)))
		draw_info((str+i));
	    return;
	}
  }
  strncpy(info[infopos],str,info_chars);
  info[infopos][info_chars] = '\0';
  XDrawImageString(display,win_info,
    gc_info,FONTWIDTH,(infoline+1)*FONTHEIGHT,
    info[infopos], strlen(info[infopos]));

  infopos = (infopos+1)% maxinfolines ;

  if(++(infoline)>=maxinfolines){
    if (scroll_info_window) {
	XCopyArea(display,win_info,win_info,
	    gc_info,0,FONTHEIGHT,info_chars*FONTWIDTH,
	    maxinfolines*FONTHEIGHT,0,0);
	infoline--;
    }
    else
 	infoline=0;
    info_full=1;
  }
  strncpy(info[infopos],blanks,info_chars);
  info[infopos][info_chars] = '\0';
  XDrawImageString(display,win_info,
	gc_info,FONTWIDTH,(infoline+1)*FONTHEIGHT,blanks,strlen(blanks));
  if (cpl.input_state==Reply_Many) {
    strncpy(info[infopos], cpl.input_text, info_chars);
    info[infopos][info_chars] = '\0';
     
    XDrawImageString(display,win_info,gc_info,
	FONTWIDTH,(infoline+1)*FONTHEIGHT,
	info[infopos], strlen(info[infopos]));
  }
  if ((int)strlen(str) > (int)info_chars)
    draw_info(str+info_chars);
}

/* This is pretty much print_message, but the name is changed, and some
 * unnecessary code has been removed.
 */

void draw_color_info(int colr, const char *buf){
  if (iscolor){
    XSetForeground(display,gc_info,discolor[colr].pixel);
    draw_info(buf);
    XSetForeground(display,gc_info,discolor[0].pixel);
  }
  else{
    draw_info("==========================================");
    draw_info(buf);
    draw_info("==========================================");
  }
}


/*
 * draw_all_info is only needed for redraws
 */

static void draw_all_info() {
  int i;

  XClearWindow(display,win_info);
  if ((scroll_info_window) && (info_full)) {
     int j;
     for (i=0; i<maxinfolines; i++) {
	j = (infopos+i + 1) % maxinfolines;
	XDrawImageString(display,win_info,
	    gc_info,FONTWIDTH,(i+1)*FONTHEIGHT,
	    info[j], strlen(info[j]));
     }
  }
  else 
      for(i=0;i<maxinfolines;i++)
	XDrawImageString(display,win_info,
	    gc_info,FONTWIDTH,(i+1)*FONTHEIGHT,
	    info[i],strlen(info[i]));
}


static void resize_win_info(XEvent *event) {
  int chars=(event->xconfigure.width/FONTWIDTH)-1;
  int lines=(event->xconfigure.height/FONTHEIGHT);
  int i;
  char **new_info;

  if(chars==info_chars && lines==maxinfolines)
    return;

  if(chars<3 || lines<3)
    return;

  new_info=(char **) malloc(sizeof(char *) * lines);

  /* If the new window is smaller, we want to lose the oldest lines
   * in the window.  This is not necessarily the same thing as the
   * bottom lines of the display.
   * If the buffer hasn't filled up yet (infofull==0), then
   * the oldest lines are those starting at 0.
   * If in wrap mode, the oldest lines are those after infoline.  If
   * in scroll mode, the oldest are those after infopos.
   * I had to re-write this function pretty majorly to do this.
   * Mark S. Wedel (master@rahul.net)
   */
  if (lines<maxinfolines) {
	int diff = maxinfolines - lines,start;

	if (!info_full) {
		if (infoline>lines) start=infoline-lines;
		else start=0;
	}
	else if (!scroll_info_window) start = infoline + diff;
	else start = infopos + diff;

	for (i=0; i<lines; i++) {
	    new_info[i]= (char *) malloc(sizeof(char) * (chars+1));
	    strncpy(new_info[i], info[(i+start) % maxinfolines],chars);
	    new_info[i][chars]='\0';
	}
	if (infoline>=lines) infoline = lines-1;
	if (infopos>=lines) infopos = lines-1;
	info_full = 0;

  }
  else {
  /* If the window is getting bigger, things are a little simpler
   * than above.  However, for the data in scroll mode to make
   * sense, we do need to re-copy it in a make sense way.
   * MSW (master@cats.ucsc.edu)
   */
    int start;

    if (!info_full) start=0;
    else {
	if (scroll_info_window)
		start = infopos;
	else start = infoline;
	/* the way we copy the lines makes it so the oldest is
 	 * line 0, and the newest is the last line.  so update
	 * infoline accordingly
	 */
	infoline = lines-1; 
	infopos = lines - 1;
   }

	for (i=0; i<maxinfolines; i++) {
	    new_info[i]= (char *) malloc(sizeof(char) * (chars+1));
	    strncpy(new_info[i], info[(i+start) % maxinfolines], chars);
	    new_info[i][chars]='\0';
	}
	for (i=maxinfolines; i<lines; i++) {
	    new_info[i]= (char *) malloc(sizeof(char) * (chars+1));
	    new_info[i][0]='\0';
	}
	info_full = 0;
  }
  for(i=0;i<maxinfolines;i++)
    free(info[i]);
  free(info);

  info=new_info;
  info_chars=chars;
  maxinfolines=lines;
  draw_all_info();
}




/***********************************************************************
 *
 * Stats window functions follow
 *
 ***********************************************************************/

static int get_stats_display() {
    XSizeHints stathint;

    stathint.x=303;
    stathint.y=0;
    stathint.width=273;
    stathint.height=100;
    stathint.min_width=stathint.max_width=stathint.width;
    stathint.min_height=stathint.max_height=stathint.height;
    stathint.flags=PPosition | PSize;
    win_stats=XCreateSimpleWindow(display,win_root,
	stathint.x,stathint.y,stathint.width,stathint.height,2,
	foreground,background);
    XSetWindowColormap(display, win_stats, colormap);
    icon=XCreateBitmapFromData(display,win_stats,
	(_Xconst char *) crossfire_bits,
	(unsigned int) crossfire_width, (unsigned int)crossfire_height);
    XSetStandardProperties(display,win_stats,"Crossfire - status",
	"crosstatus",icon,gargv,gargc, &(stathint));

    gc_stats=XCreateGC(display,win_stats,0,0);
    XSetForeground(display,gc_stats,foreground);
    XSetBackground(display,gc_stats,background);
    font=XLoadFont(display,font_text_stats);
    XSetFont(display,gc_stats,font);
    XSelectInput(display,win_stats,KeyPressMask|KeyReleaseMask|ExposureMask);
    XMapRaised(display,win_stats);
   return 0;
}

/* This draws the stats window.  If redraw is true, it means
 * we need to redraw the entire thing, and not just do an
 * updated.
 */

void draw_stats(int redraw) {
  char buff[MAX_BUF];
  static Stats last_stats = {0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
  static char last_name[MAX_BUF]="", last_range[MAX_BUF]="";
#if 0
  static uint8 last_spell;
  static rangetype last_shoottype;
#endif
    if (strcmp(cpl.title, last_name) || redraw) {
	strcpy(last_name,cpl.title);
	strcpy(buff,cpl.title);
 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,10, buff,strlen(buff));
    }

    if(redraw || cpl.stats.exp!=last_stats.exp ||
      cpl.stats.level!=last_stats.level) {
	sprintf(buff,"Score: %5d  Level: %d",cpl.stats.exp,
	    cpl.stats.level);
 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,24, buff,strlen(buff));

	last_stats.exp = cpl.stats.exp;
	last_stats.level = cpl.stats.level;
      }

    if(redraw || 
      cpl.stats.hp!=last_stats.hp || cpl.stats.maxhp!=last_stats.maxhp ||
      cpl.stats.sp!=last_stats.sp || cpl.stats.maxsp!=last_stats.maxsp) {

	sprintf(buff,"Hp: %d/%d  Sp: %d/%d",
	    cpl.stats.hp, cpl.stats.maxhp,
	    cpl.stats.sp, cpl.stats.maxsp);

 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,38, buff,strlen(buff));
	last_stats.hp=cpl.stats.hp;
	last_stats.maxhp=cpl.stats.maxhp;
	last_stats.sp=cpl.stats.sp;
	last_stats.maxsp=cpl.stats.maxsp;
    }

    if(redraw || cpl.stats.Dex!=last_stats.Dex ||
      cpl.stats.Con!=last_stats.Con || cpl.stats.Str!=last_stats.Str ||
      cpl.stats.Int!=last_stats.Int || cpl.stats.Wis!=last_stats.Wis ||
      cpl.stats.Cha!=last_stats.Cha) {

	sprintf(buff,"St%2d Dx%2d Co%2d In%2d Wi%2d Ch%2d",
	    cpl.stats.Str,cpl.stats.Dex,cpl.stats.Con,
	    cpl.stats.Int,cpl.stats.Wis,cpl.stats.Cha);

 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,52, buff,strlen(buff));

	last_stats.Str=cpl.stats.Str;
	last_stats.Con=cpl.stats.Con;
	last_stats.Dex=cpl.stats.Dex;
	last_stats.Int=cpl.stats.Int;
	last_stats.Wis=cpl.stats.Wis;
	last_stats.Cha=cpl.stats.Cha;
      }

    if(redraw || cpl.stats.wc!=last_stats.wc ||
      cpl.stats.ac!=last_stats.ac ||
      cpl.stats.armor!=last_stats.armor ||
      cpl.stats.dam!=last_stats.dam) {

	sprintf(buff,"Wc:%3d Dam:%3d Ac:%3d Arm:%3d",
	    cpl.stats.wc,cpl.stats.dam,cpl.stats.ac,
	    cpl.stats.armor);
 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,66, buff,strlen(buff));

	last_stats.wc=cpl.stats.wc;
	last_stats.ac=cpl.stats.ac;
	last_stats.dam=cpl.stats.dam;
	last_stats.armor = cpl.stats.armor;
      }

  if(redraw || last_stats.speed!=cpl.stats.speed ||
     cpl.stats.food!=last_stats.food ||
     cpl.stats.weapon_sp != last_stats.weapon_sp) {
	/* since both speed and weapon speed have been multiplied by
	 * the same value, to get proper weapon, we only need to divide
	 * by speed - the multiplication/division factor on both factors
	 * out.
	 */
	float weap_sp = (float) cpl.stats.weapon_sp /
		((float)cpl.stats.speed);

	if(cpl.stats.food<100 && (cpl.stats.food&4))
	    sprintf(buff,"Speed: %3.2f (%1.2f) Food: *%d* HUNGRY!",
		(float)cpl.stats.speed/FLOAT_MULTF,
		weap_sp,cpl.stats.food);
	else
	    sprintf(buff,"Speed: %3.2f (%1.2f)  Food: %3d",
		(float)cpl.stats.speed/FLOAT_MULTF,
		weap_sp, cpl.stats.food);

 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,80, buff,strlen(buff));

	last_stats.food=cpl.stats.food;
	last_stats.speed = cpl.stats.speed;
	last_stats.weapon_sp = cpl.stats.weapon_sp;
     }
    if (redraw || strcmp(cpl.range, last_range)) {
	strcpy(last_range, cpl.range);
	strcpy(buff,cpl.range);
 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,94, buff,strlen(buff));
    }
#if 0
    if (redraw || last_shoottype!=cpl.shoottype ||
      (cpl.shoottype==range_magic && cpl.ready_spell!=last_spell)) {

	switch(cpl.shoottype) {
	    case range_none:
		strcpy(buff,"Range: nothing");
		break;

	    case range_bow: {
		char buf2[MAX_BUF];

		if (cpl.ranges[range_bow]==NULL) break;
		strcpy(buf2, cpl.ranges[range_bow]->name);
		if (strcmp(buf2 + strlen(buf2)-10, " (readied)") == 0)
		    buf2[strlen(buf2) - 10] = 0;
	        sprintf (buff, "Range: %s", buf2);
		break;
	    }
	case range_magic:
	    sprintf(buff,"Range: spell (%s)", cpl.spells[cpl.ready_spell]);
	    break;

	case range_wand:
	case range_rod:
	case range_horn:
	    sprintf(buff,"Range: %s",
		cpl.ranges[cpl.shoottype]?
		cpl.ranges[cpl.shoottype]->name:"null");
	    break;

	case range_scroll:
	    sprintf(buff,"Range: golem (%s)",
		cpl.ranges[cpl.shoottype]?
		cpl.ranges[cpl.shoottype]->name:"null");
	    break;

	    case range_steal:
		strcpy(buff, "Range: steal");
		break;

	    default:
		strcpy(buff,"Range: illegal");
	}
	last_spell= cpl.ready_spell;
	last_shoottype=cpl.shoottype;

 	strcat(buff,"                     ");
	XDrawImageString(display,win_stats,
	    gc_stats,10,94, buff,strlen(buff));
      }
#endif
}

/***********************************************************************
*
* Handles the message window
*
***********************************************************************/

static int get_message_display() {
    XSizeHints messagehint;

    messagehint.x=303;
    messagehint.y=378;
    messagehint.width=273;
    messagehint.height=101;
    messagehint.max_width=messagehint.min_width=messagehint.width;
    messagehint.max_height=messagehint.min_height=messagehint.height;
    messagehint.flags=PPosition | PSize;
    win_message=XCreateSimpleWindow(display,win_root,
	messagehint.x,messagehint.y,messagehint.width,
	messagehint.height,2,foreground,background);
    XSetWindowColormap(display, win_message, colormap);
    icon=XCreateBitmapFromData(display,win_message,
	(_Xconst char *) crossfire_bits,
	(unsigned int) crossfire_width, (unsigned int)crossfire_height);
    XSetStandardProperties(display,win_message,"Crossfire - vitals",
	"crossvitals",icon, gargv,gargc,&(messagehint));
    gc_message=XCreateGC(display,win_message,0,0);
    XSetForeground(display,gc_message,foreground);
    XSetBackground(display,gc_message,background);
    font=XLoadFont(display,font_text_info);
    XSetFont(display,gc_message,font);
    XSelectInput(display,win_message,
	       ButtonPressMask|KeyPressMask|KeyReleaseMask|ExposureMask);
    XMapRaised(display,win_message);
   return 0;
}

static void xwritedown(char *txt,int x) {
  int y=13;
  for(;*txt!='\0';txt++,y+=13)
    XDrawImageString(display,win_message,
		     look_list.gc_text,x,y,txt,1);
}

#define MAX_BARS_MESSAGE 80

static void draw_stat_bar(int bar_pos, int height, int is_alert)
{
  if(height!=MAX_BARS_MESSAGE)	/* clear the top of the bar */
    XClearArea(display,win_message, bar_pos, 4,
	10, MAX_BARS_MESSAGE-height, 0);

  if(height==0)			/* empty bar */
    return;

  if(is_alert && iscolor) /* this should have its own gc */
	XSetForeground(display,look_list.gc_text,
			discolor[3].pixel);

  XFillRectangle(display,win_message,
	 look_list.gc_text, bar_pos, 4+MAX_BARS_MESSAGE-height, 10, height);

  if(is_alert && iscolor)
	 XSetForeground(display,look_list.gc_text,
			foreground);
}


/* This updates the status bars.  If redraw, then redraw them
 * even if they have not changed
 */

void draw_message_window(int redraw) {
  int bar,is_alert;

  static uint16 scrollsize_hp=0, scrollsize_sp=0, scrollsize_food=0;
  static uint8 scrollhp_alert=FALSE, scrollsp_alert=FALSE,
	scrollfood_alert=FALSE;

  /* draw hp bar */
  if(cpl.stats.maxhp>0)
    {
      bar=(cpl.stats.hp*MAX_BARS_MESSAGE)/cpl.stats.maxhp;
      if(bar<0)
	bar=0;
      is_alert=(cpl.stats.hp <= cpl.stats.maxhp/4);
    }
  else
    {
      bar=0;
      is_alert=0;
    }
  if (redraw || scrollsize_hp!=bar || scrollhp_alert!=is_alert)
    draw_stat_bar(40, bar, is_alert);
  scrollsize_hp=bar;
  scrollhp_alert=is_alert;

  /* draw sp bar.  spellpoints can go above max
   * spellpoints via supercharging with the transferrance spell,
   * or taking off items that raise max spellpoints.
   */
  if (cpl.stats.sp>cpl.stats.maxsp)
	bar = MAX_BARS_MESSAGE;
  else
	bar=(cpl.stats.sp*MAX_BARS_MESSAGE)/cpl.stats.maxsp;
  if(bar<0) 
    bar=0;

  is_alert=(cpl.stats.sp <= cpl.stats.maxsp/4);

  if (redraw || scrollsize_sp!=bar || scrollsp_alert!=is_alert)
    draw_stat_bar(80, bar, is_alert);

  scrollsize_sp=bar;
  scrollsp_alert=is_alert;
  
  /* draw food bar */
  bar=(cpl.stats.food*MAX_BARS_MESSAGE)/999;
  if(bar<0) 
    bar=0;
  is_alert=(cpl.stats.food <= 999/4);

  if (redraw || scrollsize_food!=bar || scrollfood_alert!=is_alert)
    draw_stat_bar(120, bar, is_alert);

  scrollsize_food=bar;
  scrollfood_alert=is_alert;
}

static void draw_all_message() {

  XClearWindow(display,win_message);
  xwritedown("HP",26);
  XDrawRectangle(display,win_message,
		 look_list.gc_text,38,2,14,MAX_BARS_MESSAGE+4);
  xwritedown("SP",66);
  XDrawRectangle(display,win_message,
		 look_list.gc_text,78,2,14,MAX_BARS_MESSAGE+4);
  xwritedown("Food",106);
  XDrawRectangle(display,win_message,
		 look_list.gc_text,118,2,14,MAX_BARS_MESSAGE+4);
  draw_message_window(1);
}

/****************************************************************************
 *
 * Inventory window functions follow
 *
 ****************************************************************************/

#if 0
#define draw_face(win,gc,x,y,face) \
    XSetClipMask(display, gc, pixmaps[face].mask), \
    XSetClipOrigin(display, gc, x, y), \
    XCopyArea(display, pixmaps[face].pixmap, win, gc, 0, 0, 24, 24, x, y);
#endif

#define draw_status_icon(l,x,y,face) \
do { \
    XClearArea(display, l->win, x, y, 24, 6, False); \
    if (face) { \
	XSetClipMask(display, l->gc_status, icons[face].mask), \
	XSetClipOrigin(display, l->gc_status, x, y), \
	XCopyArea(display, icons[face].pixmap, l->win, l->gc_status, \
		  0, 0, 24, 6, x, y); \
    } \
} while (0)

/* This will need to be changed to do the 'right thing' for different display
 * modes (use bitmap data if we are using bitmaps for the game.
 */

static void create_status_icons ()
{
#include "pixmaps/locked.xpm"
#include "pixmaps/unpaid.xpm"
#include "pixmaps/applied.xpm"
#include "pixmaps/magic.xpm"
#include "pixmaps/damned.xpm"
#include "pixmaps/cursed.xpm"
#include "pixmaps/close.xpm"

    if (XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				locked_xpm, &(icons[locked_icon].pixmap), 
				&(icons[locked_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				applied_xpm, &(icons[applied_icon].pixmap), 
				&(icons[applied_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				unpaid_xpm, &(icons[unpaid_icon].pixmap), 
				&(icons[unpaid_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				magic_xpm, &(icons[magic_icon].pixmap), 
				&(icons[magic_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				damned_xpm, &(icons[damned_icon].pixmap), 
				&(icons[damned_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				cursed_xpm, &(icons[cursed_icon].pixmap), 
				&(icons[cursed_icon].mask), NULL)
	|| XpmCreatePixmapFromData(display, RootWindow(display, screen_num), 
				close_xpm, &(icons[close_icon].pixmap), 
				&(icons[close_icon].mask), NULL)) {
	fprintf(stderr, "Unable to create icon pixmaps.\n");
	exit (0);
    }
}

/*
 *  draw_list redraws changed item to the window and updates a scrollbar
 */
static void draw_list (itemlist *l)
{
    int i, items, size, pos;
    item *tmp;
    char buf[MAX_BUF], buf2[MAX_BUF];

    /* draw title */
    if(l->env->weight < 0 || l->show_weight == 0)
	sprintf (buf, l->format_n, l->title);
    else
	sprintf (buf, l->format_nw, l->title, l->env->weight);

    if (strcmp (buf, l->old_title)) {
	XClearArea(display, l->win, 2, 2, 24, 13, False);
	if (l->env->open) {
	    XSetClipMask(display, l->gc_status, icons[close_icon].mask);
	    XSetClipOrigin(display, l->gc_status, 2, 2);
	    XCopyArea(display, icons[close_icon].pixmap, l->win, l->gc_status,
		      0, 0, 24, 13, 2, 2); \
	}
	strcpy (l->old_title, buf);
	XDrawImageString(display, l->win, l->gc_text, (l->show_icon ? 56 : 32),
			 13, buf, strlen(buf));
    }

    /* draw items */
    for(tmp = l->env->inv, items=0; tmp ;tmp=tmp->next)
	items++;

    if(l->item_pos > items - l->size)
	l->item_pos = items - l->size;
    if(l->item_pos < 0)
	l->item_pos = 0;

    for(tmp = l->env->inv, i=l->item_pos; tmp && i; tmp=tmp->next, i--)
	;

    for(i=0; tmp && i < l->size; tmp=tmp->next, i++) {
	/* draw face */
	if(l->faces[i] != tmp->face) {
	    l->faces[i] = tmp->face;
	    XClearArea(display, l->win, 4, 16 + 24 * i, 24, 24, False);
	    gen_draw_face (l->win, tmp->face,4, 16 + 24 * i);
	}
	/* draw status icon */
	if (l->show_icon) {
	    sint8 tmp_icon;
	    tmp_icon = tmp->locked ? locked_icon : 0;
	    if (l->icon1[i] != tmp_icon) {
		l->icon1[i] = tmp_icon;
		draw_status_icon (l, 28, 16 + 24 * i, tmp_icon);
	    }
	    tmp_icon = tmp->applied ? applied_icon :
		       tmp->unpaid  ? unpaid_icon : 0;
	    if (l->icon2[i] != tmp_icon) {
		l->icon2[i] = tmp_icon;
		draw_status_icon (l, 28, 22 + 24 * i, tmp_icon);
	    }
	    tmp_icon = tmp->magical ? magic_icon : 0;
	    if (l->icon3[i] != tmp_icon) {
		l->icon3[i] = tmp_icon;
		draw_status_icon (l, 28, 28 + 24 * i, tmp_icon);
	    }
	    tmp_icon = tmp->damned ? damned_icon : 
		       tmp->cursed ? cursed_icon : 0;
	    if (l->icon4[i] != tmp_icon) {
		l->icon4[i] = tmp_icon;
		draw_status_icon (l, 28, 34 + 24 * i, tmp_icon);
	    }
	}
	/* draw name */
	strcpy (buf2, tmp->name);
	if (l->show_icon == 0)
	    strcat (buf2, tmp->flags);

	if(tmp->weight < 0 || l->show_weight == 0)
	    sprintf(buf, l->format_n, buf2);
	else
	    sprintf(buf, l->format_nw, buf2, tmp->weight);

	if(!l->names[i] || strcmp(buf, l->names[i])) {
	    copy_name (l->names[i], buf);
	    XDrawImageString(display, l->win, l->gc_text, (l->show_icon?56:32),
			     34 + 24 * i, buf, strlen(buf));
	}
    }

    /* If there are not enough items to fill in the display area,
     * then set the unused ares to nothing.
     */
    if(items < l->item_used) {
	XClearArea(display, l->win, 0, 16 + 24 * i, l->width - 23,
		   24 * (l->size - i) , False);
	while (i < l->item_used) {
	    copy_name (l->names[i], "");
	    l->faces[i] = 0;
	    l->icon1[i] = l->icon2[i] = l->icon3[i] = l->icon4[i] = 0;
	    i++;
	}
    }
    l->item_used = items > l->size ? l->size : items;

    /* draw the scrollbar now */
    if(items < l->size)
	items = l->size;

    size = (long) l->bar_length * l->size / items;
    pos = (long) l->bar_length * l->item_pos / items;

    if(l->bar_size != size || pos != l->bar_pos) {

	XClearArea(display, l->win, l->width-20, 17 + l->bar_pos, 16, 
		   l->bar_size, False);
	l->bar_size = size;
	l->bar_pos  = pos;

	XFillRectangle(display, l->win, l->gc_text, l->width - 20,
		       17 + l->bar_pos, 16, l->bar_size);
    }
}

/*
 * draw_all_list clears a window and after that draws all objects 
 * and a scrollbar
 */
static void draw_all_list(itemlist *l)
{
    int i;

    strcpy (l->old_title, "");

    for(i=0; i<l->size; i++) {
	copy_name(l->names[i], "");
	l->faces[i] = 0;
	l->icon1[i] = 0;
	l->icon2[i] = 0;
	l->icon3[i] = 0;
	l->icon4[i] = 0;
    }

    XClearWindow(display, l->win);
    XDrawRectangle(display, l->win, l->gc_text, l->width - 22, 15, 20,
		   l->bar_length + 4);
    l->item_pos = 0;
    l->item_used = 0;
    l->bar_size = 1;    /* so scroll bar is drawn */
    draw_list (l);
}

void open_container (item *op) 
{
    look_list.env = op;
    sprintf (look_list.title, "%s:", op->name);
    draw_list (&look_list);
}

void close_container (item *op) 
{
    look_list.env = cpl.below;
    strcpy (look_list.title, "You see:");
    draw_list (&look_list);
}

static void resize_list_info(itemlist *l, int w, int h)
{
    int i;

    if (l->faces) 
	free(l->faces);
    if (l->icon1) 
	free(l->icon1);
    if (l->icon2) 
	free(l->icon2);
    if (l->icon3) 
	free(l->icon3);
    if (l->icon4) 
	free(l->icon4);
    if (l->names) {
	for (i=0; i < l->size; i++)
	    if (l->names[i])
		free(l->names[i]);
	free(l->names);
    }
    l->width  = w;
    l->height = h;
    l->size = (l->height - FONTHEIGHT - 8) / 24;
    l->text_len = (l->width - (l->show_icon ? 84 : 60)) / FONTWIDTH;
    l->bar_length = l->size * 24;
    sprintf (l->format_nw, "%%-%d.%ds%%6.1f", l->text_len-6, l->text_len-6);
    sprintf (l->format_n, "%%-%d.%ds", l->text_len, l->text_len);

    if ((l->faces = malloc (sizeof (*(l->faces)) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    if ((l->icon1 = malloc (sizeof (*(l->icon1)) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    if ((l->icon2 = malloc (sizeof (*(l->icon2)) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    if ((l->icon3 = malloc (sizeof (*(l->icon3)) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    if ((l->icon4 = malloc (sizeof (*(l->icon4)) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    if ((l->names = malloc (sizeof (char *) * l->size )) == NULL) {
	printf ("Can't allocate memory.\n");
	exit (0);
    }
    for (i=0; i < l->size; i++) {
	if ((l->names[i] = malloc (NAME_LEN)) == NULL) {
	    printf ("Can't allocate memory.\n");
	    exit (0);
	}
    }
    draw_all_list(l);	/* this also initializes above allocated tables */
}

static void get_list_display(itemlist *l, int x, int y, int w, int h, 
		   char *t, char *s) 
{
    XSizeHints hint;

    l->faces=NULL;
    l->names=NULL;
    hint.x = x;
    hint.y = y;
    hint.width  = w;
    hint.height = h;
    hint.min_width  = 60 + 10 * FONTWIDTH;
    hint.min_height = FONTHEIGHT + 8 + 24 * 2;
    hint.flags = PPosition | PSize;
    l->win = XCreateSimpleWindow(display, win_root, hint.x, hint.y, hint.width,
			       hint.height, 2, foreground, background);
    XSetWindowColormap(display, l->win, colormap);
    icon = XCreateBitmapFromData(display, l->win, 
			       (_Xconst char *) crossfire_bits,
			       (unsigned int) crossfire_width, 
			       (unsigned int)crossfire_height);
    XSetStandardProperties(display, l->win, t, s, icon, gargv, gargc, &(hint));
    l->gc_text = XCreateGC (display, l->win, 0, 0);
    l->gc_icon = XCreateGC (display, l->win, 0, 0);
    l->gc_status = XCreateGC (display, l->win, 0, 0);
    XSetForeground (display, l->gc_text, foreground);
    XSetBackground (display, l->gc_text, background);
    XSetForeground (display, l->gc_icon, foreground);
    XSetBackground (display, l->gc_icon, background);
    XSetForeground (display, l->gc_status, foreground);
    XSetBackground (display, l->gc_status, background);
    XSetGraphicsExposures (display, l->gc_icon, False);
    XSetGraphicsExposures (display, l->gc_status, False);
    font=XLoadFont (display, font_inv_text);
    XSetFont (display, l->gc_text, font);
    XSelectInput (display, l->win, ButtonPressMask|KeyPressMask|KeyReleaseMask|
		ExposureMask|StructureNotifyMask);
    XMapRaised(display,l->win);
    create_status_icons();
    resize_list_info(l, w, h);
}

static int get_inv_display()
{
    inv_list.env = cpl.ob;
    strcpy (inv_list.title, "Inventory:");
    inv_list.show_weight = 1;
    get_list_display ( &inv_list, 0, 0, 300, 310, "Crossfire - inventory",
		      "crossinventory");
    return 0;
}

static int get_look_display() 
{
    look_list.env = cpl.below;
    strcpy (look_list.title, "You see:");
    look_list.show_weight = 1;
    get_list_display ( &look_list, 0, 313, 300, 166, "Crossfire - look",
		    "crosslook");
    return 0;
}

/*
 *  draw_lists() redraws inventory and look windows when necessary
 */
void draw_lists ()
{
    if (inv_list.env->inv_updated) {
	draw_list (&inv_list);
	inv_list.env->inv_updated = 0;
    }
    if (look_list.env->inv_updated) {
	draw_list (&look_list);
	look_list.env->inv_updated = 0;
    }
}

/*
 *  Prints players inventory, contain extra information for debugging purposes
 */
void print_inventory (item *op)
{
    char buf[MAX_BUF];
    char buf2[MAX_BUF];
    item *tmp;
    static int l = 0;

    if (l == 0) {
	sprintf (buf, "%s's inventory (%d):", op->name, op->tag);
	sprintf (buf2, "%-*s%6.1f kg", info_chars - 10, buf, op->weight);
	draw_info (buf2);
    }

    l += 2;
    for (tmp = op->inv; tmp; tmp=tmp->next) {
	sprintf (buf, "%*s- %d %s%s (%d)", l - 2, "", tmp->nrof, tmp->name,
		 tmp->flags, tmp->tag);
	sprintf (buf2, "%-*s%6.1f kg", info_chars - 8 - l, buf, tmp->weight);
	draw_info (buf2);
	if (tmp->inv)
	    print_inventory (tmp);
    }
    l -= 2;
}

void set_show_icon (char *s)
{
    if (s == NULL || *s == 0 || strncmp ("inventory", s, strlen(s)) == 0) {
	inv_list.show_icon = ! inv_list.show_icon; /* toggle */
	resize_list_info(&inv_list, inv_list.width, inv_list.height);
    } else if (strncmp ("look", s, strlen(s)) == 0) {
	look_list.show_icon = ! look_list.show_icon; /* toggle */
	resize_list_info(&look_list, look_list.width, look_list.height);
    }
}

void set_show_weight (char *s)
{
    if (s == NULL || *s == 0 || strncmp ("inventory", s, strlen(s)) == 0) {
	inv_list.show_weight = ! inv_list.show_weight; /* toggle */
	draw_list (&inv_list);
    } else if (strncmp ("look", s, strlen(s)) == 0) {
	look_list.show_weight = ! look_list.show_weight; /* toggle */
	draw_list (&look_list);
    }
}



#if 0
/* Support in the protocol to have the client clear the info window
 * should probably be added at some time.
 */
void clear_win_info(object *op) {
  int i;

	/* don't clear if in scroll mode MSW (master@rahul.net) */
  if((op->type==PLAYER)  &&  (op->contr->scroll)) return;

  for(i=0;i<op->contr->maxinfolines;i++) {
    (void) memset((void *)op->contr->info[i],' ',op->contr->info_chars);
    op->contr->info[i][op->contr->info_chars]='\0';
  }
  XClearWindow(display,win_info);
/*refresh_win_info(op); */
  op->contr->infoline=0;
  op->contr->infopos=0;
}

#endif



/* Magic mapping code needs to be added.
 * The way I see it, server will send along data of the squares and what
 * color they should be in single character codes.  So if we are mapping a
 * 10x10 area, it would be something like: "1155123620", 1296639403",
 * etc.
 */


/***********************************************************************
 *
 * Here is the start of event handling functions
 *
 ***********************************************************************/

static void parse_game_button_press(int button, int x, int y)
{
    int dx=(x-2)/24-5,dy=(y-2)/24-5,i;

    switch (button) {
	case 1:
	{
	    if(dx<WINLEFT||dx>WINRIGHT||dy<WINUPPER||dy>WINLOWER) return;
	    look_at(dx,dy);
	}
	break;
	case 2:
	case 3:
	    if (x<115)
		i = 0;
	    else if (x>149)
	       i = 6;
	    else i =3;

	    if (y>152)
	      i += 2;
	    else if (y>115)
	      i++;

	    if (button==2) {
		switch (i) {
		      case 0: fire_dir (8);break;
		      case 1: fire_dir (7);break;
		      case 2: fire_dir (6);break;
		      case 3: fire_dir (1);break;
		      case 5: fire_dir (5);break;
		      case 6: fire_dir (2);break;
		      case 7: fire_dir (3);break;
		      case 8: fire_dir (4);break;
		}
		/* Only want to fire once */
		stop_fire();
	    }
	    else switch (i) {
	      case 0: move_player (8);break;
	      case 1: move_player (7);break;
	      case 2: move_player (6);break;
	      case 3: move_player (1);break;
	      case 5: move_player (5);break;
	      case 6: move_player (2);break;
	      case 7: move_player (3);break;
	      case 8: move_player (4);break;
	    }
    }
}


/* Handles key presses.  Note that the client really doesn't know
 * much about states.  But there are at least a couple that the
 * client needs to know about - game play, where it parses keystrokes,
 * applies any bindings, and sends the command to the server,
 * and reply states.  The only time reply's are used right now is for rolling
 * up a character - sending 'SAY' commands is not appropriate.
 *
 * After input is completed in either of the reply states, the program
 * sets the game back to Play state.  May not be appropriate, but let
 * the server ignore the commands if they are not proper - a hack
 * client could always send such commands anyways.
 * If the server expect additional reply's, it should send an query.
 * Note that these can not be stacked up, (ie, server can not send
 * 5 queries and expect 5 replies).   But in no place does the server
 * do this anyways, so it is not a problem.
 */

static void do_key_press()
{
    KeySym gkey;
    char text[10];

    if(!XLookupString(&event.xkey,text,10, &gkey,NULL)) {
/*
 *	This happens quite a bit - most modifier keys (shift, control, etc)
 *	can not be mapped to a value.
 */
 /*
	fprintf(stderr,"XLookupString failed, xkey.code=%d, gkey=%ld\n",
		event.xkey.keycode, gkey);
*/
	text[0]='\0';
    }
    switch(cpl.input_state) {
	case Playing:
	    parse_key(text[0],event.xkey.keycode,gkey);
	    break;

	case Reply_One:
	    text[1]='\0';
	    send_reply(text);
	    cpl.input_state = Playing;
	    break;

	case Configure_Keys:
	    configure_keys(event.xkey.keycode, gkey);
	    break;

	case Reply_Many:
	case Command_Mode:
	    if (text[0]==13) {
		enum Input_State old_state=cpl.input_state;
		if (cpl.input_state==Reply_Many)
		    send_reply(cpl.input_text);
		else
		    extended_command(cpl.input_text);

		/* Only set state to playing if the state has otherwise
		 * not changed - this check is needed because 'bind
		 * changes the state, and we don't want to change to playing
		 * again.
		 */
		if (old_state==cpl.input_state)
		    cpl.input_state = Playing;
		cpl.input_text[0]='\0';
		cpl.no_echo=0;	/* By default, start echoing thing again */
	    }
	    else {
		write_ch(text[0]);
	    }
	    break;

	default:
	    fprintf(stderr,"Unknown input state: %d\n", cpl.input_state);
    }
}


/*
 *  buttonpress_in_list handles mouse button event in list window.
 *  It updates scrollbar or calls right function for item
 *  (apply/examine/move). It's probably better move calling the 
 *  functions to the upper level.
 */

static int buttonpress_in_list (itemlist *l, XButtonEvent *xbutton)
{
    item *tmp;
    int y = xbutton->y - 16, x=xbutton->x, button = xbutton->button;
    int items, pos, dy;

    if (y < 0 && l->env->open) /* close the sack */
	client_send_apply (l->env->tag);
 
    if (y < 0 || y > 24 * l->size)
	return 1;

    if (x > l->width-23) {    /* scrollbar */

	dy = y / 24 > 0 ? y / 24 : 1;
      
	switch(button) {
	  case 1:
	    pos = l->item_pos - dy;
	    break;

	  case 2:
	    for(tmp=l->env->inv, items=0; tmp; tmp=tmp->next)
		items++;
	    pos = y * items / l->bar_length;
	    break;

	  case 3:
	    pos = l->item_pos + dy;
	    break;

	}
	if (pos != l->item_pos) {
	    l->item_pos = pos;
	    draw_list (l);
	}
	return 1;
    }

    pos = l->item_pos + y / 24;
    for(tmp=l->env->inv, items=0; tmp && items < pos; tmp=tmp->next)
	items++;

    if (tmp) {
	switch(button) {
	  case 1:
	    if (xbutton->state == ShiftMask)
		toggle_locked(tmp);
	    else
		client_send_examine (tmp->tag);
	    break;
	  case 2:
	    client_send_apply (tmp->tag);
	    break;
	  case 3:
	    if (tmp->locked) {
		draw_info ("This item is locked.");
	    } else if (l == &inv_list)
		client_send_move (look_list.env->tag, tmp->tag, 0);
	    else
		client_send_move (inv_list.env->tag, tmp->tag, 0);
	    break;
	}
    }
    return 1;
}

/* This function handles the reading of the X Events and then
 * doing the appropriate action.  For most input events, it is calling
 * another function.
 *
 * It can also call display functions to make sure the information is
 * correct - in this way, updates will not be done so often (like
 * for every ITEM command received), but less frequently but still
 * update the display fully.  All the functions above are optimized to
 * only draw stuff that needs drawing.  So calling them a lot is not
 * going to draw stuff too much.
 */

void check_x_events() {
  char text[10];
  KeySym gkey;

  draw_lists();		/* testing if this can work this way */

  while (XPending(display)!=0) {
    XNextEvent(display,&event);
    switch(event.type) {

    case ConfigureNotify:
      if(event.xconfigure.window==win_info)
	resize_win_info(&event);
      else if(event.xconfigure.window==inv_list.win)
	resize_list_info(&inv_list, event.xconfigure.width,
			 event.xconfigure.height);
      else if(event.xconfigure.window==look_list.win)
	resize_list_info(&look_list, event.xconfigure.width,
			 event.xconfigure.height);
      break;

    case Expose:
      /* No point redrawing windows if there are more Expose's to
       * to come.
       */
      if (event.xexpose.count!=0) continue;
      if(event.xexpose.window==win_stats) {
	XClearWindow(display,win_stats);
	draw_stats(1);
      } else if(event.xexpose.window==win_info)
	draw_all_info();
      else if(event.xexpose.window==inv_list.win)
	draw_all_list(&inv_list);
      else if(event.xexpose.window==look_list.win)
	draw_all_list(&look_list);
      else if(event.xexpose.window==win_message)
	draw_all_message();
      else if(event.xexpose.window==win_game)
#if 0
	refresh();
#else
	;
#endif
      else if(split_windows==FALSE && event.xexpose.window==win_root) {
	XClearWindow(display,win_root);
      }
      break;

    case MappingNotify:
      XRefreshKeyboardMapping(&event.xmapping);
      break;


    case ButtonPress:
	if(event.xbutton.window==win_game) {
	  parse_game_button_press(event.xbutton.button,event.xbutton.x,
		event.xbutton.y);

	} else if(event.xbutton.window==inv_list.win) {
		buttonpress_in_list(&inv_list, &event.xbutton);

	} else if(event.xbutton.window==look_list.win) {
	  buttonpress_in_list(&look_list, &event.xbutton);
	}
      break;

    case KeyRelease:
#if 1
	XLookupString(&event.xkey,text,10,&gkey,NULL);
	parse_key_release(event.xkey.keycode, gkey);
#else
	{
	char text[10];
	KeySym keysym;
	if(!XLookupString(&event.xkey,text,10, &keysym,NULL)) {
		text[0]='\0';
	}
	client_send_keyrelease(event.xkey.keycode,keysym,text[0]);
	}
#endif
	break;

    case KeyPress:
#if 1
	do_key_press();
#else
	{
	char text[10];
	KeySym keysym;
	if(!XLookupString(&event.xkey,text,10, &keysym,NULL)) {
		text[0]='\0';
	}
	client_send_keypress(event.xkey.keycode,keysym,text[0]);
	}
#endif
	break;
    }
  }
  /* Since we cycle through all the Xevents, there is no need to do
   * keyboard flushing.  All commands get sent to the server.  There really
   * does need to be some decent way to flush the commands we have sent,
   * but there is no real way to do that.  This is at least somewhat
   * true because the client and server will never be completely 
   * synchronized.
   */
}


/***********************************************************************
 *
 * Here starts the X11 init functions.  It will call other
 * functions that were grouped previously by window
 *
 ***********************************************************************/

/* Usage routine.  All clients should support server, port and
 * display options, with -pix and -xpm also suggested.  -split
 * does not need to be supported - it is in this copy because
 * the old code supported it.
 */

static void usage(char *progname)
{
    puts("Usage of cfclient:\n\n");
    puts("-server <name>   - Connect to <name> instead of localhost.");
    puts("-port <number>   - Use port <number> instead of the standard port number");
    puts("-display <name>  - Use <name> instead if DISPLAY environment variable.\n");
    puts("-split           - Use split windows.");
    puts("-echo            - Echo the bound commands");
    puts("-pix             - Use bitmaps instead of the font.");
#ifdef Xpm_Pix
    puts("-xpm             - Use color pixmaps (XPM) for display.");
#endif
    puts("-showicon        - Print status icons in inventory window");
    puts("-image           - get all images from server at startup");
    puts("-sync            - Synchronize on display");
    puts("-help            - Display this message.");
    exit(0);
}

/* init_windows:  This initiliazes all the windows - it is an
 * interface routine.  The command line arguments are passed to
 * this function to interpert.  Note that it is not in fact
 * required to parse anything, but doing at least -server and
 * -port would be a good idea.
 *
 * This function returns 0 on success, nonzero on failure.
 */

int init_windows(int argc, char **argv)
{
    int on_arg=1;
    char *display_name="";

    for (on_arg=1; on_arg<argc; on_arg++) {
	if (!strcmp(argv[on_arg],"-display")) {
	    if (++on_arg == argc) {
		fprintf(stderr,"-display requires a display name\n");
		return 1;
	    }
	    display_name = argv[on_arg];
	    continue;
	}
	if (strcmp(argv[on_arg],"-sync")==0) {
	    sync_display = 1;
	    continue;
	}
	if (!strcmp(argv[on_arg],"-port")) {
	    if (++on_arg == argc) {
		fprintf(stderr,"-port requires a port number\n");
		return 1;
	    }
	    port_num = atoi(argv[on_arg]);
	    continue;
	}
	if (!strcmp(argv[on_arg],"-server")) {
	    if (++on_arg == argc) {
		fprintf(stderr,"-server requires a host name\n");
		return 1;
	    }
	    server = argv[on_arg];
	    continue;
	}
	if (!strcmp(argv[on_arg],"-xpm"))
#ifdef Xpm_Pix
	    display_mode = Xpm_Display;
#else
	{
	    fprintf(stderr,"Client not configured with Xpm display mode enabled\n");
	    fprintf(stderr,"Ignoring -xpm flag\n");
	}
#endif
	else if (!strcmp(argv[on_arg],"-pix"))
	    display_mode = Pix_Display;
	else if (!strcmp(argv[on_arg],"-split"))
	    split_windows=TRUE;
	else if (!strcmp(argv[on_arg],"-showicon"))
	    inv_list.show_icon = TRUE;
	else if (!strcmp(argv[on_arg],"-image"))
	    init_load_image=TRUE;
	else if (!strcmp(argv[on_arg],"-echo"))
	    cpl.echo_bindings=TRUE;
	else if (!strcmp(argv[on_arg],"-savedata")) {
	    /* Must have someplace to save it if this is set. */
	    if (client_libdir==NULL)
		fprintf(stderr,"No LIBDIR is set for the client, -savedata has no effect.\n");
	    else 
		save_new_data=TRUE;
	}
	else if (!strcmp(argv[on_arg],"-help"))
	    usage(argv[0]);
	else {
	    fprintf(stderr,"Do not understand option %s\n", argv[on_arg]);
	    usage(argv[0]);
	    return 1;
	}
    }
    /* Finished parsing all the command line options.  Now start
     * working on the display.
     */
    if (display_mode==Font_Display)
	init_load_image=FALSE;
    gargc=argc;
    gargv=argv;
    if (get_root_display(display_name) ||
	get_game_display() ||
	get_stats_display() ||
	get_info_display() ||
	get_inv_display() ||
	get_look_display() ||
	get_message_display())
		return 1;

    init_keys();
    return 0;
}


void display_map_clearcell(long x,long y)
{
  the_map.cells[x][y].count = 0;
}

void display_map_addbelow(long x,long y,long face)
{
  the_map.cells[x][y].faces[the_map.cells[x][y].count] = face&0xFFFF;
  the_map.cells[x][y].count ++;
}

void display_mapcell_pixmap(int ax,int ay)
{
  int k;
  XFillRectangle(display,xpm_pixmap,gc_clear_xpm,0,0,24,24);
  
  for(k=the_map.cells[ax][ay].count-1;k>=0;k--) {
    gen_draw_face(xpm_pixmap,the_map.cells[ax][ay].faces[k],
		  0,0);

#if 0
    if (pixmaps[the_map.cells[ax][ay].faces[k]].mask == None) {
	XCopyArea(display, pixmaps[the_map.cells[ax][ay].faces[k]].pixmap,
		xpm_pixmap,gc_floor,
		0,0,24,24,0,0);
    } else {
	int gcnum,i;
	Pixmap mask;
	GC gc;
	total++;
	mask = pixmaps[the_map.cells[ax][ay].faces[k]].mask;
	for(gcnum=0;gcnum<XPMGCS;gcnum++) {
	if (xpm_masks[gcnum] == mask)
	  break;
	}
	if (gcnum == XPMGCS) {
	misses++;
	gcnum--;
	  gc = gc_xpm[gcnum];
	XSetClipMask(display,gc,mask);
	} 
	gc = gc_xpm[gcnum];
	for(i=gcnum-1;i>=0;i--) {
	xpm_masks[i+1] = xpm_masks[i];
	gc_xpm[i+1] = gc_xpm[i];
	}
	xpm_masks[0] = mask;
	gc_xpm[0] = gc;
	XCopyArea(display, pixmaps[the_map.cells[ax][ay].faces[k]].pixmap,
		xpm_pixmap,gc_xpm[0],
		0,0,24,24,0,0);
    }
#endif
  }
  XCopyArea(display,xpm_pixmap,win_game,gc_game,0,0,24,24,2+24*ax,2+24*ay);
}

void display_mapcell_bitmap(int ax,int ay)
{
  gen_draw_face(win_game,the_map.cells[ax][ay].faces[0],
		2+24*ax,2+24*ay);
#if 0
  int face = the_map.cells[ax][ay].faces[0];
  XSetForeground(display,gc_game,discolor[pixmaps[face].fg].pixel);
  XSetBackground(display,gc_game,discolor[pixmaps[face].bg].pixel);
  XCopyPlane(display,pixmaps[face].bitmap,win_game,gc_game,
	     0,0,24,24,2+24*ax,2+24*ay,1);
#endif
}

int display_usebitmaps()
{
  return display_mode == Pix_Display;
}

int display_noimages()
{
  return display_mode == Font_Display;
}

void display_map_doneupdate()
{
  int ax,ay;

  XSetClipMask(display,gc_floor,None);
  for(ax=0;ax<11;ax++) {
    for(ay=0;ay<11;ay++) { 
	if (the_map.cells[ax][ay].count==0) {
	XFillRectangle(display,win_game,gc_blank,2+24*ax,2+24*ay,24,24);
	continue;
	} 
	if (display_mode == Xpm_Display) {
	display_mapcell_pixmap(ax,ay);
	} else if (display_mode == Pix_Display) {
	display_mapcell_bitmap(ax,ay);
	} else {
	fprintf(stderr,"Unknown display mode '%d'\n",display_mode);
	abort();
	}
    }
  }
  XFlush(display);
  /*  printf("misses: %d/%d\n",misses,total);*/
}



void display_mapscroll(int dx,int dy)
{
  int x,y;
  struct Map newmap;

  for(x=0;x<11;x++) {
    for(y=0;y<11;y++) {
      newmap.cells[x][y].count = 0;
      if (x+dx < 0 || x+dx >= 11)
      continue;
      if (y+dy < 0 || y+dy >= 11)
      continue;
      bcopy((char*)&(the_map.cells[x+dx][y+dy]),
	  (char*)&(newmap.cells[x][y]),sizeof(struct MapCell));
    }
  }
  bcopy((char*)&newmap,(char*)&the_map,sizeof(struct Map));
/*  display_map_doneupdate();*/
}

void display_newpixmap(long face,char *buf,long buflen)
{
  FILE *tmpfile;
  char tmpfilename[200];
  Pixmap pixmap, mask;
  XpmAttributes xpm_attr;

  xpm_attr.colormap = colormap;
  xpm_attr.valuemask = XpmColormap;
  sprintf(tmpfilename,"/tmp/xclient.%d",getpid());
  tmpfile = fopen(tmpfilename,"w");
  fprintf(tmpfile,"%s",buf);
  fclose(tmpfile);
		/* was RootWindow(display,screen_num),tmpfilename,*/
  if (XpmReadFileToPixmap(display,win_game,tmpfilename, 
		&pixmap,&mask,&xpm_attr)!= XpmSuccess) {
    fprintf(stderr,"Unable to read %s as pixmap file\n",tmpfilename);
    abort();
  			  }
/*  printf("newpixmap #%ld: (%ld,%ld)\n",face,pixmap,mask);*/
  pixmaps[face].pixmap = pixmap;
  pixmaps[face].mask = mask;
}


void display_newbitmap(long face,long fg,long bg,char *buf)
{
  Pixmap bitmap;

  bitmap = XCreateBitmapFromData(display,
                               RootWindow(display,DefaultScreen(display)),
                               buf,24,24);
  pixmaps[face].bitmap = bitmap;
  pixmaps[face].fg = fg;
  pixmaps[face].bg = bg;
}


void redisplay_stats()
{
  int i;
  for(i=0;i<7;i++) {
    XDrawImageString(display,win_stats,
		   gc_stats,10,i*14+10, stats_buff[i],strlen(stats_buff[i]));
}
  XFlush(display);
}

void display_map_startupdate()
{
}
#if 0
long display_ks2kc(long ks)
{
  return XKeysymToKeycode(display,ks);
}
#endif

#if 0
#define client_draw_face(win,gc,x,y,face) \
{ \
  if (display_mode==Xpm_Display) \
  { \
    XSetClipMask(display, gc, masks[(face)]); \
    XSetClipOrigin(display, gc, x, (y) - 24); \
    XCopyArea(display, pixmaps[(face)], win, gc, 0, 0, 24, 24, \
      (unsigned int) (x), (unsigned int) ((y) - 24)); \
  } \
  else if(display_mode==Pix_Display) \
  { \
    XCopyPlane(display,pixmaps[(face)],win,gc,0,0,24,24, \
             (unsigned int) (x),(unsigned int) ((y) - 24),1); \
  } \
  else \
  { \
    XChar buf; \
    buf = FontindexToXChar((Fontindex) (face)); \
    XDRAWIMAGESTRING(display,win,gc,x,y,&buf,1); \
  } \
}

#endif
