/***
 *** probe.c
 *** Copyright (c) 1995 by Koen Gadeyne (kmg@barco.be)
 ***
 *** All kinds of probing fuctions (currently just the clock probe)
 ***
 ***
 ***/
 

#include <stdio.h>
#include <string.h>
#include <values.h>
#include <signal.h>
#include <stdlib.h>
#ifndef DOS
#include <unistd.h>
#include <sys/types.h>
#include <sys/time.h>
#include <asm/io.h>
#else
#include <conio.h>
#include <dos.h>
#define inb inp
#define outb outp
#endif

#include "misc.h"
#include "vga_prg.h"
#include "configfile.h"
#include "messages.h"

#ifdef DOS

#define TimerResolution    1193181.667

void cardinal(long l, double *result)
{
  *result = ((l < 0) ? 4294967296.0 + (long) l : (long) l);
}

void elapsedtime(long start, long stop, double *result)
{
  double r;

  cardinal(stop - start,&r);
  *result = (1000.0 * r) / TimerResolution;

}

void initializetimer(void)
{

  outportb(0x043,0x034);
  asm jmp short NullJump1

NullJump1:;

  outportb(0x040,0x000);
  asm jmp short NullJump2

NullJump2:;

  outportb(0x040,0x000);

}

void restoretimer(void)
{
  outportb(0x043,0x036);
  asm jmp short NullJump1

NullJump1:;

  outportb(0x040,0x000);
  asm jmp short NullJump2

NullJump2:;

  outportb(0x040,0x000);

}

long readtimer(void)
{
  asm cli             /* Disable interrupts */
  asm mov  dx,020h     /* Address PIC ocw3   */
  asm mov  al,00Ah     /* Ask to read irr    */
  asm out  dx,al
  asm mov  al,00h     /* Latch timer 0 */
  asm out  043h,al
  asm in   al,dx      /* Read irr      */
  asm mov  di,ax      /* Save it in DI */
  asm in   al,040h     /* Counter --> bx*/
  asm mov  bl,al      /* LSB in BL     */
  asm in   al,040h
  asm mov  bh,al      /* MSB in BH     */
  asm not  bx         /* Need ascending counter */
  asm in   al,021h     /* Read PIC imr  */
  asm mov  si,ax      /* Save it in SI */
  asm mov  al,00FFh    /* Mask all interrupts */
  asm out  021h,al
  asm mov  ax,040h     /* read low word of time */
  asm mov  es,ax      /* from BIOS data area   */
  asm mov  dx,es:[06Ch]
  asm mov  ax,si      /* Restore imr from SI   */
  asm out  021h,al
  asm sti             /* Enable interrupts */
  asm mov  ax,di      /* Retrieve old irr  */
  asm test al,001h     /* Counter hit 0?    */
  asm jz   done       /* Jump if not       */
  asm cmp  bx,0FFh     /* Counter > 0x0FF?    */
  asm ja   done       /* Done if so        */
  asm inc  dx         /* Else count int req. */
done:;
  asm mov ax,bx   /* set function result */
}
#endif



/*
 * The Clock probe. Given the hor. and vert. REAL TOTAL screen size, it returns the pixel clock in MHz
 * 
 * 'REAL' means you have to input the total screen width (which is read from VGA regs in CHARS)
 * multiplied by the font size (8 or 9).
 *
 * NOTE: this function ASSUMES it has the rights to read VGA register STATUS1 !
 *
 * Try to measure CURRENT pixel clock.
 * Should work on ANY VGA card, since it only uses standard VGA registers
 *
 * - No need to disable interrupts to be able to measure! (the probe in the X-server does, because it doesn't use timers). 
 * - Can give slightly inaccurate results on heavily loaded machines (but normally not VERY wrong)
 * - Due to unexplained "glitches" in the vertical sync register, some timing attempts go wrong.
 *   this is detected in the program, and it tries again.
 *   Does anyone know WHY those glitches are there, and how to circumvent them?
 * - has the tendency to over-estimate the clock rate by about 0.1 MHz. No real clue why... (or is it just on MY machine?)
 *
 * There should even be a possibility to measure H-frequencies using input status 1 bit 0 (STATUS1, bit 0).
 *
 */
 
#define REALLY_SLOW_IO

/* number of frames to count for timing measurement */
#define NFRAMES    200

static int drriing=FALSE;

/* alarm fuction for when probe times out on too slow V-sync */
void badsync(int signal)
{
  PWARNING(("Slow sync. Possibly no, or very slow clock (vertical refresh < 1 Hz). Assuming 0 MHz"));
  drriing=TRUE;
}

#ifdef DOS
void restoretimerandexit(int signal)
{
  restoretimer();
  exit(0);
}
#endif

/* number sorting funstion for sort() */
static int compar(long* a, long* b)
{
  return(*a - *b);
}

#if 1
#define waitframe  while ( !(inb(STATUS1) & 0x08) ); /* wait during active display */ \
                   while ( (inb(STATUS1) & 0x08) ) /* now wait during vert retrace */
#define waitframe_alarm \
                   while ( !(inb(STATUS1) & 0x08) && !drriing ); /* wait during active display */ \
                   while ( (inb(STATUS1) & 0x08) && !drriing ) /* now wait during vert retrace */
#else
#define waitframe  while ( (inb(STATUS1) & 0x09) != 9 ); /* wait during active display */ \
                   while ( (inb(STATUS1) & 0x09) == 9 ) /* now wait during vert retrace */
#endif
                   
#define M_RETRY      3            /* maximum number of retries */
#define TIME_BAND    5           /* in usec, defines the time-band size used for building a histogram */ 
#define VALID_MEASR  NFRAMES*2/3  /* this number of measurements must be valid, or measurement will be bad */


float pixclock(int hsize, int vsize)
{
#ifndef DOS
  struct timeval tv;
#endif
  int i;
  long measure[NFRAMES+1];
  long center, num;
  double scanrate;
  int retries=0;
  double av;
  long current, centernum;

  int (*compfunc) ();
  compfunc=compar;
  
  
  /* check if vertical refresh is "reasonable" , and avoid "hangup" when something is really wrong */
#ifndef DOS
  PDEBUG(("Checking for Slow Sync..."));
  signal(SIGALRM, badsync);
  alarm(2);
  waitframe_alarm;
  alarm(0);
  signal(SIGALRM, SIG_DFL);
  if (drriing) return(0.0);
#endif

#ifdef DOS
  signal(SIGINT, restoretimerandexit);
#endif
  
  /* measure */
  do
  {
    /*** Try to do a measurement ***/
    PDEBUG(("Measurement attempt #%d",retries));
#ifdef DOS  
    initializetimer();
#endif
    waitframe;    /* synchronize */
    for (i=0; i<NFRAMES+1; i++)
    {
      waitframe;    /* measure */
#ifndef DOS
      gettimeofday(&tv, NULL); 
      measure[i] = tv.tv_usec;  /* this will go wrong if we are task-switched out for > 1 sec ... */
#else
      measure[i] = readtimer(); /* these are _not_ microseconds */
#endif
    }
#ifdef DOS
    restoretimer();
#endif

    av = 0;
    /*** convert absolute timer intervals to relative intervals ***/
    for (i=0; i<NFRAMES; i++)
    {
#ifndef DOS      
      measure[i] = measure[i+1] - measure[i];
      if (measure[i]<0) measure[i] += 1000000;
#else
     elapsedtime(measure[i],measure[i+1],&av);
      measure[i] = av*1000; /* _now_ they are microseconds */
#endif      
    }
    
    
    /*** sort measurements ***/
    qsort(measure, NFRAMES, sizeof(long), compfunc);

    /*** find value at peak of histogram ***/
    /*** Use that to filter out values out of bounds ***/
    current = measure[0];
    center = current + TIME_BAND/2; centernum = 0;
    i = 0;
    do
    {
      num = 0;
      while ((measure[i]-current < TIME_BAND) && (i<NFRAMES))
      {
        num++; i++;
      }
      if (num > centernum)
      {
        centernum = num;
        center = current + TIME_BAND/2;
      }
      if (num>0) PDEBUG(("Time slot: %ld..%ld usec, number: %d", current, current+TIME_BAND, num));
      current += TIME_BAND;
    } while (i<NFRAMES);
    PDEBUG(("Center = %ld usec", center));

    /*** Use histogram peak as center value, filter out values out of bounds ***/
    av = 0; num = 0;
    for (i=0; i<NFRAMES; i++)
    {
      if (measure[i]-center < (TIME_BAND*3))
      {
        av+=measure[i];
        num++;
      }
    }
    av /= num;
    
    retries++;
    PDEBUG(("Measurement: valid measurements: %ld/%d", num, NFRAMES));
  }
  while ( (retries<M_RETRY) && (num<VALID_MEASR) );

  
  scanrate = 1000000/av;
  PDEBUG(("average framerate (from valid measurements only) = %5.1f usec", av));
  if (num < VALID_MEASR)
  {
    PWARNING(("Only %ld%% of measurements were considered 'valid',", num*100/NFRAMES));
    PWARNING(("Your system is probably under heavy load."));
    PWARNING(("The measurement you get can be inaccurate!"));
  }

  PDEBUG(("Real total H = %d , total V = %d", hsize, vsize));

  return((scanrate * hsize * vsize) / 1000000);
}
