/***
 *** SVGA clock programming functions for SVGATextMode
 *** (c) 1995 Koen Gadeyne (kmg@barco.be)
 ***
 ***/

#include <stdio.h>
#include <unistd.h>
#ifndef DOS
#include <asm/io.h>
#else
#include <dos.h>
#include <conio.h>
#define outb(data,port) outp(port,data)
#define inb(port) inp(port)
#define ioperm(x,y,z) (0)
#define usleep(x) delay(x/1000) /* Note lack of resolution */
#endif
#include <math.h>
#include "misc.h"
#include "vga_prg.h"
#include "confdefs.h"
#include "setclock.h"
#include "messages.h"  
#include "XFREE/xfree_compat.h"
#include "XFREE/common_hw/xf86_HWlib.h"


/*****************************************************************************************************************************/

void clock_check(int result)
{
  if (result < 0)
  {
    switch(result)
    {
      case CLKSEL_DONOTHING:     PWARNING(("Clock selection: Warning: Clock not changed"));
                                 break;
      case CLKSEL_ILLEGAL_NUM:   PERROR(("Clock selection: illegal clock number."));
                                 break;
      case CLKSEL_NULLCLOCK:     PERROR(("Clock selection: 0 MHz clock selected! It would lock up the video card. Aborting..."));
                                 break;
      case CLKSEL_OUT_OF_BOUNDS: PERROR(("Clock selection: Requested clock frequency out of bounds"));
                                 break;
      default: PERROR(("Clock selection: unknown error"));
    }
  }
}

/*****************************************************************************************************************************/

void LegendClockSelect(int no)
 /* used in both S3 and ET4000 cards (according to XFREE 3.1.1) */
{
  PDEBUG(("Using Sigma Legend clock selection"));
  outb((inb(VGA_MISC_R) & 0xF3) | ((no & 0x10) >> 1) | (no & 0x04), VGA_MISC_W);
  Outb_CRTC(0x34, 0x00);
  Outb_CRTC(0x34, 0x02);
  Outb_CRTC(0x34, (no & 0x08) >> 2);
  SET_CLOCKBITS_0_1(no);
}


void TVGAClockSelect(int chipset, int num_clocks, int no)
{
    PDEBUG(("TVGAClockSelect: clock no %d (of total %d) for chipset %d.", no, num_clocks, chipset));
    SET_CLOCKBITS_0_1(no);
    Outb_SEQ(0x0B, 0); Inb_SEQ(0x0B);  /* Select "new mode regs" for CS2 and TVGA9000 CS3. */
#if 0
    Outb_SEQ(0x0D, Inb_SEQ(0x0D) & 0xF9);  /* set clock dividers to no division */
#else
    Outbit_SEQ(0x0D, 1, 0);  /* set clock dividers (div 2) to no division */
    Outbit_SEQ(0x0D, 2, 0);  /* set clock dividers (div 4 ?) to no division */
#endif    
    Outbit_SEQ(0x0D, 0, no & 0x04);  /* bit 2 of clock no */
    if (chipset == CS_TVGA9000)
    {
       Outbit_SEQ(0x0D, 6, no & 0x08);  /* bit 3 of clock no */
    }
    if ((chipset == CS_TVGA8900) && (num_clocks > 7))
    {
       Outb_SEQ(0x0B, 0);    /* Switch to Old Mode for CS3 */
       Outbit_SEQ(0x0E, 4, no & 0x08);      /* bit 3 of clock no */
    }
}


void s3ClockSelect(int no, int optmask)
  /* clock number is supposed to be in the allowable range by now. error checking should have been done before */
{
   if (optmask & OPT_LEGEND) LegendClockSelect(no);
   else
   {
     SET_CLOCKBITS_0_1(3);    /* select clock #3 to allow extra clocks to be used */
     Outb_CRTC(0x42, no & 0x0f);
   }
}


void ATIClockSelect(int chipset, int no)
{
   /* for MACH32 chipsets, bits 2 and 3 of the clock selection number must be
      exchanged to comply with the XFREE MACH server clock ordering.
      Derived through experimentation */
   if (chipset==CS_ATIMACH32) no = ( no & 0xfffffff3) | ((no & 0x04)<<1) | ((no & 0x08)>>1);
   ATI_PUTEXTREG(0xBE, (ATI_GETEXTREG(0xBE) & 0xEF) | ((no << 2) & 0x10));  /* clock bit 2 */
   ATI_PUTEXTREG(0xB9, (ATI_GETEXTREG(0xB9) & 0xFD) | ((no >> 2) & 0x02));  /* clock bit 3 */
   ATI_PUTEXTREG(0xB8, (ATI_GETEXTREG(0xB8) & 0x3F) | ((no << 2) & 0xC0));  /* clock divider */
   SET_CLOCKBITS_0_1(no);    /* clock bits no 0 and 1 , must be set LAST */
 }



void WDCClockSelect(int chipset, int num_clocks, int no, int optmask)
/* almost literal copy of the WDC driver clock selection code from XFREE, (C) Thomas Roell */
{
  int MClkIndex = 8;
  int save_cs2 = 0;

  if ((chipset == CS_WDC90C1X) || (chipset == CS_WDC90C2X) || (chipset ==CS_WDC90C3X))
    if (optmask & OPT_SWAP_HIBIT) save_cs2 = 0; else save_cs2 = 4;
  else
    if (optmask & OPT_SWAP_HIBIT) save_cs2 = 4; else save_cs2 = 0;

  if (num_clocks > 8) MClkIndex = num_clocks - 1;

  if ((no == MClkIndex) && (chipset != CS_PVGA1))
  {
    /*
     * On all of the chipsets after PVGA1, you can feed MCLK as VCLK.
     */
    PDEBUG(("WDCClockSelect: Selecting MCLOCK as VCLOCK"));
    Outbit_CRTC(0x2E, 4, 1);
  }
  else
  {
    /*
     * Disable feeding MCLK to VCLK
     */
    if (chipset != CS_PVGA1) Outbit_CRTC(0x2E, 4, 0);

    SET_CLOCKBITS_0_1(no);
    
  if ((chipset != CS_PVGA1) && (chipset != CS_WDC90C0X))
    Outbit_GR_CTL(0x0C, 1, (no & 0x04) ^ save_cs2); /* bit 2 of clock no */ 
  if (chipset==CS_WDC90C3X)
    Outbit_SEQ(0x12, 2, (no & 0x08) ^ 0x08); /* inverse clock bit 3 */
  }
}

void ET4000ClockSelect(int no, int optmask)
{
   int hibit = 1;
   if (optmask & OPT_HIBIT_HIGH) hibit = 1;
   if (optmask & OPT_HIBIT_LOW) hibit = 0;
   PDEBUG(("ET4000 Clock selection: hibit option = hibit_%s", (hibit==0) ? "low" : "high"));
   /* program clock */
   if (optmask & OPT_LEGEND) LegendClockSelect(no);
   else
   {
     SET_CLOCKBITS_0_1(no);
     Outbit_CRTC(0x34, 1, no & 0x04);                                   /* bit 2 of clock no */
     if (!(optmask & OPT_ET4000_ALTCLK))
     {
       Outbit_SEQ(7, 6, (hibit == 0) ? (no & 0x08) : (no & 0x08)^0x08);   /* bit 3 of clock no */
     }
     else
     {
       PDEBUG(("ET4000 clock selection: Using alternate clock selection mechanism"));
       Outbit_SEQ(7, 6, 0); /* just to make sure */
       Outbit_CRTC(0x31, 6, (hibit == 0) ? (no & 0x08) : (no & 0x08)^0x08);   /* bit 3 of clock no */
       Outbit_CRTC(0x31, 7, no & 0x10);    /* bit 4 of clock no */
     }
   }
}



void CirrusClockSelect(float freq, int optmask)
{
  if ((freq>57.9) && (freq<58.1)) freq = 57.9;  /* bug in cirrus clockchip code */

  /* set DRAM speed (MCLK) = (14318 * val / 8) */
  if (optmask & OPT_XFAST_DRAM) Outb_SEQ(0x1f,0x25);   /* 66.2 MHz */
  if (optmask & OPT_FAST_DRAM)  Outb_SEQ(0x1f,0x22);   /* 60.8 MHz */
  if (optmask & OPT_MED_DRAM)   Outb_SEQ(0x1f,0x1f);   /* 55.5 MHz */
  if (optmask & OPT_SLOW_DRAM)  Outb_SEQ(0x1f,0x1c);   /* 50.1 MHz */

  SET_CLOCKBITS_0_1(3);
  CirrusSetClock( (int) (freq * 1000));  /* must be AFTER MCLK setting, since CirrusSetClock() uses MCLK */
}



void Video7ClockSelect(int no)
/* from XFREE3.0 server */
{
  if (no < 2) SET_CLOCKBITS_0_1(no);
  else SET_CLOCKBITS_0_1(2);
  outw(0x3C4, 0xA4 | (no << 10));
}



int findclosestclock(float *gclocks, int num_gclocks, float req_clock, float *closest_clock, int optmask)
{
   /* returns closest clock NUMBER when one is found, error code otherwise */

   int i, closest=0;
   int num=num_gclocks;

   *closest_clock = req_clock;  /* start with both equal */

   if (req_clock == 0) return(CLKSEL_NULLCLOCK);
   
   /* create 1/2 clocks when option is set */
   if (optmask & OPT_CLOCKDIV2)
   {
     for (i=0; i<num; i++) gclocks[i+num] = gclocks[i]/2;
     num *= 2;
   }

   /* for (i=0; i<num; i++) PDEBUG(("CLOCK #%d = %f",i, gclocks[i])); */
   
   /* find closest clock frequency */
   for (i=0 ; i < num ; i ++)
   {
      if ( fabs(gclocks[i] - req_clock) < fabs(gclocks[closest] - req_clock) ) { closest = i; }
   }

   *closest_clock = gclocks[closest];
   PDEBUG(("findclosestclock: closest clock nr %d = %1.3f MHz.",closest, *closest_clock));
   if (closest>num_gclocks) PDEBUG((" (clock index #%d is actually index #%d divided by 2)", closest, closest-num_gclocks));
   
   if (closest < 0) return(CLKSEL_DONOTHING);
   if (closest > num) return(CLKSEL_ILLEGAL_NUM);
   if (*closest_clock == 0.0) return(CLKSEL_NULLCLOCK);
   
   return(closest);
}



int GetClock(int chipset, float* clocks, int num_clocks, float freq, float *closestfreq, int optmask, int report_error)
 /* look for correct clock index (for indexed clock chips), and check for out-of-bound clocks */
{
   int result;
   
   switch(chipset)
   {
     case CS_CIRRUS: /* should do some range checking here */
                     result = 0; /* suppose it can make any frequency */
                     break;
     default: result = findclosestclock(clocks, num_clocks, freq, closestfreq, optmask);
   }
   if (report_error) clock_check(result);
   return(result);
}



void SetClock(int chipset, float* clocks, int num_clocks, float freq, int optmask)
  /* "freq" is in MHz ! */
{
   int result, divby2=0;
   
   result = GetClock(chipset, clocks, num_clocks, freq, &freq, optmask, TRUE);
  /* clock number is supposed to be in the allowable range by now.
     error checking should have been done before.
     No error checking will be done in clock-setting routine! */
     
   if ((optmask & OPT_CLOCKDIV2) && (result > num_clocks-1))
   {
      divby2 = 1;
      result -= num_clocks; /* make clock selection routine pick the real clock (before division by 2) */
   }

   PDEBUG(("Setting Clock to %f MHz", freq));
   SYNCRESET_SEQ;
   switch(chipset)
   {
    case CS_VGA : SET_CLOCKBITS_0_1(result);
                  break;
    case CS_CIRRUS : CirrusClockSelect(freq, optmask);
                     break;
    case CS_S3     : s3ClockSelect(result, optmask);
                     break;
    case CS_ET4000 : ET4000ClockSelect(result,optmask);
                     break;
    case CS_TVGA8900 :
    case CS_TVGA9000 : TVGAClockSelect(chipset, num_clocks, result);
                       break;
    case CS_PVGA1:
    case CS_WDC90C0X:
    case CS_WDC90C1X:
    case CS_WDC90C2X:
    case CS_WDC90C3X: WDCClockSelect(chipset, num_clocks, result, optmask);
                      break;
    case CS_ATI:
    case CS_ATIMACH32: ATIClockSelect(chipset, result);
                       break;
    case CS_VIDEO7: Video7ClockSelect(result);
                    break;
    default: PERROR(("SetClock: unknown chip set #%d", chipset));
   }

   if (optmask & OPT_CLOCKDIV2)
   {
      Outbit_SEQ(1,3,divby2);
      if (divby2) PDEBUG(("Clock (%f) needed 'division by 2' feature.", freq));
   }

   ENDRESET_SEQ;
   usleep(50000); /* let PLL clock synthesizer stabilize */
   clock_check(result);
}

/*****************************************************************************************************************************/

void set_clockchip_clock(int chipset, int clockchip, long freq, int optmask)   /* "freq" is in Hz ! */
{
  int result=TRUE;
  
  PDEBUG(("Setting clock for chipset #%d through clock chip #%d. Freq = %ld Hz. Optionmask = 0x%x", chipset, clockchip, freq, optmask));
  switch(clockchip)
  {
     case CLKCHIP_ICS9161A:
     case CLKCHIP_DCS2834:   /* is this true ? the XFREE manual says this clock chip exists, but they don't use it. */
     case CLKCHIP_ICD2061A:
                           /* setting exactly 120 MHz doesn't work all the time */
                           if (freq > 119900000) freq = 119900000;
                           AltICD2061SetClock(freq, 2);
                           AltICD2061SetClock(freq, 2);
                           AltICD2061SetClock(freq, 2);
                           s3ClockSelect((optmask & OPT_SPEA_MERCURY) ? 6 : 2, optmask); /* select the clock */
                           break;
    case CLKCHIP_SC11412:
                           result = SC11412SetClock((long)freq/1000);
                           s3ClockSelect((optmask & OPT_SPEA_MERCURY) ? 6 : 2, optmask); /* select the clock */
                           break;
    case CLKCHIP_ICS2595:
                           result = ICS2595SetClock((long)freq/1000);
                           result = ICS2595SetClock((long)freq/1000);
                           break;
    case CLKCHIP_ICS5300:
    case CLKCHIP_S3_SDAC:
    case CLKCHIP_S3GENDAC:
                           (void) S3gendacSetClock((long)freq/1000, 2); /* can't fail */
                           s3ClockSelect(2, optmask); /* select the clock */
                           break;
    case CLKCHIP_S3TRIO:  
                           (void) S3TrioSetClock((long)freq/1000, 2); /* can't fail */
                           break;
    case CLKCHIP_ICS5342:
                           (void) ICS5342SetClock((long)freq/1000, 2); /* can't fail */
                           s3ClockSelect(2, optmask); /* select the clock */
                           break;
    case CLKCHIP_TI3025:
                           if (freq < 20000000) PERROR(("Ti3025SetClock: Min. clock is 20 MHz."));
                           xfree_Ti3025_init(optmask);
                           PDEBUG(("TI3025 Init done"));
                           Ti3025SetClock((long)freq/1000, 2);
                           s3ClockSelect(2, optmask); /* select the clock */
                           break;
    case CLKCHIP_CH8391:
                           if ((freq < 8500000) || (freq > 135000000))
                             PERROR(("Chrontel 8391 Clock chip: clock must be in range from 8.5 to 135 MHz"));
                           (void) Chrontel8391SetClock(freq, 2); /* can't fail */
                           s3ClockSelect(2, optmask); /* select the clock */
                           break;
    case CLKCHIP_ICS5341:
                           (void) ET4000gendacSetClock((long)freq/1000, 2); /* can't fail */
                           SET_CLOCKBITS_0_1(2);
                           Outbit_CRTC(0x34, 1, 0);  /* bit 2 of clock no */
                           Outbit_SEQ(7, 6, 0);      /* disable clock div 2 */
                           Outbit_SEQ(7, 0, 0);      /* disable clock div 4 */
#if 0
                             Outbit_CRTC(0x34, 4, 1); /* enable PCI burst transfer --- This should NOT be here! (special()?) */
#endif
                           break;
    case CLKCHIP_CIRRUS:
                           if ((freq>57900000) && (freq<58100000)) freq = 57900000;  /* bug in cirrus clockchip code */
                           CirrusClockSelect( (float)freq/1000000, optmask);
                           break;

    default: PERROR(("Unknown clock chip #%d.", clockchip));
  }
  if (result == FALSE) PWARNING(("ClockChip: error while programming clock chip"));
  usleep(50000);
}

 
