#!/bin/sh
# generic login/logout file for a slip line.  sliplogin invokes this with
# the parameters:
#
#      1        2      3      4         5          6        7    8    9-n
#   slipunit ttyspeed pid loginname local-addr remote-addr mask mode opt-args
#
debug=0

PATH=/sbin:/usr/sbin:/etc:/bin:/usr/bin
export PATH

# if you want to use (proxy-)ARP to announce routes, put your Ether-Addr here:
Ether=

log=/var/log/sliplog
if [ ${debug:-0} != 0 ]; then
  log=/dev/tty
  test=echo
fi

system=`uname -s`

# set correct Interface name (some Systems give us just a unit number)
slif=$1
if [ `expr "$slif" : sl` = 0 ]; then slif=sl$1; fi
speed=$2
pid=$3
user=$4
shift 4
confopt="$@"
laddr=$1
raddr=$2
mask=$3
shift 3
mode=${1:-}
if [ $# -ge 1 ]; then
  shift
fi
options="$@"

route="`expr $0 : '\(.*slip\)\.log'`.route"
if [ -f $route ]; then
  routes=`grep "^$user[^.]" $route | awk ' { mask="host";
		if ($3) mask=$3; print $2, mask; } '`
fi    

if [ ${Ether:-NULL} = NULL ]; then
    # try to calculate with ifconfig (Linux & FreeBSD)
    case `uname -s` in
      Linux )	name=eth0 ;;
      FreeBSD)	name=ed0  ;;
      * )	name=le0  ;;
    esac
    conf=`ifconfig $name`
    Ether=`expr "$conf" : '.* \([^ ]*:[^ ]*:[^ ]*:[^ ]*:[^ ]*:[^ ]*\)'`
	
    if [ ${Ether:-NULL} = NULL ]; then
      #  try to calculate with arp (SunOS)
      arp=`arp -a | grep " $laddr "`
      if [ ${arp:-NULL} = NULL ]; then
	arp=`arp -a | grep " $user "`; # match user / hostname
      fi
      Ether=`expr "${arp:-0}" : '.* \([^ ]*:[^ ]*:[^ ]*:[^ ]*:[^ ]*:[^ ]*\)'`
    fi
fi

login () {
    options="metric 1 mtu 1500 -trailers $options"
    case $system in
      Linux)
	# Linux: full character interfacename given,
	#	 special syntax for ifconfig, route explicit nessessary
	netmask="netmask $mask"
	if [ $mask = 0xffffffff -o $mask = 255.255.255.255 ]; then
	  netmask=
	fi
	$test ifconfig $slif $laddr pointopoint $raddr \
		    $netmask $options up >>$log 2>&1
	$test route add $raddr $slif >>$log 2>&1
	;;

      FreeBSD)
	case `uname -r` in
	  1.*)
	    $test ifconfig $slif $laddr $raddr netmask $mask \
			$options up >>$log 2>&1
	    ;;
	  *)
	    # FreeBSD > 2.0 will get the line policy mode = compress | link2
	    if expr "${mode:-normal}" : 'auto' >/dev/null; then
	      mode=link2
	    fi
	    $test ifconfig $slif $laddr $raddr netmask $mask \
			$options ${mode:-normal} up >>$log 2>&1
	    ;;
	esac
	;;

      *)
	# unknown system; guess BSD style
	$test ifconfig $slif $laddr $raddr netmask $mask \
			$options up >>$log 2>&1
	;;
    esac

    #in case you have an ethernet card this will announce the slip client
    if [ ${Ether:-NULL} != NULL ]; then
      $test arp -s $raddr $Ether pub
    fi

    # routes for other hosts
    set ${routes:-0}
    while [ $# -ge 2 ]; do
	type="-host"
	mask=""
	if [ $2 != host ]; then
	  type="-net"
	  # BSD style does not support netmasks
	  case $system in
	    Linux )	mask="netmask $2";;
	    FreeBSD )	mask="-netmask $2";;
	    SunOS )	type="net";;
	  esac
        else
	  case $system in
	    SunOS )	type="host";;
	  esac
	fi
	case $system in
	  Linux )	options="$1 $mask gw $raddr metric 2";;
	  FreeBSD )	options="$1 $mask $raddr";;
	  *)		options="$1 $raddr 2";;
	esac
	echo "	route add $type $options" >>$log
	$test route add $type $options >>$log 2>&1
	shift 2
    done
}

logout () {
    del=delete
    if [ $system = Linux ]; then del=del; fi

    # routes for other hosts
    set ${routes:-0}
    while [ $# -ge 2 ]; do
        type="-host"
        if [ $2 != host ]; then type="-net"; fi

	echo "	route $del $type $1" >>$log
	$test route $del $1 >>$log 2>&1
	shift 2
    done

    $test route $del $raddr >>$log 2>&1
    $test ifconfig $slif down >>$log 2>&1

    #in case you have an ethernet card     
    if [ ${Ether:-NULL} != NULL ]; then
      $test arp -d $raddr
    fi
}

logout=`expr "$0" : '.*logout'`
logout=${logout:-0}

case $logout in
    0 )	# login procedure
	echo "`date`: login for $user" >>$log
	echo "	options: $slif $confopt" >>$log
	login
	;;

    * ) # logout procedure
	echo "`date`: logout for $user at $slif" >>$log
	logout
	;;
esac

#Really important ! Don't delete this exit 0
exit 0
