/*
 * IceWM
 *
 * Copyright (C) 1997 Marko Macek
 */

#include "icewm.h"

YFrameClient::YFrameClient(YWindow *parent, YFrameWindow *frame, Window win): YFrameControl(parent, frame, win) {
    fClient = win ? win : handle();
    fBorder = 0;
    fProtocols = 0;
    fWindowTitle = 0;
    fIconTitle = 0;
    fColormap = None;
    fTransientFor = 0;
    fTransient = 0;
    fNextTransient = 0;
    fOwner = 0;
    fShaped = 0;
    fHints = 0;
    fSizeHints = XAllocSizeHints();
    fClassHint = XAllocClassHint();
    fWorkspaces = 1 << app->activeWorkspace();
    fAllWorkspaces = 0;
#ifndef NO_MWM_HINTS
    fMwmHints = 0;
#endif
    
    getProtocols();
    {
        XTextProperty prop;

        if (XGetWMName(app->display(), handle(), &prop))
            setWindowTitle(prop.value);
        if (XGetWMIconName(app->display(), handle(), &prop))
            setIconTitle(prop.value);
    }
    getSizeHints();
    getClassHint();
#ifndef NO_MWM_HINTS
    getMwmHints();
#endif

    getTransient();
    if ((fHints = XGetWMHints(app->display(), handle())) != 0) {
    }
#ifdef SHAPE
    if (shapesSupported) {
        XShapeSelectInput (app->display(), handle(), ShapeNotifyMask);
        queryShape();
    }
#endif
    getClientWorkspaces();
    XSaveContext(app->display(),
                 fClient,
                 frame ? frameContext : clientContext,
                 (XPointer)(frame ? (YWindow *)frame : (YWindow *)this));
}

YFrameClient::~YFrameClient() {
    XDeleteContext(app->display(),
                   fClient,
                   frame() ? frameContext : clientContext);
    removeAsTransient();
    removeTransients();
    free(fWindowTitle);
    free(fIconTitle);
    if (fSizeHints) { XFree(fSizeHints); fSizeHints = 0; }
    if (fClassHint) { XFree(fClassHint); fClassHint = 0; }
    if (fHints) { XFree(fHints); fHints = 0; }
    if (fMwmHints) { XFree(fMwmHints); fMwmHints = 0; }
}

void YFrameClient::getProtocols() {
    Atom *wmp = 0;
    int count;

    if (this == windowList) {
        fProtocols = wpDeleteWindow;
        return ;
    }
    
    // ???: never remove this one
    fProtocols = fProtocols & wpDeleteWindow; // = 0;

    if (XGetWMProtocols (app->display(),
                         handle(),
                         &wmp, &count))
    {
        for (int i = 0; i < count; i++) {
            if (wmp[i] == _XA_WM_DELETE_WINDOW) fProtocols |= wpDeleteWindow;
            if (wmp[i] == _XA_WM_TAKE_FOCUS) fProtocols |= wpTakeFocus;
	}
        XFree (wmp);
    }
}

void YFrameClient::getSizeHints() {
    if (fSizeHints) {
        long supplied;
        
        if (!XGetWMNormalHints(app->display(),
                               handle(),
                               fSizeHints, &supplied))
            fSizeHints->flags = 0;

        if (fSizeHints->flags & PResizeInc) {
            if (fSizeHints->width_inc == 0) fSizeHints->width_inc = 1;
            if (fSizeHints->height_inc == 0) fSizeHints->height_inc = 1;
        } else
            fSizeHints->width_inc = fSizeHints->height_inc = 1;


        if (!(fSizeHints->flags & PBaseSize)) {
            if (fSizeHints->flags & PMinSize) {
                fSizeHints->base_width = fSizeHints->min_width;
                fSizeHints->base_height = fSizeHints->min_height;
            } else
                fSizeHints->base_width = fSizeHints->base_height = 0;
        }
        if (!(fSizeHints->flags & PMinSize)) {
            fSizeHints->min_width = fSizeHints->base_width;
            fSizeHints->min_height = fSizeHints->base_height;
        }
        if (!(fSizeHints->flags & PMaxSize)) {
            fSizeHints->max_width = app->maxWidth();
            fSizeHints->max_height = app->maxHeight();
        }
        if (fSizeHints->max_width < fSizeHints->min_width)
            fSizeHints->max_width = app->maxWidth();
        if (fSizeHints->max_height < fSizeHints->min_height)
            fSizeHints->max_height = app->maxHeight();
        
        if (fSizeHints->min_height <= 0)
            fSizeHints->min_height = 1;
        if (fSizeHints->min_width <= 0)
            fSizeHints->min_width = 1;

        if (!(fSizeHints->flags & PWinGravity)) {
            fSizeHints->win_gravity = NorthWestGravity;
            fSizeHints->flags |= PWinGravity;
        }
    }
}

void YFrameClient::getClassHint() {
    if (fClassHint) {
        if (fClassHint->res_name) {
            XFree(fClassHint->res_name);
            fClassHint->res_name = 0;
        }
        if (fClassHint->res_class) {
            XFree(fClassHint->res_class);
            fClassHint->res_class = 0;
        }
        XGetClassHint(app->display(), handle(), fClassHint);
    }
}

void YFrameClient::constrainSize(int &w, int &h, int flags) {
    if (fSizeHints) {
        int wm = fSizeHints->min_width;
        int hm = fSizeHints->min_height;
        int wM = fSizeHints->max_width;
        int hM = fSizeHints->max_height;
        int wb = fSizeHints->base_width;
        int hb = fSizeHints->base_height;
        int wf = fSizeHints->width_inc;
        int hf = fSizeHints->height_inc;

        if (fSizeHints->flags & PAspect) { // aspect ratios
            int xm = fSizeHints->min_aspect.x;
            int ym = fSizeHints->min_aspect.y;
            int xM = fSizeHints->max_aspect.x;
            int yM = fSizeHints->max_aspect.y;

            // !!! fix handling of KeepX and KeepY together
            if (xm * h > ym * w) { // min aspect
                if (flags & csKeepX) {
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    h = w * ym / xm;
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    w = h * xm / ym;
                } else {
                    if (w > wM) w = wM; // maximum size
                    if (w < wm) w = wm; // minimum size
                    w = h * xm / ym;
                    if (w < wm) w = wm; // minimum size
                    if (w > wM) w = wM;
                    h = w * ym / xm;
                }
            }
            if (xM * h < yM * w) { // max aspect
                if (flags & csKeepX) {
                    if (h < hm) h = hm;
                    if (h > hM) h = hM;
                    h = w * yM / xM;
                    if (h > hM) h = hM;
                    if (h < hm) h = hm;
                    w = h * xM / yM;
                } else {
                    if (w > wM) w = wM; // maximum size
                    if (w < wm) w = wm; // minimum size
                    w = h * xM / yM;
                    if (w < wm) w = wm; // minimum size
                    if (w > wM) w = wM;
                    h = w * yM / xM;
                }
            }
        }

        if (w < wm) w = wm; // minimum size
        if (h < hm) h = hm;

        if (w > wM) w = wM; // maximum size
        if (h > hM) h = hM;

        if (w >= int(app->maxWidth())) w = app->maxWidth();
        if (h >= int(app->maxHeight())) h = app->maxHeight();

        w = wb + (w - wb + ((flags & csRound) ? wf / 2 : 0)) / wf * wf;
        h = hb + (h - hb + ((flags & csRound) ? hf / 2 : 0)) / hf * hf;
    }
    if (w <= 0) w = 1;
    if (h <= 0) h = 1;
}

struct _gravity_offset 
{
  int x, y;
};

void YFrameClient::gravityOffsets (int &xp, int &yp) {
    xp = 0;
    yp = 0;

    if (fSizeHints == 0)
        return ;

    static struct {
        int x, y;
    } gravOfsXY[11] = {
        {  0,  0 },  /* ForgetGravity */
        { -1, -1 },  /* NorthWestGravity */
        {  0, -1 },  /* NorthGravity */
        {  1, -1 },  /* NorthEastGravity */
        { -1,  0 },  /* WestGravity */
        {  0,  0 },  /* CenterGravity */
        {  1,  0 },  /* EastGravity */
        { -1,  1 },  /* SouthWestGravity */
        {  0,  1 },  /* SouthGravity */
        {  1,  1 },  /* SouthEastGravity */
        {  0,  0 },  /* StaticGravity */
    };

    int g = fSizeHints->win_gravity;
  
    if (!(g < ForgetGravity || g > StaticGravity)) {
        xp = (int)gravOfsXY[g].x;
        yp = (int)gravOfsXY[g].y;
    }
}

void YFrameClient::getTransient() {
    Window newTransientFor;
            
    if (XGetTransientForHint(app->display(),
                             handle(),
                             &newTransientFor))
    {
        if (newTransientFor == app->root()->handle() || /* bug in xfm */
            newTransientFor == handle()                 /* bug in fdesign */
            /* !!! TODO: check for recursion */
           )
            newTransientFor = 0;

        if (newTransientFor != fTransientFor) {
            if (fTransientFor)
                removeAsTransient();
            fTransientFor = newTransientFor;
            if (fTransientFor)
                addAsTransient();
        }
    }
}

void YFrameClient::addAsTransient() {
    if (fTransientFor) {
        fOwner = app->findClient(fTransientFor);
        if (fOwner == 0)
            fOwner = app->manageClient(fTransientFor);
        MSG(("transient for 0x%lX: 0x%lX", fTransientFor, fOwner));
        if (fOwner) {
            fNextTransient = fOwner->transient();
            fOwner->setTransient(this);
        } else {
            fTransientFor = 0; // ?

            fNextTransient = 0;
            fOwner = 0;
        }
    }
}

void YFrameClient::removeAsTransient() {
    if (fOwner) {
        MSG(("removeAsTransient"));
        
        YFrameClient *c = fOwner->transient(), *cp = 0;

        while (c) {
            if (c == this) {
                if (cp)
                    cp->setNextTransient(nextTransient());
                else
                    fOwner->setTransient(nextTransient());
            }
            c = c->nextTransient();
        }
        fOwner = 0;
        fNextTransient = 0;
    }
}

void YFrameClient::removeTransients() {
    if (transient()) {
        MSG(("removeTransients"));
        YFrameClient *c = transient(), *n;

        while (c) {
            n = c->nextTransient();
            c->setNextTransient(0);
            c->setOwner(0);
            c = n;
        }
        fTransient = 0;
    }
}

void YFrameClient::sendMessage(Atom msg, Time timeStamp) {
    XClientMessageEvent xev;
  
    xev.type = ClientMessage;
    xev.window = handle();
    xev.message_type = _XA_WM_PROTOCOLS;
    xev.format = 32;
    xev.data.l[0] = msg;
    xev.data.l[1] = timeStamp;
    XSendEvent (app->display(), handle(), False, 0L, (XEvent *) &xev);
}

void YFrameClient::setFrame(YFrameWindow *newFrame) {
    XDeleteContext(app->display(),
                   fClient,
                   frame() ? frameContext : clientContext);
    YFrameControl::setFrame(newFrame);
    if (newFrame == 0) {
        //msg("remove frame");
        //setWMState(WithdrawnState);
    } else {
        //msg("got frame");
        getClientWorkspaces();
        setClientWorkspaces();
    }
    XSaveContext(app->display(),
                 fClient,
                 frame() ? frameContext : clientContext,
                 (XPointer)(frame() ? (YWindow *)frame() : (YWindow *)this));
}

void YFrameClient::setWMState(FrameState state) {
    unsigned long arg[2];

    arg[0] = (unsigned long) state;
    arg[1] = (unsigned long) None;

    XChangeProperty(app->display(), handle(),
                    _XA_WM_STATE, _XA_WM_STATE,
                    32, PropModeReplace,
                    (unsigned char *)arg, 2);

    if (state == WithdrawnState && phase == phaseRunning)
        XDeleteProperty(app->display(), handle(), _XA_WM_WORKSPACES);
    else
        setClientWorkspaces();
}

FrameState YFrameClient::getWMState() {
    FrameState st = WithdrawnState;
    Atom type;
    int format;
    unsigned long nitems, lbytes;
    unsigned char *propdata;

    if (XGetWindowProperty(app->display(), handle(),
                           _XA_WM_STATE, 0, 3, False, _XA_WM_STATE,
                           &type, &format, &nitems, &lbytes,
                           &propdata) == Success)
        if (propdata) {
            st = *(long *)propdata;
            XFree(propdata);
        }
    return st;
}

void YFrameClient::configureClient(const XConfigureRequestEvent &configureRequest) {
    int cx, cy, cwidth, cheight;

    setBorder((configureRequest.value_mask & CWBorderWidth) ? configureRequest.border_width : border());
    cx = (configureRequest.value_mask & CWX) ? configureRequest.x + DO_BORDER(border()) : x();
    cy = (configureRequest.value_mask & CWY) ? configureRequest.y + DO_BORDER(border()) : y();
    cwidth = (configureRequest.value_mask & CWWidth) ? configureRequest.width : width();
    cheight = (configureRequest.value_mask & CWHeight) ? configureRequest.height : height();

    setGeometry(cx,
                cy,
                cwidth,
                cheight);

    if (configureRequest.value_mask & CWStackMode) {
        YFrameWindow *sibling;
        XWindowChanges xwc;

        if ((configureRequest.value_mask & CWSibling) &&
            XFindContext(app->display(),
                         configureRequest.above,
                         frameContext,
                         (XPointer *)&sibling) == 0)
            xwc.sibling = sibling->handle();
        else
            xwc.sibling = configureRequest.above;

        xwc.stack_mode = configureRequest.detail;

        XConfigureWindow (app->display(),
                          handle(),
                          configureRequest.value_mask & (CWSibling | CWStackMode),
                          &xwc);

    }
}

void YFrameClient::handleUnmap(const XUnmapEvent &unmap) {
    MSG(("Unmap: unmapped %d visible %d", unmapped(), visible()));
    if (!unmapped()) {
        MSG(("UnmapWindow"));
        XGrabServer(app->display());
        XSync(app->display(), 0);
        XEvent ev;
        if (XCheckTypedWindowEvent(app->display(), unmap.window, DestroyNotify, &ev)) {
            app->destroyedWindow(unmap.window);
            XUngrabServer(app->display());
            return ; // gets destroyed
        } else {
            setWMState(WithdrawnState);
            app->unmapWindow(unmap.window);
        }
        XUngrabServer(app->display());
    }
    YWindow::handleUnmap(unmap);
}

void YFrameClient::handleProperty(const XPropertyEvent &property) {
    int propDelete = (property.state == PropertyDelete) ? 1 : 0;
        
    switch (property.atom) {
    case XA_WM_NAME:
        {
            XTextProperty prop;

            if (propDelete)
                setWindowTitle(0);
            else if (XGetWMName(app->display(), handle(), &prop))
                setWindowTitle(prop.value);
        }
        break;
        
    case XA_WM_ICON_NAME:
        {
            XTextProperty prop;

            if (propDelete)
                setIconTitle(0);
            else if (XGetWMIconName(app->display(), handle(), &prop))
                setIconTitle(prop.value);
        }
        break;

    case XA_WM_CLASS:
        getClassHint();
        if (frame())
            frame()->getFrameHints();
        break;
        
    case XA_WM_HINTS:
        if (fHints)
            XFree(fHints);
        if ((fHints = XGetWMHints(app->display(), handle())) != 0) {
        }
        break;
        
    case XA_WM_NORMAL_HINTS:
        getSizeHints();
        break;
        
    case XA_WM_TRANSIENT_FOR:
        getTransient();
        break;
       
    default:
        if (property.atom == _XA_WM_PROTOCOLS)
            getProtocols();
#ifndef NO_MWM_HINTS
        else if (property.atom == _XATOM_MWM_HINTS) {
            getMwmHints();
            if (frame()) {
                frame()->getFrameHints();
                frame()->configure(frame()->x(),
                                   frame()->y(),
                                   frame()->width(),
                                   frame()->height());
            }
        }
#endif
        break;
    }
}

void YFrameClient::handleColormap(const XColormapEvent &colormap) {
    setColormap(colormap.colormap); //(colormap.state == ColormapInstalled && colormap.c_new == True) 
//                ? colormap.colormap 
//                : None);
}


void YFrameClient::handleDestroyWindow(const XDestroyWindowEvent &destroyWindow) {
    //msg("DESTROY: %lX", destroyWindow.window);
    YWindow::handleDestroyWindow(destroyWindow);

    if (destroyed())
        app->destroyedWindow(destroyWindow.window);
}

#ifdef SHAPE
void YFrameClient::handleShapeNotify(const XShapeEvent &shape) {
    if (shapesSupported) {
        if (shape.kind == ShapeBounding) {
            fShaped = shape.shaped ? 1 : 0;
            if (frame())
                frame()->setShape();
        }
    }
}
#endif

void YFrameClient::setWindowTitle(unsigned char *aWindowTitle) {
    free(fWindowTitle);
    fWindowTitle = aWindowTitle ? (unsigned char *)strdup((char *)aWindowTitle) : 0;
    if (frame())
        frame()->updateTitle();
}

void YFrameClient::setIconTitle(unsigned char *aIconTitle) {
    free(fIconTitle);
    fIconTitle = aIconTitle ? (unsigned char *)strdup((char *)aIconTitle) : 0;
    if (frame())
        frame()->updateIconTitle();
}

void YFrameClient::setColormap(Colormap cmap) {
    fColormap = cmap;
    if (frame() && app->colormapWindow() == frame())
        app->installColormap(cmap);
}

#ifdef SHAPE
void YFrameClient::queryShape() {
    fShaped = 0;
    
    if (shapesSupported) {
        int xws, yws, xbs, ybs;
        unsigned wws, hws, wbs, hbs;
        Bool boundingShaped, clipShaped;

        XShapeQueryExtents(app->display(), handle(),
                           &boundingShaped, &xws, &yws, &wws, &hws,
                           &clipShaped, &xbs, &ybs, &wbs, &hbs);
        fShaped = boundingShaped ? 1 : 0;
  }
}
#endif

#ifndef NO_MWM_HINTS
void YFrameClient::getMwmHints() {
    int retFormat;
    Atom retType;
    unsigned long retCount, remain;

    if (fMwmHints) {
        XFree(fMwmHints);
        fMwmHints = 0;
    }
    if (XGetWindowProperty(app->display(), handle(),
                           _XATOM_MWM_HINTS, 0L, 20L, False, _XATOM_MWM_HINTS,
                           &retType, &retFormat, &retCount,
                           &remain,(unsigned char **)&fMwmHints) == Success)
        if (retCount >= PROP_MWM_HINTS_ELEMENTS)
            return;
    fMwmHints = 0;
}

CARD32 YFrameClient::mwmFunctions() {
    CARD32 functions = ~0U;
    
    if (fMwmHints && (fMwmHints->flags & MWM_HINTS_FUNCTIONS)) {
        if (fMwmHints->functions & MWM_FUNC_ALL)
            functions = ~fMwmHints->functions;
        else
            functions = fMwmHints->functions;
    }
    functions &= (MWM_FUNC_RESIZE | MWM_FUNC_MOVE |
                  MWM_FUNC_MINIMIZE | MWM_FUNC_MAXIMIZE |
                  MWM_FUNC_CLOSE);
    return functions;
}

CARD32 YFrameClient::mwmDecors() {
    CARD32 decors = ~0U;
    CARD32 func = mwmFunctions();
    
    if (fMwmHints && (fMwmHints->flags & MWM_HINTS_DECORATIONS)) {
        if (fMwmHints->decorations & MWM_FUNC_ALL)
            decors = ~fMwmHints->decorations;
        else
            decors = fMwmHints->decorations;
    }
    decors &= (MWM_DECOR_BORDER | MWM_DECOR_RESIZEH |
               MWM_DECOR_TITLE | MWM_DECOR_MENU |
               MWM_DECOR_MINIMIZE | MWM_DECOR_MAXIMIZE);

    decors &=
        ~(((func & MWM_FUNC_RESIZE) ? 0 : MWM_DECOR_RESIZEH) |
          ((func & MWM_FUNC_MINIMIZE) ? 0 : MWM_DECOR_MINIMIZE) |
          ((func & MWM_FUNC_MAXIMIZE) ? 0 : MWM_DECOR_MAXIMIZE));

    return decors;
}
#endif

int YFrameClient::focusable() {
    if (!fHints)
        return 1;
    if (!(fHints->flags & InputHint))
        return 1;
    if (fHints->input)
        return 1;
    if (protocols() & wpTakeFocus)
        return 1;
    return 0;
}

void YFrameClient::setClientWorkspaces() {
    Atom wk[MAXWORKSPACES];
    int nwk = 0;

    if (fAllWorkspaces)
        wk[nwk++] = workspaceAtomAll;
    else
        for (int i = 0; i < workspaceCount; i++)
            if (workspaces() & (1 << i))
                wk[nwk++] = workspaceAtoms[i];

    XChangeProperty(app->display(),
                    handle(),
                    _XA_WM_WORKSPACES,
                    XA_ATOM,
                    32, PropModeReplace,
                    (unsigned char *)wk, nwk);
}

void YFrameClient::getClientWorkspaces() {
    Atom r_type;
    int r_format;
    unsigned long count;
    unsigned long bytes_remain;
    unsigned char *prop;

    fWorkspaces = 1 << app->activeWorkspace();
    fAllWorkspaces = 0;
    if (XGetWindowProperty(app->display(),
                           handle(),
                           _XA_WM_WORKSPACES,
                           0, MAXWORKSPACES, False, XA_ATOM,
                           &r_type, &r_format,
                           &count, &bytes_remain, &prop) == Success)
    {
        if (r_type == XA_ATOM && r_format == 32 && count > 0U) {
            Atom *atom = (Atom *)prop;
            if (count == 1U && atom[0] == workspaceAtomAll)
                //setAllWorkspaces(1);
                fAllWorkspaces = 1;
            else if (count <= 32) {
                unsigned long ws = 0;

                // could be optimized, if we are certain they are in order ???
                for (unsigned int a = 0; a < count; a++)
                    for (int w = 0; w < workspaceCount; w++)
                        if (atom[a] == workspaceAtoms[w])
                            ws |= (1 << w);
                //setWorkspaces(ws);
                fWorkspaces = ws ? ws : 1;
            }
        }
        XFree(prop);
    }
    MSG(("%s all: %d, workspaces: %lX\n", windowTitle(), fAllWorkspaces, fWorkspaces));
}

void YFrameClient::setWorkspaces(unsigned long workspaces) {
    if (workspaces != fWorkspaces) {
        fWorkspaces = workspaces;
        setClientWorkspaces();
        if (frame()) {
            if (frame()->visibleOn(app->activeWorkspace()))
                frame()->workspaceShow();
            else
                frame()->workspaceHide();
        }
    }
}

void YFrameClient::setAllWorkspaces(int allWorkspaces) {
    if (fAllWorkspaces != allWorkspaces) {
        fAllWorkspaces  = allWorkspaces;
        setClientWorkspaces();
        if (frame()) {
            if (frame()->visibleOn(app->activeWorkspace()))
                frame()->workspaceShow();
            else
                frame()->workspaceHide();
        }
    }
}

YClientContainer::YClientContainer(YWindow *parent, YFrameWindow *frame)
:YFrameControl(parent, frame)
{
    haveButtonGrab = 0;
}

YClientContainer::~YClientContainer() {
    releaseButtons();
}


void YClientContainer::handleButton(const XButtonEvent &button) {
    if (haveButtonGrab) {
        if (haveButtonGrab)
            MSG(("buttonGrab"));

        if (!(button.state & ControlMask)) {
            if (focusOnClickClient)
                frame()->activate();
            if (raiseOnClickClient)
                frame()->wmRaise();
        }
        if (passFirstClickToClient)
            XAllowEvents(app->display(), ReplayPointer, CurrentTime);
        else
            XAllowEvents(app->display(), AsyncPointer, CurrentTime);
    }
    return ;
    // YFrameControl::handleButton(button);
}

// manage button grab on frame window to capture clicks to client window
// we want to keep the grab when:
//    focusOnClickClient && not focused
// || raiseOnClickClient && not on top

// !!! fix ('not on top'  should be  'can be raised').
//     the difference is when we have transients and explicitFocus

void YClientContainer::grabButtons() {
    if (!haveButtonGrab &&
        (clickFocus ||
         focusOnClickClient || raiseOnClickClient))
    {
        haveButtonGrab = 1;

        XGrabButton(app->display(),
                    AnyButton, AnyModifier,
                    handle(), True,
                    ButtonPressMask,
                    GrabModeSync, GrabModeAsync, None, None);
    }
}

void YClientContainer::releaseButtons() {
    if (haveButtonGrab) {
        haveButtonGrab = 0;

        XUngrabButton(app->display(), AnyButton, AnyModifier,
                      handle());
    }
}

void YClientContainer::handleConfigureRequest(const XConfigureRequestEvent &configureRequest) {
    MSG(("configure request in frame"));

    if (configureRequest.window == frame()->client()->clientWindow()) {
        XConfigureRequestEvent cre = configureRequest;
        
        cre.x -= frame()->borderX();
        cre.y -= frame()->borderY() + frame()->titleY();
        frame()->configureClient(cre);
    }
}

void YClientContainer::handleMapRequest(const XMapRequestEvent &mapRequest) {
    if (mapRequest.window == frame()->client()->clientWindow()) {
        if (frame()->minimized())
            frame()->wmMinimize();
        if (frame()->shaded())
            frame()->wmShade();
        frame()->changeState(NormalState);
        if (focusOnMap)
            frame()->activate();
    }
}

void YFrameClient::handleMap(const XMapEvent &map) {
    YWindow::handleMap(map);
    if (map.window == clientWindow() && frame()) {
        if (windowList && windowList->handle() != map.window)
            return ;
        if (frame()->minimized())
            frame()->wmMinimize();
        if (frame()->shaded())
            frame()->wmShade();
        frame()->changeState(NormalState);
        if (focusOnMap)
            frame()->activate();
    }
}

void YClientContainer::handleCrossing(const XCrossingEvent &crossing) {
    if (frame() && pointerColormap) {
        if (crossing.type == EnterNotify)
            app->setColormapWindow(frame());
#ifndef SLOPPY_FOCUS
        else if (crossing.type == LeaveNotify &&
                 crossing.detail != NotifyInferior &&
                 crossing.mode == NotifyNormal &&
                 app->colormapWindow() == frame())
        {
            app->setColormapWindow(0);
        }
#endif
    }
}
