/*****************************************************************************/
/*  config_file.c - config file routines                                     */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static char *fix_colons (char *str);
void write_comment (FILE *fd, const char *comment);
static int copyfile (char *source, char *dest);
static gint hash_compare (gconstpointer path1, gconstpointer path2);
static guint hash_function (gconstpointer key);

struct config_vars {
   char *key, /* Our config file variable name */
        *description, /* How this field will show up in the options dialog */
        type; /* Type Character or Integer or AlphaNumeric */
   void *var; /* Pointer to our variable */
   int general_option : 1; /* Will this go into the general settings tab */
   char *comment; /* Comment to write out to the config file */
   GtkWidget *widget;
}; 

struct proxy_type_tag {
   char key[30];
   char desc[70];
};

static void proxy_toggle(GtkList *list, GtkWidget *child, gpointer data);
static void apply_changes (GtkWidget *widget, gpointer data);
static char *get_proxy_config (void);

static GtkWidget *proxy_text;
static char *custom_proxy;
static int proxy_num;

struct config_vars config_file_vars[] = {
/* You MUST keep the firewall stuff up here as the first entries and in the 
   same order. The options dialog relies on it being here */
   {"firewall_host", 		"Proxy hostname:", 		'C', &firewall_host, 		0, "Firewall hostname", NULL},
   {"firewall_port", 		"Proxy port:", 			'A', &firewall_port,		0, "Port to connect to on the firewall", NULL},
   {"firewall_username", 	"Proxy username:", 		'C', &firewall_username,	0, "Your firewall username", NULL},
   {"firewall_password", 	"Proxy password:", 		'C', &firewall_password, 	0, "Your firewall password", NULL},
   {"firewall_account", 	"Proxy account:", 		'C', &firewall_account,		0, "Your firewall account (optional)", NULL},
   {"proxy_config",		"Proxy config",			'C', &proxy_config,		0, "This specifies how your proxy server expects us to log in", NULL},
   {"email", 			"Email address:", 		'C', &emailaddr, 		1, "Enter your email address here", NULL},
   {"view_program", 		"View program:", 		'C', &view_program, 		1, "The default program used to view files. If this is blank, the internal file viewer will be used", NULL},
   {"edit_program", 		"Edit program:", 		'C', &edit_program, 		1, "The default program used to edit files.", NULL},
   {"connect_timeout",		"Connect timeout:",		'A', &timeout,			1, "The max timeout for the connection", NULL},
   {"retries",			"Connect retries:",		'A', &retries,			1, "The number of auto-retries to do. Set this to 0 to retry indefinately", NULL},
   {"sleep_time",		"Retry sleep time:",		'A', &sleep_time,		1, "The number of seconds to wait between retries", NULL},
   {"maxkbs",			"Max KB/S:",			'F', &maxkbs,			1, "The maximum KB/s a file transfer can get. (Set to 0 to disable)", NULL},
   {"reconnect_diag",		"Bring up reconnect dialog",	'I', &reconnect_diag,		1, "Bring up the reconnect dialog after login failure", NULL},
   {"confirm_delete",		"Confirm delete",		'I', &confirm_delete,		1, "Confirm when deleting files", NULL},
   {"one_transfer", 		"Do one transfer at a time",	'I', &do_one_transfer_at_a_time,1, "Do only one transfer at a time?", NULL},
   {"passive_transfer",		"Passive file transfers", 	'I', &passive_transfer,		1, "Send PASV command or PORT command for data transfers", NULL},
   {"preserve_permissions",	"Preserve permissions",		'I', &preserve_attribs,		1, "Save the permissions of the transfered files", NULL},
   {"refresh_files",		"Refresh after each file transfer", 'I', &refresh_files,	1, "Refresh the listbox after each file is transfered", NULL},
   {"save_geometry", 		"Save geometry", 		'I', &save_geometry, 		1, "Save the size of each widget for next startup", NULL},
   {"show_hidden_files",	"Show hidden files",		'I', &show_hidden_files,	1, "Show hidden files in the listboxes", NULL},
   {"sort_dirs_first", 		"Sort directories first", 	'I', &sort_dirs_first, 		1, "Put the directories first then the files", NULL},
   {"start_transfers",		"Start file transfers", 	'I', &start_file_transfers, 	1, "Automatically start the file transfers when they get queued?", NULL},
   {"usecache", 		"Use cache",	 		'I', &use_cache, 		1, "Do you want to use the cache?", NULL},
   {"use_default_dl_types", 	"", 				'I', &use_default_dl_types, 	0, "(*) If this is set, and there is a ext= line below for the file extension, it will download the file as specified below", NULL},
   {"listbox_local_width",	"", 				'I', &listbox_local_width, 	0, "The default width of the local files listbox", NULL},
   {"listbox_remote_width", 	"", 				'I', &listbox_remote_width, 	0, "The default width of the remote files listbox", NULL},
   {"listbox_file_height", 	"", 				'I', &listbox_file_height, 	0, "The default height of the local/remote files listboxes", NULL},
   {"transfer_height", 		"", 				'I', &transfer_height, 		0, "The default height of the transfer listbox", NULL},
   {"log_height", 		"", 				'I', &log_height, 		0, "The default height of the logging window", NULL},
   {"filename_width", 		"", 				'I', &listbox_filename_width, 	0, "(*) The width of the filename column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"size_width", 		"", 				'I', &listbox_size_width, 	0, "(*) The width of the size column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"user_width", 		"", 				'I', &listbox_user_width, 	0, "(*) The width of the user column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"group_width", 		"", 				'I', &listbox_group_width, 	0, "(*) The width of the group column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"date_width", 		"", 				'I', &listbox_date_width, 	0, "(*) The width of the date column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"attribs_width", 		"", 				'I', &listbox_attribs_width, 	0, "(*) The width of the attribs column in the file listboxes. Set this to 0 to have this column automagically resize. Set this to -1 to disable this column", NULL},
   {"", "", '\0', NULL, 0, "", NULL}};

struct proxy_type_tag proxy_type[] = {
   {"none", 		""},
   {"SITE command", 	"USER %pu\nPASS %pp\nSITE %hh\nUSER %hu\nPASS %hp\n"},
   {"user@host", 	"USER %pu\nPASS %pp\nUSER %hu@%hh\nPASS %hp\n"},
   {"user@host:port",	"USER %hu@%hh:%ho\nPASS %hp\n"},
   {"AUTHENTICATE",	"USER %hu@%hh\nPASS %hp\nSITE AUTHENTICATE %pu\nSITE RESPONSE %pp\n"},
   {"user@host port",	"USER %hu@%hh %ho\nPASS %hp\n"},
   {"user@host NOAUTH", "USER %hu@%hh\nPASS %hp\n"},
   {"HTTP Proxy",	"http"},
   {"Custom",		""},
   {"", ""}};

#define CUSTOM_PROXY_NUM	8

void options_dialog (gpointer data) {
   GtkWidget *tempwid, *dialog, *notebook, *box, *intbox, *chartbl, *table, 
             *label, *proxy_combo;
   char tempstr[30], *pos, *endpos, *oldstr;
   GList *proxy_list;
   int num, tbl_len;

   proxy_list = NULL;   
   dialog = gtk_dialog_new ();
   gtk_window_set_title (GTK_WINDOW (dialog), _("Options"));
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox), 10);
   gtk_container_border_width (GTK_CONTAINER (GTK_DIALOG (dialog)->action_area), 5);
   gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 2);
   gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
   gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
   gtk_signal_connect_object (GTK_OBJECT (dialog), "delete_event", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_signal_connect_object (GTK_OBJECT (dialog), "destroy", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   
   notebook = gtk_notebook_new ();
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->vbox), notebook, TRUE, TRUE, TRUE);
   gtk_widget_show (notebook);
   
   box = gtk_vbox_new (FALSE, 0);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("General"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);   

   tbl_len = 1;
   chartbl = gtk_table_new (tbl_len, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (chartbl), 5);
   gtk_table_set_col_spacings (GTK_TABLE (chartbl), 5);
   gtk_box_pack_start (GTK_BOX (box), chartbl, FALSE, FALSE, FALSE);
   gtk_widget_show (chartbl);
   
   intbox = gtk_vbox_new (FALSE, 0);
   gtk_box_pack_start (GTK_BOX (box), intbox, TRUE, TRUE, TRUE);
   gtk_widget_show (intbox);

   num = 0;
   while (config_file_vars[num].var != NULL) {
      if (config_file_vars[num].general_option) {
         if (config_file_vars[num].type == 'I') {
            tempwid = gtk_check_button_new_with_label (_(config_file_vars[num].description));
            gtk_box_pack_start (GTK_BOX (intbox), tempwid, FALSE, FALSE, FALSE);
            gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (tempwid), *(int *) config_file_vars[num].var);
            gtk_widget_show (tempwid);
            config_file_vars[num].widget = tempwid;
         }
         else {
            tbl_len++;
            gtk_table_resize (GTK_TABLE (chartbl), tbl_len, 2);
            
            tempwid = gtk_label_new (_(config_file_vars[num].description));
            gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
            gtk_table_attach_defaults (GTK_TABLE (chartbl), tempwid, 0, 1, tbl_len-1, tbl_len);
            gtk_widget_show (tempwid);

            tempwid = gtk_entry_new ();
            gtk_table_attach_defaults (GTK_TABLE (chartbl), tempwid, 1, 2, tbl_len-1, tbl_len);
            if (config_file_vars[num].type == 'A') {
               g_snprintf (tempstr, sizeof (tempstr), "%d", *(int *) config_file_vars[num].var);
               gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
            }
            else if (config_file_vars[num].type == 'F') {
               g_snprintf (tempstr, sizeof (tempstr), "%.2f", *(float *) config_file_vars[num].var);
               gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
            }
            else {
               gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[num].var);
            }
            gtk_widget_show (tempwid);
            config_file_vars[num].widget = tempwid; 
         }
      }
      num++;
   }

   box = gtk_vbox_new (FALSE, 5);
   gtk_container_border_width (GTK_CONTAINER (box), 10);
   gtk_widget_show (box);

   tempwid = gtk_label_new (_("Proxy Server"));
   gtk_widget_show (tempwid);
   gtk_notebook_append_page (GTK_NOTEBOOK (notebook), box, tempwid);   

   table = gtk_table_new (8, 2, FALSE);
   gtk_table_set_row_spacings (GTK_TABLE (table), 5);
   gtk_table_set_col_spacings (GTK_TABLE (table), 5);
   gtk_box_pack_start (GTK_BOX (box), table, FALSE, FALSE, FALSE);
   gtk_widget_show (table);

   tempwid = gtk_label_new (_(config_file_vars[0].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 0, 1);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[0].var);
   gtk_widget_show (tempwid);
   config_file_vars[0].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[1].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 1, 2);
   g_snprintf (tempstr, sizeof (tempstr), "%d", *(int *) config_file_vars[1].var);
   gtk_entry_set_text (GTK_ENTRY (tempwid), tempstr);
   gtk_widget_show (tempwid);
   config_file_vars[1].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[2].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 2, 3);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[2].var);
   gtk_widget_show (tempwid);
   config_file_vars[2].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[3].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 3, 4);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[3].var);
   gtk_entry_set_visibility (GTK_ENTRY (tempwid), FALSE);
   gtk_widget_show (tempwid);
   config_file_vars[3].widget = tempwid;

   tempwid = gtk_label_new (_(config_file_vars[4].description));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 4, 5);
   gtk_widget_show (tempwid);

   tempwid = gtk_entry_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 4, 5);
   gtk_entry_set_text (GTK_ENTRY (tempwid), *(char **) config_file_vars[4].var);
   gtk_entry_set_visibility (GTK_ENTRY (tempwid), FALSE);
   gtk_widget_show (tempwid);
   config_file_vars[4].widget = tempwid;

   tempwid = gtk_label_new (_("Proxy server type"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 5, 6);
   gtk_widget_show (tempwid);

   proxy_combo = gtk_combo_new ();
   gtk_table_attach_defaults (GTK_TABLE (table), proxy_combo, 1, 2, 5, 6);
   gtk_widget_show (proxy_combo);

   num = 0;
   while (*proxy_type[num].key != '\0') {
      label = gtk_list_item_new_with_label (_(proxy_type[num].key));
      gtk_widget_show (label);
      proxy_list = g_list_append (proxy_list, label);
      num++;
   }
   gtk_list_prepend_items (GTK_LIST (GTK_COMBO(proxy_combo)->list), proxy_list);

   custom_proxy = g_malloc (1);
   *custom_proxy = '\0';
   if (proxy_config == NULL || *proxy_config == '\0') {
      proxy_num = 0;
   }
   else {
      pos = proxy_config;
      while ((endpos = strstr (pos, "%n"))) {
         *endpos = '\0';
         oldstr = custom_proxy;
         custom_proxy = g_strconcat (custom_proxy, pos, "\n", NULL);
         g_free (oldstr);
         *endpos = '%';
         pos = endpos + 2;
      }
      if (strlen (pos) > 0) {
         oldstr = custom_proxy;
         custom_proxy = g_strconcat (custom_proxy, pos, NULL);
         g_free (oldstr);
      }

      for (proxy_num = 1; proxy_num < CUSTOM_PROXY_NUM; proxy_num++) {
         if (strcmp (proxy_type[proxy_num].desc, custom_proxy) == 0) {
            break;
         }
      }
   }

   proxy_text = gtk_text_new (NULL, NULL);
   gtk_widget_set_usize (proxy_text, -2, 125);
   gtk_text_set_editable (GTK_TEXT (proxy_text), TRUE);
   gtk_table_attach_defaults (GTK_TABLE (table), proxy_text, 0, 2, 6, 7);
   gtk_widget_show (proxy_text);

   tempwid = gtk_table_new (5, 2, TRUE);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 2, 7, 8);
   gtk_widget_show (tempwid);
   table = tempwid;
   
   tempwid = gtk_label_new (_("%pu = proxy user"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 0, 1);
   gtk_widget_show (tempwid);
   
   tempwid = gtk_label_new (_("%hu = host user"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 0, 1);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%pp = proxy pass"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%hp = host pass"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 1, 2);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ph = proxy host"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%hh = host"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 2, 3);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%po = proxy port"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ho = host port"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 3, 4);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%pa = proxy account"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 0, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 0, 1, 4, 5);
   gtk_widget_show (tempwid);

   tempwid = gtk_label_new (_("%ha = host account"));
   gtk_misc_set_alignment (GTK_MISC (tempwid), 1, 0.5);
   gtk_table_attach_defaults (GTK_TABLE (table), tempwid, 1, 2, 4, 5);
   gtk_widget_show (tempwid);
   
   tempwid = gtk_button_new_with_label (_("OK"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, TRUE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (apply_changes), (gpointer) NULL);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("  Cancel  "));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, TRUE);
   gtk_signal_connect_object (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (gtk_widget_destroy), GTK_OBJECT (dialog));
   gtk_widget_show (tempwid);

   tempwid = gtk_button_new_with_label (_("Apply"));
   GTK_WIDGET_SET_FLAGS (tempwid, GTK_CAN_DEFAULT);
   gtk_box_pack_start (GTK_BOX (GTK_DIALOG (dialog)->action_area), tempwid, TRUE, TRUE, TRUE);
   gtk_signal_connect (GTK_OBJECT (tempwid), "clicked", GTK_SIGNAL_FUNC (apply_changes), (gpointer) NULL);
   gtk_widget_grab_default (tempwid);
   gtk_widget_show (tempwid);

   gtk_signal_connect (GTK_OBJECT (GTK_COMBO (proxy_combo)->list), "select_child", GTK_SIGNAL_FUNC (proxy_toggle), NULL);
   gtk_list_select_item (GTK_LIST (GTK_COMBO (proxy_combo)->list), proxy_num);
   gtk_widget_show (dialog);
}
/*****************************************************************************/
static void proxy_toggle(GtkList *list, GtkWidget *child, gpointer data) {
   gtk_text_set_point (GTK_TEXT (proxy_text), 0);
   gtk_text_forward_delete (GTK_TEXT (proxy_text), gtk_text_get_length (GTK_TEXT (proxy_text)));

   proxy_num = gtk_list_child_position (list, child);
   if (proxy_num == CUSTOM_PROXY_NUM) {
      gtk_text_insert (GTK_TEXT (proxy_text), NULL, NULL, NULL,
      	custom_proxy, strlen (custom_proxy));
   }
   else {
      gtk_text_insert (GTK_TEXT (proxy_text), NULL, NULL, NULL, 
      	proxy_type[proxy_num].desc, strlen (proxy_type[proxy_num].desc));
   }
}
/*****************************************************************************/
static void apply_changes (GtkWidget *widget, gpointer data) {
   char *tempstr;
   int num;
   
   num = 0;
   while (config_file_vars[num].var != NULL) {
      if (config_file_vars[num].widget != NULL) {
         if(config_file_vars[num].type == 'I') {
            *(int *) config_file_vars[num].var = GTK_TOGGLE_BUTTON (config_file_vars[num].widget)->active;
         }
         else {
            tempstr = gtk_entry_get_text (GTK_ENTRY (config_file_vars[num].widget));
            if (config_file_vars[num].type == 'A') {
               *(int *) config_file_vars[num].var = strtol (tempstr, NULL, 10);
            }
            else if (config_file_vars[num].type == 'F') {
               *(float *) config_file_vars[num].var = strtod (tempstr, NULL);
            }
            else {
               g_free (*(char **) config_file_vars[num].var);
               *(char **) config_file_vars[num].var = g_malloc (strlen (tempstr) + 1);
               strcpy (*(char **) config_file_vars[num].var, tempstr);
            }
         }
      }
      num++;
   }
   proxy_config = get_proxy_config ();
   gftp_set_proxy_config (window2.hdata->ftpdata, proxy_config);
   window2.hdata->protocol = strcmp (proxy_config, "http") == 0 ? http : ftp;
   gftp_set_proxy_hostname (window2.hdata->ftpdata, firewall_host);
   gftp_set_proxy_port (window2.hdata->ftpdata, firewall_port);
   gftp_set_proxy_username (window2.hdata->ftpdata, firewall_username);
   gftp_set_proxy_password (window2.hdata->ftpdata, firewall_password);
   gftp_set_proxy_account (window2.hdata->ftpdata, firewall_account);
   GFTP_SET_TRANSFER_TYPE (window2.hdata->ftpdata,
   	passive_transfer ? gftp_transfer_passive : gftp_transfer_active);
   gtk_widget_set_sensitive (firewall_btn, proxy_num > 0);
   write_config_file ();
}
/*****************************************************************************/
static char *get_proxy_config (void) {
   char *newstr, *oldstr, *pos, *endpos, tempchar;
   unsigned int len;
   
   newstr = g_malloc (1);
   *newstr = '\0';
   pos = GTK_TEXT (proxy_text)->text.ch;
   len = gtk_text_get_length (GTK_TEXT (proxy_text));
   tempchar = pos[len];
   pos[len] = '\0';
   do {
      if ((endpos = strchr (pos, '\n')) != NULL) {
         if (*(endpos - 1) == '\r') *(endpos - 1) = '\0';
         *endpos = '\0';
      }
      oldstr = newstr;
      if (endpos != NULL) {
         newstr = g_strconcat (newstr, pos, "%n", NULL);
      }
      else {
         newstr = g_strconcat (newstr, pos, NULL);
      }
      g_free (oldstr);
      if (endpos != NULL) {
         *endpos = '\n';
         pos = endpos + 1;
      }
   } while (endpos != NULL);
   
   pos = GTK_TEXT (proxy_text)->text.ch;
   pos[len] = tempchar;
   return (newstr);
}
/*****************************************************************************/
void read_config_file (void) {
   struct conn_categories *newentry, *preventry, *folderentry, *endentry;
   char tempstr[MAXSTR], temp1str[MAXSTR], *curpos, tempchar, *str;
   struct pix_ext *tempext;
   FILE *conffile;
   int line, pos;
   
   line = 0;
   hostcat = g_malloc0 (sizeof (struct conn_categories));
   hostcat->isfolder = 1;
   hostcat->path = g_malloc (1);
   *hostcat->path = '\0';
   bookmarks_htable = g_hash_table_new (hash_function, hash_compare);

   registered_exts = NULL;
   preventry = NULL;
   if (!expand_path (CONFIG_FILE, tempstr, sizeof (tempstr))) {
      printf (_("gFTP Error: Bad config file name %s\n"), CONFIG_FILE);
      exit (0);
   }
   if (access (tempstr, F_OK) == -1) {
      expand_path (BASE_CONF_DIR, temp1str, sizeof (temp1str));
      if (access (temp1str, F_OK) == -1) {
         if (mkdir (temp1str, 488) != 0) {
            printf (_("gFTP Error: Could not make directory %s: %s\n"), temp1str, g_strerror(errno));
            exit (-1);
         }
      }
      g_snprintf (temp1str, sizeof(temp1str), "%s/gftprc", SHARE_DIR);
      temp1str[sizeof (temp1str) - 1] = '\0';
      if (access (temp1str, F_OK) == -1) {
         printf (_("gFTP Error: Cannot find master config file %s\n"), temp1str);
         printf (_("Did you do a make install?\n"));
         exit (-1);
      }
      copyfile (temp1str, tempstr);
   }
   chmod (tempstr, S_IRUSR | S_IWUSR);
   conffile = fopen (tempstr, "r");
   if (conffile == NULL) {
      printf (_("gFTP Error: Cannot open config file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   while (fgets (tempstr, sizeof (tempstr), conffile)) {
      if (tempstr[strlen (tempstr) - 1] == '\n') {
         tempstr[strlen (tempstr) - 1] = '\0';
         if (tempstr[strlen (tempstr) - 2] == '\r') tempstr[strlen (tempstr) - 2] = '\0';
      }
      line++;
      if (strncmp (tempstr, "host=", 5) == 0) {
         curpos = tempstr+5;
         while (*curpos == '/') curpos++;
         newentry = g_malloc0 (sizeof (struct conn_categories));
         newentry->isfolder = 0;
         parse_args (curpos, 8, line, &newentry->path, &newentry->hostname,
         	&newentry->port, &newentry->dir, &newentry->user,
         	&newentry->pass, &newentry->firewall, &newentry->acct);
         newentry->save_password = *newentry->pass != '\0';

         /* We have to create the folders. For example, if we have 
            Debian Sites/Debian, we have to create a Debian Sites entry */
         preventry = hostcat;
         if (preventry->children != NULL) { 
            endentry = preventry->children;
            while (endentry->next != NULL) endentry = endentry->next;
         }
         else endentry = NULL;
         curpos = newentry->path;
         while ((curpos = strchr (curpos, '/')) != NULL) {
            tempchar = *curpos;
            *curpos = '\0';
            /* See if we already made this folder */
            if ((folderentry = (struct conn_categories *) 
            	g_hash_table_lookup (bookmarks_htable, newentry->path)) == NULL) {

               /* Allocate the individual folder. We have to do this for the edit 
                  bookmarks feature */
               folderentry = g_malloc0 (sizeof (struct conn_categories));
               folderentry->path = g_malloc (strlen (newentry->path) + 1);
               strcpy (folderentry->path, newentry->path);
               folderentry->prev = preventry;
               folderentry->isfolder = 1;
               g_hash_table_insert (bookmarks_htable, folderentry->path, folderentry);
               if (preventry->children == NULL) preventry->children = folderentry;
               else endentry->next = folderentry;
               preventry = folderentry;
               endentry = NULL;
            }
            else {
               preventry = folderentry;
               if (preventry->children != NULL) {
                  endentry = preventry->children;
                  while (endentry->next != NULL) endentry = endentry->next;
               }
               else endentry = NULL;
            }
            *curpos = tempchar;
            curpos++;
         }

         /* Get the parent node */
         if ((curpos = strrchr (newentry->path, '/')) == NULL) preventry = hostcat;
         else {
            tempchar = *curpos;
            *curpos = '\0';
            preventry = (struct conn_categories *) 
            		g_hash_table_lookup (bookmarks_htable, newentry->path);
            *curpos = tempchar;
         }
         
         if (preventry->children != NULL) {
            endentry = preventry->children;
            while (endentry->next != NULL) endentry = endentry->next;
            endentry->next = newentry;
         }
         else preventry->children = newentry;
         newentry->prev = preventry;
         g_hash_table_insert (bookmarks_htable, newentry->path, newentry);
      }
      else if (strncmp (tempstr, "ext=", 4) == 0) {
         curpos = tempstr+4;
         tempext = g_malloc (sizeof (struct pix_ext));
         parse_args (curpos, 4, line, &tempext->ext, &tempext->filename,
         	&tempext->ascii_binary, &tempext->view_program);
         g_snprintf (tempstr, sizeof (tempstr), "%s/%s", BASE_CONF_DIR, tempext->filename);
         tempstr[sizeof (tempstr) - 1] = '\0';
         if (!tempext->filename || !expand_path (tempstr, temp1str, sizeof (temp1str))) {
            printf (_("gFTP Error: Bad file name %s in config file on line %d\n"), tempext->filename, line);
            exit (0);
         }
         if (*tempext->filename != '\0' && access (temp1str, F_OK) != 0) {
            g_snprintf (tempstr, sizeof (tempstr), "%s/%s", SHARE_DIR, tempext->filename);
            tempstr[sizeof (tempstr) - 1] = '\0';
            if (!expand_path (tempstr, temp1str, sizeof (temp1str))) {
               g_snprintf (tempstr, sizeof (tempstr), "/usr/share/icons/%s", tempext->filename);
               tempstr[sizeof (tempstr) - 1] = '\0';
               if (!expand_path (tempstr, temp1str, sizeof (temp1str))) {
                  printf (_("gFTP Error: Bad file name %s in config file on line %d\n"), tempext->filename, line);
                  exit (0);
               }
            }
            if (access (temp1str, F_OK) != 0) {
               printf (_("gFTP Error: Error on line %d: %s doesn't exist in %s or %s\n"), line, tempext->filename, SHARE_DIR, BASE_CONF_DIR);
               exit (0);
            }
         }
         tempext->stlen = strlen (tempext->ext);
         tempext->next = registered_exts;
         registered_exts = tempext;
      }         
      else if (strncmp (tempstr, "localhistory=", 13) == 0) {
         curpos = tempstr + 13;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         window1.history = g_list_append (window1.history, str);
         window1.histlen++;
      }
      else if (strncmp (tempstr, "remotehistory=", 14) == 0) {
         curpos = tempstr + 14;
         str = g_malloc (strlen (curpos) + 1);
         strcpy (str, curpos);
         window2.history = g_list_append (window2.history, str);
         window2.histlen++;
      }
      else if (*tempstr != '#' && *tempstr != '\0') {
         pos = 0;
         while (config_file_vars[pos].var != NULL) {
            if (strncmp (config_file_vars[pos].key, tempstr, strlen (config_file_vars[pos].key)) == 0) {
               curpos = tempstr + strlen (config_file_vars[pos].key) + 1;
               if (config_file_vars[pos].type == 'C') {
                  *(char **) config_file_vars[pos].var = g_malloc (strlen (curpos) + 1);
                  strcpy (*(char **) config_file_vars[pos].var, curpos);
                  break;
               }
               else if (config_file_vars[pos].type == 'F') {
                  *(float *) config_file_vars[pos].var = strtod (curpos, NULL);
                  break;
               }
               else {
                  *(int *) config_file_vars[pos].var = strtol (curpos, NULL, 10);
                  break;
               }
            }
            pos++;
         }

         if (config_file_vars[pos].var == NULL) {
            printf (_("gFTP Warning: Skipping line %d in config file: %s\n"), line, tempstr);
         }
      }
   }
   make_nonnull (&proxy_config);
   make_nonnull (&firewall_host);
   make_nonnull (&firewall_username);
   make_nonnull (&firewall_password);
   make_nonnull (&firewall_account);
   make_nonnull (&view_program);
   make_nonnull (&edit_program);
   return;
}
/*****************************************************************************/
void write_config_file (void) {
   const char *hdr = "# Config file for gFTP\n# Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>\n# Warning: Any comments that you add to this file WILL be overwritten\n# If a entry has a (*) in it's comment, you can't change it inside gFTP\n\n";
   const char *hosthdr = "host=menu path:hostname:port:start dir:username:password:firewall:account If you set the password to @EMAIL@, gFTP will automatically change that to your email address. If you set the firewall argument to 1, gFTP will try to connect to your FTP proxy if you have one. It is usually best to leave this set at 1 because gFTP won't use it if you don't have a FTP proxy";
   const char *exthdr = "ext=file extenstion:XPM file:Ascii or Binary (A or B):viewer program. Note: All arguments except the file extension are optional";
   const char *histhdr = "This section has what will be shown in the local and remote history boxes. Syntax: localhistory=entry and remotehistory=entry";
   struct conn_categories *tempentry;
   struct pix_ext *tempext;
   char tempstr[MAXSTR], *str;
   FILE *conffile;
   GList *tempnode;
   int pos;

   if (!expand_path (CONFIG_FILE, tempstr, sizeof (tempstr))) {
      printf (_("gFTP Error: Bad config file name %s\n"), CONFIG_FILE);
      exit (0);
   }
   conffile = fopen (tempstr, "w+");
   if (conffile == NULL) {
      printf (_("gFTP Error: Cannot open config file %s: %s\n"), CONFIG_FILE, g_strerror (errno));
      exit (0);
   }
   fwrite (_(hdr), 1, strlen (hdr), conffile);

   pos = 0;
   while (config_file_vars[pos].var != NULL) {
      if (*config_file_vars[pos].comment != '\0') {
         write_comment (conffile, _(config_file_vars[pos].comment));
      }
      if (config_file_vars[pos].type == 'C') {
         g_snprintf (tempstr, sizeof (tempstr), "%s=%s\n\n", 
            config_file_vars[pos].key, *(char **) config_file_vars[pos].var);
      }
      else if (config_file_vars[pos].type == 'F') {
         g_snprintf (tempstr, sizeof (tempstr), "%s=%.2f\n\n",
            config_file_vars[pos].key, *(float *) config_file_vars[pos].var);
      }
      else {
         g_snprintf (tempstr, sizeof(tempstr), "%s=%d\n\n", 
            config_file_vars[pos].key, *(int *) config_file_vars[pos].var);
      }
      tempstr[sizeof (tempstr) - 1] = '\0';
      fwrite (tempstr, 1, strlen (tempstr), conffile);
      pos++;
   }

   write_comment (conffile, _(hosthdr));
   tempentry = hostcat->children;
   while (tempentry != NULL) {
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      str = tempentry->path;
      while (*str == '/') str++;
      tempentry->hostname = fix_colons (tempentry->hostname);
      tempentry->dir = fix_colons (tempentry->dir);
      tempentry->user = fix_colons (tempentry->user);
      tempentry->pass = fix_colons (tempentry->pass);
      tempentry->acct = fix_colons (tempentry->acct);
      g_snprintf (tempstr, sizeof (tempstr), "host=%s:%s:%s:%s:%s:%s:%c:%s\n", 
      	str, tempentry->hostname == NULL ? "" : tempentry->hostname, 
      	tempentry->port == NULL ? "" : tempentry->port, 
      	tempentry->dir == NULL ? "" : tempentry->dir, 
      	tempentry->user == NULL ? "" : tempentry->user,
      	!tempentry->save_password || tempentry->pass == NULL ? "" : tempentry->pass, 
      	tempentry->firewall == NULL || tempentry->firewall[0] == '\0' ? '1' : tempentry->firewall[0],
      	tempentry->acct == NULL ? "" : tempentry->acct);
      tempstr[sizeof (tempstr) - 1] = '\0';
      fwrite (tempstr, 1, strlen (tempstr), conffile);
      if (tempentry->next == NULL) {
         tempentry = tempentry->prev;
         while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
         tempentry = tempentry->next;
      }
      else tempentry = tempentry->next;
   }

   fwrite ("\n", 1, 1, conffile);
   write_comment (conffile, _(exthdr));
   tempext = registered_exts;
   while (tempext != NULL) {
      g_snprintf (tempstr, sizeof (tempstr), "ext=%s:%s:%c:%s\n", 
      	tempext->ext, tempext->filename, 
      	*tempext->ascii_binary == '\0' ? ' ' : *tempext->ascii_binary, 
      	tempext->view_program);
      tempstr[sizeof (tempstr) - 1] = '\0';
      fwrite (tempstr, 1, strlen (tempstr), conffile);
      tempext = tempext->next;
   }

   fwrite ("\n", 1, 1, conffile);
   write_comment (conffile, _(histhdr));
   tempnode = window1.history;
   while (tempnode != NULL) {
      g_snprintf (tempstr, sizeof (tempstr), "localhistory=%s\n", (char *) tempnode->data);
      tempstr[sizeof (tempstr) - 1] = '\0';
      fwrite (tempstr, 1, strlen (tempstr), conffile);
      tempnode = tempnode->next;
   }

   fwrite ("\n", 1, 1, conffile);
   tempnode = window2.history;
   while (tempnode != NULL) {
      g_snprintf (tempstr, sizeof (tempstr), "remotehistory=%s\n", (char *) tempnode->data);
      tempstr[sizeof (tempstr) - 1] = '\0';
      fwrite (tempstr, 1, strlen (tempstr), conffile);
      tempnode = tempnode->next;
   }

   fclose (conffile);
}
/*****************************************************************************/
void write_comment (FILE *fd, const char *comment) {
   const char *pos, *endpos;
   
   fwrite ("# ", 1, 2, fd);
   pos = comment;
   while (strlen (pos) > 76) {
      for (endpos = pos + 76; *endpos != ' ' && endpos > pos; endpos--);
      if (endpos == pos) {
         for (endpos = pos + 76; *endpos != ' ' && *endpos != '\0'; endpos++);
      }
      fwrite (pos, 1, endpos - pos, fd);
      fwrite ("\n# ", 1, 3, fd);
      if (*endpos == '\0') {
         pos = endpos;
         break;
      }
      else pos = endpos + 1;
   }
   if (strlen (pos) > 1) {
      fwrite (pos, 1, strlen (pos), fd);
      fwrite ("\n", 1, 1, fd);
   }
}
/*****************************************************************************/
static char *fix_colons (char *str) {
   char *pos, *oldpos, *newstr;
   int colons;
   
   if (str == NULL) return (NULL);
   colons = 0;
   for (pos = str; *pos != '\0'; pos++) {
      if (*pos == ':') colons++;
   }
   if (colons == 0) return (str);
   newstr = g_malloc (strlen (str) + colons + 1);
   pos = newstr;
   for (oldpos = str; *oldpos != '\0';) {
      if (*oldpos == ':') *pos++ = '\\';
      *pos++ = *oldpos++;
   }
   *pos = '\0';
   g_free (str);
   return (newstr);
}
/*****************************************************************************/
static int copyfile (char *source, char *dest) {
   FILE *srcfd, *destfd;
   char buf[8192];
   size_t n;
   
   if ((srcfd = fopen (source, "rb")) == NULL) {
      return (0); 
   }
   if ((destfd = fopen (dest, "wb")) == NULL) {
      fclose (srcfd);
      return (0);
   }

   while ((n = fread (buf, 1, sizeof (buf), srcfd)) > 0) {
      fwrite (buf, 1, n, destfd);
   }

   fclose (srcfd);
   fclose (destfd);
   return (1);
}
/*****************************************************************************/
int parse_args (char *str, int numargs, int lineno, char **first, ...) {
   char *curpos, *endpos, *pos, **dest, tempchar;
   int ret, has_colon;
   va_list argp;
   
   ret = 1;
   va_start (argp, first);
   curpos = str;
   dest = first;
   *dest = NULL;
   while (numargs > 0) {
      has_colon = 0;
      if (numargs > 1) {
         if ((endpos = strchr (curpos, ':')) == NULL) {
            printf (_("gFTP Warning: Line %d doesn't have enough arguments\n"), lineno);
            ret = 0;
            endpos = curpos + strlen (curpos);
         }
         else {
            /* Allow colons inside the fields. If you want a colon inside a field,
               just put 2 colons in there */
            while (endpos != NULL && *(endpos - 1) == '\\') {
               endpos = strchr (endpos + 1, ':');
               has_colon = 1;
            }
         }
      }
      else endpos = curpos + strlen (curpos);

      *dest = g_malloc (endpos - curpos + 1);
      tempchar = *endpos;
      *endpos = '\0';
      strcpy (*dest, curpos);
      *endpos = tempchar;
      if (has_colon) {
         pos = *dest;
         curpos = *dest;
         while (*pos != '\0') {
            if (*pos != '\\' && *(pos+1) != ':') *curpos++ = *pos++;
            else pos++;
         }
         *curpos = '\0';
      }
      if (*endpos == '\0') break;
      curpos = endpos + 1;
      if (numargs > 1) {
         dest = va_arg (argp, char **);
         *dest = NULL;
      }
      numargs--;
   }

   while (numargs > 1) {
      dest = va_arg (argp, char **);
      *dest = g_malloc (1);
      **dest = '\0';
      numargs--;
   }
   va_end (argp);
   return (1);
}
/*****************************************************************************/
GHashTable *build_hash_table (struct conn_categories *entry) {
   struct conn_categories *tempentry;
   GHashTable *htable;
   
   htable = g_hash_table_new (hash_function, hash_compare);
   tempentry = entry;
   while (tempentry != NULL) {
      g_hash_table_insert (htable, tempentry->path, tempentry);
      if (tempentry->children != NULL) {
         tempentry = tempentry->children;
         continue;
      }
      while (tempentry->next == NULL && tempentry->prev != NULL) tempentry = tempentry->prev;
      tempentry = tempentry->next;
   }
   return (htable);
}
/*****************************************************************************/
static gint hash_compare (gconstpointer path1, gconstpointer path2) {
   return (strcmp ((char *) path1, (char *) path2) == 0);
}
/*****************************************************************************/
static guint hash_function (gconstpointer key) {
   return (((char *) key)[0] + ((char *) key)[1]);
}
/*****************************************************************************/
