/*****************************************************************************/
/*  ftp.c - contains most of the core network routines                       */
/*  Copyright (C) 1998-1999 Brian Masney <masneyb@seul.org>                  */
/*                                                                           */
/*  This program is free software; you can redistribute it and/or modify     */
/*  it under the terms of the GNU General Public License as published by     */
/*  the Free Software Foundation; either version 2 of the License, or        */
/*  (at your option) any later version.                                      */
/*                                                                           */
/*  This program is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            */
/*  GNU General Public License for more details.                             */
/*                                                                           */
/*  You should have received a copy of the GNU General Public License        */
/*  along with this program; if not, write to the Free Software              */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "ftp.h"

static void *getdir_thread (void *data);
static void *connect_thread (void *data);
static void sig_quit (int signo);

static sigjmp_buf envir;
static gftp_readline_buffer readln_buffer;

/* This struct will be used more in later versions of gftp */
struct task_thread {
   int id; /* What we are doing */
   void *ptr1; /* Pointer to some data */
   void *ptr2;
};

#define ID_GET_DIRECTORY	1

int ftp_list_files (struct ftp_window_data *wdata, int load_from_cache) {
   struct task_thread tthread;
   void *success;

   wdata->totalitems = wdata->numselected = 0;
   wdata->local = 0;

   update_ftp_info (wdata);
   if (wdata->hdata->files == NULL) {
      if (wdata->local == 2 && !ftp_connect (wdata->hdata, 0)) return (0);
      wdata->hdata->getdir = 1;
      gtk_widget_set_sensitive (stop_btn, 1);
      gtk_clist_freeze (GTK_CLIST (wdata->listbox));
      wdata->hdata->stopable = 1;
      tthread.id = ID_GET_DIRECTORY;
      tthread.ptr1 = wdata;
      tthread.ptr2 = &load_from_cache;
      pthread_create (&wdata->hdata->tid, NULL, getdir_thread, &tthread);

      while (wdata->hdata->stopable) {
         fix_display();
         sleep (0);
      }

      gtk_clist_thaw (GTK_CLIST (wdata->listbox));
      gtk_widget_set_sensitive (stop_btn, 0);
      pthread_join (wdata->hdata->tid, &success);
      wdata->hdata->tid = 0;
   }
   
   if (wdata->hdata->files == NULL) {
      wdata->local = 2;
      return (0);
   }
   wdata->flags &= ~(WINDOW_SORTED);
   sortrows (GTK_CLIST (wdata->listbox), wdata->sortcol, (gpointer) wdata);
   gtk_clist_select_row (GTK_CLIST (wdata->listbox), 0, 0);
   update_ftp_info (wdata);
   return (1);
}
/*****************************************************************************/
static void *getdir_thread (void *data) {
   struct ftp_window_data *wdata;
   struct task_thread *tthread;
   gftp_logging_func old_func;
   int sj;
   
   tthread = (struct task_thread *) data;
   wdata = (struct ftp_window_data *) tthread->ptr1;
   old_func = wdata->hdata->ftpdata->logging_function;
   wdata->hdata->ftpdata->logging_function = queue_log;

   sj = sigsetjmp (envir, 1);
   signal (SIGINT, sig_quit);
   signal (SIGALRM, sig_quit);
   if ((sj == 0 || sj == 2) && wdata->hdata->getdir && tthread->id == ID_GET_DIRECTORY) {
      wdata->hdata->files = get_remote_files (wdata->hdata, NULL, &wdata->totalitems, *(int *) tthread->ptr2, NULL);
   }
   if (sj != 1 && (wdata->hdata->files == NULL && *(int *) tthread->ptr2)) {
      *(int *) tthread->ptr2 = 0;
      siglongjmp (envir, 2);
   }
   wdata->hdata->ftpdata->logging_function = old_func;
   wdata->hdata->stopable = 0;
   return (NULL);
}
/*****************************************************************************/
struct ftp_file_data *get_remote_files (struct ftp_host_data *hdata, char *path, int *total, int load_from_cache, GtkLabel *up_wid) {
   char logstr[MAXSTR], description[MAXSTR], commastr[20], *curdir, *tempstr, *dir, *str;
   struct ftp_file_data *files, *newfle, *lastfle;
   gftp_file fle;
   int isdotdot, got;
   size_t dltotal;
   time_t lasttime;
   FILE *fd;
   
   *total = 0;
   isdotdot = 0;
   curdir = NULL;
   fd = NULL;
   if (path != NULL) {
      curdir = g_malloc (strlen (GFTP_GET_DIRECTORY (hdata->ftpdata)) + 1);
      strcpy (curdir, GFTP_GET_DIRECTORY (hdata->ftpdata));
      if (gftp_set_directory (hdata->ftpdata, path) != 0) return (NULL);
   }

   if (up_wid != NULL) {
      update_ftp_info (hdata->wdata);
      gtk_label_set (up_wid, _("Receiving file names..."));
      fix_display ();
   }
   
   g_snprintf (description, sizeof (description), "%s://%s:%s@%s:%d%s", 
   	hdata->protocol == ftp ? "ftp" : "http",
   	GFTP_GET_USERNAME (hdata->ftpdata), GFTP_GET_PASSWORD (hdata->ftpdata),
   	GFTP_GET_HOSTNAME (hdata->ftpdata), GFTP_GET_PORT (hdata->ftpdata),
   	path == NULL ? GFTP_GET_DIRECTORY (hdata->ftpdata) : path);
   description[sizeof (description)-1] = '\0';
   remove_double_slashes (description+6);

   if (use_cache && load_from_cache && (fd = find_cache_entry (description)) != NULL) {
      files = parse_local_file (fd, 0, total, hdata->protocol);
      fclose (fd);
      if (files && files->next == NULL && strcmp (files->file, "..") == 0) {
         free_file_list (files);
         return (NULL);
      }
      else {
         if (hdata->wdata->hdata == hdata) hdata->wdata->flags |= WINDOW_CACHED;
         return (files);
      }
   }

   if (hdata->wdata->hdata == hdata && hdata->wdata->local == 2 && 
   	!ftp_connect (hdata, 0)) {
      return (NULL);
   }

   if (use_cache) fd = new_cache_entry (description);
   if (hdata->protocol == ftp) {
      if (gftp_list_files (hdata->ftpdata) != 0) return (NULL);
   }
   else {
      dir = GFTP_GET_DIRECTORY (hdata->ftpdata);
      if (dir == NULL) {
         gftp_set_directory (hdata->ftpdata, "/");
         dir = g_malloc (2);
         strcpy (dir, "/");
      }
      if (!http_connect (hdata)) return (NULL);
   
      str = GFTP_GET_USERNAME (hdata->ftpdata);
      if (str == NULL || strcmp (str, ANON_LOGIN) == 0) {
         tempstr = g_strconcat ("GET ftp://", 
         	GFTP_GET_HOSTNAME (hdata->ftpdata), dir, " HTTP/1.1\r\n", NULL);
      }
      else {
         tempstr = g_strconcat ("GET ftp://", str, 
         	":", GFTP_GET_PASSWORD (hdata->ftpdata), "@", GFTP_GET_HOSTNAME (hdata->ftpdata), 
         	dir, " HTTP/1.1\r\n", NULL);
      }
      remove_double_slashes (tempstr + 10);
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_send, hdata->ftpdata->user_data, tempstr);
      }
      write (GFTP_GET_DATA_FD (hdata->ftpdata), tempstr, strlen (tempstr));
      write (GFTP_GET_DATA_FD (hdata->ftpdata), "\n", 1);
      g_free (tempstr);
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_misc, hdata->ftpdata->user_data, 
         	_("Retrieving directory listing...\n"));
      }
   }
   
   memset (&readln_buffer, 0, sizeof (readln_buffer));
   files = lastfle = NULL;
   lasttime = time (NULL);
   dltotal = 0;
   while (hdata->protocol == ftp ? (got = gftp_get_next_file (hdata->ftpdata, &fle)) > 0 :
   	(got = http_get_next_file (hdata->ftpdata, &fle, fd)) > 0) {
      dltotal += got;
      if (!fle.file) { /* Invalid entry */
         if (hdata->protocol == ftp && hdata->ftpdata->logging && 
         	strncmp ("total", GFTP_GET_LAST_DIRENT (hdata->ftpdata), 5) != 0) {
            hdata->ftpdata->logging_function (gftp_logging_error, hdata->ftpdata->user_data, 
            	_("Warning: Cannot parse listing %s\n"), GFTP_GET_LAST_DIRENT (hdata->ftpdata));
         }
         continue;
      }
      if (strcmp (fle.file, ".") == 0) continue;
      if (strcmp (fle.file, "..") == 0) isdotdot = 1;

      if (up_wid != NULL && time (NULL)-lasttime >= 1) {
         insert_commas (dltotal, commastr, sizeof (commastr));
         g_snprintf (logstr, sizeof (logstr), _("Retrieving file names...%s bytes"), commastr);
         logstr[sizeof (logstr)-1] = '\0';
         gtk_label_set (up_wid, logstr);
         fix_display ();
         lasttime = time (NULL);
      }
      if (use_cache && hdata->protocol == ftp && fd != NULL) {
         fwrite (GFTP_GET_LAST_DIRENT (hdata->ftpdata), 1, strlen (GFTP_GET_LAST_DIRENT (hdata->ftpdata)), fd);
         fwrite ("\n", 1, 1, fd);
      }
      newfle = g_malloc0 (sizeof (struct ftp_file_data));
      memcpy (newfle, &fle, sizeof (gftp_file));
      newfle->flags = 0;
      if (strchr (newfle->attribs, 'd') != NULL) newfle->flags |= FILE_ISDIR;
      if (strchr (newfle->attribs, 'l') != NULL) newfle->flags |= FILE_ISLINK;
      if ((strchr (newfle->attribs, 'x') != NULL) && !(newfle->flags & FILE_ISDIR)
      		&& !(newfle->flags & FILE_ISLINK)) newfle->flags |= FILE_ISEXE;
      newfle->next = NULL;
      if (lastfle == NULL) files = newfle;
      else lastfle->next = newfle;
      lastfle = newfle;
      (*total)++;
   }
   
   if (fd != NULL) fclose (fd);
   if (hdata->protocol == ftp) gftp_end_transfer (hdata->ftpdata);
   else {
      close (hdata->ftpdata->datafd);
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_misc, hdata->ftpdata->user_data, 
         	_("Finished retrieving directory listing\n"));
      }
   }
   
   if (!isdotdot) {
      newfle = g_malloc0 (sizeof(struct ftp_file_data));
      newfle->file = g_malloc (3);
      strcpy (newfle->file, "..");
      newfle->user = g_malloc (1);
      *newfle->user = '\0';
      newfle->group = g_malloc (1);
      *newfle->group = '\0';
      newfle->attribs = g_malloc (1);
      *newfle->attribs = '\0';

      newfle->flags = FILE_ISDIR;
      newfle->next = NULL;
      if (lastfle == NULL) files = newfle;
      else lastfle->next = newfle;
      lastfle = newfle;
      (*total)++;
   }

   if (hdata->wdata->hdata == hdata) hdata->wdata->flags &= ~(WINDOW_CACHED);
   if (path != NULL) if (gftp_set_directory (hdata->ftpdata, curdir) != 0) return (NULL);
   return (files);
}
/*****************************************************************************/
int http_get_next_file (gftp_request *request, gftp_file *fle, FILE *cachefd) {
   char *tempstr;

   if (GFTP_GET_LAST_DIRENT (request)) g_free (GFTP_GET_LAST_DIRENT (request));
   if ((tempstr = gftp_read_line (GFTP_GET_DATA_FD (request), &readln_buffer)) == NULL) {
      GFTP_GET_LAST_DIRENT (request) = NULL;
      return (0);
   }
   if (cachefd != NULL) {
      fwrite (tempstr, 1, strlen (tempstr), cachefd);
      fwrite ("\n", 1, 1, cachefd);
   }
   if (parse_html_line (tempstr, fle) == 0) {
      gftp_file_destroy (fle);
      memset (fle, 0, sizeof (gftp_file));
   }
   GFTP_GET_LAST_DIRENT (request) = tempstr;

   return (strlen (tempstr));
}
/*****************************************************************************/
int http_connect (struct ftp_host_data *hdata) {
   struct sockaddr_in proxyaddr;
   struct hostent *host;
   int proxyfd, curhost;

   if ((proxyfd = socket (AF_INET, SOCK_STREAM, IPPROTO_TCP)) == -1) {
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_error, hdata->ftpdata->user_data, _("Failed to create a socket: %s\n"),
      		g_strerror (errno));
      }
      return (0);
   }

   if (hdata->ftpdata->logging) {
      hdata->ftpdata->logging_function (gftp_logging_misc, hdata->ftpdata->user_data, _("Looking up %s...\n"),
      		GFTP_GET_PROXY_HOSTNAME (hdata->ftpdata));
   }
   memset (&proxyaddr, 0, sizeof (proxyaddr));
   proxyaddr.sin_family = AF_INET;
   proxyaddr.sin_port = htons (GFTP_GET_PROXY_PORT (hdata->ftpdata));
   if ((host = gethostbyname (GFTP_GET_PROXY_HOSTNAME (hdata->ftpdata))) == NULL) {
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_error, hdata->ftpdata->user_data, _("Cannot look up hostname %s: %s\n"), 
         	GFTP_GET_PROXY_HOSTNAME (hdata->ftpdata), g_strerror (errno));
      }
      close (proxyfd);
      return (0);
   }
   curhost = 0;
   while (host->h_addr_list[curhost] != NULL) {
      if (hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_misc, hdata->ftpdata->user_data, 
         	_("Trying %s:%d...\n"), host->h_name, GFTP_GET_PROXY_PORT (hdata->ftpdata));
      }
      memcpy (&proxyaddr.sin_addr, host->h_addr_list[curhost], host->h_length);
      if (connect (proxyfd, (struct sockaddr *) &proxyaddr, sizeof (proxyaddr)) == -1 && hdata->ftpdata->logging) {
         hdata->ftpdata->logging_function (gftp_logging_error, hdata->ftpdata->user_data, _("Cannot connect to %s: %s\n"), 
         	host->h_name, g_strerror (errno));
      }
      else break;
      curhost++;
   }

   if (host->h_addr_list[curhost] == NULL) {
      close (proxyfd);
      return (0);
   }
   if (hdata->ftpdata->logging) {
      hdata->ftpdata->logging_function (gftp_logging_misc, hdata->ftpdata->user_data, _("Connected to %s:%d\n"), host->h_name, GFTP_GET_PROXY_PORT (hdata->ftpdata));
   }
   hdata->ftpdata->datafd = proxyfd;
   return (1);
}
/*****************************************************************************/
int http_get_file (struct ftp_host_data *hdata, char *filename) {
   char *tempstr, *str;
   
   if (!http_connect (hdata)) return (0);
   str = GFTP_GET_USERNAME (hdata->ftpdata);
   if (strcmp (str, ANON_LOGIN) == 0) {
      tempstr = g_strconcat ("GET ftp://", GFTP_GET_HOSTNAME (hdata->ftpdata), 
        "/", filename, " HTTP/1.1\r\n", NULL);
   }
   else {
      tempstr = g_strconcat ("GET ftp://", str, 
      	":", GFTP_GET_PASSWORD (hdata->ftpdata), "@", GFTP_GET_HOSTNAME (hdata->ftpdata), 
        "/", filename, " HTTP/1.1\r\n", NULL);
   }
   remove_double_slashes (tempstr + 10);
   if (hdata->ftpdata->logging) {
      hdata->ftpdata->logging_function (gftp_logging_send, hdata->ftpdata->user_data, tempstr);
   }
   write (GFTP_GET_DATA_FD (hdata->ftpdata), tempstr, strlen (tempstr));
   write (GFTP_GET_DATA_FD (hdata->ftpdata), "\n", 1);
   g_free (tempstr);
   if (hdata->ftpdata->logging) {
      hdata->ftpdata->logging_function (gftp_logging_send, hdata->ftpdata->user_data, _("Retrieving file: %s...\n"), filename);
   }
   return (1);
}
/*****************************************************************************/
int parse_html_line (char *tempstr, gftp_file *fle) {
   char *stpos, *pos, *tempfpos, month[4];
   struct tm t;
   
   memset (fle, 0, sizeof (gftp_file));

   if ((pos = strstr (tempstr, "<A HREF=")) == NULL) {
      return (0);
   }

   /* Find the filename */
   while (*pos != '"' && *pos != '\0') pos++;
   if (*pos == '\0') return (0);
   pos++;
   
   for (stpos = pos; *pos != '"' && *pos != '\0'; pos++);
   if (*pos == '\0') return (0);
   *pos = '\0';

   /* Copy file attributes. Just about the only thing we can get is whether it
      is a directory or not */
   fle->attribs = g_malloc (11);
   strcpy (fle->attribs, "----------");
   if (*(pos-1) == '/') {
      *(pos-1) = '\0';
      *fle->attribs = 'd';
   }

   /* Copy filename */
   if (strncmp (stpos, "FTP://", 6) == 0 || strncmp (stpos, "HTTP://", 7) == 0 ||
   	strncmp (stpos, "ftp://", 6) == 0 || strncmp (stpos, "http://", 7) == 0) {
      return (0);
   }
   if (*stpos == '/') stpos++;
   if ((tempfpos = strrchr (stpos, '/')) == NULL) tempfpos = stpos;
   else tempfpos++;
   fle->file = g_malloc (strlen (tempfpos) + 1);
   strcpy (fle->file, tempfpos);
   if (*(pos-1) == '\0') *(pos-1) = '/';
   *pos = '"';
   pos++;

   /* Skip whitespace and html tags after file and before date */
   stpos = pos;
   if ((pos = strstr (stpos, "</A>")) == NULL) {
      if ((pos = strstr (stpos, "</a>")) == NULL) {
         return (0);
      }
   }
   pos += 4;

   while (*pos == ' ' || *pos == '.' || *pos == '<') {
      if (*pos == '<') {
         if (strncmp (pos, "<A ", 3) == 0 || strncmp (pos, "<a ", 3) == 0) {
            stpos = pos;
            if ((pos = strstr (stpos, "</A>")) == NULL) {
               if ((pos = strstr (stpos, "</a>")) == NULL) {
                  return (0);
               }
            }
            pos += 4;
         }
         else {
            while (*pos != '>' && *pos != '\0') pos++;
            if (*pos == '\0') return (0);
         }
      }
      pos++;
   }

   /* Get the size */
   if ((stpos = strchr (pos, 'k')) == NULL) return (1); /* Return successfully
                                                           since we got the file */
   while (*(stpos-1) != ' ' && *(stpos-1) != '\t' && stpos > tempstr) stpos--;
   fle->size = strtol (stpos, NULL, 10) * 1024;
   
   /* Now get the date */
   memset (&t, 0, sizeof (t));
   memset (month, 0, sizeof (month));
   if (strchr (pos, ':') != NULL) {
      if (*pos == '[') pos++;
      sscanf (pos, "%02d-%3s-%04d %02d:%02d", &t.tm_mday, month, &t.tm_year, 
      		&t.tm_hour, &t.tm_min);
      while (*pos != ' ' && *pos != '\0') pos++;
      if (*pos == '\0') return (1);
      while (*pos == ' ') pos++;
      while (*pos != ' ' && *pos != '\0') pos++;
      if (*pos == '\0') return (1);
      while (*pos == ' ') pos++;
   }
   else {
      pos++;
      strncpy (month, pos, 3);
      pos += 3;
      
      while (*pos == ' ') pos++;
      t.tm_mday = strtol (pos, NULL, 10);
      while (*pos != ' ' && *pos != '\0') pos++;

      while (*pos == ' ') pos++;
      t.tm_year = strtol (pos, NULL, 10);
      while (*pos != ' ' && *pos != '\0') pos++;
   }
   fle->datetime = mktime (&t);
   return (1);
}
/*****************************************************************************/
int ftp_connect (struct ftp_host_data *hdata, int getdir) {
   void *ret;
   int success;
   
   ret = 0;
   if (getdir && hdata->wdata->hdata == hdata) hdata->getdir = 1;
   else hdata->getdir = 0;
   if (hdata->wdata->hdata == hdata) {
      hdata->wdata->local = 0;
      gtk_clist_freeze (GTK_CLIST (hdata->wdata->listbox));
      gtk_widget_set_sensitive (stop_btn, 1);
      hdata->stopable = 1;
      pthread_create (&hdata->tid, NULL, connect_thread, hdata);

      while (hdata->stopable) {
         fix_display ();
         sleep (0);
      }

      gtk_widget_set_sensitive (stop_btn, 0);
      gtk_clist_thaw (GTK_CLIST (hdata->wdata->listbox));
      pthread_join (hdata->tid, &ret);
   }
   else ret = connect_thread ((void *) hdata);
   success = (int) ret;
   hdata->tid = 0;

   if (hdata->wdata->hdata == hdata) {
      if (hdata->getdir && (success || hdata->wdata->hdata->files != NULL)) {
         success = ftp_list_files (hdata->wdata, 1);
      }
      else if (!success) {
         hdata->wdata->totalitems = hdata->wdata->numselected = 0;
         hdata->wdata->local = -1;
         if (reconnect_diag) {
            hdata->wait = 1;
            reconnect_dialog (hdata);
         }
      }
      
      while (hdata->wait) {
         fix_display ();
         sleep (0);
      }
   }

   if (GFTP_CONNECTED (hdata->ftpdata)) success = 1;
   return (success);
}
/*****************************************************************************/
static void *connect_thread (void *data) {
   struct ftp_host_data *hdata;
   gftp_logging_func old_func;
   int ret;
   int sj;
   
   hdata = (struct ftp_host_data *) data;
   old_func = hdata->ftpdata->logging_function;
   hdata->ftpdata->logging_function = queue_log;

   hdata->connection_number = 0;
   sj = sigsetjmp (envir, 1);
   ret = 0;
   if (hdata->wdata->hdata == hdata) {
      signal (SIGINT, sig_quit);
      signal (SIGALRM, sig_quit);
   }
   if (sj != 0)  {
      ret = 0;
      if (hdata->ftpdata->sockfd >= 0) {
         close (hdata->ftpdata->sockfd);
         hdata->ftpdata->sockfd = -1;
      }
      if (hdata->wdata->hdata->files == NULL) {
         free_file_list (hdata->files);
      }
   }
   while (sj != 1 && (retries == 0 || hdata->connection_number < retries)) {
      hdata->connection_number++;
      if (timeout > 0) alarm (timeout);
      if (hdata->protocol == ftp && GFTP_GET_CONTROL_FD (hdata->ftpdata) == -1) {
         ret = gftp_connect (hdata->ftpdata) == 0;
      }
      else ret = 1;
      alarm (0);
      if (hdata->wdata->hdata == hdata) {
         hdata->wdata->totalitems = hdata->wdata->numselected = 0;
         hdata->wdata->local = 0;
      }
      if (ret) {
         if (hdata->getdir) {
            hdata->wdata->hdata->files = get_remote_files (hdata->wdata->hdata, NULL, &hdata->wdata->totalitems, 1, NULL);
         }
         break;
      }
      else if (retries == 0 || hdata->connection_number < retries) {
         queue_log (gftp_logging_misc, NULL, _("Waiting %d seconds until trying to connect again\n"), sleep_time);
         alarm (sleep_time);
         pause ();
      }
   }
   
   hdata->ftpdata->logging_function = old_func;
   hdata->stopable = 0;
   hdata->wait = 0;
   return ((void *) ret);
}
/*****************************************************************************/
static void sig_quit (int signo) {
   signal (signo, sig_quit);
   siglongjmp (envir, signo == SIGINT ? 1 : 2);
}
/*****************************************************************************/
void stop_button (GtkWidget *widget, struct ftp_window_data *wdata) {
   if (window2.hdata->tid > 0) {
      pthread_kill (window2.hdata->tid, SIGINT);
   }
}
/*****************************************************************************/
void disconnect (struct ftp_window_data *wdata) {
   gftp_disconnect (wdata->hdata->ftpdata);
   gftp_set_username (wdata->hdata->ftpdata, ANON_LOGIN);
   gftp_set_password (wdata->hdata->ftpdata, emailaddr);
   delete_ftp_file_info (wdata);
   wdata->local = -1;
   update_ftp_info (wdata);
}
/*****************************************************************************/
unsigned long get_file_transfer_mode (char *filename) {
   struct pix_ext *tempext;
   unsigned long flags;
   int stlen;

   flags = 0;   
   stlen = strlen (filename);
   tempext = registered_exts;
   while (use_default_dl_types && tempext != NULL) {
      if (stlen >= tempext->stlen &&
      		strcmp (&filename[stlen-tempext->stlen], tempext->ext) == 0) {
         if (toupper (*tempext->ascii_binary == 'A')) flags = FILE_ASCII;
         else if (toupper (*tempext->ascii_binary != 'B')) {
            tempext = NULL;
            break;
         }
         break;
      }
      tempext = tempext->next;
   }
   if (tempext == NULL && GFTP_GET_DATA_TYPE (window2.hdata->ftpdata) == gftp_type_ascii) {
      flags = FILE_ASCII;
   }
   return (flags);
}
/*****************************************************************************/
void add_file_transfer (struct ftp_transfer_data *tdata) {
   struct ftp_transfer_data *temptdata;
   
   pthread_mutex_lock (&transfer_mutex);
   if (file_transfers == NULL) file_transfers = tdata;
   else {
      temptdata = file_transfers;
      while (temptdata->next != NULL) temptdata = temptdata->next;
      temptdata->next = tdata;
   }
   pthread_mutex_unlock (&transfer_mutex);
}
/*****************************************************************************/
struct ftp_transfer_data *transfer_one_file (char *local_file, char *remote_file, int dl_or_up) {
   struct ftp_transfer_data *newtdata;

   newtdata = new_tdata ();
   copy_hdata_struct (window2.hdata, newtdata->hdata);
   newtdata->hdata->files = g_malloc0 (sizeof (struct ftp_file_data));
   newtdata->hdata->files->remote_file = g_malloc (strlen (remote_file) + 1);
   strcpy (newtdata->hdata->files->remote_file, remote_file);
   newtdata->hdata->files->file = g_malloc (strlen (local_file) + 1);
   strcpy (newtdata->hdata->files->file, local_file);
   newtdata->hdata->files->next = NULL;
   newtdata->hdata->files->size = 0;
   newtdata->curfle = newtdata->hdata->files;
   newtdata->hdata->totalfiles = 1;
   newtdata->flags = TRANSFER_SHOW;
   if (dl_or_up) {
      /* We're downloading this file */
      newtdata->flags |= TRANSFER_DIRECTION;
      newtdata->hdata->files->flags |= get_file_transfer_mode (remote_file);
   }
   else {
      /* We're uploading this file */
      newtdata->hdata->files->flags |= get_file_transfer_mode (local_file);
   }
   return (newtdata);
}
/*****************************************************************************/
