/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include <glide.h>
#include <X11/Xlib.h>

#include "global.h"
#include "window.h"
#include "render.h"
#include "window.h"
#include "control.h"
#include "window_state.h"
#include "gxf.h"
#include "tux.h"
#include "bird.h"
#include "lighting.h"

#define CHROMA_COLOR_16 0xF81F
#define CHROMA_COLOR_32 0x00FF00FF

extern Vector LightPos;
extern Vector LightDir;
extern Vector CameraDir;

static void
render_sky (Demo_State *state);

static void
render_wave (Demo_State *state);

void
render_newframe (Demo_State *state)
{
  /* Don't bother clearing the screen if I going to draw a background */
  if (state->need_clears)
  {
    grBufferClear (CHROMA_COLOR_32 | 0xFF000000, 0, GR_WDEPTHVALUE_FARTHEST);
  }
}

void
depthify_desktop (Display *display, Demo_State *state, Main_Window *mw)
{
  int i;
  static int old_wincount = -1;


#ifdef GPROG

#ifndef RENDER_WINDOW_NUMBERS
  /* Setup the glide state for no texturing constant color */
  grConstantColorValue (CHROMA_COLOR_32);
  grColorCombine (GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		  GR_COMBINE_LOCAL_CONSTANT, GR_COMBINE_OTHER_NONE, FXFALSE);
  /* Render the triangles in black since it is the chroma color */
  /* Setup the alpha blend function for invisible triangles */
  grAlphaBlendFunction (GR_BLEND_ONE, GR_BLEND_ZERO,
			GR_BLEND_ONE, GR_BLEND_ZERO);
#else
  grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
		  GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE, FXFALSE);
#endif
  /* Setup Depthbuffering */
  grDepthBufferFunction (GR_CMP_LESS);

  /* build_window_list builds the list back to front so we render backwards */
  for (i = win_count-1; i >= 0; i--)
  {
#ifdef RENDER_WINDOW_NUMBERS
  /* Decal Texturing */
    grTexCombine (GR_TMU0, 
		  GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		  GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		  FXFALSE, FXFALSE);
    
    grTexSource (GR_TMU0, state->numbers_tex[i],
		 GR_MIPMAPLEVELMASK_BOTH, 
		 &state->numbers[i]);
#endif
    /*ConstantColorValue (color[i]);*/
    grDrawTriangle (&vtx[i][0], &vtx[i][1], &vtx[i][2]);
    grDrawTriangle (&vtx[i][0], &vtx[i][2], &vtx[i][3]);    
  }
#endif

  old_wincount = win_count;
}

/* Half Period of triangle movement, measured in microseconds */
#define HP 1000000

static void
render_spider (Demo_State *state, int ws_index)
{
  Vector position;
  Matrix translate, save;

  /* Save current model matrix */
  MatrixCopy (save, state->model);

  /* Setup the current matrix */  
  position[0] = (state->wstates[ws_index]->spider_position[0] +
		 state->wstates[ws_index]->world_upper_left[0]);
  position[1] = (state->wstates[ws_index]->spider_position[1] +
		 state->wstates[ws_index]->world_upper_left[1]);
  position[2] = (state->wstates[ws_index]->spider_position[2] +
		 state->wstates[ws_index]->world_upper_left[2]);
  /*#define TRACE_VERTS*/
#ifdef TRACE_VERTS
  printf ("unprojected corner is %f %f %f %f, w = %f\n",
	  state->wstates[ws_index]->world_upper_left[0],
	  state->wstates[ws_index]->world_upper_left[1],
	  state->wstates[ws_index]->world_upper_left[2],
	  state->wstates[ws_index]->world_upper_left[3],
	  1.f / vtx[ state->wstates[ws_index]->index ][0].oow);
  printf ("translate to %f %f %f\n", 
	  position[0], position[1], position[2]);
#endif

  /* Build the translation matrix */  

  TranslateMat (translate, position[0], position[1], position[2]);
  MatMultMat4x4 (state->model, state->spider_orientation, state->model);
  MatMultMat4x4 (state->model,  state->wstates[ws_index]->spider_align,
  		 state->model);
  MatMultMat4x4 (state->model, translate, state->model);

  /* Render the spider and increment the frame counter */
  draw_scene (state, state->spider_walk, state->wstates[ws_index]->spider_frame++);
  if (state->wstates[ws_index]->spider_frame > SPIDER_MAX_FRAME)
    state->wstates[ws_index]->spider_frame = SPIDER_MIN_FRAME;

  /* restore  current model matrix */
  MatrixCopy (state->model, save);
}

static void
render_tux (Demo_State *state)
{
  Matrix save, translate;
  Vector position;
  /* Save current model matrix */
  MatrixCopy (save, state->model);

  if (!state->tux_window)
    return;

  /* Place in his right place */
  position[0] = state->tux_position[0] + 
    state->tux_window->world_upper_left[0];
  position[1] = state->tux_position[1] + 
    state->tux_window->world_upper_left[1];
  position[2] = state->tux_position[2] + 
    state->tux_window->world_upper_left[2];

  TranslateMat (translate, position[0], position[1], position[2]);    
  MatMultMat4x4 (state->model, state->tux_orientation, state->model);
  MatMultMat4x4 (state->model, state->tux_scale, state->model);
  MatMultMat4x4 (state->model, translate, state->model);

  /* Draw the correct animation */
  if (state->tux_anim[state->tux_state])
  {
    draw_scene (state, state->tux_anim[state->tux_state], 
		state->tux_frame);
  }
  else
  {
    printf ("invalid animation\n");
  }

  /* Update tux's animation for the next frame */
  if (state->tux_window)
    update_tux (state);

  /* restore old  model matrix */
  MatrixCopy (state->model, save);
}

static void
render_bird (Demo_State *state)
{
  Matrix save, translate;
  Vector position;
  /* Save current model matrix */
  MatrixCopy (save, state->model);

  if (!state->bird_window)
    return;

  /* Place the bird in its right place */
  position[0] = state->bird_position[0] + 
    state->bird_window->world_upper_left[0];
  position[1] = state->bird_position[1] + 
    state->bird_window->world_upper_left[1];
  position[2] = state->bird_position[2] + 
    state->bird_window->world_upper_left[2];


  //#define TRACE_VERTS
#ifdef TRACE_VERTS
  printf ("translate to %f %f %f\n", 
	  position[0], position[1], position[2]);
#endif

  TranslateMat (translate, position[0], position[1], position[2]);    
  MatMultMat4x4 (state->model, state->bird_orientation, state->model);
  MatMultMat4x4 (state->model, state->bird_align, state->model);
  MatMultMat4x4 (state->model, state->bird_scale, state->model);
  MatMultMat4x4 (state->model, translate, state->model);


  draw_scene (state, state->bird_fly, state->bird_frame);

  /* Update tux's animation for the next frame */
  if (state->bird_window)
    update_bird (state);

  /* restore old  model matrix */
  MatrixCopy (state->model, save);  
}

void
render_decorations (Demo_State *state, Main_Window *mw)
{
  int i; 
  int ws_index;


  /* Process all of the windows */
  for (i = 0; i < win_count; i++)
  {
    /* Impose a minimum size on decorated windows */
    if (vtx[i][1].x - vtx[i][0].x >= MIN_WIN_WIDTH &&
    	vtx[i][2].y - vtx[i][0].y >= MIN_WIN_HEIGHT)
    {
      /* FIXME: this stuff should move out of render_decorations as it
       * needs to do work for lots of different effects.  */
      ws_index = ordered_states[i];

      /* Draw a spider on the window */
      render_spider (state, ws_index);
    }
  }


  /* Just render one tux */
  render_tux (state);

  /* And render one bird */
  render_bird (state);
}


void
render_background (Demo_State *state)
{
  switch (state->back)
  {
  case BACK_NONE:
    break;
  case BACK_SKY:
    render_sky (state);
    break;
  case BACK_WAVE:
    render_wave (state);
    break;
  case BACK_LAST:
    /* Don't use BACK_LAST */
    break;
  }
}

static void
render_sky (Demo_State *state)
{
  /* GrVertex
       x,y,z,
       r, g, b,
       ooz,
       a,
       oow,
       tmuvtx[]
     
     GrTmuVertex
       oow,
       sow,
       tow
  */
	



  /* Sky Polygons */
  static GrVertex vt[4] = {
    /* Upper Left */
    {
      0, 0, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{-128, 96,
	 1.f,
	},
	{ 0, 0,
	  1,
	}
      }
    },
    /* Upper Right */
    {
      DESKTOP_WIDTH, 0, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{ 128, 96,
	  1.f / (3000)
	},
	{ 128, 96,
	  1
	}
      }      
    },
    /* Lower Right */
    {
      DESKTOP_WIDTH, DESKTOP_HEIGHT, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{ 128, -96,
	  1.f / (3000)
	},
	{ 128, -96,
	  1.f / (3000)
	}
      }            
    },
    /* Lower Left */
    {
      0, DESKTOP_HEIGHT, 0,
      0, 255, 255,
      1,
      255,
      1.f / (3000),
      {
	{ -128, -96,
	  1.f / (3000)
	},
	{ -128, -96,
	  1
	}
      }                  
    }
  };
  static GrVertex v[4] = {
    /* Upper Left */
    {
      0, 0, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{ -1, 1,
	 1.f,
	},
	{ -1, 1,
	  1,
	}
      }
    },
    /* Upper Right */
    {
      DESKTOP_WIDTH, 0, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{ 1, 1,
	  1.f / (3000)
	},
	{ 1, 1,
	  1
	}
      }      
    },
    /* Lower Right */
    {
      DESKTOP_WIDTH, DESKTOP_HEIGHT, 0,
      0, 255, 255, 
      1,
      255,
      1.f / (3000),
      {
	{ 1, -1,
	  1.f / (3000)
	},
	{ 1, -1,
	  1.f / (3000)
	}
      }            
    },
    /* Lower Left */
    {
      0, DESKTOP_HEIGHT, 0,
      0, 255, 255,
      1,
      255,
      1.f / (3000),
      {
	{ -1, -1,
	  1.f / (3000)
	},
	{ -1, -1,
	  1
	}
      }                  
    }
  };

  static float theta = 0;
  int i;
  float sin_th, cos_th;
  /* The angular velocity in radians per microsecond */
#define AV (M_PI / (24 * 1000000))

  /* Transform texture cordinates for rotating background */
  theta += AV * state->delta_time;
  sin_th = sin(theta);
  cos_th = cos(theta);

  for (i = 0; i < 4; i++)
  {
    /* Set the depth to the desktop_w calculated when querying windows */
    
    vt[i].oow = state->one_over_desktop_w;
    vt[i].tmuvtx[0].sow = (v[i].tmuvtx[0].sow * 
			   state->background_texture_half_width * cos_th -
			   v[i].tmuvtx[0].tow * 
			   state->background_texture_half_height * sin_th + 
			   state->background_texture_half_width)
      * vt[i].oow;
    vt[i].tmuvtx[0].tow = (v[i].tmuvtx[0].sow *     
			   state->background_texture_half_width * sin_th + 
			   v[i].tmuvtx[0].tow 
			   * state->background_texture_half_height * cos_th 
			   + state->background_texture_half_width)
      * vt[i].oow;
    vt[i].tmuvtx[0].oow = vt[i].oow;
  }
  
  /* This background is used to clear the screen so always draw it */
  grDepthBufferFunction (GR_CMP_ALWAYS);

  /* Turn on texturing */
  grTexCombine (GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		FXFALSE, FXFALSE);
  grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
		  GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE,
		  FXFALSE);
  /* Tile the texture */
  grTexClampMode (GR_TMU0, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);

  /* Assume texture has already been downloaded to card */
  grTexSource (GR_TMU0, state->background_tex[state->background_index], 
	       GR_MIPMAPLEVELMASK_BOTH, 
	       &state->background[state->background_index]);

  grDrawTriangle (&vt[0], &vt[1], &vt[2]);
  grDrawTriangle (&vt[0], &vt[2], &vt[3]);

  /* Clear only the depth buffer.  This way my depth values don't
   * really matter.  I am always in back.  */
  grColorMask (FXFALSE, FXFALSE);
  grBufferClear (0, 0, GR_WDEPTHVALUE_FARTHEST);
  grColorMask (FXTRUE, FXTRUE);
}

static void
render_wave (Demo_State *state)
{
#define WAVE_WIDTH  10
#define WAVE_HEIGHT 8

  //static GrVertex surface [(WAVE_WIDTH + 1) * (WAVE_HEIGHT + 1)];
  int i, j;
  FxU32 offset = 0;
  //float dx = (DESKTOP_WIDTH)  / (float)(WAVE_WIDTH);
  const float dnx = 2 * M_PI / (float)(WAVE_WIDTH);
  //float dy = (DESKTOP_HEIGHT) / (float)(WAVE_HEIGHT);
  const float dny = 2 * M_PI / (float)(WAVE_HEIGHT);
  //float ds = 2 * state->background_texture_half_width / (float)(WAVE_WIDTH);
  //float dt = 2 * state->background_texture_half_height / (float)(WAVE_HEIGHT);

  float nx;
  float ny = 0;
  float w, oow;

  static float cur_time = 0;
  float time;
  float base_w = DESKTOP_DEPTH;
    //1.f / (state->one_over_desktop_w);

  cur_time += state->dt;//delta_time / 1000000.f;
  time = 0.25 * M_PI * cur_time;
  // (state->cur_frame_time.tv_sec + state->cur_frame_time.tv_usec/(1000000));


  /* Build the geometry */
  for (j = 0; j <= WAVE_HEIGHT; j++)
  {
    //x = 0;
    nx = 0;
    //s = 0;
    for (i = 0; i <= WAVE_WIDTH; i++)
    {
      /* Compute w */
      w = base_w + state->wave_amplitude * (sin (nx + time) + cos (ny + time));
      oow = 1.f / w;

      //wave[offset].x = x;
      //wave[offset].y = y;

      wave[offset].oow = oow;

      /*oow = state->one_over_desktop_w;*/
      //wave[offset].tmuvtx[0].sow = s * state->one_over_desktop_w;
      //wave[offset].tmuvtx[0].tow = t * state->one_over_desktop_w;

      //x += dx;      
      nx += dnx;
      //s += ds;

      offset++;
    }
    //y += dy;
    ny += dny;
    //t += dt;
  }

  /* This background is used to clear the screen so always draw it */
  grDepthBufferFunction (GR_CMP_ALWAYS);

  /* Turn on texturing */
  grTexCombine (GR_TMU0, GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		GR_COMBINE_FUNCTION_LOCAL, GR_COMBINE_FACTOR_NONE,
		FXFALSE, FXFALSE);
  grColorCombine (GR_COMBINE_FUNCTION_SCALE_OTHER, GR_COMBINE_FACTOR_ONE,
		  GR_COMBINE_LOCAL_NONE, GR_COMBINE_OTHER_TEXTURE,
		  FXFALSE);
  /* Tile the texture */
  grTexClampMode (GR_TMU0, GR_TEXTURECLAMP_WRAP, GR_TEXTURECLAMP_WRAP);

  /* Assume texture has already been downloaded to card */
  grTexSource (GR_TMU0, state->background_tex[state->background_index], 
	       GR_MIPMAPLEVELMASK_BOTH, 
	       &state->background[state->background_index]);


  /* Draw the mesh */
  offset = 0;
  for (j = 0; j < WAVE_HEIGHT; j++)
  {    
    for (i = 0; i < WAVE_WIDTH; i++)
    {
      grDrawTriangle (&wave[offset], &wave[offset+1],
		                     &wave[offset+1+WAVE_WIDTH+1]);
      grDrawTriangle (&wave[offset], 
		      &wave[offset+WAVE_WIDTH+1], &wave[offset+1+WAVE_WIDTH+1]);
      offset++;
    }
    /* Skip last vertex in each row since it doesn't start a triangle */
    offset++;
  }
  
  /* Clear only the depth buffer.  This way my depth values don't
   * really matter.  I am always in back.  */
  grColorMask (FXFALSE, FXFALSE);
  grBufferClear (0, 0, GR_WDEPTHVALUE_FARTHEST);
  grColorMask (FXTRUE, FXTRUE);
}

