

#include "types.h"
#include "vbrender.h"
#include "i810log.h"

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include "mm.h"
#include "i810lib.h"
#include "i810dd.h"
#include "i810clear.h"
#include "i810state.h"
#include "i810tris.h"
#include "i810ioctl.h"

#define DEPTH_SCALE 65535.0F

static void i810_clear_buffer( GLcontext *ctx, 
			       GLboolean all,
			       GLint cx, GLint cy, 
			       GLint cwidth, 
			       GLint cheight,
			       GLushort value,
			       GLuint offset ) 
{
   i810ContextPtr imesa = I810_CONTEXT( ctx );
   i810ScreenPrivate *i810Screen = imesa->i810Screen;
   __DRIdrawablePrivate *dPriv = imesa->driDrawable;

   int _nc = imesa->numClipRects;
   XF86DRIClipRectPtr box = imesa->pClipRects;

   cy = dPriv->h-cy-cheight;
   cx += imesa->drawX;
   cy += imesa->drawY;
         
   while (_nc--) {	 
      GLint x = box[_nc].x1;
      GLint y = box[_nc].y1;
      GLint width = box[_nc].x2 - x;
      GLint height = box[_nc].y2 - y;

      if (!all) {
	 if (x < cx) width -= cx - x, x = cx; 
	 if (y < cy) height -= cy - y, y = cy;
	 if (x + width > cx + cwidth) width = cx + cwidth - x;
	 if (y + height > cy + cheight) height = cy + cheight - y;
	 if (width <= 0) continue;
	 if (height <= 0) continue;
      }

      if (I810_DEBUG&DEBUG_VERBOSE_2D)
	 fprintf(stderr, "clear color rect %d,%d %dx%d\n", 
		 (int) x, (int) y, (int) width, (int) height);


      {
	 int start = (offset +  
		      y * i810Screen->backPitch + 
		      x * i810Screen->cpp);
	    
	 BEGIN_BATCH( imesa, 6 );
	    
	 OUT_BATCH( BR00_BITBLT_CLIENT | BR00_OP_COLOR_BLT | 0x3 );
	 OUT_BATCH( BR13_SOLID_PATTERN | 
		    (0xF0 << 16) |
		    i810Screen->backPitch );
	 OUT_BATCH( (height << 16) | (width * i810Screen->cpp));
	 OUT_BATCH( start );
	 OUT_BATCH( value );
	 OUT_BATCH( 0 );

	 ADVANCE_BATCH();
      }
   }
}




/*
 * i810Clear
 *
 * Clear the color and/or depth buffers.  If 'all' is GL_TRUE, clear
 * whole buffer, otherwise clear the region defined by the remaining
 * parameters.  
 */
GLbitfield i810Clear( GLcontext *ctx, GLbitfield mask, GLboolean all,
		      GLint cx, GLint cy, GLint cwidth, GLint cheight ) 
{
   i810ContextPtr imesa = I810_CONTEXT( ctx );
   const GLuint colorMask = *((GLuint *) &ctx->Color.ColorMask);

   if (I810_DEBUG&DEBUG_VERBOSE_API)
      fprintf(stderr, "i810Clear( %d, %d, %d, %d, %d )\n",
	      (int)mask, (int)cx, (int)cy, (int)cwidth, (int)cheight );
	
   FLUSH_BATCH(imesa);
   
   LOCK_HARDWARE(imesa);
   i810GetGeneralDmaBufferLocked( imesa );

   {
      BEGIN_BATCH( imesa, 2 );   
      OUT_BATCH( INST_PARSER_CLIENT | INST_OP_FLUSH );
      OUT_BATCH( 0 );
      ADVANCE_BATCH();      
   }

   if ( (mask & DD_FRONT_LEFT_BIT) && colorMask == ~0) {
      i810_clear_buffer( ctx, all, cx, cy, cwidth, cheight,
			 imesa->ClearColor,
			 imesa->i810Screen->fbOffset);
      mask &= ~DD_FRONT_LEFT_BIT;
   }

   if ( (mask & DD_BACK_LEFT_BIT) && colorMask == ~0) {
      i810_clear_buffer( ctx, all, cx, cy, cwidth, cheight, 
			 imesa->ClearColor,
			 imesa->i810Screen->backOffset);
      mask &= ~DD_BACK_LEFT_BIT;
   }

   if ( (mask & DD_DEPTH_BIT) && ctx->Depth.Mask ) {
      i810_clear_buffer( ctx, all, cx, cy, cwidth, cheight, 
			 ctx->Depth.Clear * DEPTH_SCALE,
			 imesa->i810Screen->depthOffset);
      mask &= ~DD_DEPTH_BIT;
   }      

   {
      BEGIN_BATCH( imesa, 2 );   
      OUT_BATCH( INST_PARSER_CLIENT | INST_OP_FLUSH );
      OUT_BATCH( 0 );
      ADVANCE_BATCH();      
   }

   i810FlushGeneralLocked(imesa);
   UNLOCK_HARDWARE( imesa );
   return mask;
}

