/*
    libparted - a library for manipulating disk partitions
    Copyright (C) 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <parted/parted.h>

static const PedConstraint* ped_constraint_none = NULL;

int
ped_constraint_init (
	PedConstraint* constraint,
	const PedAlignment* start_align,
	const PedAlignment* end_align,
	const PedGeometry* start_range,
	const PedGeometry* end_range,
	PedSector min_size)
{
	PED_ASSERT (constraint != NULL, return 0);
	PED_ASSERT (start_range != NULL, return 0);
	PED_ASSERT (end_range != NULL, return 0);
	PED_ASSERT (min_size > 0, return 0);

	if (start_range->start > end_range->end)
		return 0;
	if (end_range->end - start_range->start + 1 < min_size)
		return 0;

	constraint->start_align = ped_alignment_duplicate (start_align);
	constraint->end_align = ped_alignment_duplicate (end_align);
	constraint->start_range = ped_geometry_duplicate (start_range);
	constraint->end_range = ped_geometry_duplicate (end_range);
	constraint->min_size = min_size;

	return 1;
}

PedConstraint*
ped_constraint_new (
	const PedAlignment* start_align,
	const PedAlignment* end_align,
	const PedGeometry* start_range,
	const PedGeometry* end_range,
	PedSector min_size)
{
	PedConstraint*	constraint;

	constraint = (PedConstraint*) ped_malloc (sizeof (PedConstraint));
	if (!constraint)
		goto error;
	if (!ped_constraint_init (constraint, start_align, end_align,
			          start_range, end_range, min_size))
		goto error_free_constraint;
	return constraint;

error_free_constraint:
	ped_free (constraint);
error:
	return 0;
}

PedConstraint*
ped_constraint_duplicate (const PedConstraint* constraint)
{
	PED_ASSERT (constraint != NULL, return NULL);

	return ped_constraint_new (
		constraint->start_align,
		constraint->end_align,
		constraint->start_range,
		constraint->end_range,
		constraint->min_size);
}

PedConstraint*
ped_constraint_intersect (const PedConstraint* a, const PedConstraint* b)
{
	PedAlignment*	start_align;
	PedAlignment*	end_align;
	PedGeometry*	start_range;
	PedGeometry*	end_range;
	PedSector	min_size;
	PedConstraint*	constraint;

	if (!a || !b)
		return NULL;

	start_align = ped_alignment_intersect (a->start_align, b->start_align);
	if (!start_align)
		goto empty;
	end_align = ped_alignment_intersect (a->end_align, b->end_align);
	if (!end_align)
		goto empty_destroy_start_align;
	start_range = ped_geometry_intersect (a->start_range, b->start_range);
	if (!start_range)
		goto empty_destroy_end_align;
	end_range = ped_geometry_intersect (a->end_range, b->end_range);
	if (!end_range)
		goto empty_destroy_start_range;
	min_size = PED_MAX (a->min_size, b->min_size);

	constraint = ped_constraint_new (
			start_align, end_align, start_range, end_range,
			min_size);
	if (!constraint)
		goto empty_destroy_end_range;

	ped_alignment_destroy (start_align);
	ped_alignment_destroy (end_align);
	ped_geometry_destroy (start_range);
	ped_geometry_destroy (end_range);
	return constraint;

error:
	return NULL;

empty_destroy_end_range:
	ped_geometry_destroy (end_range);
empty_destroy_start_range:
	ped_geometry_destroy (start_range);
empty_destroy_end_align:
	ped_alignment_destroy (end_align);
empty_destroy_start_align:
	ped_alignment_destroy (start_align);
empty:
	return NULL;
}

void
ped_constraint_done (PedConstraint* constraint)
{
	PED_ASSERT (constraint != NULL, return);

	ped_alignment_destroy (constraint->start_align);
	ped_alignment_destroy (constraint->end_align);
	ped_geometry_destroy (constraint->start_range);
	ped_geometry_destroy (constraint->end_range);
}

void
ped_constraint_destroy (PedConstraint* constraint)
{
	if (constraint) {
		ped_constraint_done (constraint);
		ped_free (constraint);
	}
}

/* this could be easily implemented on top of ped_constraint_solve_nearest,
 * by passing the maximum geometry, with:
 * 	start == constraint->start_range->start, and
 * 	end == end_range->end
 *
 * However, this might make ped_constraint_solve_nearest() easier to
 * understand ;-)
 */
PedGeometry*
ped_constraint_solve_max (const PedConstraint* constraint)
{
	PedSector	start;
	PedSector	end;
	PedGeometry*	result;

	PED_ASSERT (constraint != NULL, return NULL);

	start = ped_alignment_align_down (
			constraint->start_align,
			constraint->start_range,
			constraint->start_range->start);

	end = ped_alignment_align_up (
			constraint->end_align,
			constraint->end_range,
			constraint->end_range->end);

	if (start == -1 || end == -1)
		return NULL;
	if (end - start + 1 < constraint->min_size)
		return NULL;

	result = ped_geometry_new (constraint->start_range->disk,
				   start, end - start + 1);
	PED_ASSERT (ped_constraint_is_solution (constraint, result),
		    return NULL);
	return result;
}

PedGeometry*
ped_constraint_solve_nearest (
	const PedConstraint* constraint, const PedGeometry* geom)
{
	PedSector	first_start_soln;
	PedSector	last_end_soln;
	PedSector	max_start;
	PedSector	min_end;
	PedGeometry*	start_range;
	PedGeometry*	end_range;
	PedGeometry	allowable_end;
	PedSector	start;
	PedSector	end;
	PedGeometry*	result;

	PED_ASSERT (constraint != NULL, return NULL);
	PED_ASSERT (geom != NULL, return NULL);
	PED_ASSERT (constraint->start_range->disk == geom->disk, return NULL);

	first_start_soln = ped_alignment_align_up (
			constraint->start_align,
			constraint->start_range,
			constraint->start_range->start);
	last_end_soln = ped_alignment_align_up (
			constraint->end_align,
			constraint->end_range,
			constraint->end_range->end);
	if (first_start_soln == -1 || last_end_soln == -1)
		goto error;

	max_start = last_end_soln - constraint->min_size + 1;
	min_end = first_start_soln + constraint->min_size - 1;

	if (max_start < constraint->start_range->start
	    || min_end > constraint->end_range->end)
		goto error;

	start_range = ped_geometry_duplicate (constraint->start_range);
	if (!start_range)
		goto error;
	ped_geometry_set_end (start_range,
			      PED_MIN (start_range->end, max_start));

	end_range = ped_geometry_duplicate (constraint->end_range);
	if (!end_range)
		goto error_free_start_range;
	ped_geometry_set_start (end_range, PED_MAX(end_range->start, min_end));

	start = ped_alignment_align_nearest (
			constraint->start_align, start_range, geom->start);
	end = ped_alignment_align_nearest (
			constraint->end_align, end_range,
			PED_MAX (geom->end, start + constraint->min_size - 1));

	PED_ASSERT (start != -1, goto error_free_start_range);
	PED_ASSERT (end != -1, goto error_free_start_range);

	/* hack - end may have been rounded down, so min_size constraint may
	 * not be satisfied - needs to be bumped up by
	 * constraint->end_align->grain_size
	 */
	if (end - start + 1 < constraint->min_size)
		end += constraint->end_align->grain_size;

	result = ped_geometry_new (start_range->disk, start, end - start + 1);
	if (!result)
		goto error_free_end_range;

	ped_geometry_destroy (start_range);
	ped_geometry_destroy (end_range);

	PED_ASSERT (ped_constraint_is_solution (constraint, result),
		    return NULL);
	return result;

error_free_end_range:
	ped_geometry_destroy (end_range);
error_free_start_range:
	ped_geometry_destroy (start_range);
error:
	return NULL;
}

int
ped_constraint_is_solution (const PedConstraint* constraint,
	       		    const PedGeometry* geom)
{
	PED_ASSERT (constraint != NULL, return 0);
	PED_ASSERT (geom != NULL, return 0);

	if (!ped_alignment_is_aligned (constraint->start_align, NULL,
				       geom->start))
		return 0;
	if (!ped_alignment_is_aligned (constraint->end_align, NULL, geom->end))
		return 0;
	if (!ped_geometry_test_sector_inside (constraint->start_range,
					      geom->start))
		return 0;
	if (!ped_geometry_test_sector_inside (constraint->end_range, geom->end))
		return 0;
	if (geom->length < constraint->min_size)
		return 0;
	return 1;
}

PedConstraint*
ped_constraint_any (const PedDisk* disk)
{
	PedGeometry	full_disk;

	if (!ped_geometry_init (&full_disk, disk, 0, disk->dev->length))
		return NULL;

	return ped_constraint_new (
			ped_alignment_any,
		       	ped_alignment_any,
			&full_disk,
			&full_disk,
		       	1);
}

PedConstraint*
ped_constraint_exact (const PedGeometry* geom)
{
	PedAlignment	start_align;
	PedAlignment	end_align;
	PedGeometry	start_sector;
	PedGeometry	end_sector;

	ped_alignment_init (&start_align, geom->start, 0);
	ped_alignment_init (&end_align, geom->end, 0);
	ped_geometry_init (&start_sector, geom->disk, geom->start, 1);
	ped_geometry_init (&end_sector, geom->disk, geom->end, 1);

	return ped_constraint_new (&start_align, &end_align,
				   &start_sector, &end_sector, 1);
}

