/*
 * linux/drivers/ide/arm/rapide.c
 *
 * Copyright (c) 2004 Peter Naulls
 *
 * Based loosely upon ARM Linux ICS IDE driver by Russell King and NetBSD RapIDE
 * driver by Mark Brinicombe, and information from the RISC OS sources
 * provided by Chris Honey
 *
 * Minor modifications to work with kernel 2.6 by Jim Hawkins
 */

#include <linux/module.h>
#include <linux/slab.h>
#include <linux/blkdev.h>
#include <linux/errno.h>
#include <linux/ide.h>
#include <linux/init.h>

#include <asm/ecard.h>

/* IDE drive registers */

#define PRIMARY_DRIVE_REGISTERS_OFFSET		(0x400080 >> 2)
#define PRIMARY_AUX_REGISTER_OFFSET		(0x400298 >> 2)
#define PRIMARY_DATA_REGISTER_OFFSET		(0x600080 >> 2)

#define SECONDARY_DRIVE_REGISTERS_OFFSET	(0x400000 >> 2)
#define SECONDARY_AUX_REGISTER_OFFSET		(0x400218 >> 2)
#define SECONDARY_DATA_REGISTER_OFFSET		(0x600000 >> 2)

#define CONTROL_REGISTERS_OFFSET                (0x200000 >> 2)


#define IRQ_MASK_REGISTER_OFFSET		0
#define IRQ_STATUS_REGISTER_OFFSET		0
#define IRQ_REQUEST_REGISTER_OFFSET		1

#define PRIMARY_IRQ_MASK			0x01
#define SECONDARY_IRQ_MASK			0x02
#define IRQ_MASK				(PRIMARY_IRQ_MASK | SECONDARY_IRQ_MASK)



struct cardinfo {
	unsigned int primary;
	unsigned int aux;
	unsigned int data;
	unsigned int mask;
};


static struct cardinfo __initdata rapide_regs[] = {
	{ PRIMARY_DRIVE_REGISTERS_OFFSET, PRIMARY_AUX_REGISTER_OFFSET,
	  PRIMARY_DATA_REGISTER_OFFSET, PRIMARY_IRQ_MASK  },
	{ SECONDARY_DRIVE_REGISTERS_OFFSET, SECONDARY_AUX_REGISTER_OFFSET,
	  SECONDARY_DATA_REGISTER_OFFSET, SECONDARY_IRQ_MASK }
};  


struct rapide_card {
	ide_hwif_t *hwif[2];
	int channel;
	int enabled;
	unsigned long base;
	unsigned long control;
	unsigned long data[2];
};


extern void rapide_bs_rm_4(char *data, char *buffer, u32 wcount);
extern void rapide_bs_wm_4(char *data, char *buffer, u32 wcount);


void rapide_input_data (ide_drive_t *drive, void *buffer, u32 wcount)
{
	ide_hwif_t *hwif	= HWIF(drive);
	char *data = (void *)((struct rapide_card *)hwif->hwif_data)->data[hwif->channel];

	rapide_bs_rm_4(data, buffer, wcount);
}


void rapide_output_data (ide_drive_t *drive, void *buffer, u32 wcount)
{
	ide_hwif_t *hwif	= HWIF(drive);
	char *data = (void *)((struct rapide_card *)hwif->hwif_data)->data[hwif->channel];

	rapide_bs_wm_4(data, buffer, wcount);
}


static void rapide_maskproc(ide_drive_t *drive, int mask)
{
	ide_hwif_t *hwif = HWIF(drive);
	struct rapide_card *card = hwif->hwif_data;
	unsigned long flags;

	local_irq_save(flags);

	card->channel = hwif->channel;

	if (card->enabled && !mask) {
 	   outb(IRQ_MASK, card->control + IRQ_MASK_REGISTER_OFFSET);
	} else {
           outb(0, card->control + IRQ_MASK_REGISTER_OFFSET);
	}

	local_irq_restore(flags);
}


static ide_hwif_t *rapide_find_hwif(unsigned long dataport)
{
	ide_hwif_t *hwif;
	int index;

	for (index = 0; index < MAX_HWIFS; ++index) {
		hwif = &ide_hwifs[index];
		if (hwif->io_ports[IDE_DATA_OFFSET] == dataport)
			return hwif;
	}

	for (index = 0; index < MAX_HWIFS; ++index) {
		hwif = &ide_hwifs[index];
		if (!hwif->io_ports[IDE_DATA_OFFSET])
			return hwif;
	}

	return NULL;
}


static inline int rapide_register(struct rapide_card *card, int irq, unsigned long regs, unsigned long control,
                                  ide_hwif_t **hwif_p)
{
        unsigned long port = card->base;
        int ide_reg;
        hw_regs_t *hw;
        ide_hwif_t *hwif = rapide_find_hwif(regs);

        if (!hwif) return -1;
        hw = &hwif->hw;

        memset(hw, 0, sizeof(hw_regs_t));
            
        for (ide_reg = IDE_DATA_OFFSET; ide_reg <= IDE_STATUS_OFFSET; ide_reg++) {
                hwif->io_ports[ide_reg] =
                hw->io_ports[ide_reg]   = port + regs;
                port++;
        }
        hwif->io_ports[IDE_CONTROL_OFFSET] =
        hw->io_ports[IDE_CONTROL_OFFSET]   = card->base + control;
        hw->dma = NO_DMA;

        hwif->irq =
        hw->irq      = irq;

        hwif->hwif_data   = card;
        hwif->channel     = card->channel;
        hwif->maskproc    = rapide_maskproc;

        hwif->noprobe     = 0;
        hwif->chipset     = ide_acorn;

        hwif->ata_input_data     = rapide_input_data;
        hwif->ata_output_data    = rapide_output_data;

        hwif->serialized = 1;

        card->hwif[card->channel] = hwif;

        probe_hwif_init(hwif);
        create_proc_ide_interfaces();        
        
        *hwif_p = hwif;
        return 0;

}


static void rapide_irqenable(struct expansion_card *ec, int irqnr)
{
        struct rapide_card *card = ec->irq_data;

        card->enabled = 1;

        outb(IRQ_MASK, card->control + IRQ_MASK_REGISTER_OFFSET);
}


static void rapide_irqdisable(struct expansion_card *ec, int irqnr)
{
        struct rapide_card *card = ec->irq_data;

        card->enabled = 0;

        outb(0, card->control + IRQ_MASK_REGISTER_OFFSET);
}


static int rapide_irqpending(struct expansion_card *ec)
{
	struct rapide_card *card = ec->irq_data;

	return inb(card->control + IRQ_REQUEST_REGISTER_OFFSET);
}


static const expansioncard_ops_t rapide_ops = {
	.irqenable	= rapide_irqenable,
	.irqdisable	= rapide_irqdisable,
	.irqpending	= rapide_irqpending
};


static int __init rapide_probe(struct expansion_card *ec, const struct ecard_id *id)
{
        int interface;
        struct rapide_card *card;
        unsigned long base;

        printk("RapIDE: IDE card driver for Yellowstone IDE Podule v2\n");

        if (!(card = kmalloc(sizeof(struct rapide_card), GFP_KERNEL)))
           return -ENOMEM;

                card->base             = base = ecard_address(ec, ECARD_EASI, ECARD_SYNC);
                card->control          = base + CONTROL_REGISTERS_OFFSET;

                ec->ops      = &rapide_ops;
                ec->irq_data = card;

                ecard_set_drvdata(ec, card);

                /* Enable IRQs */
                outb(~0, card->control + IRQ_MASK_REGISTER_OFFSET);

                for (interface = 0; interface < 2; interface++) {
                        ide_hwif_t *hwif;

                        card->data[interface] = ioaddr(base + rapide_regs[interface].data);
                        card->channel         = interface;

                        if (rapide_register(card, ec->irq, rapide_regs[interface].primary,
                                                                    rapide_regs[interface].aux, &hwif) != -1) {

                                printk("RapIDE: interface found for channel %d\n", hwif->channel);

                        }
                }
        return 0;

}

static void __devexit rapide_remove(struct expansion_card *ec) {
        struct rapide_card *card = ecard_get_drvdata(ec);

        if (card->hwif[0])
                card->hwif[0]->io_ports[IDE_DATA_OFFSET] = 0;
        if (card->hwif[1])
                card->hwif[1]->io_ports[IDE_DATA_OFFSET] = 0;

        kfree(card);
}


static void rapide_shutdown(struct expansion_card *ec)
{
        struct rapide_card *card = ecard_get_drvdata(ec);

        /* Disable interrupts */
        outb(0, card->control + IRQ_MASK_REGISTER_OFFSET);
}


static struct ecard_id __initdata rapide_cids[] = {
	{ MANU_YELLOWSTONE, PROD_YELLOWSTONE_RAPIDE32 },
	{ 0xffff, 0xffff }
};


static struct ecard_driver rapide_driver = {
	.probe		= rapide_probe,
	.remove		= __devexit_p(rapide_remove),
	.shutdown	= rapide_shutdown,
	.id_table	= rapide_cids,
};


int __init rapide_init(void)
{
	return ecard_register_driver(&rapide_driver);
}


__initcall(rapide_init);
