/*
 *
 * Beryl hardware/software detection system
 *
 * Copyright : (C) 2006 by Quinn Storm
 * E-mail    : quinn@beryl-project.org
 *
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 */
//libberylsettings header
#ifndef LIB_BERYL_SETTINGS
#define LIB_BERYL_SETTINGS

#define BERYL_SETTINGS_RELOAD_ATOM "_BERYL_SETTINGS_RELOAD"

#include <glib.h>
#include <beryl.h>


// as noted, all values in the structs are in the current locale, not C
// libberylsettings transparently handles translation back to C for
//        the options of type STRING and LIST[STRING] that have a list
//        of allowed values set.

typedef struct _BerylSettingsBackend
{
	gchar * name;
	gchar * short_desc;
	gboolean supports_integration;
} BerylSettingsBackend;

typedef struct _BerylSettingsPluginCategory
{
	const gchar * name;
	const gchar * short_desc;
	const gchar * long_desc;
	GSList * plugins;
} BerylSettingsPluginCategory;

typedef struct _BerylSettingsContext BerylSettingsContext;
typedef struct _BerylSetting BerylSetting;

typedef gboolean (*BerylSettingsContextReadInitFunc) (BerylSettingsContext * context);
typedef void (*BerylSettingsContextReadDoneFunc) (BerylSettingsContext * context);
typedef gboolean (*BerylSettingsContextWriteInitFunc) (BerylSettingsContext * context);
typedef void (*BerylSettingsContextWriteDoneFunc) (BerylSettingsContext * context);
typedef void (*BerylSettingsContextReadSettingFunc)
    (BerylSettingsContext * context, BerylSetting * setting);
typedef void (*BerylSettingsContextWriteSettingFunc)
    (BerylSettingsContext * context, BerylSetting * setting);

typedef gboolean (*BerylSettingGetIsIntegratedFunc) (BerylSetting * setting);
typedef gboolean (*BerylSettingGetIsReadOnlyFunc) (BerylSetting * setting);

typedef gboolean (*BerylSettingsInitBackendFunc) (BerylSettingsContext * context);
typedef gboolean (*BerylSettingsFiniBackendFunc) (BerylSettingsContext * context);

typedef GSList * (*BerylSettingsGetExistingProfilesFunc) (void);
typedef gboolean (*BerylSettingsDeleteProfileFunc) (gchar *);

typedef void (*BerylSettingChangedNotifyFunc) (BerylSettingsContext * context);

struct _BerylSettingsContext
{
    GSList   *  plugins;    // list_of(BerylSettingsPlugin *), including the core pseudo-plugin
                            //      which is identified by name==NULL
	BerylSettingsPluginCategory	 *	categories;	// array of (BerylSettingsPluginCategoryList *)
    gpointer    private_ptr;

    gpointer	backend_private_ptr;
	BerylSettingGetIsIntegratedFunc 		get_setting_is_integrated;
	BerylSettingGetIsReadOnlyFunc 			get_setting_is_read_only;
    BerylSettingsContextReadInitFunc		read_init;
    BerylSettingsContextReadDoneFunc		read_done;
    BerylSettingsContextWriteInitFunc		write_init;
    BerylSettingsContextWriteDoneFunc		write_done;
    BerylSettingsContextReadSettingFunc		read_setting;
    BerylSettingsContextWriteSettingFunc	write_setting;
	BerylSettingsInitBackendFunc	        backend_init;
	BerylSettingsFiniBackendFunc		backend_fini;
	BerylSettingsGetExistingProfilesFunc	get_existing_profiles;
	BerylSettingsDeleteProfileFunc			delete_profile;
	BerylSettingChangedNotifyFunc			setting_changed;
    gchar * backend;
    void * backend_dlhand;

	gchar		*profile; 			// settings profile name
	gboolean	de_integration; 	// integrate settings into the desktop environment
									// settings system
	GSList * changed_settings;		// list of BerylSetting* changed by last context_read()
	gboolean plugins_changed;
};



typedef struct _BerylSettingsSubGroup
{
    gchar 	* name;		//in current locale
	gchar   * desc;		//in current locale
    GSList	* settings;	//list of BerylSetting
} BerylSettingsSubGroup;

typedef struct _BerylSettingsGroup
{
    gchar	* name;		//in current locale
	gchar   * desc;		//in current locale
    GSList	* subGroups;	//list of BerylSettingsSubGroup
} BerylSettingsGroup;

typedef struct _BerylSettingsPlugin
{
    gchar    *  gettext_domain;     // gettext domain for the plugin
    gchar    *  name;
    gchar    *  hints;
    gchar    *  short_desc;         	// in current locale
    gchar    *  long_desc;          	// in current locale
	gchar	 *	category;				// simple name
	gchar	 *  filename;				// filename of the so
    GSList   *  load_after;         	// list_of(gchar *)
    GSList   *  load_before;        	// list_of(gchar *)
    GSList   *  provides;           	// list_of(gchar *)
    GSList   *  requires;           	// list_of(gchar *)
    GSList   *  settings;           	// list_of(BerylSetting)
    GSList   *  groups;		    		// list_of(BerylSettingsGroup)
    gpointer    private_ptr;
	BerylSettingsContext	*	context;
} BerylSettingsPlugin;

typedef enum _BerylSettingConflictType
{
	BERYL_SETTING_CONFLICT_TYPE_KEY,
	BERYL_SETTING_CONFLICT_TYPE_BUTTON,
	BERYL_SETTING_CONFLICT_TYPE_EDGE,
	BERYL_SETTING_CONFLICT_TYPE_ANY
} BerylSettingConflictType;

typedef struct _BerylSettingConflict
{
	GSList * settings; // settings that conflict over the binding
	BerylSettingConflictType type; // type of the conflict, note that a setting may show up again in another
								   // list for a different type
} BerylSettingConflict;

typedef enum _BerylSettingType
{
    BERYL_SETTING_TYPE_BOOL,
    BERYL_SETTING_TYPE_INT,
    BERYL_SETTING_TYPE_FLOAT,
    BERYL_SETTING_TYPE_STRING,
    BERYL_SETTING_TYPE_COLOR,
    BERYL_SETTING_TYPE_BINDING,
    BERYL_SETTING_TYPE_LIST,
    BERYL_SETTING_TYPE_COUNT
} BerylSettingType;

union _BerylSettingInfo;

typedef struct _BerylIntSettingInfo
{
    gint min;
    gint max;
} BerylIntSettingInfo;

typedef struct _BerylFloatSettingInfo
{
    gdouble min;
    gdouble max;
    gdouble precision;
} BerylFloatSettingInfo;

typedef struct _BerylStringSettingInfo
{
    GSList * allowed_values; //list_of(gchar *) in current locale
    GSList * raw_values;     //list_of(gchar *) in C locale
} BerylStringSettingInfo;

typedef struct _BerylBindingSettingInfo
{
    gboolean key;
    gboolean button;
    gboolean bell;
    gboolean edge;
} BerylBindingSettingInfo;

typedef struct _BerylBindingSettingArrayInfo
{
    gboolean array[4];
} BerylBindingSettingArrayInfo;

typedef struct _BerylListSettingInfo
{
    BerylSettingType    list_of_type;
    union _BerylSettingInfo    * list_of_info;
} BerylListSettingInfo;

typedef union _BerylSettingInfo
{
    BerylIntSettingInfo             for_int;
    BerylFloatSettingInfo           for_float;
    BerylStringSettingInfo          for_string;
    BerylBindingSettingInfo         for_bind;
    BerylBindingSettingArrayInfo    for_bind_as_array;
    BerylListSettingInfo            for_list;
} BerylSettingInfo;

typedef struct _BerylSettingColorValueColor
{
    gint red;
    gint green;
    gint blue;
    gint alpha;
} BerylSettingColorValueColor;

typedef struct _BerylSettingColorValueArray
{
    gint array[4];
} BerylSettingColorValueArray;

typedef union _BerylSettingColorValue
{
    BerylSettingColorValueColor color;
    BerylSettingColorValueArray array;
} BerylSettingColorValue;

typedef struct _BerylSettingValueBindingIsSetValue
{
    gboolean key;
    gboolean button;
} BerylSettingValueBindingIsSetValue;

typedef struct _BerylSettingValueBindingIsSetArray
{
    gboolean array[2];
} BerylSettingValueBindingIsSetArray;

typedef union _BerylSettingValueBindingIsSet
{
    BerylSettingValueBindingIsSetValue value;
    BerylSettingValueBindingIsSetArray array;
} BerylSettingValueBindingIsSet;

typedef struct _BerylSettingBindingValue
{
    BerylSettingValueBindingIsSet   enabled;
    gint                            button;
    gint                            button_mod_mask;
    gint                            keysym;
    gint                            key_mod_mask;
    gboolean                        on_bell;
    gint                            edge_mask;
} BerylSettingBindingValue;

typedef union _BerylSettingValueUnion
{
    gboolean                    as_bool;
    gint                        as_int;
    gdouble                     as_float;
    gchar                     * as_string;
    BerylSettingBindingValue    as_binding;
    BerylSettingColorValue      as_color;
    GSList                    * as_list;        // list_of(BerylSettingValue *)
} BerylSettingValueUnion;

typedef struct _BerylSettingValue
{
    BerylSettingValueUnion      value;
    BerylSetting              * parent;
    gboolean                    is_list_child;
} BerylSettingValue;

struct _BerylSetting
{
    BerylSettingType        type;
    gboolean                is_screen;        // support the 'screen/display' thing
    BerylSettingInfo        info;
    gchar                 * name;
    gchar                 * short_desc;        // in current locale
    gchar                 * long_desc;        // in current locale
    gchar	 	  * group;		// in current locale
    gchar		  * subGroup;		// in current locale
    gchar		  * displayHints;	// in current locale
	gboolean				advanced;
    BerylSettingValue       value;
    BerylSettingValue       default_value;
    gboolean                is_default; // never read setting->value unless this is set to FALSE
                                        // for composite values, if a set occurs and this is TRUE,
                                        // a copy will occur first into setting->value, then
                                        // the set will occur and this of course will be set to FALSE
    BerylSettingsPlugin   * parent;
    gpointer                private_ptr;
};

BerylSettingsContext * beryl_settings_context_new(void);
/*        This function creates a SettingsContext, and
        populates it with BerylSetting items from both
        core and all of the available plugins in
        PLUGINDIR and $HOME/.beryl/plugins, ignoring
        plugins in PLUGINDIR that have name conflicts
        with plugins in $HOME/.beryl/plugins
        The context is localized to the currently set locale
        whenever this function is called
        %return     A new BerylSettingsContext representing
                        the currently available plugins & core*/

gboolean beryl_settings_context_set_backend(BerylSettingsContext * context,
	gchar * backend);
/*	  Pass this function the canonical name of the
 *	backend you want to load, returns if it could.
 *	looks first in $HOME/.beryl/backends then in
 *	PLUGINDIR/backends/
 *	$context	the context to set the backend for
 *	$backend	the base name of the backend to load
 *	%return		wether a backend of that name could
 *			be found */

void beryl_settings_context_destroy(BerylSettingsContext * context);
/*        This function will not usually be needed, but
        will free the context and all its contents.
        $context    context to free.*/


GSList * beryl_settings_plugin_get_provides(BerylSettingsPlugin * plugin);
GSList * beryl_settings_plugin_get_requires(BerylSettingsPlugin * plugin);
GSList * beryl_settings_plugin_get_load_before(BerylSettingsPlugin * plugin);
GSList * beryl_settings_plugin_get_load_after(BerylSettingsPlugin * plugin);

GSList * beryl_settings_plugin_get_groups(BerylSettingsPlugin * plugin);
/*	This function returns a list of BerylSettingsGroup
 *	$plugin		the plugin to get the groups for
 *	%return		the list of groups */

BerylSettingsGroup * beryl_settings_plugin_find_group(BerylSettingsPlugin * plugin, gchar * group);
//FIXME NODOC

gchar * beryl_settings_group_get_name(BerylSettingsGroup * group);
/*	Get the name
 *	$group		the group
 *	%return		its name in current locale */

GSList * beryl_settings_group_get_subgroups(BerylSettingsGroup * group);
/*	This function returns a list of BerylSettingsSubGroup
 *	$group		the group to get subgroups of
 *	%return		the list of subgroups */

BerylSettingsSubGroup * beryl_settings_group_find_subgroup(BerylSettingsGroup * group, gchar * subgroup);
//FIXME NODOC

gchar * beryl_settings_subgroup_get_name(BerylSettingsSubGroup * group);
/*	Get the name
 *	$group		the subgroup
 *	%return		its name in current locale */

GSList * beryl_settings_subgroup_get_settings(BerylSettingsSubGroup * subGroup);
/*	This function returns the settings in a subgroup
 *	$group		the subgroups to get settings for
 *	%return		the list of settings */

BerylSetting * beryl_settings_subgroup_find_setting(BerylSettingsSubGroup * subGroup, gchar * setting);
	//FIXME NODOC

gboolean beryl_setting_has_hint(BerylSetting * setting, gchar * hint);
/*	This function returns wether a given display-hint is set
 *	on the setting in question
 *	$setting	the setting to look at
 *	$hint		the hint to look for
 *	%return		wether it is there */

BerylSettingsPlugin * beryl_settings_context_find_plugin(
                        BerylSettingsContext * context, gchar * name);
/*          This function looks up a given plugin in the SettingsContext.
    $context    settings context to look in
    $name       name of plugin to find, NULL to return a pseudo-
                    plugin representing core (it can be used anywhere
                    a regular BerylSettingsPlugin can)
    %return     a BerylSettingsPlugin or NULL if no such name exists
*/

BerylSetting * beryl_settings_plugin_find_setting(
                        BerylSettingsPlugin * plugin, gchar * name, gboolean is_screen);
/*          This function looks up a setting in a given plugin.
    $plugin     plugin to look the setting up in
    $name       name of the setting to look up
    $is_screen  wether the setting is a 'screen' setting or not
    %return     the setting structure matching the search criteria,
                    or NULL if it isn't found.*/

void beryl_settings_context_read(BerylSettingsContext * context);
/*          Reads settings in from $HOME/.beryl/settings
    $context    context to read settings into */

void beryl_settings_context_write(BerylSettingsContext * context);
/*          Writes any non-default values in context out to the
        $HOME/.beryl/settings, but does not send SIGUSR1 to any beryl
        processes.
    $context    context to write*/

gboolean beryl_settings_context_comp_get_option_value(BerylSettingsContext * context,
        gchar * plugin, gchar * name, gboolean is_screen, CompOptionValue * value);
/*          Retrieves the value of a setting into a CompOptionValue from
 *      a settings-plugin context
 *  $context    the context to get from
 *  $plugin     the plugin to look for, or NULL for the 'core'
 *  $name       the name of the option to get
 *  $is_screen  TRUE if a 'screen' type else FALSE
 *  $value      the value to set with the relevant info
 *  %return     TRUE if $value was set, FALSE if the option could not be found */

gboolean beryl_settings_context_comp_set_option_value(BerylSettingsContext * context,
        gchar * plugin, gchar * name, gboolean is_screen, CompOptionValue * value);
/*          Sets the value of a setting from a CompOptionValue
 *  $context    the context to set in
 *  $plugin     the plugin to look for, or NULL for 'core'
 *  $name       the name of the option to set
 *  $is_screen  TRUE if a 'screen' type, else FALSE
 *  $value      the value to set the setting to
 *  %return     TRUE if the setting was found and updated, else FALSE */

gboolean beryl_settings_send_reload_signal(void);
/*          Looks for running beryl processes owned by the calling UID,
        sends them SIGUSR1 if found.  The reason this is a separate
        function from write_options is for tools that choose not to
        or situations where it is adventageous not to reload beryl's
        options at the time of writing.
    %return     TRUE if any signals were sent, else FALSE*/

void beryl_setting_reset_to_default(BerylSetting * setting);
/*          frees any non-default value(s) attached to the setting
        and returns the setting to 'default' state, so it would
        not be written if settings were written out
    $setting    the setting to operate on*/

void beryl_setting_list_clear(BerylSetting * setting);
/*          Clears and frees the list of values attached to setting
      $setting  the setting (type LIST[*]) to clear*/

BerylSettingValue * beryl_setting_list_append(BerylSetting * setting);
/*    $setting  the LIST[*] setting to append a value to
      %return   the new, unset list value, to be used in
                    the set/get functions*/

gint beryl_setting_list_length(BerylSetting * setting);
/*    $setting  the LIST[*] setting to get the count from
      %return   the count of items in the LIST[*] setting*/

gint beryl_setting_list_value_index(BerylSettingValue * value);
/*    $value    the value, of a LIST[*] setting to get the index of
      %return   the index in the list of the given value*/

void beryl_setting_list_value_swap_with(BerylSettingValue * value, guint n);
/*    $value    the value, of a LIST[*] setting to swap
      $n        the index of the value to swap it with,
                    must be 0 -- (count(list)-1), if == index of
                    $value, does nothing.*/

void beryl_setting_list_value_move_before(BerylSettingValue * value,
                guint n);
/*    $value    the value, of a LIST[*] setting, to move
      $n        where in the list to reinsert the value,
                    if $n == index_of($value), does not move
                    the value, if $n == 0, inserts at top of list,
                    if $n == count(list), inserts at end of list.*/

void beryl_setting_list_value_remove(BerylSettingValue * value);
/*    $value    the value to remove from its LIST[*] option,
                    the value and its contents will be freed, so
                    do NOT attempt to operate on it after this.*/

BerylSettingValue * beryl_setting_get_primary_value(BerylSetting * setting);
/*    $setting  a BerylSetting to get the primary value for
      %return   a pointer to the primary BerylSettingValue for
                    the given setting, essentially meaningless for
                    LIST[*] types*/

/*gboolean beryl_setting_value_[get|set]_[$type](
                    BerylSettingValue * value, $value_type * data);
            these are functions that set a BerylSetting to
        a given value, or retrieve the value of a BerylSetting,
        performing proper input validation on the values,
        and return FALSE if something went wrong (bad input mostly)
        IT IS IMPORTANT TO USE THESE, as they do important things like
        free and alloc and copy strings.
        $value              the BerylSettingValue to operate on
        $data               the typed data to set the value to, anything
                            passed to a set function may be subsequently
                            freed as it will be copied, while anything
                            returned from a get function is internally owned
                            and should NEVER be altered or freed */

//these are value get functions
gboolean beryl_setting_value_get_int(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_float(BerylSettingValue * value, gdouble * data);
gboolean beryl_setting_value_get_bool(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_get_string(BerylSettingValue * value, const gchar ** data);
gboolean beryl_setting_value_get_keysym(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_keymods(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_button(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_buttonmods(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_bell(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_get_edgemask(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_get_key_enabled(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_get_button_enabled(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_get_color(BerylSettingValue * value, BerylSettingColorValue * data);

//returns list of BerylSettingValue's that is the list value
gboolean beryl_setting_value_get_value_list(BerylSettingValue * value, GSList ** data);

//these are value set functions
gboolean beryl_setting_value_set_int(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_float(BerylSettingValue * value, gdouble * data);
gboolean beryl_setting_value_set_bool(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_set_string(BerylSettingValue * value, const gchar ** data);
gboolean beryl_setting_value_set_raw_string(BerylSettingValue * value, const gchar ** data);
gboolean beryl_setting_value_set_keysym(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_keymods(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_button(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_buttonmods(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_bell(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_set_edgemask(BerylSettingValue * value, gint * data);
gboolean beryl_setting_value_set_key_enabled(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_set_button_enabled(BerylSettingValue * value, gboolean * data);
gboolean beryl_setting_value_set_color(BerylSettingValue * value, BerylSettingColorValue * data);

//these are info get functions
gboolean beryl_setting_get_can_set_key(BerylSetting * value, gboolean * data);
gboolean beryl_setting_get_can_set_button(BerylSetting * value, gboolean * data);
gboolean beryl_setting_get_can_set_edgemask(BerylSetting * value, gboolean * data);
gboolean beryl_setting_get_can_set_bell(BerylSetting * value, gboolean * data);
gboolean beryl_setting_get_allowed_strings(BerylSetting * value, GSList ** data);
gboolean beryl_setting_get_raw_allowed_strings(BerylSetting * value, GSList ** data);
gboolean beryl_setting_get_int_min(BerylSetting * value, gint * data);
gboolean beryl_setting_get_int_max(BerylSetting * value, gint * data);
gboolean beryl_setting_get_float_min(BerylSetting * value, gdouble * data);
gboolean beryl_setting_get_float_max(BerylSetting * value, gdouble * data);
gboolean beryl_setting_get_float_precision(BerylSetting * value, gdouble * data);
gboolean beryl_setting_get_list_type(BerylSetting * value, BerylSettingType * data);

//these are info functions
BerylSettingType beryl_setting_get_type(BerylSetting * setting);
const gchar * beryl_setting_get_short_desc(BerylSetting * setting);
const gchar * beryl_setting_get_long_desc(BerylSetting * setting);
const gchar * beryl_setting_get_group(BerylSetting * setting);
const gchar * beryl_setting_get_subgroup(BerylSetting * setting);
const gchar * beryl_setting_get_name(BerylSetting * setting);
const gchar * beryl_settings_plugin_get_short_desc(BerylSettingsPlugin * plugin);
const gchar * beryl_settings_plugin_get_long_desc(BerylSettingsPlugin * plugin);
const gchar * beryl_settings_plugin_get_name(BerylSettingsPlugin * plugin);

//context functions
GSList * beryl_settings_context_get_plugins(BerylSettingsContext * context);
GSList * beryl_settings_plugin_get_settings(BerylSettingsPlugin * plugin);

//privates
gpointer beryl_settings_context_get_private(BerylSettingsContext * context);
gpointer beryl_settings_plugin_get_private(BerylSettingsPlugin * plugin);
gpointer beryl_setting_get_private(BerylSetting * setting);
void beryl_settings_context_set_private(BerylSettingsContext * context, gpointer private_ptr);
void beryl_settings_plugin_set_private(BerylSettingsPlugin * plugin, gpointer private_ptr);
void beryl_setting_set_private(BerylSetting * setting, gpointer private_ptr);

//utility functions
unsigned int beryl_settings_get_mods_and_endptr(gchar * src, gchar ** ret);
gchar * beryl_settings_mods_to_string(unsigned int mods);

//FIXME - move where they belong
gboolean beryl_setting_get_is_advanced(BerylSetting * setting);

gboolean beryl_settings_context_import_from_file(BerylSettingsContext * context, gchar * filename, gboolean overwrite);
void beryl_settings_context_export_to_file(BerylSettingsContext * context, gchar * filename);

int beryl_settings_get_plugin_category_count(void);
BerylSettingsPluginCategory * beryl_settings_context_get_nth_plugin_category(BerylSettingsContext * context, int i);
BerylSettingsPluginCategory * beryl_settings_plugin_get_category(BerylSettingsPlugin * plugin);
const gchar * beryl_settings_plugin_category_get_name(BerylSettingsPluginCategory * category);
const gchar * beryl_settings_plugin_category_get_short_desc(BerylSettingsPluginCategory * category);
const gchar * beryl_settings_plugin_category_get_long_desc(BerylSettingsPluginCategory * category);
GSList * beryl_settings_plugin_category_get_plugins(BerylSettingsPluginCategory * category);

void beryl_settings_context_set_profile(BerylSettingsContext * context, gchar * profile);
gchar * beryl_settings_context_get_profile(BerylSettingsContext * context);

void beryl_settings_context_set_de_integration_enabled(BerylSettingsContext * context, gboolean enable_integration);

gboolean beryl_settings_context_get_de_integration_enabled(BerylSettingsContext * context);

gboolean beryl_settings_context_get_setting_is_read_only(BerylSettingsContext * context, BerylSetting * setting);
gboolean beryl_settings_context_get_setting_is_de_integrated(BerylSettingsContext * context, BerylSetting * setting);

gchar * beryl_settings_context_get_backend(BerylSettingsContext * context);

GSList * beryl_settings_context_get_existing_profiles(BerylSettingsContext * context);
	//the NULL profile is ALWAYS available so don't bother listing it
	//MAKE SURE TO FREE THE RESULTS! (use the following function ^-^)
void beryl_settings_free_profile_list(GSList * list);

void beryl_settings_set_codeset(const char*);

gchar * beryl_settings_group_get_desc(BerylSettingsGroup * group);
gchar * beryl_settings_subgroup_get_desc(BerylSettingsSubGroup * subGroup);

GSList * beryl_settings_context_find_conflicts(BerylSettingsContext * context, BerylSettingConflictType type);
GSList * beryl_settings_context_find_conflicts_for_setting(BerylSettingsContext * context, BerylSetting * setting, BerylSettingConflictType type);
void beryl_settings_free_conflict_list(GSList * list);
BerylSetting * beryl_settings_context_find_first_edge_owner(BerylSettingsContext * context, int edgemask);
BerylSettingsPlugin * beryl_setting_get_plugin(BerylSetting * setting);
BerylSettingConflictType beryl_setting_conflict_get_type(BerylSettingConflict * c);
GSList * beryl_setting_conflict_get_settings(BerylSettingConflict * c);
GSList * beryl_settings_context_get_active_plugins(BerylSettingsContext * c);
gboolean beryl_settings_plugin_enable(BerylSettingsPlugin * p);
gboolean beryl_settings_plugin_disable(BerylSettingsPlugin * p);
gboolean beryl_settings_plugin_get_is_enabled(BerylSettingsPlugin * p);
gchar * beryl_settings_plugin_get_filename(BerylSettingsPlugin * p);
GSList * beryl_settings_context_get_settings_changed(BerylSettingsContext * c);
gboolean beryl_settings_context_get_if_plugins_changed(BerylSettingsContext * c);
GSList * beryl_settings_get_backends(void);
gchar * beryl_settings_backend_get_name(BerylSettingsBackend * b);
gchar * beryl_settings_backend_get_short_desc(BerylSettingsBackend * b);
gboolean beryl_settings_backend_get_supports_integration(BerylSettingsBackend * b);
gboolean beryl_settings_delete_profile(BerylSettingsContext * c, gchar * profile);


void beryl_setting_add_nofify(BerylSettingsContext * context, BerylSettingChangedNotifyFunc function);

// for backend use only
gboolean beryl_setting_changed(BerylSetting *setting);

#endif
