"""
Stores application settings

Part of this code copied from Gimmie (c) Alex Gravely

Copyright: John Stowers, 2006
License: GPLv2
"""
import re
import os
import gobject

try:
    import gconf
except ImportError:
    from gnome import gconf

import logging
log = logging.getLogger("Settings")

#import gnomekeyring
#import conduit

#these dicts are used for mapping config setting types to type names
#and back again (isnt python cool...)
TYPE_TO_TYPE_NAME = {
    int     :   "int",
    bool    :   "bool",
    str     :   "string",
    list    :   "list"
}
STRING_TO_TYPE = {
    "int"       :   lambda x: int(x),
    "bool"      :   lambda x: string_to_bool(x),
    "string"    :   lambda x: str(x),
    "list"      :   lambda x: string_to_list(x)
}
TYPE_TO_STRING = {
    int     :   lambda x: str(x),
    bool    :   lambda x: str(x),
    str     :   lambda x: str(x),
    list    :   lambda x: list_to_string(x)
}

def string_to_bool(stringy):
    #Because bool("False") doesnt work as expected when restoring strings
    if stringy == "True":
        return True
    else:
        return False
    
def list_to_string(listy):
    s = ""
    if type(listy) is list:
        s = ",".join(listy) #cool
    return s
    
def string_to_list(string, listInternalVtype=str):
    l = string.split(",")
    internalTypeName = TYPE_TO_TYPE_NAME[listInternalVtype]
    for i in range(0, len(l)):
        l[i] = STRING_TO_TYPE[internalTypeName](l[i])
    return l

class Settings(gobject.GObject):
    """
    Class for storing conduit.GLOBALS.settings. Keys of type str, bool, int, 
    and list of strings supported at this stage.
    
    Also stores the special proxy settings.
    """
    __gsignals__ = {
        'changed' : (gobject.SIGNAL_RUN_LAST | gobject.SIGNAL_DETAILED, gobject.TYPE_NONE, ()),
    }

    #Default values for conduit settings
    DEFAULTS = {
        'show_splashscreen'         :   True,           #The splashscreen can be quite useful on slow computers
        'show_dp_description'       :   False,          #Should the treeview show the dataprovider description
        'show_status_icon'          :   True,           #Show an icon in the notification area indicating if a sync is running
        'save_on_exit'              :   True,           #Is the sync set saved on exit automatically?
        'enable_network'            :   True,           #Should conduit look for other conduits on the local network
        'enable_removable_devices'  :   True,           #Should conduit support iPods, USB keys, etc
        'default_policy_conflict'   :   "ask",          #Default conflict policy for new Conduits, ask,replace,skip
        'default_policy_deleted'    :   "ask",          #Default deleted policy for new Conduits, ask,replace,skip
        'gui_expanded_rows'         :   [],             #list of expanded column paths in the treeview
        'gui_restore_expanded_rows' :   True,           #Shoud we expand columns at startup
        'gui_hpane_postion'         :   250,            #The hpane seperating the canvas and treeview position
        'gui_window_size'           :   [],             #W,H
        'gui_minimize_to_tray'      :   False,          #Behaviour when one minimizes the main window, should it iconify?
        'gui_initial_canvas_height' :   450,            #Reduce to ~300 for eepc, etc
        'gui_initial_canvas_width'  :   450,            #Reduce for eepc, etc
        'gui_use_rgba_colormap'     :   False,          #Seems to corrupt gtkmozembed on some systems
        'web_login_browser'         :   "gtkmozembed"   #When loggin into websites use: "system","gtkmozembed","webkit","gtkhtml"
    }
    CONDUIT_GCONF_DIR = "/apps/conduit/"
        
    def __init__(self):
        """
        @param xmlSettingFilePath: The path to the xml file in which to store
        the per-conduit settings
        @type xmlSettingFilePath: C{string}
        """
        gobject.GObject.__init__(self)
        self.client = gconf.client_get_default()
        #Preload gconf directories
        self.client.add_dir(self.CONDUIT_GCONF_DIR[:-1], gconf.CLIENT_PRELOAD_RECURSIVE)  
        self.notifications = []
        #also keep an internal dict of settings which have been overridden
        #for this session
        self.overrides = {}

    def _fix_key(self, key):
        """
        Appends the CONDUIT_GCONF_PREFIX to the key if needed
        
        @param key: The key to check
        @type key: C{string}
        @returns: The fixed key
        @rtype: C{string}
        """
        if not key.startswith(self.CONDUIT_GCONF_DIR):
            return self.CONDUIT_GCONF_DIR + key
        else:
            return key
            
    def _key_changed(self, client, cnxn_id, entry, data=None):
        """
        Callback when a gconf key changes
        """
        key = self._fix_key(entry.key)
        detailed_signal = 'changed::%s' % key
        self.emit(detailed_signal)
        
    def set_overrides(self, **overrides):
        self.overrides = overrides

    def get(self, key, vtype=None, default=None):
        """
        Returns the value of the key or the default value if the key is 
        not yet in gconf
        """
        #check if the setting has been overridden for this session
        if key in self.overrides:
            try:
                #try and cast to correct type
                return type(self.DEFAULTS[key])(self.overrides[key])
            except:
                return self.overrides[key]

        if key in self.DEFAULTS:
            #function arguments override defaults
            if default is None:
                default = self.DEFAULTS[key]
            if vtype is None:
                vtype = type(default)

        #for gconf refer to the full key path
        key = self._fix_key(key)

        if key not in self.notifications:
            self.client.notify_add(key, self._key_changed)
            self.notifications.append(key)
        
        value = self.client.get(key)
        if not value:
            self.set(key, default, vtype)
            return default

        if vtype is bool:
            return value.get_bool()
        elif vtype is str:
            return value.get_string()
        elif vtype is int:
            return value.get_int()
        elif vtype in [list, tuple]:
            l = []
            for i in value.get_list():
                l.append(i.get_string())
            return l
            
        log.warn("Unknown gconf key: %s" % key)
        return None

    def set(self, key, value, vtype=None):
        """
        Sets the key value in gconf and connects adds a signal 
        which is fired if the key changes
        """
        #overidden settings only apply for this session, and are
        #not set
        if key in self.overrides:
            return True

        log.debug("Settings %s -> %s" % (key, value))
        if key in self.DEFAULTS and not vtype:
            vtype = type(self.DEFAULTS[key])

        #for gconf refer to the full key path
        key = self._fix_key(key)

        if vtype is bool:
            self.client.set_bool(key, value)
        elif vtype is str:
            self.client.set_string(key, value)
        elif vtype is int:
            self.client.set_int(key, value)
        elif vtype in [list, tuple]:
            #Save every value as a string
            strvalues = [str(i) for i in value]
            self.client.set_list(key, gconf.VALUE_STRING, strvalues)
        else:
            log.warn("Unknown gconf type (k:%s v:%s t:%s)" % (key,value,vtype))
            return False

        return True
        
    def proxy_enabled(self):
        """
        @returns: True if the user has specified a http proxy via
        the http_proxy environment variable, or in gconf
        """
        return os.environ.has_key("http_proxy") or \
                self.client.get_bool("/system/http_proxy/use_http_proxy")
        
    def get_proxy(self):
        """
        Returns the details of the configured http proxy. 
        The http_proxy environment variable overrides the GNOME setting
        @returns: host,port,user,password
        """
        if self.proxy_enabled():
            #env vars have preference
            if os.environ.has_key("http_proxy"):
                #re taken from python boto
                pattern = re.compile(
                    '(?:http://)?' \
                    '(?:(?P<user>\w+):(?P<pass>.*)@)?' \
                    '(?P<host>[\w\-\.]+)' \
                    '(?::(?P<port>\d+))?'
                )
                match = pattern.match(os.environ['http_proxy'])
                if match:
                    return (match.group('host'),
                            int(match.group('port')),
                            match.group('user'),
                            match.group('pass'))
            #now try gconf
            if self.client.get_bool("/system/http_proxy/use_authentication"):
                return (self.client.get_string("/system/http_proxy/host"),
                        self.client.get_int("/system/http_proxy/port"),
                        self.client.get_string("/system/http_proxy/authentication_user"),
                        self.client.get_string("/system/http_proxy/authentication_password"))
            else:
                return (self.client.get_string("/system/http_proxy/host"),
                        self.client.get_int("/system/http_proxy/port"),
                        "",
                        "")

        return ("",0,"","")
                            
                
        

        
        

