/*
 *
 * nucleo/network/HttpMessage.java --
 *
 * Copyright (C) Nicolas Roussel
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 */

package fr.lri.insitu.nucleo.network ;

// import gnu.regexp.* ;

import java.util.* ;
import java.util.regex.* ;

public class HttpMessage {

    public static final int NEED_STARTLINE = 0 ;
    public static final int NEED_MP_BOUNDARY = 1 ;
    public static final int NEED_MP_HEADERS = 2 ;
    public static final int NEED_BODY = 3 ;
    public static final int COMPLETE = 4 ;
    public static final String[] STATES = {"NEED_STARTLINE","NEED_MP_BOUNDARY","NEED_MP_HEADERS","NEED_BODY","COMPLETE"} ;

    private static final String oneCR = "\015" ;
    private static final String twoCR = "\015\015" ;
    private static final String oneLF = "\012" ;
    private static final String twoLF = "\012\012" ;
    private static final String oneCRLF = "\015\012" ;
    private static final String twoCRLF = "\015\012\015\012" ;

    private static final Hashtable _parseHeaders(String textblock, String eol) {
	   Hashtable h = new Hashtable() ;
	   try {
		  StringTokenizer hlines = new StringTokenizer(textblock,eol) ;
		  Pattern headerExp = Pattern.compile("^([^:]+):[ \t]*(.+)$") ;
		  while (hlines.hasMoreTokens()) {
			 String header = hlines.nextToken() ;
			 Matcher m = headerExp.matcher(header) ;
			 if (m.matches()) h.put(m.group(1), m.group(2)) ;
			 else System.out.println("Problem with pseudo-header '"+header+"'") ;
		  }
	   } catch (Exception e) {
		  System.out.println("Merde : "+e) ;
	   }
	   return h ;
    }
  
    // ------------------------------------------------------------------

    private String _data = "" ;
    private int _state = NEED_BODY ;

    private boolean _isMultipart = false ;
    private String _multipartBoundary = null ;
    private Hashtable _multipartHeaders = null ;

    private String _eol = null ;
    private String _meol = null ;
    private int _contentLength = -1 ;
  
    private String _startline = null ;
    private Hashtable _headers = null ;
    private String _body = null ; 

    // ------------------------------------------------------------------

    public HttpMessage() {
	   _eol = oneCRLF ; 
	   _meol = oneCRLF ; 
	   reset() ;
    }

    // ------------------------------------------------------------------

    private int _parseStartLineAndHeaders() {
	   if (_isMultipart) return NEED_MP_BOUNDARY ;

	   int p = _data.indexOf(twoCRLF) ;
	   if (p!=-1) _eol = oneCRLF ;
	   else {
		  p = _data.indexOf(twoLF) ;
		  if (p!=-1) _eol = oneLF ;
		  else {
			 p = _data.indexOf(twoCR) ;
			 if (p!=-1) _eol = oneCR ; 
			 else return _state ;
		  }
	   }

	   String textblock = _data.substring(0,p) ;
	   _data = _data.substring(p+2*_eol.length()) ;

	   p = textblock.indexOf(_eol) ;
	   if (p!=-1) {
		  _startline = textblock.substring(0,p) ;
		  textblock = textblock.substring(p+_eol.length()) ;
		  _headers = _parseHeaders(textblock, _eol) ;
	   } else _startline = textblock ;

	   return NEED_MP_BOUNDARY ;
    }

    private int _skipBoundary() {
	   if (_isMultipart) {
		  int p = _data.indexOf(_multipartBoundary) ;
		  if (p!=-1) {
			 _data = _data.substring(p+_multipartBoundary.length()) ;
			 return NEED_MP_HEADERS ;
		  }
		  return _state ;
	   }
	   return NEED_MP_HEADERS ;
    }

    private int _parseMultipartHeaders() {
	   if (_isMultipart) {
		  int p = _data.indexOf(twoCRLF) ;
		  if (p!=-1) _meol = oneCRLF ;
		  else {
			 p = _data.indexOf(twoLF) ;
			 if (p!=-1) _meol = oneLF ;
			 else {
				p = _data.indexOf(twoCR) ;
				if (p!=-1) _meol = oneCR ; 
				else return _state ;
			 }
		  }

		  String textblock = _data.substring(0,p) ;
		  _data = _data.substring(p+2*_meol.length()) ;

		  _multipartHeaders = _parseHeaders(textblock, _meol) ;
		  return NEED_BODY ;

	   }
	   return NEED_BODY ;
    }

    int _parseBody() {
	   if (_contentLength!=-1) {
		  int needed = _contentLength-_body.length() ;
		  int ready = _data.length() ;
		  if (needed>=ready) {
			 _body = _body + _data ;
			 _data = "" ;
		  } else {
			 _body = _body + _data.substring(0,needed) ;
			 _data = _data.substring(needed+1) ;
		  }
		  if (_body.length()==_contentLength) return COMPLETE ;
	   } else if (_isMultipart) {
		  int p = _data.indexOf(_multipartBoundary) ;
		  if (p!=-1) {
			 _body = _body + _data.substring(0,p) ;
			 _data = _data.substring(p+_multipartBoundary.length()) ;
		  } else {
			 _body = _body + _data ;
			 _data = "" ;
		  }
	   } else {
		  _body = _body + _data ;
		  _data = "" ;
		  if (_startline.indexOf("GET")!=-1 || _startline.indexOf("HEAD")!=-1) return COMPLETE ;
	   }
	   return _state ;
    }

    // ------------------------------------------------------------------

    public void reset() {
	   _state = NEED_STARTLINE ;

	   _isMultipart = false ;
	   _multipartBoundary = null ;
	   _multipartHeaders = new Hashtable() ;
  
	   _contentLength = -1 ;

	   _startline = "" ;
	   _headers = new Hashtable() ;
	   _body = "" ;
    }

    public void next() {
	   if (_isMultipart) {
		  _state = NEED_STARTLINE ;
		  _multipartHeaders = new Hashtable() ;
		  _contentLength = -1 ;
		  _body = "" ;
	   } else reset() ;
    }

    public void feed(String s) {
	   _data = _data + s ;
    }

    public void feed(byte[] buffer, int n) {
	   String s = new String(buffer, 0, n) ;
	   _data = _data + s ;
    }

    public int parse() {
	   for (;;) {
		  int newstate = _state ;
		  // System.out.println(STATES[_state]) ;

		  switch (_state) {
		  case NEED_STARTLINE:
			 newstate = _parseStartLineAndHeaders() ;
			 break ;
		  case NEED_MP_BOUNDARY:
			 newstate = _skipBoundary() ;	 
			 break ;
		  case NEED_MP_HEADERS:
			 newstate = _parseMultipartHeaders() ;	 
			 break ;
		  case NEED_BODY:
			 newstate = _parseBody() ;
			 break ;
		  case COMPLETE:
			 newstate = COMPLETE ;
			 break ;
		  }

		  if (newstate==_state) break ;
	 
		  switch(newstate) {
		  case NEED_STARTLINE:
			 break ;
		  case NEED_MP_BOUNDARY: {
			 if (! _isMultipart) {
				Object value = getHeaderValue("Content-Type") ;
				if (value!=null) {
				    String v = value.toString() ;
				    int p = v.indexOf("multipart/x-mixed-replace") ;
				    if (p!=-1) {
					   v = v.substring(p) ;
					   p = v.indexOf("boundary=") ;
					   if (p!=-1) {
						  _isMultipart = true ;
						  v = v.substring(p+9) ;
						  v.trim() ;
						  _multipartBoundary = v ;
					   }
				    }
			   
				}
			 }
		  } break ;
		  case NEED_MP_HEADERS:
			 break ;
		  case NEED_BODY: {
			 Object value = getHeaderValue("Content-Length") ;
			 if (value!=null) _contentLength = Integer.parseInt(value.toString()) ;
		  } break ;
		  case COMPLETE:
			 break ;
		  }

		  _state = newstate ;
	   }
	   return _state ;
    }

    public Object getHeaderValue(String key) {
	   Object value = null ;

	   if (_isMultipart) {
		  value = _multipartHeaders.get(key) ;
		  if (value!=null) return value ;
		  for (Enumeration e=_multipartHeaders.keys(); e.hasMoreElements(); ) {
			 String k = String.valueOf(e.nextElement()) ;
			 if (k.compareToIgnoreCase(key)==0) return _multipartHeaders.get(k) ;
		  }
	   }

	   value = _headers.get(key) ;
	   if (value!=null) return value ;
	   for (Enumeration e=_headers.keys(); e.hasMoreElements(); ) {
		  String k = String.valueOf(e.nextElement()) ;
		  if (k.compareToIgnoreCase(key)==0) return _headers.get(k) ;
	   }

	   return null ;
    }

    public void setHeaderValue(String key, Object value) {
	   // XXX Incomplete...
	   _headers.put(key,value) ;
    }

    public byte[] getBodyBytes() {
	   return _body.getBytes() ;
    }

    public void debug() {
	   System.out.println("---------------------------------------------------------------") ;

	   System.out.print("_eol      : ") ;
	   if (_eol.equals(oneCRLF)) System.out.println("CRLF") ;
	   else if (_eol.equals(oneCR)) System.out.println("CR") ;
	   else if (_eol.equals(oneLF)) System.out.println("LF") ;
	   System.out.print("_meol     : ") ;
	   if (_meol.equals(oneCRLF)) System.out.println("CRLF") ;
	   else if (_meol.equals(oneCR)) System.out.println("CR") ;
	   else if (_meol.equals(oneLF)) System.out.println("LF") ;

	   System.out.println() ;
	   System.out.println("STARTLINE : "+_startline) ;
	   System.out.println("HEADERS   : "+_headers) ;
	   if (_isMultipart) {
		  System.out.println("MBOUNDARY : "+_multipartBoundary) ;
		  System.out.println("MHEADERS  : "+_multipartHeaders) ;
	   }
	   System.out.println("CLENGTH   : "+_contentLength) ;
	   System.out.println("BODY      : ") ;
	   System.out.println(_body) ;

	   System.out.println("---------------------------------------------------------------") ;
    }
}
