/*
 * QtCurve GTK engine
 *
 * This is a hacked/modified version of the B***Curve GTK engine. Modified to use
 * ~/.qt/qtrc to get color information to recolor widgets.
 *
 * Changes are (C) Craig Drummond 2003 - Craig.Drummond@lycos.co.uk
 *
 */

#include "common.h"
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#define QTC_MAX_FILENAME_LEN   1024
#define QTC_MAX_INPUT_LINE_LEN 1024
#define QTC_FILE               "qtcurvestylerc"

static const char * getHome()
{
    static const char *home=NULL;

    if(!home)
    {
        struct passwd *p=getpwuid(geteuid());

        if(p)
            home=p->pw_dir;
        else
        {
            char *env=getenv("HOME");

            if(env)
                home=env;
        }

        if(!home)
            home="/tmp";
    }

    return home;
}

#ifdef CONFIG_READ
static int ctoh(char ch)
{
    return (ch>='0' && ch<='9') ? ch-'0' :
           (ch>='a' && ch<='f') ? 10+(ch-'a') :
           (ch>='A' && ch<='F') ? 10+(ch-'A') :
           0;
}

#define ATOH(str) ((ctoh(*str)<<4)+ctoh(*(str+1)))

static void set_rgb(qtc_color *col, char *str)
{
    if(str && strlen(str)>5)
    {
#ifdef __cplusplus
        col->setRgb(ATOH(str), ATOH(&str[2]), ATOH(&str[4]));
#else
        col->red=ATOH(str)<<8;
        col->green=ATOH(&str[2])<<8;
        col->blue=ATOH(&str[4])<<8;
        col->pixel=0;
#endif
    }
    else
#ifdef __cplusplus
        col->setRgb(0, 0, 0);
#else
        col->red=col->green=col->blue=col->pixel=0;
#endif
}

static EDefBtnIndicator qtc_to_ind(const char *str)
{
    if(0==memcmp(str, "fontcolor", 9) || 0==memcmp(str, "fontcolour", 10) || 0==memcmp(str, "border", 6))
        return IND_FONT_COLOR;
    if(0==memcmp(str, "none", 4))
        return IND_NONE;
    if(0==memcmp(str, "corner", 6))
        return IND_CORNER;
    return IND_COLORED;
}

static ELine qtc_to_line(const char *str)
{
    if(0==memcmp(str, "raised", 6))
        return LINE_RAISED;

    if(0==memcmp(str, "dashes", 6))
        return LINE_DASHES;

    if(0==memcmp(str, "none", 4))
        return LINE_NONE;

    if(0==memcmp(str, "sunken", 6))
        return LINE_SUNKEN;
    return LINE_DOTS;
}

static ETBarBorder qtc_to_tbar_border(const char *str)
{
    if(0==memcmp(str, "dark", 4))
        return TB_DARK;
    if(0==memcmp(str, "none", 4))
        return TB_NONE;
    return TB_LIGHT;
}

static ELvExpander qtc_to_lv_expander(const char *str)
{
    return 0==memcmp(str, "arrow", 5) ? LV_EXP_ARR : LV_EXP_PM;
}

static ELvLines qtc_to_lv_lines(const char *str)
{
    if(0==memcmp(str, "solid", 5))
        return LV_LINES_SOLID;
    if(0==memcmp(str, "dotted", 6))
        return LV_LINES_DOTTED;
    return LV_LINES_NONE;
}

static EAppearance qtc_to_appearance(const char *str)
{
    if(0==memcmp(str, "flat", 4))
        return APPEARANCE_FLAT;
    if(0==memcmp(str, "gradient", 8) || 0==memcmp(str, "lightgradient", 13))
        return APPEARANCE_GRADIENT;
    if(0==memcmp(str, "glass", 5) || 0==memcmp(str, "shinyglass", 10))
        return APPEARANCE_SHINY_GLASS;
    if(0==memcmp(str, "dullglass", 9))
        return APPEARANCE_DULL_GLASS;
    if(0==memcmp(str, "inverted", 8))
        return APPEARANCE_INVERTED; 
    return APPEARANCE_BEVELLED;
}

static EShade qtc_to_shade(const char *str, qtc_bool allowDarken)
{
    /* true/false is from 0.25... */
    if(0==memcmp(str, "true", 4) || 0==memcmp(str, "TRUE", 4) || 0==memcmp(str, "selected", 8))
        return SHADE_SELECTED;
    if(allowDarken && 0==memcmp(str, "darken", 6))
        return SHADE_DARKEN;
    if(0==memcmp(str, "custom", 6))
        return SHADE_CUSTOM;
    return SHADE_NONE;
}

static EHeaderColor qtc_to_header_color(const char *str)
{
    if(0==memcmp(str, "background", 10))
        return HEADER_COLOR_BGND;
    if(0==memcmp(str, "custom", 6))
        return HEADER_COLOR_CUSTOM;
    return HEADER_COLOR_BTN;
}

static EFocus qtc_to_focus(const char *str)
{
    if(0==memcmp(str, "dots", 4))
        return FOCUS_DOTS;
    return FOCUS_COLORED;
}

/* Prior to 0.42 round was a bool - so need to read 'false' as 'none' */
static ERound qtc_to_round(const char *str)
{
    if(0==memcmp(str, "none", 4) || 0==memcmp(str, "false", 5))
        return ROUND_NONE;
    if(0==memcmp(str, "slight", 6))
        return ROUND_SLIGHT;
    return ROUND_FULL;
}

static qtc_bool get_bool(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? 0==memcmp("true", eq, 4) || 0==memcmp("qtc_true", eq, 4) : qtc_false;
}

static void get_color(char *line, qtc_color *col)
{
    char *eq=strstr(line, "=#");

    if(strlen(eq)>7) /* =#RRGGBB */
        set_rgb(col, &eq[2]); /* Skip past=# */
}

static ELine get_line(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? qtc_to_line(eq) : LINE_DOTS;
}

static EAppearance get_appearance(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? qtc_to_appearance(eq) : APPEARANCE_BEVELLED;
}

static EHeaderColor get_header_color(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? qtc_to_header_color(eq) : HEADER_COLOR_BGND;
}

static EFocus get_focus(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? qtc_to_focus(eq) : FOCUS_COLORED;
}

static ERound get_round(char *line)
{
    char *eq=strstr(line, "=");

    return eq && ++eq ? qtc_to_round(eq) : ROUND_FULL;
}
#endif

#ifdef CONFIG_WRITE
#ifdef __cplusplus
#include <kstandarddirs.h>
#endif
#endif

static const char *xdgConfigFolder()
{
    static char xdgDir[QTC_MAX_FILENAME_LEN]={'\0'};

    if(!xdgDir[0])
    {
        static const char *home=NULL;

        char *env=getenv("XDG_CONFIG_HOME");

        /*
            Check the setting of XDG_CONFIG_HOME
            For some reason, sudo leaves the env vars set to those of the
            caller - so XDG_CONFIG_HOME would point to the users setting, and
            not roots.

            Therefore, check that home is first part of XDG_CONFIG_HOME
        */

        if(env && 0==getuid())
        {
            if(!home)
                home=getHome();
            if(home && home!=strstr(env, home))
                env=NULL;
        }

        if(!env)
        {
            if(!home)
                home=getHome();

            sprintf(xdgDir, "%s/.config", home);
        }
        else
            strcpy(xdgDir, env);

#ifdef CONFIG_WRITE
#ifdef __cplusplus
        KStandardDirs::makeDir(xdgDir);
#endif
#endif
    }

    return xdgDir;
}

#ifdef CONFIG_READ
static void defaultSettings(Options *opts)
{
    opts->contrast=7;
    opts->highlightFactor=DEFAULT_HIGHLIGHT_FACTOR;
    opts->round=ROUND_FULL;
    opts->lighterPopupMenuBgnd=qtc_true;
    opts->vArrow=qtc_false;
    opts->stripedProgress=qtc_true;
    opts->animatedProgress=qtc_true;
    opts->highlightTab=qtc_true;
    opts->highlightEdits=qtc_true;
    opts->crLabelHighlight=qtc_false;
    opts->embolden=qtc_false;
    opts->menubarRoundTopOnly=qtc_true;
    opts->fillSlider=qtc_true;
    opts->focus=FOCUS_COLORED;
    opts->lvDark=qtc_false;
    opts->drawStatusBarFrames=qtc_false;
    opts->appearance=APPEARANCE_BEVELLED;
    opts->menubarAppearance=APPEARANCE_GRADIENT;
    opts->toolbarAppearance=APPEARANCE_GRADIENT;
    opts->selectedTabAppearance=APPEARANCE_GRADIENT;
    opts->normalTabAppearance=APPEARANCE_GRADIENT;
    opts->defBtnIndicator=IND_COLORED;
    opts->sliderThumbs=LINE_DOTS;
    opts->handles=LINE_DOTS;
    opts->shadeSliders=SHADE_SELECTED;
    opts->shadeMenubars=SHADE_DARKEN;
    opts->toolbarBorders=TB_NONE;
    opts->lvExpander=LV_EXP_ARR;
    opts->lvLines=LV_LINES_NONE;
    opts->lvHeaderAppearance=APPEARANCE_BEVELLED;
    opts->lvHeaderColor=HEADER_COLOR_BGND;
    opts->toolbarSeparators=LINE_DOTS;
    opts->splitters=LINE_DOTS;
#ifdef __cplusplus
    opts->customMenubarsColor.setRgb(0, 0, 0);
    opts->customSlidersColor.setRgb(0, 0, 0);
    opts->customLvHeaderColor.setRgb(0, 0, 0);
#else
    opts->customMenubarsColor.red=opts->customMenubarsColor.green=opts->customMenubarsColor.blue=0;
    opts->customSlidersColor.red=opts->customSlidersColor.green=opts->customSlidersColor.blue=0;
    opts->customLvHeaderColor.red=opts->customLvHeaderColor.green=opts->customLvHeaderColor.blue=0;
#endif
#ifdef QTC_CONFIG_DIALOG
    opts->readOnly=qtc_true;
#endif
}

#ifdef __cplusplus
#define QTC_IS_BLACK(A) (0==(A).red() && 0==(A).green() && 0==(A).blue())
#else
#define QTC_IS_BLACK(A) (0==(A).red && 0==(A).green && 0==(A).blue)
#endif

static void check_color(EShade *s, qtc_color *c)
{
    if(SHADE_CUSTOM==*s && QTC_IS_BLACK(*c))
        *s=SHADE_NONE;
}

qtc_bool static readConfig(const char *rc, Options *opts)
{
    if(!rc)
    {
        static const char *xdg=NULL;

        if(!xdg)
            xdg=xdgConfigFolder();

        if(!readConfig("/etc/qt4/"QTC_FILE, opts))
            if(!readConfig("/etc/qt3/"QTC_FILE, opts))
                readConfig("/etc/qt/"QTC_FILE, opts);

        if(xdg)
        {
            char filename[QTC_MAX_FILENAME_LEN];

            sprintf(filename, "%s/"QTC_FILE, xdg);
            return readConfig(filename, opts);
        }
    }
    else
    {
        enum
        {
            SECT_NONE,
            SECT_DESCR,
            SECT_SETTINGS
        }          section=SECT_NONE;
        char       line[QTC_MAX_INPUT_LINE_LEN+1];
        qtc_bool   darkerMenubarBgnd=qtc_false,
                   gotLvApp=qtc_false,
                   gotMenuBarShade=qtc_false,
                   gotNTabApp=qtc_false,
                   gotSTabApp=qtc_false,
                   foundSettings=qtc_false,
#ifdef QTC_CONFIG_DIALOG
                   foundDescription=qtc_false,
#endif
                   finished=qtc_false;
        FILE       *f=fopen(rc, "r");

        if(f)
        {
            while(!finished && NULL!=fgets(line, QTC_MAX_INPUT_LINE_LEN, f))
                if(line[0]=='[')
                {
                    if(0==memcmp(line, "["QTC_GROUP"]", QTC_GROUP_LEN+2))
                    {
                        foundSettings=qtc_true;
                        section=SECT_SETTINGS;
                    }
#ifdef QTC_CONFIG_DIALOG
                    else if(0==memcmp(line, "["QTC_DESCR_GROUP"]", QTC_DESCR_GROUP_LEN+2))
                    {
                        foundDescription=qtc_true;
                        section=SECT_DESCR;
                    }
                    else
                    {
                        section=SECT_NONE;
                        if(foundSettings && foundDescription)
                            finished=qtc_true;
                    }
#else
                    else
                    {
                        section=SECT_NONE;
                        if(foundSettings)
                            finished=qtc_true;
                    }
#endif
                }
#ifdef QTC_CONFIG_DIALOG
                else if(SECT_DESCR==section)
                {
                    if(0==memcmp(line, "name=", 5))
                    {
                        int len=strlen(line);

                        if('\n'==line[len-1])
                            len--;

                        opts->name=QString::fromUtf8(&line[5], len);
                        opts->readOnly=(0!=access(rc, W_OK));
                        opts->file=rc;
                    }
                }
#endif
                else if(SECT_SETTINGS==section)
                {
                    if(0==memcmp(line, "darkerMenubarBgnd=", 18))
                        darkerMenubarBgnd=get_bool(line);
                    else if(0==memcmp(line, "lighterPopupMenuBgnd=", 21))
                        opts->lighterPopupMenuBgnd=get_bool(line);
                    else if(0==memcmp(line, "round=", 6))
                        opts->round=get_round(line);
                    else if(0==memcmp(line, "menubarRoundTopOnly=", 20))
                        opts->menubarRoundTopOnly=get_bool(line);
                    else if(0==memcmp(line, "toolbarBorders=", 15))
                    {
                        char *eq=strstr(line, "=");

                        if(eq && ++eq)
                            opts->toolbarBorders=qtc_to_tbar_border(eq);
                    }
                    else if(0==memcmp(line, "vArrow=", 7))
                        opts->vArrow=get_bool(line);
                    else if(0==memcmp(line, "fillSlider=", 11))
                        opts->fillSlider=get_bool(line);
                    else if(0==memcmp(line, "stripedProgress=", 16))
                        opts->stripedProgress=get_bool(line);
                    else if(0==memcmp(line, "animatedProgress=", 17))
                        opts->animatedProgress=get_bool(line);
                    else if(0==memcmp(line, "embolden=", 9))
                        opts->embolden=get_bool(line);
                    else if(0==memcmp(line, "crLabelHighlight=", 17))
                        opts->crLabelHighlight=get_bool(line);
                    else if(0==memcmp(line, "highlightTab=", 13))
                        opts->highlightTab=get_bool(line);
                    else if(0==memcmp(line, "selectedTabAppearance=", 22))
                    {
                        opts->selectedTabAppearance=get_appearance(line);
                        if(APPEARANCE_BEVELLED==opts->selectedTabAppearance)
                            opts->selectedTabAppearance=APPEARANCE_GRADIENT;
                        gotSTabApp=qtc_true;
                    }
                    else if(0==memcmp(line, "normalTabAppearance=", 20))
                    {
                        opts->normalTabAppearance=get_appearance(line);
                        if(APPEARANCE_BEVELLED==opts->normalTabAppearance)
                            opts->normalTabAppearance=APPEARANCE_GRADIENT;
                        gotNTabApp=qtc_true;
                    }
                    else if(0==memcmp(line, "drawStatusBarFrames=", 20))
                        opts->drawStatusBarFrames=get_bool(line);
                    else if(0==memcmp(line, "highlightFactor=", 16))
                    {
                        char *l=strchr(line, '=');
                        int  val=0;
                        l++;
                        sscanf(l, "%i", &val);
                        if(val<=MAX_HIGHLIGHT_FACTOR && val>=MIN_HIGHLIGHT_FACTOR)
                            opts->highlightFactor=((double)(val+100))/100.0;
                        else if(val<=(100+MAX_HIGHLIGHT_FACTOR) &&
                                val>=(100+MIN_HIGHLIGHT_FACTOR))
                            opts->highlightFactor=((double)val)/100.0;
                    }
                    else if(0==memcmp(line, "shadeSliders=", 13))
                    {
                       char *eq=strstr(line, "=");

                        if(eq && ++eq)
                            opts->shadeSliders=qtc_to_shade(eq, qtc_false);
                    }
                    else if(0==memcmp(line, "shadeMenubars=", 14))
                    {
                       char *eq=strstr(line, "=");

                        if(eq && ++eq)
                        {
                            opts->shadeMenubars=qtc_to_shade(eq, qtc_true);
                            gotMenuBarShade=qtc_true;
                        }
                    }
                    else if(0==memcmp(line, "customSlidersColor=", 19))
                        get_color(line, &opts->customSlidersColor);
                    else if(0==memcmp(line, "customMenubarsColor=", 20))
                        get_color(line, &opts->customMenubarsColor);
                    else if(0==memcmp(line, "customLvHeaderColor=", 20))
                        get_color(line, &opts->customLvHeaderColor);
                    else if(0==memcmp(line, "highlightEdits=", 15))
                        opts->highlightEdits=get_bool(line);
                    else if(0==memcmp(line, "menubarAppearance=", 18))
                    {
                        opts->menubarAppearance=get_appearance(line);
                        if(APPEARANCE_BEVELLED==opts->menubarAppearance)
                            opts->menubarAppearance=APPEARANCE_GRADIENT;
                    }
                    else if(0==memcmp(line, "toolbarAppearance=", 18))
                    {
                        opts->toolbarAppearance=get_appearance(line);
                        if(APPEARANCE_BEVELLED==opts->toolbarAppearance)
                            opts->toolbarAppearance=APPEARANCE_GRADIENT;
                    }
                    else if(0==memcmp(line, "toolbarSeparators=", 18))
                        opts->toolbarSeparators=get_line(line);
                    else if(0==memcmp(line, "splitters=", 10))
                        opts->splitters=get_line(line);
                    else if(0==memcmp(line, "defBtnIndicator=", 16))
                    {
                        char *eq=strstr(line, "=");

                        if(eq && ++eq)
                            opts->defBtnIndicator=qtc_to_ind(eq);
                    }
                    else if(0==memcmp(line, "sliderThumbs=", 13))
                        opts->sliderThumbs=get_line(line);
                    else if(0==memcmp(line, "handles=", 8))
                        opts->handles=get_line(line);
                    else if(0==memcmp(line, "appearance=", 11))
                        opts->appearance=get_appearance(line);
                    else if(0==memcmp(line, "lvHeaderAppearance=", 13))
                    {
                        opts->lvHeaderAppearance=get_appearance(line);
                        gotLvApp=qtc_true;
                    }
                    else if(0==memcmp(line, "lvHeaderColor=", 13))
                        opts->lvHeaderColor=get_header_color(line);
                    else if(0==memcmp(line, "focus=", 6))
                        opts->focus=get_focus(line);
                    else if(0==memcmp(line, "lvExpander=", 11))
                    {
                        char *eq=strstr(line, "=");

                        if(eq && ++eq)
                            opts->lvExpander=qtc_to_lv_expander(eq);
                    }
                    else if(0==memcmp(line, "lvLines=", 8))
                    {
                        char *eq=strstr(line, "=");

                        if(eq && ++eq)
                            opts->lvLines=qtc_to_lv_lines(eq);
                    }
                }
            fclose(f);
        }

        if(foundSettings)
        {
            /* Old config file had a bool entry for darkerMenubarBgnd */
            if(darkerMenubarBgnd && !gotMenuBarShade)
                opts->shadeMenubars=SHADE_DARKEN;

            if(!gotLvApp)
                opts->lvHeaderAppearance=opts->appearance;

            if(!gotNTabApp)
                if(gotSTabApp)
                    opts->normalTabAppearance=opts->selectedTabAppearance;
                else if(APPEARANCE_BEVELLED==opts->appearance)
                    opts->normalTabAppearance=APPEARANCE_GRADIENT;
                else
                    opts->normalTabAppearance=opts->appearance;

            if(!gotSTabApp)
                if(gotNTabApp)
                    opts->selectedTabAppearance=opts->normalTabAppearance;
                else if(APPEARANCE_BEVELLED==opts->appearance)
                    opts->selectedTabAppearance=APPEARANCE_GRADIENT;
                else
                    opts->selectedTabAppearance=opts->appearance;

            check_color(&opts->shadeMenubars, &opts->customMenubarsColor);
            check_color(&opts->shadeSliders, &opts->customSlidersColor);

            if(HEADER_COLOR_CUSTOM==opts->lvHeaderColor &&
               QTC_IS_BLACK(opts->customLvHeaderColor))
                opts->lvHeaderColor=HEADER_COLOR_BGND;

            if(opts->animatedProgress && !opts->stripedProgress)
                opts->animatedProgress=qtc_false;
        }
#ifdef QTC_CONFIG_DIALOG
        return foundSettings && foundDescription;
#else
        return foundSettings;
#endif
    }
    return qtc_false;
}
#endif

#ifdef CONFIG_WRITE
static const char *toStr(EDefBtnIndicator ind)
{
    switch(ind)
    {
        case IND_NONE:
            return "none";
        case IND_FONT_COLOR:
            return "fontcolor";
        case IND_CORNER:
            return "corner";
        default:
            return "colored";
    }
}

static const char *toStr(ELine ind, qtc_bool none)
{
    switch(ind)
    {
        case LINE_RAISED:
            return "raised";
        case LINE_DOTS:
            return "dots";
        case LINE_DASHES:
            return none ? "none" : "dashes";
        default:
            return "sunken";
    }
}

static const char *toStr(ETBarBorder ind)
{
    switch(ind)
    {
        case TB_DARK:
            return "dark";
        case TB_NONE:
            return "none";
        default:
            return "light";
    }
}

static const char *toStr(ELvExpander exp)
{
    switch(exp)
    {
        case LV_EXP_PM:
            return "plusminus";
        default:
            return "arrow";
    }
}

static const char *toStr(ELvLines exp)
{
    switch(exp)
    {
        case LV_LINES_NONE:
            return "none";
        case LV_LINES_DOTTED:
            return "dotted";
        default:
            return "solid";
    }
}

static const char *toStr(EAppearance exp)
{
    switch(exp)
    {
        case APPEARANCE_FLAT:
            return "flat";
        case APPEARANCE_GRADIENT:
            return "gradient";
        case APPEARANCE_DULL_GLASS:
            return "dullglass";
        case APPEARANCE_BEVELLED:
            return "bevelled";
        case APPEARANCE_INVERTED:
            return "inverted";
        default:
            return "shinyglass";
    }
}

static const char *toStr(EShade exp)
{
    switch(exp)
    {
        default:
        case SHADE_NONE:
            return "none";
        case SHADE_SELECTED:
            return "selected";
        case SHADE_CUSTOM:
            return "custom";
        case SHADE_DARKEN:
            return "darken";
    }
}

static const char *toStr(EHeaderColor exp)
{
    switch(exp)
    {
        default:
        case HEADER_COLOR_BGND:
            return "background";
        case HEADER_COLOR_BTN:
            return "button";
        case HEADER_COLOR_CUSTOM:
            return "custom";
    }
}

static const char *toStr(EFocus exp)
{
    switch(exp)
    {
        case FOCUS_DOTS:
            return "dots";
        default:
        case FOCUS_COLORED:
            return "colored";
    }
}

static const char *toStr(ERound exp)
{
    switch(exp)
    {
        case ROUND_NONE:
            return "none";
        case ROUND_SLIGHT:
            return "slight";
        default:
        case ROUND_FULL:
            return "full";
    }
}

qtc_bool static writeConfig(KConfig *cfg, const Options &opts)
{
    if(!cfg)
    {
        static const char *xdg=xdgConfigFolder();

        if(xdg)
        {
            char filename[QTC_MAX_FILENAME_LEN];

            sprintf(filename, "%s/"QTC_FILE, xdg);

            KConfig defCfg(filename, false, false);

            return writeConfig(&defCfg, opts);
        }
    }
    else
    {
        cfg->setGroup(QTC_GROUP);
        cfg->writeEntry("round", toStr(opts.round));
        cfg->writeEntry("highlightFactor", ((int)(opts.highlightFactor*100))-100);
        cfg->writeEntry("toolbarBorders", toStr(opts.toolbarBorders));
        cfg->writeEntry("appearance", toStr(opts.appearance));
        cfg->writeEntry("focus", toStr(opts.focus));
        cfg->writeEntry("vArrow", opts.vArrow);
        cfg->writeEntry("fillSlider", opts.fillSlider);
        cfg->writeEntry("stripedProgress", opts.stripedProgress);
        cfg->writeEntry("animatedProgress", opts.animatedProgress);
        cfg->writeEntry("lighterPopupMenuBgnd", opts.lighterPopupMenuBgnd);
        cfg->writeEntry("embolden", opts.embolden);
        cfg->writeEntry("crLabelHighlight", opts.crLabelHighlight);
        cfg->writeEntry("defBtnIndicator", toStr(opts.defBtnIndicator));
        cfg->writeEntry("sliderThumbs", toStr(opts.sliderThumbs, qtc_true));
        cfg->writeEntry("handles", toStr(opts.handles, qtc_false));
        cfg->writeEntry("lvExpander", toStr(opts.lvExpander));
        cfg->writeEntry("lvLines", toStr(opts.lvLines));
        cfg->writeEntry("lvDark", opts.lvDark);
        cfg->writeEntry("drawStatusBarFrames", opts.drawStatusBarFrames);
        cfg->writeEntry("lvHeaderAppearance", toStr(opts.lvHeaderAppearance));
        cfg->writeEntry("lvHeaderColor", toStr(opts.lvHeaderColor));
        cfg->writeEntry("highlightTab", opts.highlightTab);
        cfg->writeEntry("shadeSliders", toStr(opts.shadeSliders));
        cfg->writeEntry("shadeMenubars", toStr(opts.shadeMenubars));
        cfg->writeEntry("menubarAppearance", toStr(opts.menubarAppearance));
        cfg->writeEntry("menubarRoundTopOnly", opts.menubarRoundTopOnly);
        cfg->writeEntry("toolbarAppearance", toStr(opts.toolbarAppearance));
        cfg->writeEntry("toolbarSeparators", toStr(opts.toolbarSeparators, qtc_true));
        cfg->writeEntry("splitters", toStr(opts.splitters, qtc_false));
        cfg->writeEntry("highlightEdits", opts.highlightEdits);
        cfg->writeEntry("normalTabAppearance", toStr(opts.normalTabAppearance));
        cfg->writeEntry("selectedTabAppearance", toStr(opts.selectedTabAppearance));

        char colorStr[8];

        sprintf(colorStr, "#%02X%02X%02X", opts.customSlidersColor.red(),
                                           opts.customSlidersColor.green(),
                                           opts.customSlidersColor.blue());
        cfg->writeEntry("customSlidersColor", colorStr);
        sprintf(colorStr, "#%02X%02X%02X", opts.customMenubarsColor.red(),
                                           opts.customMenubarsColor.green(),
                                           opts.customMenubarsColor.blue());
        cfg->writeEntry("customMenubarsColor", colorStr);
        sprintf(colorStr, "#%02X%02X%02X", opts.customLvHeaderColor.red(),
                                           opts.customLvHeaderColor.green(),
                                           opts.customLvHeaderColor.blue());
        cfg->writeEntry("customLvHeaderColor", colorStr);

        if(!opts.name.isEmpty())
        {
            cfg->setGroup(QTC_DESCR_GROUP);
            cfg->writeEntry("name", opts.name);
        }

        cfg->sync();
        return qtc_true;
    }
    return qtc_false;
}
#endif

#if defined CONFIG_WRITE && defined CONFIG_READ
static void defaultStyleSettings(Options *opts)
{
    defaultSettings(opts);
    readConfig("/etc/qt/"QTC_FILE, opts);
    readConfig("/etc/qt3/"QTC_FILE, opts);
    readConfig("/etc/qt4/"QTC_FILE, opts);
}
#endif
