/*
 * QtCurve GTK engine
 *
 * This is a hacked/modified version of the B***Curve GTK engine. Modified to use
 * ~/.qt/qtrc to get color information to recolor widgets.
 *
 * Changes are (C) Craig Drummond 2003 - Craig.Drummond@lycos.co.uk
 *
 */

#include "common.h"
#include "config.h"
#define CONFIG_READ
#include "config_file.c"
#include <gtk/gtk.h>
#include <time.h>
#ifdef HAVE_LOCALE_H
#include <locale.h>
#endif

/*
#define QTC_DEBUG
*/

static int strcmp_i(const char *s1, const char *s2)
{
    char c1,
         c2;

    for(;;)
    {
        c1=*s1++;
        c2=*s2++;
        if(!c1 || !c2)
            break;
        if(isupper(c1))
            c1=tolower (c1);
        if(isupper(c2))
            c2=tolower (c2);
        if(c1!=c2)
            break;
    }
    return (int)c2-(int)c1;
}

#if GTK_MAJOR_VERSION>1

#include <gdk/gdkcolor.h>
#include <gtk/gtkenums.h>

struct qt_toolbar
{
    GtkIconSize     size;
    GtkToolbarStyle style;
};

#else

#include <gdk/gdktypes.h>

#endif

enum qt_fcolor
{
  COLOR_BACKGROUND,
  COLOR_BUTTON,
  COLOR_SELECTED,
  COLOR_WINDOW,

  COLOR_FOREGROUND,
  COLOR_MID,
  COLOR_TEXT,
  COLOR_TEXT_SELECTED,
  COLOR_BUTTON_TEXT,
  COLOR_NONE,
  COLOR_NUMCOLORS=COLOR_NONE  /* NONE does not count! */
};

typedef enum
{
    GTK_APP_UNKNOWN,
    GTK_APP_MOZILLA,
    GTK_APP_OPEN_OFFICE,
    GTK_APP_VMPLAYER,
    GTK_APP_GIMP,
    GTK_APP_GIMP_PLUGIN
} EGtkApp;

struct qt_data
{
    GdkColor          colors[COLOR_NUMCOLORS];
    char              *font;
#if GTK_MAJOR_VERSION>1
    char              *icons,
                      *progressfont_rc;
    struct qt_toolbar toolbar;
    gboolean          button_icons;
    EGtkApp           app;
#endif
};

#include <gmodule.h>
#include <ctype.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <pwd.h>
#include <sys/types.h>

#define DEFAULT_KDE_COLORS "active=#000000^e#dddfe4^e#ffffff^e#ffffff^e#555555^e#c7c7c7^e#000000^e#ffffff^e#000000^e#ffffff^e#efefef^e#000000^e#678db2^e#ffffff^e#0000ee^e#52188b^e"
#define DEFAULT_KDE_FONT      "Sans Serif"
#define DEFAULT_KDE_FONT_SIZE 10.0
#define MAX_LINE_LEN 1024

#if GTK_MAJOR_VERSION>1
#define DEF_ICONS        "crystalsvg"
#define DEF_ICONS_LEN    10
#define DEF_ICON_DIR     "default.kde"
#define DEF_ICON_DIR_LEN 11
#endif

struct qt_data qt_settings;

enum
{
    SECT_NONE,
    SECT_PALETTE,
    SECT_GENERAL,
    SECT_KDE
#if GTK_MAJOR_VERSION>1
  , SECT_ICONS,
    SECT_TOOLBAR_STYLE,
    SECT_MAIN_TOOLBAR_ICONS
#endif
};

/*
  Qt uses the following predefined weights: 
    Light    = 25,
    Normal   = 50,
    DemiBold = 63,
    Bold     = 75,
    Black    = 87

  ...need to categorize mid way, i.e. 0->37 = light, 38->56 = normal, etc...
*/

enum
{
    WEIGHT_NORMAL   = 38,   /* Less than this = light */
    WEIGHT_DEMIBOLD = 57,
    WEIGHT_BOLD     = 69,
    WEIGHT_BLACK    = 81
};

static const char * weight_str(int w)
{
    if(w<WEIGHT_NORMAL)
        return "light";
    else if(w<WEIGHT_DEMIBOLD)
#if GTK_MAJOR_VERSION>1
        return "";
#else
        return "medium";
#endif
    else if(w<WEIGHT_BOLD)
        return "demibold";
    else if(w<WEIGHT_BLACK)
        return "bold";
    else
        return "black";
}

#if GTK_MAJOR_VERSION>1
static const char * italic_str(int i)
{
    return i ? "Italic" : "";
}
#endif

enum
{
    RD_PALETTE           = 0x1,
    RD_FONT              = 0x2,
    RD_CONTRAST          = 0x4
#if GTK_MAJOR_VERSION>1
  , RD_ICONS             = 0x10,
    RD_TOOLBAR_STYLE     = 0x20,
    RD_TOOLBAR_ICON_SIZE = 0x40,
    RD_BUTTON_ICONS      = 0x80
#endif
};

#if GTK_MAJOR_VERSION==1
static const char * iso8859_1_langs[]=
{
    "af", "afrikaans", "ca", "catalan", "da", "danish", "dansk", "de", "german", "deutsch",
    "en", "english", "es", "spanish", "eu", "basque", "fi", "finnish", "fo", "faroese",
    "faeroese", "fr", "french", "ga", "irish", "gd", "scottish", "gl", "galician", "is",
    "icelandic", "it", "italian", "nl", "dutch", "no", "norwegian", "pt", "portuguese",
    "sv", "swedish",
    NULL
};

static const char * iso8859_1_codesets[]=
{
    "ISO8859-1", "ISO_8859-1", "iso88591", "88591", "88591.en", "8859", "8859.in", "ascii",
    NULL
};

static const char * iso8859_2_langs[]=
{
    "cs", "croatian", "cz", "czech", "hr", "hu", "hungarian", "pl", "polish", "ro", "romanian",
    "rumanian", "serbocroatian", "sh", "sk", "sl", "slovak", "sr", "slovene", "slovenian",
    "sq", "albanian",
    NULL
};

static const char * iso8859_2_codesets[]=
{
    "ISO8859-2", "ISO_8859-2", "iso88592", "88592",
    NULL
};

static const char * iso8859_3_langs[] =
{
    "eo", "esperanto", "mt", "maltese",
    NULL
};

static const char * iso8859_3_codesets[]=
{
    "ISO8859-3", "ISO_8859-3", "iso88593", "88593",
    NULL
};

static const char * iso8859_5_langs[] =
{
    "be", "byelorussian", "bg", "bulgarian", "mk", "macedonian", "sp", 
    "sr", "serbian",
    NULL
};

static const char * iso8859_5_codesets[]=
{
    "ISO8859-5", "ISO_8859-5", "iso88595", "88595",
    NULL
};

static const char * iso8859_6_langs[] =
{
    "ar", "arabic",
    NULL
};

static const char * iso8859_6_codesets[]=
{
    "ISO8859-6", "ISO_8859-6", "iso88596", "88596",
    NULL
};

static const char * iso8859_7_langs[] =
{
    "el", "greek",
    NULL
};

static const char * iso8859_7_codesets[]=
{
    "ISO8859-7", "ISO_8859-7", "iso88597", "88597",
    NULL
};

static const char * iso8859_8_langs[] =
{
    "hebrew", "iw", "he",
    NULL
};

static const char * iso8859_8_codesets[]=
{
    "ISO8859-8", "ISO_8859-8", "iso88598", "88598",
    NULL
};

static const char * iso8859_9_langs[] =
{
    "tr", "tr", "turkish",
    NULL
};

static const char * iso8859_9_codesets[]=
{
    "ISO8859-9", "ISO_8859-9", "iso88599", "88599",
    NULL
};

static const char * iso8859_10_langs[] =
{
    "et", "estonian", "lt", "lithuanian", "lv", "latvian",
    NULL
};

static const char * iso8859_10_codesets[]=
{
    "ISO8859-10", "ISO_8859-10", "iso885910", "885910",
    NULL
};

static const char * iso8859_15_langs[] =
{
/*
      "fr", "french", "fi", "finnish",
*/
    NULL
};

static const char * iso8859_15_codesets[]=
{
    "ISO8859-15", "ISO_8859-15", "iso885915", "885915",
    NULL
};

static const char * koi8_r_langs[] =
{
    "ru", "russian",
    NULL
};

static const char * koi8_r_codesets[]=
{
    "KOI8-R", "KOI8_R",
    NULL
};

static const char * koi8_ru_langs[] =
{
    "uk", "ukrainian",
    NULL
};

static const char * koi8_ru_codesets[]=
{
    "KOI8-RU", "KOI8_RU",
    NULL
};

static struct
{
    const char *encoding;
    const char **languages,
               **codesets;
} x11EncodingMap[] =
    {
        { "iso8859-1",  iso8859_1_langs,  iso8859_1_codesets  },
        { "iso8859-2",  iso8859_2_langs,  iso8859_2_codesets  },
        { "iso8859-3",  iso8859_3_langs,  iso8859_3_codesets  },
        { "iso8859-5",  iso8859_5_langs,  iso8859_5_codesets  },
        { "iso8859-6",  iso8859_6_langs,  iso8859_6_codesets  },
        { "iso8859-7",  iso8859_7_langs,  iso8859_7_codesets  },
        { "iso8859-8",  iso8859_8_langs,  iso8859_8_codesets  },
        { "iso8859-9",  iso8859_9_langs,  iso8859_9_codesets  },
        { "iso8859-10", iso8859_10_langs, iso8859_10_codesets },
        { "iso8859-15", iso8859_15_langs, iso8859_15_codesets },
        { "KOI8-R",     koi8_r_langs,     koi8_r_codesets     },
        { "KOI8-RU",    koi8_ru_langs,    koi8_ru_codesets    },
        { NULL,         NULL,             NULL                }
    };

static const char * getX11Encoding()
{
    const char *enc=getenv("QT_GTK_ENC");

    if(!enc)
        enc=getenv("QTPIXMAP_GTK_ENC");   /* For backwards compatibility */

    if(!enc)
    {
        const char *locale;

        locale = getenv("LC_ALL");
        if (!locale || !*locale)
        { 
            locale = getenv("LC_CTYPE");
            if (!locale || !*locale)
                locale = getenv("LANG");
        }

        if (locale && *locale)
        {
            /* The most general syntax of a locale (not all optional parts recognized by all systems) is
               language[_territory][.codeset][@modifier][+special][,[sponsor][_revision]]
               To retrieve the codeset, search the first dot. Stop searching when
               a '@' or '+' or ',' is encountered. */
            char       *buf = (char*) malloc(strlen(locale)+1);
            const char *codeset = NULL;
            const char *cp = locale;

            for (; *cp != '\0' && *cp != '@' && *cp != '+' && *cp != ','; cp++)
            {
                if (*cp == '.')
                {
                    codeset = ++cp;

                    for (; *cp != '\0' && *cp != '@' && *cp != '+' && *cp != ','; cp++)
                        ;
                    if (*cp != '\0')
                    {
                        size_t n = cp - codeset;
                        memcpy(buf, codeset, n);
                        buf[n] = '\0';
                        codeset = buf;
                    }
                    break;
                }
            }

            if (codeset)
            {
                int i;

                for(i=0; x11EncodingMap[i].encoding && !enc; ++i)
                {
                    int j;

                    for(j=0; x11EncodingMap[i].codesets[j] && !enc; ++j)
                        if(0==strcmp_i(codeset, x11EncodingMap[i].codesets[j]))
                            enc=x11EncodingMap[i].encoding;
                }

                if(!enc)  /* Then didn't recognise codeset... */
                {
                    char *dash=strchr(codeset, '-');

                    /* Check if codeset is of the form <wibble>-<wobble> */
                    if(dash && strlen(dash)>1 && NULL==strchr(dash+1, '-'))  /* if so, then assume this is an OK X11 enc... */
                        enc=codeset;
                }
            }

            if(!enc) /* No matching/recognised codeset, so try to guess based upon language */
            {
                const char *underscore = strchr(locale, '_');
                const char *lang;
                int        i;

                if (underscore)
                {
                    size_t n = underscore - locale;
                    memcpy(buf,locale,n);
                    buf[n] = '\0';
                    lang = buf;
                }
                else
                    lang = locale;

                for(i=0; x11EncodingMap[i].encoding && !enc; ++i)
                {
                    int j;

                    for(j=0; x11EncodingMap[i].languages[j] && !enc; ++j)
                        if(0==strcmp_i(lang, x11EncodingMap[i].languages[j]))
                            enc=x11EncodingMap[i].encoding;
                }
            }

            free(buf);
        }
    }

    return enc ? enc : "iso8859-1";
}

static void getX11Res(int *resX, int *resY)
{
    gint width=gdk_screen_width(),
         height=gdk_screen_height(),
         widthMM=gdk_screen_width_mm(),
         heightMM=gdk_screen_height_mm();

    *resX=(width*254+widthMM*5)/(widthMM*10);
    *resY=(height*254+heightMM*5)/(heightMM*10);
}

int read_font_replacement(const char *file, const char **font)
{
    FILE *r=fopen(file, "r");
    int  ret=0;

    if(r)
    {
        char        line[QTC_MAX_INPUT_LINE_LEN+1];
        static char font_line[QTC_MAX_INPUT_LINE_LEN+1];

        while(NULL!=fgets(line, QTC_MAX_INPUT_LINE_LEN, r))
            if(line[0] != '#')
            {
                char *of=NULL;

                memcpy(font_line, line, QTC_MAX_INPUT_LINE_LEN+1);
                of=strtok(font_line, "=");
                if(0==strcmp_i(*font, of))
                {
                    *font=strtok(NULL, "\n");
                    ret=1;
                    break;
                }
            }
        fclose(r);
    }
    return ret;
}
#endif

#if GTK_MAJOR_VERSION>1
static char * get_kde_home()
{
    static char *kdeHome=NULL;

    if(!kdeHome)
    {
        char *env=getenv("KDEHOME");

        if(env)
            kdeHome=env;
        else
        {
            static char kdeHomeStr[QTC_MAX_FILENAME_LEN+1];

            const char *home=getHome();

            if(home && strlen(home)<(QTC_MAX_FILENAME_LEN-5))
            {
                sprintf(kdeHomeStr, "%s/.kde", home);
                kdeHome=kdeHomeStr;
            }
        }
    }

    return kdeHome;
}
#endif

static void parse_qt_colors(char *line)
{
    int  n=-1;
    char *l=strtok(line, "#");

    while(l)
    {
        if(8==strlen(l))
            switch(n)
            {
                case 0:
                    set_rgb(&qt_settings.colors[COLOR_FOREGROUND], l);
                    break;
                case 1:
                    set_rgb(&qt_settings.colors[COLOR_BUTTON], l);
                    break;
                case 5:
                    set_rgb(&qt_settings.colors[COLOR_MID], l);
                    break;
                case 6:
                    set_rgb(&qt_settings.colors[COLOR_TEXT], l);
                    break;
                case 8:
                    set_rgb(&qt_settings.colors[COLOR_BUTTON_TEXT], l);
                    break;
                case 9:
                    set_rgb(&qt_settings.colors[COLOR_BACKGROUND], l);
                    break;
                case 10:
                    set_rgb(&qt_settings.colors[COLOR_WINDOW], l);
                    break;
                case 12:
                    set_rgb(&qt_settings.colors[COLOR_SELECTED], l);
                    break;
                case 13:
                    set_rgb(&qt_settings.colors[COLOR_TEXT_SELECTED], l);
                    break;
                default:
                    break;
            }
        else
            if(n>-1)
                break;

        n++;
        if(n>13)
            break;
        l=strtok(NULL, "#");
    }
}

static int read_rc(const char *rc, int rd, Options *opts, gboolean absolute, gboolean set_default_font)
{
    const char *home=absolute ? NULL : getHome();
    int        found=0,
               weight=WEIGHT_NORMAL,
               italic=0,
               fixedW=0;
    float      size=DEFAULT_KDE_FONT_SIZE;
    const char *family=DEFAULT_KDE_FONT,
               *foundry="*";
    char       line[QTC_MAX_INPUT_LINE_LEN+1],
               font_line[QTC_MAX_INPUT_LINE_LEN+1];

    if(absolute || NULL!=home)
    {
        char     fname[256];
        FILE     *f;

        if(absolute)
            strcpy(fname, rc);
        else
            sprintf(fname, "%s/%s", home, rc);

        f=fopen(fname, "r");

        if(f)
        {
            int section=SECT_NONE;

            while(found!=rd && NULL!=fgets(line, QTC_MAX_INPUT_LINE_LEN, f))
                if(line[0]=='[')
                {
                    if(0==memcmp(line, "[Palette]", 9))
                        section=SECT_PALETTE;
                    else if(0==memcmp(line, "[General]", 9))
                        section=SECT_GENERAL;
                    else if(0==memcmp(line, "[KDE]", 5))
                        section=SECT_KDE;
#if GTK_MAJOR_VERSION>1
                    else if(0==memcmp(line, "[Icons]", 7))
                        section=SECT_ICONS;
                    else if(0==memcmp(line, "[Toolbar style]", 15))
                        section=SECT_TOOLBAR_STYLE;
                    else if(0==memcmp(line, "[MainToolbarIcons]", 18))
                        section=SECT_MAIN_TOOLBAR_ICONS;
#endif
                    else
                        section=SECT_NONE;
                }
#if GTK_MAJOR_VERSION>1
                else if (SECT_ICONS==section && rd&RD_ICONS && !(found&RD_ICONS) &&
                         0==memcmp(line, "Theme=", 6))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        unsigned int len=strlen(eq);

                        if(qt_settings.icons)
                        {
                            free(qt_settings.icons);
                            qt_settings.icons=NULL;
                        }

                        qt_settings.icons=(char *)malloc(len+1);
                        strcpy(qt_settings.icons, eq);
                        if('\n'==qt_settings.icons[len-1])
                            qt_settings.icons[len-1]='\0';
                    }

                    found|=RD_ICONS;
                }
                else if (SECT_TOOLBAR_STYLE==section && rd&RD_TOOLBAR_STYLE &&
                         !(found&RD_TOOLBAR_STYLE) && 0==memcmp(line, "IconText=", 9))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        if(0==memcmp(eq, "IconOnly", 8))
                            qt_settings.toolbar.style=GTK_TOOLBAR_ICONS;
                        else if(0==memcmp(eq, "TextOnly", 8))
                            qt_settings.toolbar.style=GTK_TOOLBAR_TEXT;
                        else if(0==memcmp(eq, "IconTextRight", 13))
                            qt_settings.toolbar.style=GTK_TOOLBAR_BOTH_HORIZ;
                        else if(0==memcmp(eq, "IconTextBottom", 14))
                            qt_settings.toolbar.style=GTK_TOOLBAR_BOTH;
                        found|=RD_TOOLBAR_STYLE;
                    }
                }
                else if (SECT_MAIN_TOOLBAR_ICONS==section && rd&RD_TOOLBAR_ICON_SIZE &&
                         !(found&RD_TOOLBAR_ICON_SIZE) && 0==memcmp(line, "Size=", 5))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        int tbarsize=atoi(eq);

                        qt_settings.toolbar.size= tbarsize <=16 ? GTK_ICON_SIZE_MENU :
                                                  tbarsize <=22 ? GTK_ICON_SIZE_LARGE_TOOLBAR :
                                                  tbarsize <=32 ? GTK_ICON_SIZE_DND
                                                                : GTK_ICON_SIZE_DIALOG;
                        found|=RD_TOOLBAR_ICON_SIZE;
                    }
                }
                else if (SECT_KDE==section && rd&RD_BUTTON_ICONS && !(found&RD_BUTTON_ICONS) &&
                         0==memcmp(line, "ShowIconsOnPushButtons=", 23))
                {
                    char *eq=strstr(line, "=");

                    if(eq && ++eq)
                    {
                        qt_settings.button_icons=0==memcmp(eq, "true", 4);
                        found|=RD_BUTTON_ICONS;
                    }
                }
#endif
                else if(SECT_KDE==section && rd&RD_CONTRAST && !(found&RD_CONTRAST) &&
                        0==memcmp(line, "contrast=", 9))
                {
                    char *l=strchr(line, '=');
                    l++;
                    sscanf(l, "%i", &(opts->contrast));
                    if(opts->contrast>10 || opts->contrast<0)
                        opts->contrast=7;
                    found|=RD_CONTRAST;
                }
                else if(SECT_PALETTE==section && rd&RD_PALETTE && !(found&RD_PALETTE) &&
                        0==memcmp(line, "active=", 7))
                {
                    parse_qt_colors(line);
                    found|=RD_PALETTE;
                }
                else if (SECT_GENERAL==section && rd&RD_FONT && !(found&RD_FONT) &&
                         0==memcmp(line, "font=", 5))
                {
                    int   n=-1,
                          rc_weight=WEIGHT_NORMAL,
                          rc_italic=0,
                          rc_fixedW=0;
                    float rc_size=12.0;
                    char  *l,
                          *rc_family=NULL,
                          *rc_foundry=NULL;

                    memcpy(font_line, line, QTC_MAX_INPUT_LINE_LEN+1);
                    l=strtok(font_line, "=");
                    found|=RD_FONT;

                    while(l)
                    {
                        switch(n)
                        {
                            case 0:  /* Family - and foundry(maybe!) (ignore X11 and XFT) */
                            {
                                char *ob=NULL,
                                     *cb=NULL;

                                if(NULL!=(ob=strchr(l, '[')) && ob!=l && NULL!=(cb=strchr(l, ']')))
                                {
                                    ob[-1]='\0';
                                    *cb='\0';
                                    rc_foundry=&(ob[1]);

                                    if(0==strcmp_i(rc_foundry, "xft") ||
                                       0==strcmp_i(rc_foundry, "x11"))
                                        rc_foundry=NULL;
                                }
                                else  /* Sometimes .kderc has "adobe-helvetica" */
                                {
                                    char *dash=NULL;

                                    if(NULL!=(dash=strchr(l, '-')))
                                    {
                                        rc_foundry=l;
                                        *dash='\0';
                                        l=++dash;
                                    }
                                }

                                rc_family=l;
                                break;
                            }
                            case 1:  /* Point size */
                                sscanf(l, "%f", &rc_size);
                                break;
                            case 4:  /* Weight */
                                sscanf(l, "%d", &rc_weight);
                                break;
                            case 5:  /* Slant */
                                sscanf(l, "%d", &rc_italic);
                                break;
                            case 8:  /* Spacing */
                                sscanf(l, "%d", &rc_fixedW);
                                break;
                            default:
                                break;
                        }

                        n++;
                        if(n>8 && NULL!=family)
                        {
                            weight=rc_weight;
                            italic=rc_italic;
                            fixedW=rc_fixedW;
                            size=rc_size;
                            family=rc_family;
                            foundry=rc_foundry;
                            break;
                        }
                        l=strtok(NULL, ",");
                    }
                }
                else if(found==rd)
                    break;

            fclose(f);
        }
    }

    if(rd&RD_PALETTE && !(found&RD_PALETTE))
    {
        strncpy(line, DEFAULT_KDE_COLORS, QTC_MAX_INPUT_LINE_LEN);
        line[QTC_MAX_INPUT_LINE_LEN]='\0';
        parse_qt_colors(line);
    }

    if(rd&RD_FONT && (found&RD_FONT || set_default_font))  /* No need to check if read in */
    {
        if(qt_settings.font)
        {
            free(qt_settings.font);
            qt_settings.font=NULL;
        }
#if GTK_MAJOR_VERSION>1
        if(qt_settings.progressfont_rc)
        {
            free(qt_settings.progressfont_rc);
            qt_settings.progressfont_rc=NULL;
        }

        qt_settings.font=(char *)malloc(
            strlen(family) + 1 +
            strlen(weight_str(weight)) + 1 +
            strlen(italic_str(italic)) + 1 +
            20+  /* point size */ +1);

        sprintf(qt_settings.font, "%s %s %s %d",
                family,
                weight_str(weight),
                italic_str(italic),
                (int)size);

        /* Qt uses a bold font for progressbars, try to mimic this... */
        if(weight>=WEIGHT_NORMAL && weight<WEIGHT_DEMIBOLD)
        {
            static const char *constPrefix="style \"qtc-pbar-font\"{ font_name=\"";
            static const char *constSuffix="\"} class \"GtkProgress\" style \"qtc-pbar-font\"\n";

            int pweight=WEIGHT_BOLD;

            qt_settings.progressfont_rc=(char *)malloc(
                strlen(constPrefix)+
                strlen(family) + 1 +
                strlen(weight_str(pweight)) + 1 +
                strlen(italic_str(italic)) + 1 +
                20+  /* point size */
                strlen(constSuffix) +1);

            sprintf(qt_settings.progressfont_rc, "%s%s %s %s %d%s",
                    constPrefix,
                    family,
                    weight_str(pweight),
                    italic_str(italic),
                    (int)size,
                    constSuffix);
        }
#else
        char       *format=(char *)"-%s-%s-%s-%s-*--*-%d-%d-%d-%s-*-%s";
        const char *encoding=getX11Encoding();
        int        resX,
                   resY;
        const char *global_name="/etc/X11/qt_gtk_fnt2fntrc";

        getX11Res(&resX, &resY);

        if(NULL!=home)
        {
            char home_name[QTC_MAX_FILENAME_LEN+1];

            snprintf(home_name, QTC_MAX_FILENAME_LEN, "%s/%s", home, "/.qt/gtk_fnt2fntrc");

            if(!read_font_replacement(home_name, &family))
                read_font_replacement(global_name, &family);
        }
        else
            read_font_replacement(global_name, &family);

        qt_settings.font=(char *)malloc(
            (foundry ? strlen(foundry) : 1) +
            strlen(family) +
            strlen(weight_str(weight))+
            1+   /* Italic */
            20+  /* point size */
            1+   /* fixed width */
            20+  /* Res X */
            20+  /* Res Y */
            strlen(format) +
            strlen(encoding));

        sprintf(qt_settings.font, format,
                foundry ? foundry : "*",
                family,
                weight_str(weight),
                italic ? "i" : "r",
                (int)(size*10),
                resX,
                resY,
                fixedW ? "m" : "p",
                encoding);
#endif
#ifdef QTC_DEBUG
printf("REQUEST FONT: %s\n", qt_settings.font);
#endif
    }

#if GTK_MAJOR_VERSION>1
    if(rd&RD_ICONS && !qt_settings.icons)
    {
        qt_settings.icons=(char *)malloc(DEF_ICONS_LEN+1);
        strcpy(qt_settings.icons, DEF_ICONS);
    }
    if(rd&RD_TOOLBAR_STYLE && !(found&RD_TOOLBAR_STYLE))
        qt_settings.toolbar.style=GTK_TOOLBAR_ICONS;
    if(rd&RD_TOOLBAR_ICON_SIZE && !(found&RD_TOOLBAR_ICON_SIZE))
        qt_settings.toolbar.size=GTK_ICON_SIZE_LARGE_TOOLBAR;
    if(rd&RD_BUTTON_ICONS && !(found&RD_BUTTON_ICONS))
        qt_settings.button_icons=TRUE;
#endif

    return found;
}

static int qt_refs=0;

#if GTK_MAJOR_VERSION>1

#define MAX_APP_NAME_LEN 32
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdio.h>
#include <dirent.h>

static const char * kde_globals()
{
    static char kg[QTC_MAX_FILENAME_LEN+1]={'\0'};

    char *kdehome=get_kde_home();

    if(kdehome && strlen(kdehome)<(QTC_MAX_FILENAME_LEN-strlen("/share/config/kdeglobals")))
        sprintf(kg, "%s/share/config/kdeglobals", kdehome);

    return kg;
}

#define HICOLOR_ICONS    "hicolor"
#define HICOLOR_LEN      7
#define ICON_FOLDER      "/share/icons/"
#define ICON_FOLDER_SLEN 13
static char * qt_geticon_path()
{
    static char *path=NULL;
    char        *kdeHome=get_kde_home();
    gboolean    nonDefIcons=qt_settings.icons && strcmp(qt_settings.icons, DEF_ICONS);
    int         len=strlen("pixmap_path \""),
                kdeHomeLen=kdeHome ? strlen(kdeHome) : 0,
                kdeIconPrefixLen=strlen(KDE_ICONS_PREFIX),
                iconLen=qt_settings.icons ? strlen(qt_settings.icons) : 0;

    if(nonDefIcons)
    {
        if(kdeHome)
        {
            len+=kdeHomeLen;
            len+=ICON_FOLDER_SLEN;
            len+=iconLen;
            len++;
        }
        len+=kdeIconPrefixLen;
        len++;
        len+=iconLen;
        len++;
    }

    if(kdeHome)
    {
        len+=kdeHomeLen;
        len+=ICON_FOLDER_SLEN;
        len+=DEF_ICONS_LEN;
        len++;
    }
    len+=kdeIconPrefixLen;
    len++;
    len+=DEF_ICON_DIR_LEN;
    len++;

    if(kdeHome)
    {
        len+=kdeHomeLen;
        len+=ICON_FOLDER_SLEN;
        len+=HICOLOR_LEN;
        len++;
    }
    len+=kdeIconPrefixLen;
    len++;
    len+=HICOLOR_LEN;
    len++;
    len++;

    if(path && len!=(strlen(path)+1))
        free(path);

    if(!path)
        path=(char *)malloc(len+1);

    strcpy(path, "pixmap_path \"");

    if(nonDefIcons)
    {
        if(kdeHome)
        {
            strcat(path, kdeHome);
            strcat(path, ICON_FOLDER);
            strcat(path, qt_settings.icons);
            strcat(path, ":");
        }
        strcat(path, KDE_ICONS_PREFIX);
        strcat(path, "/");
        strcat(path, qt_settings.icons);
        strcat(path, ":");
    }

    if(kdeHome)
    {
        strcat(path, kdeHome);
        strcat(path, ICON_FOLDER);
        strcat(path, DEF_ICONS);
        strcat(path, ":");
    }

    strcat(path, KDE_ICONS_PREFIX);
    strcat(path, "/");
    strcat(path, DEF_ICON_DIR);
    strcat(path, ":");

    if(kdeHome)
    {
        strcat(path, kdeHome);
        strcat(path, ICON_FOLDER);
        strcat(path, HICOLOR_ICONS);
        strcat(path, ":");
    }
    strcat(path, KDE_ICONS_PREFIX);
    strcat(path, "/");
    strcat(path, HICOLOR_ICONS);

    strcat(path, "\"");
    return path;
}

#define GIMP_PLUGIN "gimpplugin"

static char * qtc_get_app_name_from_pid(int pid)
{
/* CPD: There must be an easier way than this? */
    static char app_name[MAX_APP_NAME_LEN+1]="";

        int  procFile=-1;
        char cmdline[MAX_LINE_LEN+1];

        sprintf(cmdline, "/proc/%d/cmdline", pid);

        if(-1!=(procFile=open(cmdline, O_RDONLY)))
        {
            if(read(procFile, cmdline, MAX_LINE_LEN)>2)
            {
                int      len=strlen(cmdline),
                         pos=0;
                gboolean found_slash=FALSE;

                for(pos=len-1; pos>=0 && cmdline[pos] && !found_slash; --pos)
                    if('/'==cmdline[pos])
                    {
                        pos++;
                        found_slash=TRUE;
                    }

                if(!found_slash)
                    pos=0;  /* Perhaps no / */
                if(pos>=0)
                    if(NULL!=strstr(cmdline, "gimp/2.0/plug-ins"))
                        strcpy(app_name, GIMP_PLUGIN);
                    else
                    {
                        strncpy(app_name, &cmdline[pos ? pos+1 : 0], MAX_APP_NAME_LEN);
                        app_name[MAX_APP_NAME_LEN]='\0';
                    }
            }
            close(procFile);
        }

    return app_name;
}

static char * qtc_get_app_name()
{
    char *name=qtc_get_app_name_from_pid(getpid());

    if(0==strcmp(name, "perl") || 0==strcmp(name, "python"))
    {
        name=qtc_get_app_name_from_pid(getppid());
        if(!name)
            return "scriptedapp";
        else if(name==strstr(name, "gimp"))
            return GIMP_PLUGIN;

    }

    return name;
}

#ifdef QTC_MODIFY_MOZILLA

#define MAX_CSS_HOME     256
#define CSS_DEFAULT      ".default"
#define CSS_DEFAULT_ALT  "default."
#define USER_CHROME_DIR  "/chrome"
#define USER_CHROME_FILE "userChrome.css"
#define USER_CHROME_CSS  USER_CHROME_DIR"/"USER_CHROME_FILE
#define MAX_DEFAULT_NAME 16+strlen(CSS_DEFAULT)+strlen(USER_CHROME_CSS)

#define MENU_TEXT_STR "menubar > menu { color: HighlightText !important; } menubar > menu[_moz-menuactive=\"true\"] "\
                      "{ background-color : HighlightText !important; color: HighlightText !important; } "\
                      "/* Added by QtCurve -- do not remove */\n"

#define CSS_FILE_STR  "@import url(\"file://"QTC_MOZILLA_DIR"/QtCurve.css\"); /* Added by QtCurve -- do not remove */\n"

static void qtc_process_userChrome_css(char *file, gboolean add_menu_colors)
{
    FILE     *f=fopen(file, "r");
    char     *contents=NULL;
    gboolean add_css=TRUE,
             remove_menu_colors=FALSE;
    int      size=0;


    if(f)
    {
        if(0==fseek(f, 0, SEEK_END) && 0==fgetpos(f, (fpos_t*)&size) && size>0 && 0==fseek(f, 0, SEEK_SET))
        {
            size+=strlen(MENU_TEXT_STR)+strlen(CSS_FILE_STR)+3;

            contents=(char *)malloc(size);

            if(contents)
            {
                char  *line=NULL;
                size_t len=0;

                contents[0]='\0';
                while(-1!=getline(&line, &len, f))
                {
                    gboolean write_line=TRUE;

                    if(0==strcmp(line, CSS_FILE_STR))
                        add_css=FALSE;
                    else
                        if(0==strcmp(line, MENU_TEXT_STR))
                            if (add_menu_colors)
                                add_menu_colors=FALSE;
                            else
                            {
                                remove_menu_colors=TRUE;
                                write_line=FALSE;
                            }

                    if(write_line)
                        strcat(contents, line);
                }
                if (line)
                     free(line);
            }
        }

        fclose(f);
    }

    if(!contents || add_css || add_menu_colors)
    {
        if(!contents)
        {
            size=strlen(MENU_TEXT_STR)+strlen(CSS_FILE_STR)+3;

            contents=(char *)malloc(size);
            if(contents)
                contents[0]='\0';
        }

        if(contents)
        {
            if(add_css)  /* CSS needs to be on 1st line */
            {
                char *css_contents=(char *)malloc(size);

                if(css_contents)
                {
                    css_contents[0]='\0';
                    strcat(css_contents, CSS_FILE_STR);
                    strcat(css_contents, contents);
                    free(contents);
                    contents=css_contents;
                }
            }
            if(add_menu_colors)  /* This can be on last line */
            {
                int len=strlen(contents);

                if(len && contents[len-1]!='\n')
                    strcat(contents, "\n");
                strcat(contents, MENU_TEXT_STR);
            }
        }
    }

    if(contents && (add_css || add_menu_colors || remove_menu_colors))
    {
        f=fopen(file, "w");

        if(f)
        {
            fputs(contents, f);
            fclose(f);
        }
        free(contents);
    }
}

static void qtc_process_mozilla_app(gboolean add_menu_colors, char *app, gboolean under_moz)
{
    const char *home=getHome();

    if(home && (strlen(home)+strlen(app)+10+MAX_DEFAULT_NAME)<MAX_CSS_HOME)     /* 10 for .mozilla/<app>/ */
    {
        char cssHome[MAX_CSS_HOME+1];
        DIR  *dir=NULL;

        sprintf(cssHome, under_moz ? "%s/.mozilla/%s/" : "%s/.%s/", home, app);

        if((dir=opendir(cssHome)))
        {
             struct dirent *dir_ent=NULL;

             for (dir_ent=readdir(dir); dir_ent; dir_ent=readdir(dir))
             {
                 char *str=NULL;

                 if(((str=strstr(dir_ent->d_name, CSS_DEFAULT)) && str>=dir_ent->d_name &&
                    '\0'==str[strlen(CSS_DEFAULT)]) ||
                    ((str=strstr(dir_ent->d_name, CSS_DEFAULT_ALT)) && str==dir_ent->d_name &&
                    '\0'!=str[strlen(CSS_DEFAULT_ALT)]))
                 {
                     char        sub[MAX_CSS_HOME];
                     struct stat statbuf;

#ifdef QTC_MODIFY_MOZILLA_USER_JS
                     /* Add custom user.js file */
                     sprintf(sub, "%s%s/user.js", cssHome, dir_ent->d_name);
                     if(-1==lstat(sub, &statbuf))
                     {
                         FILE *in=NULL;

                         sprintf(sub, QTC_MOZILLA_DIR"/%s-user.js", app);

                         if((in=fopen(sub, "r")))
                         {
                             FILE *out=fopen(sub, "w");

                             if(out)
                             {
                                 char ch;

                                 while((ch=fgetc(in))!=EOF)
                                     fputc(ch, out);
                                 fclose(out);
                             }
                             fclose(in);
                         }
                     }
#endif

                     /* Now do userChrome.css */
                     sprintf(sub, "%s%s%s/", cssHome, dir_ent->d_name, USER_CHROME_DIR);

                     if(-1!=lstat(sub, &statbuf) && S_ISDIR(statbuf.st_mode))
                     {
                         strcat(sub, USER_CHROME_FILE);
                         qtc_process_userChrome_css(sub, add_menu_colors);
                     }
                 }
             }

             closedir(dir);
        }
    }
}
#endif

#endif

#if GTK_MAJOR_VERSION>1 && defined QTC_ADD_EVENT_FILTER
/* ... Taken from Qt engine ... */

#include <gdk/gdkx.h>
#include <X11/X.h>
#include <X11/Xlib.h>
static Atom kipcCommAtom;
static Atom desktopWindowAtom;
static Atom knwmCommAtom;

static void qt_init();
static void qt_exit();
static GdkFilterReturn qt_gdk_event_filter(GdkXEvent *xevent, GdkEvent *gevent, gpointer data)
{
    XEvent *event =(XEvent*) xevent;

    if ( (ClientMessage==event->type && kipcCommAtom==event->xclient.message_type) ||
         (PropertyNotify==event->type && knwmCommAtom==event->xclient.message_type) )
    {
        qt_refs=1;
        qt_exit();
        gtk_rc_reset_styles(gtk_settings_get_default());

        return GDK_FILTER_REMOVE;
    }

    return GDK_FILTER_CONTINUE;
}

static void qt_add_event_filter() /* GdkWindow *widget) */
{
    static int done=0;

    if(!done)
    {
        /* Create a new window, and set the KDE_DESKTOP_WINDOW property on it
           This window will then receive events from KDE when the style changes */

        GtkWindow *top=gtk_window_new(GTK_WINDOW_TOPLEVEL);

        if(top && GTK_IS_WIDGET(top))
        {
            long data = 1;

            gtk_window_iconify(GTK_WINDOW(top));
            gtk_widget_show(top);
            gtk_window_set_skip_taskbar_hint(GTK_WINDOW(top), TRUE);
            gtk_window_set_skip_pager_hint(GTK_WINDOW(top), TRUE);

            /* Get KDE related atoms from the X server */
            kipcCommAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "KIPC_COMM_ATOM", FALSE);
            knwmCommAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "_KDE_NET_WM_FRAME_STRUT", FALSE);
            desktopWindowAtom = XInternAtom(gdk_x11_get_default_xdisplay(), "KDE_DESKTOP_WINDOW", FALSE);

            XChangeProperty(gdk_x11_get_default_xdisplay(), GDK_WINDOW_XID(GTK_WIDGET(top)->window),
                            desktopWindowAtom, desktopWindowAtom, 32, PropModeReplace,
                            (unsigned char *)&data, 1);
            /* This filter will intercept those events */
            gdk_window_add_filter(GTK_WIDGET(top)->window, qt_gdk_event_filter, 0);
            gtk_widget_hide(top);
            done=1;
        }
    }
}

/* ... */

#endif

static void shade(GdkColor *a, GdkColor *b, float k);

static void qt_init(Options *opts)
{
    if(0==qt_refs)
    {
        static int lastRead=0;

        int now=time(NULL);

#if GTK_MAJOR_VERSION>1
        qt_settings.app=GTK_APP_UNKNOWN;
#endif

        if(abs(now-lastRead)>1)
        {
#if GTK_MAJOR_VERSION>1
            char        *app=NULL,
                        *path=NULL;
            GtkSettings *settings=NULL;

            qt_settings.icons=NULL;
            qt_settings.progressfont_rc=NULL;
#endif
            qt_settings.font=NULL;
            lastRead=now;

            defaultSettings(opts);
            read_rc("/etc/qt/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, TRUE, FALSE);
            read_rc("/etc/qt3/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, TRUE, FALSE);
            read_rc(".qt/qtrc", RD_PALETTE|RD_FONT|RD_CONTRAST, opts, FALSE, TRUE);
            readConfig(NULL, opts);

#if GTK_MAJOR_VERSION==1
            opts->shadeMenubars=SHADE_NONE;
            opts->menubarAppearance=APPEARANCE_FLAT;
#endif

#if GTK_MAJOR_VERSION>1
            /* Check if we're firefox... */
            if((app=qtc_get_app_name()))
                if((0==strcmp(app, "firefox-bin") || 0==strcmp(app, "thunderbird-bin") ||
                    0==strcmp(app, "swiftfox-bin") || 0==strcmp(app, "mozilla-thunderbird-bin")))
                {
#ifdef QTC_MODIFY_MOZILLA
                    GdkColor *menu_col=SHADE_CUSTOM==opts->shadeMenubars
                                        ? &opts->customMenubarsColor
                                        : &qt_settings.colors[COLOR_SELECTED];
                    gboolean add_menu_colors=FALSE;

                    if(SHADE_SELECTED==opts->shadeMenubars || (SHADE_CUSTOM==opts->shadeMenubars &&
                                                               TOO_DARK(*menu_col) ))
                        add_menu_colors=TRUE;

                    if(0==strcmp(app, "firefox-bin") || 0==strcmp(app, "swiftfox-bin"))
                        qtc_process_mozilla_app(add_menu_colors, "firefox", TRUE);
                    else if(0==strcmp(app, "thunderbird-bin"))
                        qtc_process_mozilla_app(add_menu_colors, "thunderbird", FALSE);
                    else if(0==strcmp(app, "mozilla-thunderbird-bin"))
                        qtc_process_mozilla_app(add_menu_colors, "mozilla-thunderbird", FALSE);
#endif
                    qt_settings.app=GTK_APP_MOZILLA;
                }
                else if(0==strcmp(app, "soffice.bin"))
                    qt_settings.app=GTK_APP_OPEN_OFFICE;
                else if(0==strcmp(app, "vmplayer"))
                    qt_settings.app=GTK_APP_VMPLAYER;
                else if(0==strcmp(app, GIMP_PLUGIN))
                    qt_settings.app=GTK_APP_GIMP_PLUGIN;
                else if(app==strstr(app, "gimp"))
                    qt_settings.app=GTK_APP_GIMP;

            if(GTK_APP_MOZILLA==qt_settings.app)
            {
                /* KDE's "apply colors to non-KDE apps" messes up firefox! so need to fix this! */
                char filename[QTC_MAX_FILENAME_LEN+1]={'\0'};
                char *kdehome=get_kde_home();

                if(kdehome &&
                   strlen(kdehome)<(QTC_MAX_FILENAME_LEN-strlen("/share/config/gtkrc-2.0")))
                {
                    sprintf(filename, "%s/share/config/gtkrc-2.0", kdehome);

                    char *gtkEnv=getenv("GTK2_RC_FILES");

                    /* its in the users environment... */
                    if(gtkEnv && NULL!=strstr(gtkEnv, filename))
                    {
                        struct stat statbuf;

                        /* Check if KDE's gtkrc file exists */
                        if(-1!=lstat(filename, &statbuf))
                        {
                            static const int constVersion=1;

                            sprintf(filename, "%s/share/config/gtkrc-qtc", kdehome);

                            FILE     *f=fopen(filename, "r");
                            gboolean write_file=TRUE;
                            char     version[10];

                            sprintf(version, "#%02d%02X%02X%02X",
                                         constVersion,
                                         qt_settings.colors[COLOR_TEXT_SELECTED].red>>8,
                                         qt_settings.colors[COLOR_TEXT_SELECTED].green>>8,
                                         qt_settings.colors[COLOR_TEXT_SELECTED].blue>>8);

                            /* Read any previous file, and check previous color setting! */
                            if(f)
                            {
                                char line[8];

                                write_file=NULL==fgets(line, 10, f) || memcmp(version, line, 9);
                                fclose(f);
                            }

                            /* Either it didnt exist, or the colors are different */
                            if(write_file && (f=fopen(filename, "w")))
                            {
#ifdef HAVE_LOCALE_H
                               /* We need to switch to the C locale in order
                                  to have decimal separator recognized by gtkrc engine */
                                char *locale = setlocale(LC_NUMERIC, "C");
#endif

                                fprintf(f, "%s\n"
                                           "# Fix for firefox and KDE's \"apply color to non-KDE"
                                           " apps\" setting\n"
                                           "# Please do not alter this file - it is used, and written"
                                           " by, QtCurve\n"
                                           "style \"qtcurve-text-color-fix\"\n"
                                           "{\n"
                                           "    fg[ACTIVE] = { %5.3f, %5.3f, %5.3f }\n"
                                           "    fg[PRELIGHT]={ %5.3f, %5.3f, %5.3f }\n}\n"
                                           "widget_class \"*Menu*\" style \"qtcurve-text-color-fix\"",
                                           version,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].red>>8)/255.0,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].green>>8)/255.0,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].blue>>8)/255.0,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].red>>8)/255.0,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].green>>8)/255.0,
                                           (qt_settings.colors[COLOR_TEXT_SELECTED].blue>>8)/255.0);
#ifdef HAVE_LOCALE_H
                                if(locale)
                                    setlocale(LC_NUMERIC, locale);
#endif
                                fclose(f);
                            }

                            /* Now get gtk to read this file *after* its other gtkrc files - this
                               allows us to undo the KDE settings! */
                            gtk_rc_add_default_file(filename);
                        }
                    }
                }
            }
            else if(GTK_APP_VMPLAYER==qt_settings.app)
                opts->shadeMenubars=SHADE_NONE;

            read_rc(kde_globals(), RD_ICONS|RD_TOOLBAR_STYLE|RD_TOOLBAR_ICON_SIZE|RD_BUTTON_ICONS,
                    opts, TRUE, FALSE);

            /* Tear off menu items dont seem to draw they're background, and the default background
               is drawn :-(  Fix/hack this by making that background the correct color */
            if(opts->lighterPopupMenuBgnd)
            {
                static const char *format="style \"qtcurve-menu-fix\""
                                          "{bg[NORMAL]={%5.3f, %5.3f, %5.3f}}\n"
                                          "class \"GtkMenu\" style \"qtcurve-menu-fix\"\n";
                char *menuFix=(char *)malloc(strlen(format)+32);

                if(menuFix)
                {
                    GdkColor col;

                    shade(&qt_settings.colors[COLOR_BUTTON], &col, POPUPMENU_LIGHT_FACTOR);
                    sprintf(menuFix, format, (col.red>>8)/255.0, (col.green>>8)/255.0,
                                             (col.blue>>8)/255.0);
                    gtk_rc_parse_string(menuFix);
                    free(menuFix);
                }
            }

            if((settings=gtk_settings_get_default()))
            {
                GtkSettingsValue svalue;

                if(qt_settings.font)
                    g_object_set(settings, "gtk-font-name", qt_settings.font, 0);

                g_object_set(settings, "gtk-icon-sizes",
                                       "gtk-large-toolbar=22,22:"
                                       "gtk-button=16,16:"
                                       "gtk-dnd=32,32:"
                                       "gtk-menu=16,16:"
                                       "gtk-small-toolbar=16,16:"
                                       "gtk-dialog=22,22", 0);

                svalue.origin="KDE-Settings";
                svalue.value.g_type=G_TYPE_INVALID;
                g_value_init(&svalue.value, G_TYPE_LONG);
                g_value_set_long(&svalue.value, qt_settings.toolbar.style);
                gtk_settings_set_property_value(settings, "gtk-toolbar-style", &svalue);
#ifdef QTC_DEBUG
                printf("gtk-toolbar-style %d\n", qt_settings.toolbar.style);
#endif
                g_value_set_long(&svalue.value, qt_settings.toolbar.size);
                gtk_settings_set_property_value(settings, "gtk-toolbar-icon-size", &svalue);
#ifdef QTC_DEBUG
                printf("gtk-toolbar-icon-size %d\n", qt_settings.toolbar.size);
#endif
                if(NULL==gtk_check_version(2, 4, 0)) /* The following settings only apply for GTK>=2.4.0 */
                {
                    g_value_set_long(&svalue.value, qt_settings.button_icons);
#ifdef QTC_DEBUG
                    printf("gtk-button-images %d\n", qt_settings.button_icons);
#endif
                    gtk_settings_set_property_value(settings, "gtk-button-images", &svalue);

                    if(opts->drawStatusBarFrames)
                        gtk_rc_parse_string("style \"QtCurveStyle_StatusBarSettings\""
                                            "{ GtkStatusbar::shadow-type = 1 }" /*GtkStatusbar::has-resize-grip = FALSE }" */
                                            "class \"GtkStatusbar\" style"
                                            " \"QtCurveStyle_StatusBarSettings\"\n");
                    else
                        gtk_rc_parse_string("style \"QtCurveStyle_StatusBarSettings\""
                                            "{ GtkStatusbar::shadow-type = 0 }" /*GtkStatusbar::has-resize-grip = FALSE }" */
                                            "class \"GtkStatusbar\" style"
                                            " \"QtCurveStyle_StatusBarSettings\"\n");
                }
                if(IND_COLORED==opts->defBtnIndicator && FOCUS_COLORED==opts->focus)
                    gtk_rc_parse_string("style \"QtCurveStyle_ButtonFocusSetting\""
                                            "{ GtkButton::focus-padding = 2 } " 
                                            "class \"*Button\" style \"QtCurveStyle_ButtonFocusSetting\"\n");
                if(NULL==gtk_check_version(2, 6, 0)) /* The following settings only apply for GTK>=2.6.0 */
                {
                /*GtkSettingsValue bvalue;
                    g_value_init(&bvalue.value, G_TYPE_BOOLEAN);
                    g_value_set_boolean(&bvalue.value, TRUE);
                    gtk_settings_set_property_value(settings, "gtk-alternative-button-order", &bvalue);
                */
                    g_object_set(settings, "gtk-alternative-button-order", TRUE, NULL);
                }
            }

            if((path=qt_geticon_path()))
                gtk_rc_parse_string(path);

            if(qt_settings.progressfont_rc)
            {
                gtk_rc_parse_string(qt_settings.progressfont_rc);
                free(qt_settings.progressfont_rc);
                qt_settings.progressfont_rc=NULL;
            }
#else
            if(IND_COLORED==opts->defBtnIndicator)
                opts->defBtnIndicator=IND_FONT_COLOR;
#endif
        }
    }
    qt_refs++;
}

static void qt_exit()
{
    qt_refs--;

    if(0==qt_refs)
    {
        if(qt_settings.font)
            free(qt_settings.font);
        qt_settings.font=NULL;

#if GTK_MAJOR_VERSION>1
        if(qt_settings.icons)
            free(qt_settings.icons);
        qt_settings.font=NULL;
#endif
    }
}

#define SET_COLOR(st, rc, itm, ITEM, state, QTP_COL) \
    st->itm[state]=rc->color_flags[state]&ITEM ? rc->itm[state] : qt_settings.colors[QTP_COL];

static void qt_set_colors(GtkStyle *style, GtkRcStyle *rc_style)
{
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_NORMAL, COLOR_WINDOW)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_ACTIVE, COLOR_MID)
    SET_COLOR(style, rc_style, bg, GTK_RC_BG, GTK_STATE_PRELIGHT, COLOR_WINDOW)

    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_NORMAL, COLOR_BACKGROUND)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_SELECTED, COLOR_SELECTED)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_INSENSITIVE, COLOR_WINDOW)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_ACTIVE, COLOR_SELECTED)
    SET_COLOR(style, rc_style, base, GTK_RC_BASE, GTK_STATE_PRELIGHT, COLOR_SELECTED)

    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_NORMAL, COLOR_TEXT)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_ACTIVE, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, text, GTK_RC_TEXT, GTK_STATE_PRELIGHT, COLOR_TEXT)

    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_NORMAL, COLOR_FOREGROUND)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_SELECTED, COLOR_TEXT_SELECTED)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_INSENSITIVE, COLOR_MID)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_ACTIVE, COLOR_FOREGROUND)
    SET_COLOR(style, rc_style, fg, GTK_RC_FG, GTK_STATE_PRELIGHT, COLOR_FOREGROUND)
}

static void qt_set_font(GtkRcStyle *rc_style)
{
    if(qt_settings.font)
    {
#if GTK_MAJOR_VERSION>1
        if (rc_style->font_desc)
            pango_font_description_free (rc_style->font_desc);

        rc_style->font_desc = pango_font_description_from_string (qt_settings.font);
#else
        rc_style->font_name = g_strdup(qt_settings.font);
#endif
    }
}
