//////////////////////////////////////////////////////////////////////////////
// svgdeco.cc
// -------------------
// SVG Deco window decoration for KDE
// -------------------
// Copyright (c) 2003, 2004 David Johnson
// Please see the header file for copyright and license information.
//////////////////////////////////////////////////////////////////////////////

#include <kconfig.h>
#include <kdeversion.h>
#include <kglobal.h>
#include <kglobalsettings.h>
#include <klocale.h>
#include <kpixmap.h>
#include <kimageeffect.h>
#include <kpixmapeffect.h>
#include <kpixmap.h>
#include <kstandarddirs.h>

#include <qbitmap.h>
#include <qlabel.h>
#include <qlayout.h>
#include <qpainter.h>
#include <qtooltip.h>
#include <qtimer.h>

#include "svgdeco.h"
#include "buttons.h"

using namespace svgdeco;

//////////////////////////////////////////////////////////////////////////////
// svgdecoFactory Class
//////////////////////////////////////////////////////////////////////////////
svgdecoFactory* svgFactory=NULL;

bool svgdecoFactory::initialized_              = false;
Qt::AlignmentFlags svgdecoFactory::titlealign_ = Qt::AlignHCenter;
int svgdecoFactory::titlesize_                 = 30;
int svgdecoFactory::buttonsize_                = 26;
int svgdecoFactory::framesize_                 = 4;
bool svgdecoFactory::titleshadow_              = true;
bool svgdecoFactory::animatebuttons            = true;
int svgdecoFactory::btnComboBox                = 0;

// global constants
static const int TOPMARGIN       = 4; // do not change
static const int DECOHEIGHT      = 4; // do not change
static const int SIDETITLEMARGIN = 2;
// Default button layout
const char default_left[]  = "M";
const char default_right[] = "HIAX";

static const uint TIMERINTERVAL = 50; // msec
static const uint ANIMATIONSTEPS = 4;

extern "C" KDecorationFactory* create_factory() {
    return new svgdeco::svgdecoFactory();
}

//////////////////////////////////////////////////////////////////////////////
// svgdecoFactory()
// ----------------
// Constructor

svgdecoFactory::svgdecoFactory() {
    readConfig();
    initialized_ = true;
}

//////////////////////////////////////////////////////////////////////////////
// ~svgdecoFactory()
// -----------------
// Destructor

svgdecoFactory::~svgdecoFactory() {
    initialized_ = false;
}

//////////////////////////////////////////////////////////////////////////////
// createDecoration()
// -----------------
// Create the decoration

KDecoration* svgdecoFactory::createDecoration(KDecorationBridge* b) {
    return new svgdecoClient(b, this);
}

//////////////////////////////////////////////////////////////////////////////
// reset()
// -------
// Reset the handler. Returns true if decorations need to be remade, false if
// only a repaint is necessary

bool svgdecoFactory::reset(unsigned long changed) {
    // read in the configuration
    initialized_ = false;
    bool confchange = readConfig();
    initialized_ = true;

    if (confchange ||
            (changed & (SettingDecoration | SettingButtons | SettingBorder))) {
        return true;
    } else {
        resetDecorations(changed);
        return false;
    }
}

//////////////////////////////////////////////////////////////////////////////
// readConfig()
// ------------
// Read in the configuration file

bool svgdecoFactory::readConfig() {
    // create a config object
    KConfig config("kwinsvgdecorc");
    config.setGroup("General");

    // grab settings
    QString value = config.readEntry("TitleAlignment", "AlignHCenter");
    if (value == "AlignLeft")
        titlealign_ = Qt::AlignLeft;
    else if (value == "AlignHCenter")
        titlealign_ = Qt::AlignHCenter;
    else if (value == "AlignRight")
        titlealign_ = Qt::AlignRight;

    titlesize_ = config.readNumEntry("TitleSize",30);
    buttonsize_ = config.readNumEntry("ButtonSize",26);
    framesize_ = config.readNumEntry("FrameSize",4);
    titleshadow_ = config.readBoolEntry("TitleShadow", true);
    animatebuttons = config.readBoolEntry("AnimateButtons", true);
    btnComboBox = config.readNumEntry("ButtonComboBox", 0);

    if(buttonsize_ > titlesize_ - framesize_) buttonsize_ = titlesize_-framesize_;
    return true;
}

//////////////////////////////////////////////////////////////////////////////
// svgdecoButton Class 
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// svgdecoButton()
// ---------------
// Constructor
svgdecoButton::svgdecoButton(svgdecoClient *parent, const char *name, const QString& tip, ButtonType type, int button_size, bool toggle): QButton(parent->widget(), name), 
     client_(parent), 
     type_(type), 
     size_(button_size), 
     deco_(0), 
     lastmouse_(NoButton), 
     hover_(false)
{
    setBackgroundMode(NoBackground);
    setFixedSize( ::svgFactory->buttonSize(), ::svgFactory->buttonSize());
    setCursor(arrowCursor);
    QToolTip::add(this, tip);
    setToggleButton(toggle);
    //button animation setup
    animTmr = new QTimer(this);
    connect(animTmr, SIGNAL(timeout() ), this, SLOT(animate() ) );
    animProgress = 0;
}

svgdecoButton::~svgdecoButton() {
    if ( deco_ )
        delete deco_;
}

//////////////////////////////////////////////////////////////////////////////
// sizeHint()
// ----------
// Return size hint

QSize svgdecoButton::sizeHint() const {
    return QSize(::svgFactory->buttonSize(), ::svgFactory->buttonSize());
}

//////////////////////////////////////////////////////////////////////////////
// animate()
// ----------
// Button animation timing
void svgdecoButton::animate() {
    animTmr->stop();
    if(!hover_) {
        if(animProgress > 0) {
            if (::svgFactory->animateButtons() ) {
                animProgress--;
            } else {
                animProgress = 0;
            }
            animTmr->start(TIMERINTERVAL, true); // single-shot
        }
    } else {
        if(animProgress < ANIMATIONSTEPS) {
            if (::svgFactory->animateButtons() ) {
                animProgress++;
            } else {
                animProgress = ANIMATIONSTEPS;
            }
            animTmr->start(TIMERINTERVAL, true); // single-shot
        }
    }
    repaint(false);
}
//////////////////////////////////////////////////////////////////////////////
// enterEvent()
// ------------
// Mouse has entered the button

void svgdecoButton::enterEvent(QEvent *e) {
    // we want to do mouseovers, so keep track of it here
    hover_=true; 
    repaint( hover_ );
    animate();
    // we wanted to pass on the event
    QButton::enterEvent(e);
    
}

//////////////////////////////////////////////////////////////////////////////
// leaveEvent()
// ------------
// Mouse has left the button

void svgdecoButton::leaveEvent(QEvent *e) {
    // we want to do mouseovers, so keep track of it here
    hover_=false; 
    repaint( hover_ );
    animate();
    // we wanted to pass on the event
    QButton::leaveEvent(e);
}

//////////////////////////////////////////////////////////////////////////////
// mousePressEvent()
// -----------------
// Button has been pressed

void svgdecoButton::mousePressEvent(QMouseEvent* e) {
    lastmouse_ = e->button();

    // translate and pass on mouse event
    int button = LeftButton;
    if ((type_ != ButtonMax) && (e->button() != LeftButton)) {
        button = NoButton; // middle & right buttons inappropriate
    }
    QMouseEvent me(e->type(), e->pos(), e->globalPos(),
                   button, e->state());
    QButton::mousePressEvent(&me);
}

//////////////////////////////////////////////////////////////////////////////
// mouseReleaseEvent()
// -----------------
// Button has been released

void svgdecoButton::mouseReleaseEvent(QMouseEvent* e) {
    lastmouse_ = e->button();

    // translate and pass on mouse event
    int button = LeftButton;
    if ((type_ != ButtonMax) && (e->button() != LeftButton)) {
        button = NoButton; // middle & right buttons inappropriate
    }
    QMouseEvent me(e->type(), e->pos(), e->globalPos(), button, e->state());
    QButton::mouseReleaseEvent(&me);
}

void svgdecoButton::setOn(bool on)
{
        QButton::setOn(on);
}

//////////////////////////////////////////////////////////
// getButtonImage()
// ----------------
// get the button QImage based on type and window mode
QImage svgdecoButton::getButtonImage(ButtonType type)
{
    QImage finalImage;
    QImage tempImage;
    switch(type) {
        case ButtonClose:
            finalImage = uic_findImage( "close.png" );
            break;
        case ButtonHelp:
            finalImage = uic_findImage( "help.png" );
            break;
        case ButtonMin:
            finalImage = uic_findImage( "minimize.png" );
            break;
        case ButtonMax:
            if(client_->maximizeMode() == KDecorationDefines::MaximizeFull)
            {
                finalImage = uic_findImage( "restore.png" );
            }
            else
            {
                finalImage = uic_findImage( "maximize.png" );
            }
            break;
        case ButtonSticky:
            if(client_->isOnAllDesktops())
            {
                finalImage = uic_findImage( "splat.png" );
            }
            else
            {
                finalImage = uic_findImage( "circle.png" );
            }
            break;
        case ButtonShade:
            if(client_->isShade())
            {
                tempImage = uic_findImage( "minimize.png" );
                finalImage = tempImage.mirror();
            }
            else
            {
                tempImage = uic_findImage( "minimize.png" );
                finalImage = tempImage.mirror();
            }
            break;
        case ButtonAbove:
            if(client_->keepAbove())
            {
                finalImage = uic_findImage( "keep_above_lit.png" );
            }
            else
            {
                finalImage = uic_findImage( "keep_above.png" );
            }
            break;
        case ButtonBelow:
            if(client_->keepBelow())
            {
                finalImage = uic_findImage( "keep_below_lit.png" );
            }
            else
            {
                finalImage = uic_findImage( "keep_below.png" );
            }
            break;
        default:
            finalImage = uic_findImage( "splat.png" );
            break;
    }
    return finalImage;
}

//////////////////////////////////////////////////////////
// drawButton()
// -------------------------
// draw the pixmap button

void svgdecoButton::drawButton( QPainter *painter ) {
    if ( !svgdecoFactory::initialized() )
        return ;
    
    int newWidth = width() - 2;
    int newHeight = height() - 2;
    int dx = (width() - newWidth) / 2;
    int dy = (height() - newHeight) / 2;
    //int FRAMESIZE = ::svgFactory->frameSize();
    QImage tmpResult, tmpImage;
    QImage maskImage = uic_findImage("mask.bmp").smoothScale(width(),height());
    QColorGroup group;
    bool active = client_->isActive();
    KPixmap backgroundTile = client_->getTitleBarTile(active);
    QColor whiteColor(white);
    KPixmap bgBlend;

    // paint with the titlebar color set
    group = KDecoration::options()->colorGroup(KDecoration::ColorTitleBar, client_->isActive());

    //draw the titlebar behind the buttons and app icons
    painter->drawTiledPixmap(0, 0, width(), height(), backgroundTile, 0, y());
    
    //scale to be just smaller than the button region our pixmap will reside in
    //mostly for spacing so the buttons are not too close together
    //QImage buttonImage = getButtonImage(type_).smoothScale( newWidth, newHeight);
    QImage buttonImage = getButtonImage(type_).smoothScale(width(),height());
    if (type_ == ButtonMenu) {
        //slight movement to show the menu button is depressed
        if (isDown()) {
            dx++;
            dy++;
        }
        QPixmap menuButtonPixmap(client_->icon().pixmap(QIconSet::Large, QIconSet::Normal));
        QImage menuButtonImage(menuButtonPixmap.convertToImage());
        //draw the menu button the same size as the other buttons
        //using QIconSet::Large gives us a 32x32 icon to resize, resizing larger than
        //that may produce pixilation of the image
        painter->drawImage( dx, dy, menuButtonImage.smoothScale(newWidth, newHeight) );
    } else {
        //highlight on a mouse over repaint
        double factor = animProgress * 0.13;

        if(!isDown())
        {
            switch(::svgFactory->getBtnComboBox())
            {
                case 0:
                    {
                    //draw the titlebar behind the buttons and app icons
                    KPixmap p;
                    p.convertFromImage( maskImage );
                    
                    QPainter bP(&p);
                    bgBlend = KPixmapEffect::intensity(backgroundTile, factor);
                    bP.drawTiledPixmap(0, 0, width(), height(), bgBlend, 0, y());
                    
                    if ( !p.mask() )
                    {
                        QBitmap bm;
                        bm = maskImage.createHeuristicMask();
                        bm.setMask(bm);
                        p.setMask( *bm.mask() );
                    }
                    painter->drawPixmap( 0, 0, p);
                    tmpResult = buttonImage;
                    break;
                    }
                case 1:
                    {
                    tmpResult = KImageEffect::fade( buttonImage, factor, group.background());
                    break;
                    }
            }
        }
        else
        {
            tmpResult = buttonImage;
        }
        painter->drawPixmap( 0, 0, QPixmap( tmpResult ) );
    }
}


//////////////////////////////////////////////////////////////////////////////
// svgdecoClient Class
//////////////////////////////////////////////////////////////////////////////

//////////////////////////////////////////////////////////////////////////////
// svgdecoClient()
// ---------------
// Constructor

svgdecoClient::svgdecoClient(KDecorationBridge *b, KDecorationFactory *f)
    : KDecoration(b, f),
    mainLayout_(0),
    topSpacer_(0),
    titleSpacer_(0),
    leftTitleSpacer_(0), rightTitleSpacer_(0),
    decoSpacer_(0),
    leftSpacer_(0), rightSpacer_(0),
    bottomSpacer_(0), windowSpacer_(0),
    aCaptionBuffer(0), iCaptionBuffer(0),
    aTitleBarTile(0), iTitleBarTile(0), aTitleBarTopTile(0), iTitleBarTopTile(0),
    pixmaps_created(false),
    //captionBufferDirty(true),
    //closing(false),
    s_titleHeight(0),
    s_titleFont(QFont())
    {
        aCaptionBuffer = new QPixmap();
        iCaptionBuffer = new QPixmap();
        //s_titleFont = isTool()?svgdecoFactory::titleFontTool():svgdecoFactory::titleFont();
        s_titleFont = options()->font();
        s_titleHeight = svgdecoFactory::titleSize();
    }
//////////////////////////////////////////////////////////////////////////////////
// ~svgdecoClient()
// --------------------
// Destructor
svgdecoClient::~svgdecoClient() {
    delete aCaptionBuffer;
    delete iCaptionBuffer;
}

void svgdecoClient::create_pixmaps() {
    if(pixmaps_created)
        return;
    //KPixmap alphaPixmap(QPixmap("images/titlealpha.png"));
    int FRAMESIZE = ::svgFactory->frameSize();
    QPainter painter;
    QImage alphaImage, blendImage;
    KPixmap blendPixmap;
    QColorGroup group,widgetGroup;
    // Get the color groups we need for the gradients
    group = options()->colorGroup(KDecoration::ColorTitleBar, true);
    widgetGroup = widget()->colorGroup();
    // Get the alpha png for the glossy title bar
    alphaImage = uic_findImage("titlealpha.png");
    // active title bar tile
    group = options()->colorGroup(KDecoration::ColorTitleBar, true);
    blendPixmap.resize(1,s_titleHeight+FRAMESIZE);
    blendPixmap = KPixmapEffect::gradient(blendPixmap,
                                          group.light(),
                                          group.background(),
                                          KPixmapEffect::VerticalGradient);
    aTitleBarTile = new QPixmap(1, s_titleHeight+FRAMESIZE);
    painter.begin(aTitleBarTile);
    painter.fillRect(0,0,1,s_titleHeight+FRAMESIZE,group.background());
    //painter.drawPixmap(0,0, blendPixmap);
    painter.drawImage(0, 2, alphaImage.smoothScale(1,s_titleHeight+FRAMESIZE-4));
    painter.end();
    
    // inactive title bar tile
    group = options()->colorGroup(KDecoration::ColorTitleBar, false);
    blendPixmap = KPixmapEffect::gradient(blendPixmap,
                                          group.light(),
                                          group.background(),
                                          KPixmapEffect::VerticalGradient);
    iTitleBarTile = new QPixmap(1, s_titleHeight+FRAMESIZE);
    painter.begin(iTitleBarTile);
    painter.fillRect(0,0,1,s_titleHeight+FRAMESIZE,group.background());
    //painter.drawPixmap(0,0, blendPixmap);
    painter.drawImage(0, 2, alphaImage.smoothScale(1,s_titleHeight+FRAMESIZE-4));
    painter.end();

    pixmaps_created = true;
}

void svgdecoClient::delete_pixmaps() {
    delete aTitleBarTopTile;
    aTitleBarTopTile = 0;

    delete iTitleBarTopTile;
    iTitleBarTopTile = 0;

    delete aTitleBarTile;
    aTitleBarTile = 0;

    delete iTitleBarTile;
    iTitleBarTile = 0;

    pixmaps_created = false;
}
//////////////////////////////////////////////////////////////////////////////
// init()
// ------
// Actual initializer for class

void svgdecoClient::init() {
    createMainWidget(WResizeNoErase | WRepaintNoErase);
    widget()->installEventFilter(this);
    handlebar = ::svgFactory->frameSize() < 4 ? 4 - ::svgFactory->frameSize() : 0;
    // for flicker-free redraws
    widget()->setBackgroundMode(NoBackground);

    _resetLayout();

    create_pixmaps();
}
void svgdecoClient::_resetLayout()
{
    // basic layout:
    //  _______________________________________________________________
    // |                         topSpacer                             |
    // |_______________________________________________________________|
    // | leftTitleSpacer | btns |  titlebar   | bts | rightTitleSpacer |
    // |_________________|______|_____________|_____|__________________|
    // |                         decoSpacer                            |
    // |_______________________________________________________________|
    // | |                                                           | |
    // | |                      windowWrapper                        | |
    // | |                                                           | |
    // |leftSpacer                                          rightSpacer|
    // |_|___________________________________________________________|_|
    // |                           bottomSpacer                        |
    // |_______________________________________________________________|
    //
    if (!::svgFactory->initialized()) return;
    
    delete mainLayout_;
    delete topSpacer_;
    delete titleSpacer_;
    delete leftTitleSpacer_;
    delete rightTitleSpacer_;
    delete decoSpacer_;
    delete leftSpacer_;
    delete rightSpacer_;
    delete bottomSpacer_;
    delete windowSpacer_;

    mainLayout_       = new QVBoxLayout(widget());
    // title
    QHBoxLayout *titleLayout_ = new QHBoxLayout();
    QHBoxLayout *windowLayout_ = new QHBoxLayout();
    int FRAMESIZE = ::svgFactory->frameSize();

    topSpacer_        = new QSpacerItem(1, TOPMARGIN, QSizePolicy::Expanding, QSizePolicy::Fixed);
    titlebar_         = new QSpacerItem(1, s_titleHeight+FRAMESIZE,
                                        QSizePolicy::Expanding, QSizePolicy::Fixed);
    leftTitleSpacer_  = new QSpacerItem(FRAMESIZE*2, s_titleHeight+FRAMESIZE,
                                        QSizePolicy::Fixed, QSizePolicy::Fixed);
    rightTitleSpacer_ = new QSpacerItem(FRAMESIZE*2, s_titleHeight+FRAMESIZE,
                                        QSizePolicy::Fixed, QSizePolicy::Fixed);
    decoSpacer_       = new QSpacerItem(1, DECOHEIGHT, QSizePolicy::Expanding, QSizePolicy::Fixed);
    leftSpacer_       = new QSpacerItem(::svgFactory->frameSize(), 1,
                                        QSizePolicy::Fixed, QSizePolicy::Expanding);
    rightSpacer_      = new QSpacerItem(::svgFactory->frameSize(), 1,
                                        QSizePolicy::Fixed, QSizePolicy::Expanding);
    bottomSpacer_     = new QSpacerItem(1, ::svgFactory->frameSize(),
                                        QSizePolicy::Expanding, QSizePolicy::Fixed);
    
    // sizeof(...) is calculated at compile time
    memset(button, 0, sizeof(svgdecoButton *) * ButtonTypeCount);

    // message in preview widget
    if (isPreview()) {
        windowLayout_->addWidget(
            new QLabel( i18n("<b><center>SVG Deco</center></b>"), widget() ), 1 ); 
    } else {
        windowLayout_->addItem(new QSpacerItem(0, 0));
    }

    // setup titlebar buttons
    for (int n=0; n<ButtonTypeCount; n++)
        button[n] = 0;
    //Deal with the title and buttons
    titleLayout_->addItem(leftTitleSpacer_);
    addButtons(titleLayout_,
               options()->customButtonPositions() ? options()->titleButtonsLeft() : QString(default_left),
               ::svgFactory->titleSize()-1);
    titleLayout_->addItem(titlebar_);
    addButtons(titleLayout_,
               options()->customButtonPositions() ? options()->titleButtonsRight() : QString(default_right),
               ::svgFactory->titleSize()-1);
    titleLayout_->addItem(rightTitleSpacer_);

    //Mid - left side, middle contents and right side
    QHBoxLayout * midLayout_   = new QHBoxLayout();
    midLayout_->addItem(leftSpacer_);
    midLayout_->addLayout(windowLayout_);
    midLayout_->addItem(rightSpacer_);

    //Layout order
    //mainLayout_->addItem( topSpacer_ );
    mainLayout_->addLayout( titleLayout_ );
    //mainLayout_->addItem( decoSpacer_ );
    mainLayout_->addLayout( midLayout_ );
    mainLayout_->addItem( bottomSpacer_ );
    
    // connections
    connect(this, SIGNAL(keepAboveChanged(bool)), SLOT(keepAboveChange(bool)));
    connect(this, SIGNAL(keepBelowChanged(bool)), SLOT(keepBelowChange(bool)));
}

//////////////////////////////////////////////////////////////////////////////
// addButtons()
// ------------
// Add buttons to title layout

void svgdecoClient::addButtons(QBoxLayout *layout, const QString& s, int button_size) {
    QString tip;
    if (s.length() > 0) {
        for (unsigned n=0; n < s.length(); n++) {
            switch (s[n]) {
            case 'M': // Menu button
                if (!button[ButtonMenu]) {
                    button[ButtonMenu] =
                        new svgdecoButton(this, "splat.png", i18n("Menu"),ButtonMenu,button_size);
                    connect(button[ButtonMenu], SIGNAL(pressed()),
                            this, SLOT(menuButtonPressed()));
                    layout->addWidget(button[ButtonMenu]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'S': // Sticky button
                if (!button[ButtonSticky]) {
                    if (isOnAllDesktops()) {
                        tip = i18n("Un-Sticky");
                    } else {
                        tip = i18n("Sticky");
                    }
                    button[ButtonSticky] =
                        new svgdecoButton(this, "circle.png", tip, ButtonSticky, button_size, true);
                    connect(button[ButtonSticky], SIGNAL(clicked()),
                            this, SLOT(toggleOnAllDesktops()));
                    layout->addWidget(button[ButtonSticky]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'H': // Help button
                if ((!button[ButtonHelp]) && providesContextHelp()) {
                    button[ButtonHelp] =
                        new svgdecoButton(this, "help.png", i18n("Help"), ButtonHelp, button_size);
                    connect(button[ButtonHelp], SIGNAL(clicked()),
                            this, SLOT(showContextHelp()));
                    layout->addWidget(button[ButtonHelp]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'I': // Minimize button
                if ((!button[ButtonMin]) && isMinimizable())  {
                    button[ButtonMin] =
                        new svgdecoButton(this, "minimize.png", i18n("Minimize"), ButtonMin, button_size);
                    connect(button[ButtonMin], SIGNAL(clicked()),
                            this, SLOT(minimize()));
                    layout->addWidget(button[ButtonMin]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'A': // Maximize button
                if ((!button[ButtonMax]) && isMaximizable()) {
                    if (maximizeMode() == MaximizeFull) {
                        tip = i18n("Restore");
                    } else {
                        tip = i18n("Maximize");
                    }
                    button[ButtonMax]  =
                        new svgdecoButton(this, "maximize.png", tip, ButtonMax, button_size, true);
                    connect(button[ButtonMax], SIGNAL(clicked()),
                            this, SLOT(maxButtonPressed()));
                    layout->addWidget(button[ButtonMax]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'X': // Close button
                if ((!button[ButtonClose]) && isCloseable()) {
                    button[ButtonClose] =
                        new svgdecoButton(this, "close.png", i18n("Close"), ButtonClose, button_size);
                    connect(button[ButtonClose], SIGNAL(clicked()),
                            this, SLOT(closeWindow()));
                    layout->addWidget(button[ButtonClose]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'F': // Above button
                if ((!button[ButtonAbove])) {
                    button[ButtonAbove] =
                        new svgdecoButton(this, "keep_above.png",
                                          i18n("Keep Above Others"), ButtonAbove, button_size, true);
                    connect(button[ButtonAbove], SIGNAL(clicked()),
                            this, SLOT(aboveButtonPressed()));
                    layout->addWidget(button[ButtonAbove]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'B': // Below button
                if ((!button[ButtonBelow])) {
                    button[ButtonBelow] =
                        new svgdecoButton(this, "keep_below.png",
                                          i18n("Keep Below Others"), ButtonBelow, button_size, true);
                    connect(button[ButtonBelow], SIGNAL(clicked()),
                            this, SLOT(belowButtonPressed()));
                    layout->addWidget(button[ButtonBelow]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case 'L': // Shade button
                if ((!button[ButtonShade && isShadeable()])) {
                    if ( isSetShade()) {
                        tip = i18n("Unshade");
                    } else {
                        tip = i18n("Shade");
                    }
                    button[ButtonShade] =
                        new svgdecoButton(this, "shade.png", tip, ButtonShade, button_size, true);
                    connect(button[ButtonShade], SIGNAL(clicked()),
                            this, SLOT(shadeButtonPressed()));
                    layout->addWidget(button[ButtonShade]);
                    if (n < s.length()-1) layout->addSpacing(1);
                }
                break;

            case '_': // Spacer item
                layout->addSpacing(::svgFactory->frameSize());
            }
        }
    }
}

//////////////////////////////////////////////////////////////////////////////
// activeChange()
// --------------
// window active state has changed

void svgdecoClient::activeChange() {
    for (int n=0; n<ButtonTypeCount; n++)
        if (button[n])
            button[n]->reset();
    widget()->repaint(false);
}

//////////////////////////////////////////////////////////////////////////////
// captionChange()
// ---------------
// The title has changed

void svgdecoClient::captionChange() {
    widget()->repaint(titlebar_->geometry(), false);
}

//////////////////////////////////////////////////////////////////////////////
// desktopChange()
// ---------------
// Called when desktop/sticky changes

void svgdecoClient::desktopChange() {
    bool d = isOnAllDesktops();
    if (button[ButtonSticky]) {
        QToolTip::remove(button[ButtonSticky]);
        QToolTip::add(button[ButtonSticky], d ? i18n("Un-Sticky") : i18n("Sticky"));
        button[ButtonSticky]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// iconChange()
// ------------
// The title has changed

void svgdecoClient::iconChange() {
    if (button[ButtonMenu]) {
        button[ButtonMenu]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// maximizeChange()
// ----------------
// Maximized state has changed

void svgdecoClient::maximizeChange() {
    bool m = (maximizeMode() == MaximizeFull);
    if (button[ButtonMax]) {
        QToolTip::remove(button[ButtonMax]);
        QToolTip::add(button[ButtonMax], m ? i18n("Restore") : i18n("Maximize"));
        button[ButtonMax]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// shadeChange()
// -------------
// Called when window shading changes

void svgdecoClient::shadeChange() {
    bool s = isSetShade();
    if (button[ButtonShade]) {
        QToolTip::remove(button[ButtonShade]);
        QToolTip::add(button[ButtonShade], s ? i18n("Unshade") : i18n("Shade"));
        button[ButtonShade]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// keepAboveChange()
// ------------
// The above state has changed

void svgdecoClient::keepAboveChange(bool a) {
    if (button[ButtonAbove]) {
        button[ButtonAbove]->setOn(a);
        button[ButtonAbove]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// keepBelowChange()
// ------------
// The below state has changed

void svgdecoClient::keepBelowChange(bool b) {
    if (button[ButtonBelow]) {
        button[ButtonBelow]->setOn(b);
        button[ButtonBelow]->repaint(false);
    }
}

//////////////////////////////////////////////////////////////////////////////
// borders()
// ----------
// Get the size of the borders

void svgdecoClient::borders(int &l, int &r, int &t, int &b) const {
    if(isShade())
    {
        l = r = ::svgFactory->frameSize();
        b = 0;
        t = ::svgFactory->titleSize() + (::svgFactory->frameSize()*2);
    }
    else
    {
        l = r = ::svgFactory->frameSize();
        t = ::svgFactory->titleSize() + ::svgFactory->frameSize();
        b = ::svgFactory->frameSize();
    }
}

//////////////////////////////////////////////////////////////////////////////
// resize()
// --------
// Called to resize the window

void svgdecoClient::resize(const QSize &size) {
    widget()->resize(size);
}

//////////////////////////////////////////////////////////////////////////////
// minimumSize()
// -------------
// Return the minimum allowable size for this window

QSize svgdecoClient::minimumSize() const {
    return widget()->minimumSize();
}

//////////////////////////////////////////////////////////////////////////////
// mousePosition()
// ---------------
// Return logical mouse position

KDecoration::Position svgdecoClient::mousePosition(const QPoint &point) const {
    const int corner = 24;
    Position pos;
    int fs = ::svgFactory->frameSize() + handlebar;
    //int fs = ::factory->frameSize();

    if (point.y() <= fs) {
        // inside top frame
        if (point.x() <= corner)
            pos = PositionTopLeft;
        else if (point.x() >= (width()-corner))
            pos = PositionTopRight;
        else
            pos = PositionTop;
    } else if (point.y() >= (height()-fs*2)) {
        // inside handle
        if (point.x() <= corner)
            pos = PositionBottomLeft;
        else if (point.x() >= (width()-corner))
            pos = PositionBottomRight;
        else
            pos = PositionBottom;
    } else if (point.x() <= fs ) {
        // on left frame
        if (point.y() <= corner)
            pos = PositionTopLeft;
        else if (point.y() >= (height()-corner))
            pos = PositionBottomLeft;
        else
            pos = PositionLeft;
    } else if (point.x() >= width()-fs) {
        // on right frame
        if (point.y() <= corner)
            pos = PositionTopRight;
        else if (point.y() >= (height()-corner))
            pos = PositionBottomRight;
        else
            pos = PositionRight;
    } else {
        // inside the frame
        pos = PositionCenter;
    }
    return pos;
}

//////////////////////////////////////////////////////////////////////////////
// eventFilter()
// -------------
// Event filter

bool svgdecoClient::eventFilter(QObject *obj, QEvent *e) {
    if (obj != widget())
        return false;

    switch (e->type()) {
    case QEvent::MouseButtonDblClick: {
            mouseDoubleClickEvent(static_cast<QMouseEvent *>(e));
            return true;
        }
    case QEvent::MouseButtonPress: {
            processMousePressEvent(static_cast<QMouseEvent *>(e));
            return true;
        }
    case QEvent::Paint: {
            paintEvent(static_cast<QPaintEvent *>(e));
            return true;
        }
    case QEvent::Resize: {
            resizeEvent(static_cast<QResizeEvent *>(e));
            return true;
        }
    case QEvent::Show: {
            showEvent(static_cast<QShowEvent *>(e));
            return true;
        }
    default: {
            return false;
        }
    }

    return false;
}

//////////////////////////////////////////////////////////////////////////////
// mouseDoubleClickEvent()
// -----------------------
// Doubleclick on title

void svgdecoClient::mouseDoubleClickEvent(QMouseEvent *e) {
    if (titlebar_->geometry().contains(e->pos()))
        titlebarDblClickOperation();
}

//////////////////////////////////////////////////////////////////////////////
// paintEvent()
// ------------
// Repaint the window

void svgdecoClient::paintEvent(QPaintEvent*) {
    if (!::svgFactory->initialized())
    {
        return;
    }
    
    int FRAMESIZE = ::svgFactory->frameSize();
    const uint maxCaptionLength = 300; // truncate captions longer than this!
    QString captionText(caption());
    if (captionText.length() > maxCaptionLength) {
        captionText.truncate(maxCaptionLength);
        captionText.append(" [...]");
    }

    QColor blackColor(black);
    QColor redColor(red);
    QColorGroup group,widgetGroup;
    QPainter painter(widget());
    QRect frame;
    bool active = isActive();
    //get group information first
    group = options()->colorGroup(KDecoration::ColorTitleBar, isActive());
    widgetGroup = widget()->colorGroup();

    // all the Title Bar - behind buttons and text
    QRect titleRect;
    titleRect.setRect(FRAMESIZE, 0, width()-FRAMESIZE, ::svgFactory->titleSize()+FRAMESIZE);
    painter.drawTiledPixmap(titleRect, active ? *aTitleBarTile:*iTitleBarTile);

    //use rectangles to bound where the text should be drawn
    //FIXME: change this to be a static pixmap later to reduce flicker
    // get titlebar area
    QRect title( titlebar_->geometry() );
    QRect textRect(title.x()+SIDETITLEMARGIN,title.y(),title.width()-SIDETITLEMARGIN*2,title.height());
    QRect shadowRect(textRect.left()+1,textRect.top()+1,textRect.width(),textRect.bottom()+1);
    //if we are shadowing title bar text
    if(::svgFactory->titleShadow())
    {
        //shadow text
        painter.setFont(options()->font(isActive(), false));
        painter.setPen(blackColor);
        painter.drawText(shadowRect,
                         ::svgFactory->titleAlign() | AlignVCenter | Qt::SingleLine,
                         captionText);
    }
    // draw title text
    painter.setFont(options()->font(isActive(), false));
    painter.setPen(options()->color(KDecoration::ColorFont, isActive()));
    painter.drawText(textRect,
                     ::svgFactory->titleAlign() | AlignVCenter | Qt::SingleLine,
                     captionText);
    
    //left of buttons and title
    frame.setRect(2,0,FRAMESIZE,s_titleHeight+FRAMESIZE);
    painter.fillRect(frame, group.background());
    frame.setRect(3,0,FRAMESIZE,s_titleHeight+FRAMESIZE);
    painter.drawTiledPixmap(frame, active ? *aTitleBarTile:*iTitleBarTile);
    //painter.setPen(group.background());
    //painter.drawArc(3,3,3,3, 90*16, 180*16);
    // left mid layout
    if(!isShade())
    {
        frame.setRect(0, s_titleHeight+FRAMESIZE, FRAMESIZE, height()-s_titleHeight-(FRAMESIZE*2));
        painter.fillRect(frame, widgetGroup.background());
    }

    // right of buttons and title
    frame.setRect(width()-FRAMESIZE, 0, FRAMESIZE-2, s_titleHeight);
    painter.fillRect(frame, group.background());
    frame.setRect(width()-FRAMESIZE, 0, FRAMESIZE-3, s_titleHeight);
    painter.drawTiledPixmap(frame, active ? *aTitleBarTile:*iTitleBarTile);
    // right mid layout
    if(!isShade())
    {
        frame.setRect(width()-FRAMESIZE, s_titleHeight+FRAMESIZE, FRAMESIZE, height()-titleRect.height()-(FRAMESIZE*2));
        painter.fillRect(frame, widgetGroup.background());
    }
    
    // bottom
    if(isShade())
    {
        frame.setRect(0,::svgFactory->titleSize()+FRAMESIZE, width(), FRAMESIZE);
    }
    else
    {
        frame.setRect(0, height() - (FRAMESIZE*2), width(),  (FRAMESIZE*2));
    }
    painter.fillRect(frame, widgetGroup.background());

    //painter.setPen(group.background());
    painter.setPen(blackColor);
    //draw a line between title bar and window contents
    painter.drawLine(0,titleRect.bottom(),titleRect.right(),titleRect.bottom());
    painter.drawLine(0,titleRect.bottom()-1,titleRect.right(),titleRect.bottom()-1);

    // outline outside the frame but not the corners if they are rounded
    frame = widget()->rect();
    painter.drawRect(frame);
    // this makes the outline 2 pixels instead of 1 pixel wide
    frame.setRect(frame.x()+1,frame.y()+1,frame.right()-1,frame.bottom()-1);
    painter.drawRect(frame);

    // local temp right and bottom
    int r(width());
    //painter.setPen(group.background());
    painter.setPen(blackColor);

    // Draw edge of top-left corner inside the area removed by the mask.
    painter.drawPoint(4, 0);
    painter.drawPoint(4, 1);
    painter.drawPoint(3, 1);
    painter.drawPoint(3, 2);
    painter.drawPoint(2, 2);
    painter.drawPoint(2, 3);
    painter.drawPoint(1, 3);
    painter.drawPoint(1, 4);
    painter.drawPoint(0, 4);
    // Draw edge of top-right corner inside the area removed by the mask.
    painter.drawPoint(r - 5, 0);
    painter.drawPoint(r - 5, 1);
    painter.drawPoint(r - 4, 1);
    painter.drawPoint(r - 4, 2);
    painter.drawPoint(r - 3, 2);
    painter.drawPoint(r - 3, 3);
    painter.drawPoint(r - 2, 3);
    painter.drawPoint(r - 2, 4);
    painter.drawPoint(r - 1, 4);

    // mask off which corners we dont want
    updateMask();

}

//////////////////////////////////////////////////////////////////////////////
// updateMask()
// ------------
// update the frame mask
void svgdecoClient::updateMask() {
    if ( (!options()->moveResizeMaximizedWindows() && maximizeMode() & MaximizeFull ) ) 
    {
        setMask(QRegion(widget()->rect()));
        return;
    }

    int r(width());
    int b(height());
    QRegion mask;

    mask=QRegion(widget()->rect());

    // Remove top-left corner.
    mask -= QRegion(0, 0, 5, 1);
    mask -= QRegion(0, 1, 3, 1);
    mask -= QRegion(0, 2, 2, 1);
    mask -= QRegion(0, 3, 1, 2);
    // Remove top-right corner.
    mask -= QRegion(r - 5, 0, 5, 1);
    mask -= QRegion(r - 3, 1, 3, 1);
    mask -= QRegion(r - 2, 2, 2, 1);
    mask -= QRegion(r - 1, 3, 1, 2);
    //always remove one corner pixel so it simulates a soft corner like plastik
    mask -= QRegion(0,0,1,1);
    mask -= QRegion(r-1,0,1,1);
    mask -= QRegion(0, b-1, 1,1);
    mask -= QRegion(r-1,b-1,1,1);

    setMask(mask);
}

//////////////////////////////////////////////////////////////////////////////
// resizeEvent()
// -------------
// Window is being resized

void svgdecoClient::resizeEvent(QResizeEvent *) {
    if (widget()->isShown()) {
        QRegion region = widget()->rect();
        region = region.subtract(titlebar_->geometry());
        widget()->erase(region);
    }
}

//////////////////////////////////////////////////////////////////////////////
// showEvent()
// -----------
// Window is being shown

void svgdecoClient::showEvent(QShowEvent *) {
    widget()->repaint();
}

//////////////////////////////////////////////////////////////////////////////
// maxButtonPressed()
// -----------------
// Max button was pressed

void svgdecoClient::maxButtonPressed() {
    if (button[ButtonMax]) {
#if KDE_IS_VERSION(3, 3, 0)
        maximize(button[ButtonMax]->lastMousePress());
#else

        switch (button[ButtonMax]->lastMousePress()) {
        case MidButton:
            maximize(maximizeMode() ^ MaximizeVertical);
            break;
        case RightButton:
            maximize(maximizeMode() ^ MaximizeHorizontal);
            break;
        default:
            (maximizeMode() == MaximizeFull) ? maximize(MaximizeRestore)
            : maximize(MaximizeFull);
        }
#endif

    }
}

//////////////////////////////////////////////////////////////////////////////
// shadeButtonPressed()
// -----------------
// Shade button was pressed

void svgdecoClient::shadeButtonPressed() {
    if (button[ButtonShade]) {
        setShade( !isSetShade());
    }
}

//////////////////////////////////////////////////////////////////////////////
// aboveButtonPressed()
// -----------------
// Above button was pressed

void svgdecoClient::aboveButtonPressed() {
    if (button[ButtonAbove]) {
        setKeepAbove( !keepAbove());
    }
}

//////////////////////////////////////////////////////////////////////////////
// belowButtonPressed()
// -----------------
// Below button was pressed

void svgdecoClient::belowButtonPressed() {
    if (button[ButtonBelow]) {
        setKeepBelow( !keepBelow());
    }
}

//////////////////////////////////////////////////////////////////////////////
// menuButtonPressed()
// -------------------
// Menu button was pressed (popup the menu)

void svgdecoClient::menuButtonPressed() {
    if (button[ButtonMenu]) {
        QPoint p(button[ButtonMenu]->rect().bottomLeft().x(),
                 button[ButtonMenu]->rect().bottomLeft().y());
        KDecorationFactory* f = factory();
        showWindowMenu(button[ButtonMenu]->mapToGlobal(p));
        if (!f->exists(this))
            return; // decoration was destroyed
        button[ButtonMenu]->setDown(false);
    }
}

#include "svgdeco.moc"
