/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <qtimer.h>

#include <kglobalsettings.h>

#include "cell.h"

#define MIN_CELL_WIDTH 83
#define MIN_CELL_HEIGHT 83

Cell::Cell( QWidget *parent, const char *name )
  : QLabel( parent, name )
{
  init();
}

Cell::Cell( QChar c, QWidget *parent, const char *name )
  : QLabel( parent, name )
{
  mChar = c;
  setText( c );

  init();
}

void Cell::init()
{
  mShowChar = true;
  mSelected = false;
  mSelectable = true;
  mPrevSelected = 0;
  mHighlightTimer = new QTimer( this );
  connect( mHighlightTimer, SIGNAL( timeout() ), SLOT( unhighlightForeground() ) );

  mNeighbours.setAutoDelete( false );

  setFont( QFont( "Helvetica", 32 ) );
  setAlignment( Qt::AlignCenter );
  setMinimumWidth( MIN_CELL_WIDTH );
  setMinimumHeight( MIN_CELL_HEIGHT );

  setFrameStyle( QFrame::Panel | QFrame::Raised );
  setLineWidth( 2 );

  show();
}

void Cell::setVisibility( bool b )
{
  mShowChar = b;
  if( b )
  {
    setEnabled( true );
    setText( mChar );
    if ( mSelected )
    {
      QColor bgcolor = KGlobalSettings::highlightColor().dark(120);
      setPaletteBackgroundColor( bgcolor );
      setPaletteForegroundColor( getTextColor( bgcolor ) );
    }
  }
  else
  {
    setEnabled( false );
    setText( QString::null );
    unsetPalette();
  }
}

void Cell::addNeighbour( Cell *cell )
{
  if( mNeighbours.find( cell ) == -1 )
    mNeighbours.append( cell );
  cell->neighbours()->append( this );
}

void Cell::setChar( QChar c )
{
  mChar = c;
  setVisibility( mShowChar );
}

void Cell::resizeEvent( QResizeEvent * )
{
  // make cells squary
  setMinimumHeight( width() );
}

void Cell::highlightForeground( bool special )
{
  setPaletteForegroundColor( special ? QColor( "Blue" ) : QColor( "Red" ) );

  mHighlightTimer->start( 1000, /* single shot */ true );
}

void Cell::unhighlightForeground()
{
  setPaletteForegroundColor( getTextColor( KGlobalSettings::highlightColor() ) );
}

void Cell::mousePressEvent( QMouseEvent *ev )
{
  if( mUseMouseEvents && mShowChar )
  {
    if( ev->button() == Qt::LeftButton && !mSelected && mSelectable )
    {
      emit appendChar( this );
      setSelected( true ); // sets right highlightables
    }
    else if( ev->button() == Qt::MidButton )
      emit submit();
    else if( ev->button() == Qt::RightButton && mSelected )
    {
      setSelected( false );
      emit removeChar( this );
    }
  }
}

void Cell::mouseDoubleClickEvent( QMouseEvent *ev )
{
  if( mUseMouseEvents && ev->button() == Qt::RightButton )
    emit clear();
}

void Cell::enterEvent( QEvent * )
{
  if( mUseMouseEvents && mSelectable && mShowChar )
  {
    QColor bgcolor = KGlobalSettings::highlightColor().light(120);
    setPaletteBackgroundColor( bgcolor );
    setPaletteForegroundColor( getTextColor( bgcolor ) );
  }
}

void Cell::leaveEvent( QEvent * )
{
  if( mUseMouseEvents && mShowChar )
  {
    if ( mSelected )
    {
      QColor bgcolor = KGlobalSettings::highlightColor().dark(120);
      setPaletteBackgroundColor( bgcolor );
      setPaletteForegroundColor( getTextColor( bgcolor ) );
    }
    else
      unsetPalette();
  }
}

void Cell::setSelected( bool b )
{
  QColor bgcolor;

  mSelected = b;
  if( b )
  {
    bgcolor = KGlobalSettings::highlightColor().dark( 120 );
    setPaletteBackgroundColor( bgcolor );
    setPaletteForegroundColor( getTextColor( bgcolor ) );
  }
  else if ( !mHighlightTimer->isActive() )
  {
    unsetPalette();
  }
}

void Cell::setNeighboursSelectable( bool b )
{
    for( uint i = 0; i < mNeighbours.count(); ++i )
    {
      Cell *cell = mNeighbours.at(i);
      if( !cell->mSelected )
        cell->mSelectable = b;
    }
}

QColor Cell::getTextColor(const QColor &c) const
{
  float luminance = (c.red() * 0.299) + (c.green() * 0.587) + (c.blue() * 0.114);
  return (luminance > 128.0) ? QColor( 0, 0 ,0 ) : QColor( 255, 255 ,255 );
}

#include "cell.moc"
