/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#ifndef CELL_H
#define CELL_H

#include <qlabel.h>

/**
  Represents a cell in the character grid.
  @author Bram Schoenmakers <bramschoenmakers@kde.nl>
*/
class Cell : public QLabel // public or protected?
{
  Q_OBJECT
  public:
    /** Constructor. @p c is the character which this cell represents */
    Cell( QChar c, QWidget *parent = 0, const char *name = 0 );
    /** Default constructor */
    Cell( QWidget *parent = 0, const char *name = 0 );

    typedef QPtrList<Cell> List;

    /** Returns a pointer to a the list of all neighbours of this cell */
    List *neighbours() { return &mNeighbours; }

    /**
      Adds a pointer of a neighbour to this cell. The function makes sure each neighbour is unique.
      Pre: The given cell *must* be a neighbour, no check is performed.
    */
    void addNeighbour( Cell *cell );

    void setChar( QChar c );
    QChar getChar() const { return mChar; }

    /** If @p b is false, the text will be hidden. If true, the text will be shown (default). */
    void setVisibility( bool b );

    void setMouseEvents( bool b ) { mUseMouseEvents = b; }

    void setSelectable( bool b ) { mSelectable = b; }

    /**
      Highlights a cell by making the text temporary red. If @p special is true, make it blue.
      Handy for defining the start of a word.
     */
    void highlightForeground( bool special = false );
    void setSelected( bool );
    void setNeighboursSelectable( bool );

    void setPreviousCell( Cell *cell ) { mPrevSelected = cell; }
    Cell *previousCell() const { return mPrevSelected; }

  signals:
    /** Will be emitted when the user single clicks on a cell. */
    void appendChar( Cell * );
    void removeChar( Cell * );
    /** Submit a word instead of pressing Enter, by double clicking. */
    void submit();
    void clear();

  public slots:
    void unhighlightForeground();

  protected:
    void init();
    virtual void resizeEvent( QResizeEvent * );
    virtual void mousePressEvent (QMouseEvent *);
    virtual void mouseDoubleClickEvent( QMouseEvent *);
    virtual void enterEvent( QEvent * );
    virtual void leaveEvent( QEvent * );
    QColor getTextColor( const QColor & ) const;

  private:
    QChar mChar;
    QTimer *mHighlightTimer;

    bool mUseMouseEvents;

    List mNeighbours;
    bool mShowChar;
    bool mSelected;
    bool mSelectable;

    Cell *mPrevSelected;
};

#endif // CELL_H
