/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <stdlib.h>

#include <qfile.h>
#include <qlayout.h>
#include <qtextstream.h>

#include <kapplication.h>
#include <kconfig.h>
#include <klocale.h>
#include <kspell.h>
#include <kstandarddirs.h>

#include "grid.h"

Grid::Grid( KSpell *spell, QWidget *parent, const char *name )
  : QWidget(parent, name), mLayout(0), mRows(4), mColumns(4), mShowChars( true ), mUseMouseEvents( false ), mLastCell( 0 )
{
  mCells.setAutoDelete( true );
  mSpellChecker = spell;\

  readSettings();
}

Grid::~Grid()
{
  // Only disconnect signals from this object.
  disconnect( mSpellChecker, 0, this, 0 );

  delete mLayout;
  mLayout = 0;

  mCells.clear();
}

void Grid::setDimensions( int w, int h )
{
  // Ensure we have at least a matrix of 3 by 3.
  if( w < 3 )
    mColumns = 3;
  else
    mColumns = w;

  if( h < 3 )
    mRows = 3;
  else
    mRows = h;

  randomize();
}

void Grid::randomize()
{
  mCells.clear();
  mWords.clear();

  delete mLayout;
  mLayout = new QGridLayout( this, mRows, mColumns );
  mShowChars = true;

  QStringList cubes;
  loadCubes( cubes );

  for( int row = 0 ; row < mRows ; ++row )
  {
    for( int column = 0 ; column < mColumns ; ++column )
    {
      // FIXME: Use high order bits

      int cube = KApplication::random() % cubes.count();
      QChar character = cubes[ cube ][ KApplication::random() % 6 ];
      // don't use this cube again
      cubes.remove( cubes.at( cube ) );
      // kdDebug() << row << "," << column << ":" << cubes << endl;

      Cell *cell = new Cell( character, this );
      connect( cell, SIGNAL( appendChar( Cell * ) ), SLOT( appendChar( Cell * ) ) );
      connect( cell, SIGNAL( removeChar( Cell * ) ), SLOT( removeChar( Cell * ) ) );
      connect( cell, SIGNAL( submit() ), SLOT( resetCells() ) );
      connect( cell, SIGNAL( submit() ), SIGNAL( submit() ) );
      connect( cell, SIGNAL( clear() ), SLOT( resetCells() ) );
      connect( cell, SIGNAL( clear() ), SIGNAL( clear() ) );
      cell->setVisibility( true );
      cell->setMouseEvents( mUseMouseEvents );
      mCells.append( cell );

      // neighbour: left top
      if( column != 0 && row != 0 )
        cell->addNeighbour( mCells.at(mColumns*(row - 1) + column - 1 ) );

      // neighbour: top
      if( row != 0 )
        cell->addNeighbour( mCells.at( mColumns*(row - 1) + column ) );

      // neighbour: right top
      if( row != 0 && column != mColumns - 1 )
        cell->addNeighbour( mCells.at( mColumns * (row - 1) + column + 1 ) );

      // neighbour: left
      if( column != 0 )
        cell->addNeighbour( mCells.at( (mColumns*row) + column - 1 ) );

      mLayout->addWidget( cell, row, column );
    }
  }

// FIXME: Uncomment this line if we have an algorithm for this which doesn't take 3 days
#if 0
  // only find words if the spellchecker is usable.
  if( mSpellChecker && (mSpellChecker)->status() == KSpell::Running ) {
    // Passing a null is safe, we don't use it anyway.
    makeWordList( 0 );
  }
  else if ( mSpellChecker && (mSpellChecker)->status() == KSpell::Starting ) {
    connect( mSpellChecker, SIGNAL( ready( KSpell * ) ), SLOT( makeWordList( KSpell * ) ) );
  }
#endif
}

void Grid::setCharactersVisible( bool b )
{
  if ( mShowChars == b )
    return;
  mShowChars = b;

  for( uint i = 0 ; i < mCells.count() ; ++i )
  {
    mCells.at(i)->setVisibility( b );
  }
}

#if 0
void Grid::findWordsFrom( Cell *cell )
{ // FIXME : Don't eat memory!
  kdDebug() << "Cell: " << cell->getChar() << endl;

  static Cell::List visited;
  static Possible possibility; // FIXME
  static QString word;

  visited.append( cell );

  word += cell->getChar();
  kdDebug() << "Word: " << word << endl;

  Cell::List *neighbours = cell->neighbours();
  uint count_neighbours = neighbours->count();

  // FIXME: Iterators gebruiken
  for( uint i = 0; i < count_neighbours ; ++i )
  {
    Cell *current = neighbours->at(i);
    if(  visited.find( current ) != -1 )
      continue;

    if ( possibility.list.findIndex ( word ) == -1 ) // FIXME
      return;

    findWordsFrom( current );
  }

  if( validateWord( word ) )
    mWords.append( word );

  visited.remove( cell );
  word.truncate( word.length() - 1 );
}

// FIXME: Prevent code duplication with Game
bool Grid::validateWord( const QString &word ) const // FIXME: Const
{
  if ( mSpellChecker )
    (mSpellChecker)->checkWord( word );
  else
    /* TODO */;

  return ( word.length() > 2 ) && mSpellChecker &&
     mSpellChecker->status() == KSpell::FinishedNoMisspellingsEncountered;
}

void Grid::makeWordList( KSpell * )
{
  kdDebug(5850) << "Grid::makeWordList() entered" << endl;

  // FIXME: Iterators gebruiken
  uint count = mCells.count();
  for( uint i = 0 ; i < count ; ++i )
  {
    // FIXME: Only uncomment if this one works properly.
    // findWordsFrom( mCells.at(i) );
  }
}

#endif

QString Grid::composeWord( Cell::List cells ) const
{
  QString word = QString::null;
  // FIXME: Use iterators
  for( uint i = 0 ; i < cells.count() ; ++i )
    word += cells.at(i)->getChar();

  return word;
}

bool Grid::hasWord( const QString &word, bool highlight )
{
  QString upperWord = word.upper();

  uint count = mCells.count();
  for( uint i = 0; i < count ; ++i )
  {
    Cell *cell = mCells.at(i);

    if ( cell->getChar() == upperWord.at(0) )
    {
      QString whatsLeft = upperWord.right( upperWord.length() - 1 );
      Cell::List *matchedCells = findMatchFrom( cell, whatsLeft, upperWord );

      if ( matchedCells )
      {
        if( highlight )
          highlightCells( matchedCells );
        matchedCells->clear();
        return true;
      }
    }
  }

  return false;
}

Cell::List *Grid::findMatchFrom( Cell *cell, QString whatsLeft, const QString &fullWord, bool newWord )
{
  static Cell::List visited;
  visited.setAutoDelete( false );

  if( newWord ) visited.clear();

  visited.append( cell );

//   kdDebug() << "visited=" << composeWord( visited ) << endl;

  if ( composeWord( visited ) == fullWord )
    return &visited;

  uint neighbour_count = cell->neighbours()->count();
  for( uint i = 0 ; i < neighbour_count ; ++i )
  {
    Cell *neighbour = cell->neighbours()->at(i);
    if( neighbour->getChar() == whatsLeft.at(0) && visited.find( neighbour ) == -1 )
    {
      if ( findMatchFrom( neighbour, whatsLeft.right( whatsLeft.length() - 1 ), fullWord, false ) )
        return &visited;
    }
  }

  visited.remove( cell );

  return 0;
}

/* Foreground */
void Grid::highlightCells( Cell::List *cells )
{

  if( !mHighlightWords )
    return;

  for( uint i = 0 ; i < mCells.count() ; ++i )
  {
    Cell *cell = mCells.at(i);
    int pos = cells->find( cell );
    // unhighlight already highlighted cells if they do not appear in the new word
    if( cells->find( cell ) == -1 )
      cell->unhighlightForeground();
    else
    {
      cell->highlightForeground( pos == 0 ? true : false );
    }
  }
}

void Grid::loadCubes( QStringList &cubes )
{
  QString path = KGlobal::instance()->dirs()->findResource( "data", QString("kboggle/cubes/%1.cubes").arg( mCubeSet ).latin1() );
  if ( path.isEmpty() )
    path = KGlobal::instance()->dirs()->findResource( "data", QString("kboggle/cubes/en.cubes").latin1() );

  QFile file( path );
  file.open( IO_ReadOnly );
  QTextStream stream( &file );

  for( int i = 0 ; i < mColumns * mRows && !stream.atEnd() ; ++i )
  {
    // TODO: validation
    cubes << stream.readLine();
  }

  file.close();

  // 2) Shuffle cubes
  for( int i = 0 ; i < 100 ; ++i )
  {
    QString temp = cubes[ 1 + (int)(( cubes.count() - 1) * KApplication::random() / RAND_MAX )];
    QStringList::Iterator it = cubes.find( temp );
    cubes.append( temp );
    cubes.remove( it );
  }
}

// 1+(int) (10.0*rand()/(RAND_MAX+1.0));

void Grid::readSettings()
{
  KConfig *config = kapp->config();
  config->setGroup( "Localization" );
  mCubeSet = config->readEntry( "Cubeset", "en" );

  config->setGroup( "Preferences" );
  mHighlightWords = config->readBoolEntry( "HighlightWords", true );
  mUseMouseEvents = config->readBoolEntry( "UseMouse", false );

  for( uint i = 0 ; i < mCells.count() ; ++i )
  {
    mCells.at(i)->setMouseEvents( mUseMouseEvents );
  }
}

void Grid::resetCells()
{
  mLastCell = 0;

  for( uint i = 0 ; i < mCells.count() ; ++i )
  {
    mCells.at(i)->setSelected( false );
    mCells.at(i)->setSelectable( true );
    mCells.at(i)->setPreviousCell( 0 );
    mCells.at(i)->setVisibility( true );
  }
}

void Grid::disableCells()
{
  for( uint i = 0 ; i < mCells.count() ; ++i )
  {
    mCells.at(i)->setEnabled( false );
    mCells.at(i)->setSelected( false );
    mCells.at(i)->setSelectable( false );
  }
}

void Grid::appendChar( Cell *cell )
{
  for( uint i = 0 ; i < mCells.count() ; ++i )
    mCells.at(i)->setSelectable( false );

  cell->setPreviousCell( mLastCell );
  mLastCell = cell;
  mLastCell->setNeighboursSelectable( true );

  emit appendChar( cell->getChar().lower() );
}

void Grid::removeChar( Cell *cell )
{
  mLastCell->setNeighboursSelectable( false );
  mLastCell = cell->previousCell();

  if( mLastCell)
    mLastCell->setNeighboursSelectable( true );
  else
    resetCells();

  cell->setPreviousCell( 0 );

  emit removeChar( cell->getChar().lower() );
}

#include "grid.moc"
