/*
  Copyright (C) 2005 by Bram Schoenmakers
  bramschoenmakers@kde.nl
  
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
 
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*/

#include <qlabel.h>
#include <qlayout.h>
#include <qtimer.h>
#include <qptrlist.h>

#include "wordlist.h"

WordList::WordList( QWidget *parent, const char *name )
  : QWidget( parent, name ), mLayout(0), mCurrentlyHighlighted( 0 )
{
  mLabels.setAutoDelete( true );
  mTimer = new QTimer( this );
  connect( mTimer, SIGNAL( timeout() ), SLOT( unHighlight() ) );
  init();
}

WordList::~WordList()
{
  mLabels.clear();
}

void WordList::init()
{
  mCount = 0;
  mRows = 9;
  mColumns = 1;

  mLayout = new QGridLayout( this, mRows, mColumns );
}

void WordList::addWord( const QString &word )
{
  ++mCount;
  mWords.append( word );

  QLabel *label = new QLabel( word, this );
  label->show();
  mLabels.append( label );

  if( mCount != 1 && mCount % mRows == 1 )
    addColumn();
  else
    mLayout->addWidget( label, ( ( mCount - 1 ) % mRows ), mColumns - 1 );
}

void WordList::addColumn()
{
  ++mColumns;

  delete mLayout;
  mLayout = new QGridLayout( this, mRows, mColumns);

  for( int i = 0 ; i < mCount ; ++i )
  {
    int column = (int)(i / mRows);
    mLayout->addWidget( mLabels.at( i ), i % mRows, column );
  }
}

void WordList::clear()
{
  // putting this line in init() makes KBoggle crash
  delete mLayout;
  mWords.clear();
  mLabels.clear();

  init();
}

bool WordList::hasWord( const QString &word )
{
  return mWords.find( word ) != mWords.end();
}

void WordList::highlight( const QString &word )
{
  uint count = mLabels.count();
  for( uint i = 0; i < count ; ++i )
  {
    if( mLabels.at(i)->text() == word )
    {
      //unhighlight current highlighted first
      if( mCurrentlyHighlighted )
        mCurrentlyHighlighted->setPaletteForegroundColor( QColor( "black" ) );
      mLabels.at(i)->setPaletteForegroundColor( QColor( "red" ) );
      mTimer->start( 1000, true );
      mCurrentlyHighlighted = mLabels.at(i);
    }
  }
}

void WordList::unHighlight()
{
  if( mCurrentlyHighlighted )
    mCurrentlyHighlighted->setPaletteForegroundColor( QColor( "black" ) );

  mCurrentlyHighlighted = 0;
}

#include "wordlist.moc"
