/**
  * @file            country.h  
  * @date            begin Wed Jul 18 2001
  * @author          Kleag (Gael de Chalendar ) <kleag@free.fr> \n
  *                  (C) 2001-2004 by Gael de Chalendar
  * @license         GPL 
  */

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
 
#define KDE_NO_COMPAT

#ifndef KSIRK_COUNTRY_H
#define KSIRK_COUNTRY_H

#include "Sprites/infantrysprite.h"
#include "Sprites/cavalrysprite.h"
#include "Sprites/cannonsprite.h"
#include "Sprites/animspriteslist.h"

#include <kgame/kgameproperty.h>

#include <qpoint.h>
#include <qstring.h>

#include <vector>

namespace Ksirk
{

class BackGnd;
class FlagSprite;
// class AnimSprite;

namespace GameLogic
{
class Player;
class Continent;

/**
 * Each country of the map is represented by a Country object. It has a name,
 * a point for its m_flag and points for its canon, horse, etc.
 * Also, it stores pointers to the objects that represent  its owner and the
 * sprites of its m_flag and its armies.
 */
class Country : QObject
{
  Q_OBJECT 

public:
  Country(const QString& theName,
      const QPoint& centralPoint,
      const QPoint& flagPoint, const QPoint& cannonPoint, const QPoint& cavalryPoint,
      const QPoint& infantryPoint, unsigned int id);

  virtual ~Country();

  /**
    * Removes the sprites (m_flag and soldiers), the owner, etc.
    * The sprites ared deleted.
    */
  void reset();

  /**
    * Ask each of the sprites to paint themselves
    */
  void paint();

  /**
    * Creates the sprites necessary to disblay the armies of the country.
    * Eventually removes previously existing sprites.
    */
  void createArmiesSprites(BackGnd *backGnd);

  /**
    * Creates the sprite of the contry's m_flag. Eventually removes a previously
    * existing sprite.
    */
  void flag(const QString& theFlagFileName, BackGnd *backGnd);
  
  /**
    * Test if this is a neighbour of country
    */
  bool communicateWith(const Country *country) const;

  /**
   * Returns the continent this country is in
   */
  inline Continent* continent() {return m_continent;}
  
  /**
   * Sets the continent this country is in
   */
  inline void setContinent(Continent* continent) {m_continent = continent;}
  
  /**
    * Change the owner of this to player and update the number of countries for previous and new owners
    */
  void owner(Player *player);
  
  /**
    * return a pointer to the Player owner of this
    */
  const Player* owner() const;
  Player* owner();

  /**
    * Return the number of armies in this country
    */
  unsigned int nbArmies() const;

  /**
    *  Return the number of armies that has been added during the last
    * armies movement
    */
  unsigned int nbAddedArmies();

  /**
    * Change the number of armies to nb
    */
  void nbArmies(unsigned int nb);

  /**
    * Change the number of armies that has been added during the last
    * armies movement to nb
    */
  void nbAddedArmies(unsigned int nb);

  /**
    * Add nb armies. Defaults to 1.
    */
  void incrNbArmies(unsigned int nb=1);

  /**
    * Add nb armies to the number of armies that has been added during the
    * last armies movement. Defaults to 1.
    */
  void incrNbAddedArmies(unsigned int nb=1);

  /**
    * Remove nb armies. Defaults to 1.
    */
  void decrNbArmies(unsigned int nb=1);

  /**
    * Remove nb armies to the number of armies that has been added during the
    * last armies movement. Defaults to 1.
    */
  void decrNbAddedArmies(unsigned int nb=1);

  /**
    * Return the name of the country
    */
  const QString name() const;

  /**
    * Return a point inside the country territory
    */
  const QPoint& centralPoint() const;

  /**
    * Return the point where the m_flag is displayed
    */
  const QPoint& pointFlag() const;

  /**
    * Return the point where the cannons are displayed
    */
  const QPoint& pointCannon() const;

  /**
    * Return the point where the cavalrymen are displayed
    */
  const QPoint& pointCavalry() const;

  /**
    * Return the point where the infantrymen are displayed
    */
  const QPoint& pointInfantry() const;

  /**
    * Set the point guaranted to be inside this country territory
    */
  void centralPoint(const QPoint pt);

  /**
    * Set the point where the m_flag is displayed
    */
  void pointFlag(const QPoint pt);

  /**
    * Set the point where the cannons are displayed
    */
  void pointCannon(const QPoint pt);

  /**
    * Set the point where the cavalrymen are displayed
    */
  void pointCavalry(const QPoint pt);

  /**
    * Set the point where the infantrymen are displayed
    */
  void pointInfantry(const QPoint pt);

  /**
    * Return the list of cannon sprites
    */
  AnimSpritesList< CannonSprite >& spritesCannons();

  /**
    * Return the list of Cavalrymen sprites
    */
  AnimSpritesList< CavalrySprite >& spritesCavalry();

  /**
    * Return the list of Infantrymen sprites
    */
  AnimSpritesList< InfantrySprite >& spritesInfantry();

  /** No descriptions */
  void neighbours(const std::vector<Country*>& neighboursVect);

  /** No descriptions */
  std::vector< Country* >& neighbours();

  /** No descriptions */
  const std::vector< Country* >& neighbours() const;

  void clearAllSprites();

  const QPoint& pointFor(const AnimSprite* sprite);

  void saveXml(std::ostream& xmlStream);

  void send(QDataStream& stream);
  
  unsigned int id() const {return m_id;}
  unsigned int id() {return m_id;}
  void id(unsigned int id) {m_id = id;}
  
  /**
    * Tests if there is at least one enemey adjacent to this country.
    * @return true if this country has an enemy neighbour and false otherwise.
    */
  bool hasAdjacentEnemy();

private:

  /**
   * A pointer to the continent this country is in.
   */
  Continent* m_continent;
  
  /**
    * A pointer to the Player object that holds the country. 0 if it is not
    * affected.
    */
  Player* m_belongsTo;

  /**
    * A pointer to the sprite of the country's m_flag
    */
  FlagSprite* m_flag;

  /**
    * the number of armies held by the country (used to compute the number
    * of soldiers, horses and cannons
    */
  unsigned int  m_nbArmies;
  
  /**
    * the number of armies added after the capture of the country or during an
    * armies move. It must stay positive as one cannot remove more armies than
    * he has added
    */
  unsigned int m_nbAddedArmies;
  
  /**
    * The name of the country
    */
  QString m_name;

  /** the array of neigbours of this country */
  std::vector<Country*> m_neighbours;

  /**
    * a point situated inside this country teritory such that any click on
    * this point (for example by an AI player) will be a click on this country
    */
  QPoint m_centralPoint;

  /**
    * the point of the upper left corner of the country's m_flag sprite
    */
  QPoint m_pointFlag;

  /**
    * the point of the upper left corner of the country's first cannon sprite
    * the subsequent cannons sprites are shifted by a fixed number of pixels
    */
  QPoint m_pointCannon;

  /**
    * the point of the upper left corner of the country's first cavalryman
    * sprite.The subsequent cavalrymen sprites are shifted by a fixed number
    * of pixels
    */
  QPoint m_pointCavalry;

  /**
    * the point of the upper left corner of the country's first soldier sprite
    * The subsequent soldier sprites are shifted by a fixed number of pixels
    */
  QPoint m_pointInfantry;

  /**
    * The list of the cannon sprites used to represent the armies of the
    * country
    */
  AnimSpritesList< CannonSprite > m_spritesCannons;

  /**
    * The list of the cavalrymen sprites used to represent the armies of the
    * country
    */
  AnimSpritesList< CavalrySprite > m_spritesCavalry;

  /**
    * The list of the soldiers sprites used to represent the armies of the
    * country
    */
  AnimSpritesList< InfantrySprite > m_spritesInfantry;
  
  unsigned int m_id;
};

QDataStream& operator>>(QDataStream& stream, Country* country);
  
} // closing namespace GameLogic

} // closing namespace Ksirk

#endif // KSIRK_COUNTRY_H

