//
// C++ Implementation: gameautomaton
//
// Description:
//
//
// Author: Gaël de Chalendar <kleag@free.fr>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "gameautomaton.h"
#include "kgamewin.h"
#include "kdebug.h"
#include "aiplayer.h"
#include "aiColsonPlayer.h"
#include "aiplayerio.h"
#include "onu.h"
#include "dice.h"
#include "goal.h"
#include "country.h"
#include "kstringvector.h"
#include "Dialogs/newGameDialogImpl.h"
#include "Dialogs/kplayersetupdialog.h"

#include <qvbox.h>
#include <qlayout.h>
#include <qspinbox.h>
#include <qvgroupbox.h>

#include <klocale.h>
#include <kdialogbase.h>
#include <kinputdialog.h>
#include <klineedit.h>
#include <kmessagebox.h>
#include <kgame/kmessageclient.h>
#include <kgame/kmessageserver.h>
#include <kgame/kgamechat.h>


#include <sstream>

namespace Ksirk{
namespace GameLogic {

const char* GameAutomaton::GameStateNames[] = {
    "INIT",
    "INTERLUDE",
    "NEWARMIES",
    "WAIT",
    "WAIT_RECYCLING",
    "ATTACK",
    "ATTACK2",
    "INVADE",
    "SHIFT1",
    "SHIFT2",
    "FIGHT_BRING",
    "FIGHT_ANIMATE",
    "FIGHT_BRINGBACK",
    "WAITDEFENSE",
    "EXPLOSION_ANIMATE",
    "WAIT_PLAYERS",
    "GAME_OVER",
    "INVALID"
};

const char* GameAutomaton::KsirkMessagesIdsNames[] = {
"CountryOwner", // 257
"PlayerPutsArmy", // 258
"StateChange", // 259
"PlayerChange", // 260
"RegisterCountry", // 261
"PlayerAvailArmies", // 262
"KGameWinAvailArmies", // 263
"ChangeItem", // 264
"DisplayRecyclingButtons", // 265
"DisplayNormalGameButtons", // 266
"ActionRecycling", // 267
"ClearGameActionsToolbar", // 268
"DisplayDefenseButtons", // 269
"ActionDefense", // 270
"FirstCountry", // 271
"SecondCountry", // 272
"InitCombatMovement", // 273
"AnimCombat", // 274
"DisplayInvasionButtons", // 275
"TerminateAttackSequence", // 276
"DecrNbArmies", // 277
"DisplayNextPlayerButton", // 278
"Invade", // 279
"Retreat", // 280
"NextPlayerNormal", // 281
"NextPlayerRecycling", // 282
"ShowArmiesToPlace", // 283
"PlayerPutsInitialArmy", // 284
"PlayerRemovesArmy", //285
"VoteRecyclingFinished", // 286
"CancelShiftSource", // 287
"ChangePlayerNation", // 288
"ChangePlayerName", // 289
"StartGame", // 290
"SetNation", // 291
"SetBarFlagButton", // 292
"FinishMoves", // 293
"AnimExplosion", // 294
"SetupOnePlayer", // 295
"SetupWaitedPlayer", // 296
"ValidateWaitedPlayerPassword", // 297
"ValidPassword", // 298
"InvalidPassword", // 299
"SetupCountries", // 300
"AddMsgIdPair", // 301
"CheckGoal", // 302
"SetGoalFor", // 303
"GoalForIs", // 304
"Winner", // 305
"NbPlayers", // 306
"FinalizePlayers", // 307
"Acknowledge", // 308
};

GameAutomaton* GameAutomaton::m_singleton = 0;
//GameAutomaton::GameState  GameAutomaton::m_state = INIT;

GameAutomaton::GameAutomaton() : 
    KGame(),
    m_game(0),
  m_networkPlayersNumber(0),
  m_goals(),
  m_useGoals(true),
  m_currentPlayer(""),
  m_state(INIT),
  m_savedState(INVALID),
  m_currentPlayerPlayed(false)

{
//   kdDebug() << "GameAutomaton::GameAutomaton" << endl;
//   m_stateId = m_state.registerData(dataHandler(),KGamePropertyBase::PolicyDirty,QString("m_state"));
  m_skinId = m_skin.registerData(dataHandler(),KGamePropertyBase::PolicyDirty,QString("m_skin"));
//   m_currentPlayerId = m_currentPlayer.registerData(dataHandler(),KGamePropertyBase::PolicyDirty,QString("m_currentPlayer"));
//   m_events.registerData(dataHandler(),KGamePropertyBase::PolicyDirty,QString("m_events"));
  
  // Connect the most important slot which tells us which properties are
  // changed
  connect(this,SIGNAL(signalPropertyChanged(KGamePropertyBase *,KGame *)),
          this,SLOT(slotPropertyChanged(KGamePropertyBase *,KGame *)));
  
  connect(this,SIGNAL(signalPlayerJoinedGame(KPlayer*)),
          this,SLOT(slotPlayerJoinedGame(KPlayer*)));
  
  connect(this,SIGNAL(signalNetworkData(int, const QByteArray&, Q_UINT32, Q_UINT32)),
          this,SLOT(slotNetworkData(int, const QByteArray&, Q_UINT32, Q_UINT32)));
  
  connect(this,SIGNAL(signalClientJoinedGame(Q_UINT32, KGame*)),
          this,SLOT(slotClientJoinedGame(Q_UINT32, KGame*)));
  
  connect(messageClient(),SIGNAL(connectionBroken()),
          this,SLOT(slotConnectionToServerBroken()));
  
  connect(messageServer(),SIGNAL(connectionLost(KMessageIO *)),
          this,SLOT(slotConnectionToClientBroken(KMessageIO *)));
  
  setPolicy(KGame::PolicyDirty,true);
  
//   kdDebug() << "GameAutomaton::GameAutomaton finished" << endl;
}

GameAutomaton::~GameAutomaton()
{
  delete m_singleton;
  m_singleton = 0;
}

const GameAutomaton& GameAutomaton::single()
{
  if (m_singleton == 0)
    m_singleton = new GameAutomaton();
  return *m_singleton;
}

GameAutomaton& GameAutomaton::changeable()
{
  if (m_singleton == 0)
    m_singleton = new GameAutomaton();
  return *m_singleton;
}

void GameAutomaton::init(KGameWindow* gw)
{
  if (m_singleton == 0)
    m_singleton = new GameAutomaton();
  m_singleton->m_game = gw;
}

GameAutomaton::GameState GameAutomaton::state() const
{
  return m_state;
}
    
void GameAutomaton::state(GameAutomaton::GameState state) 
{
//   kdDebug() << "GameAutomaton::state new state (id=" << state << ") is " << GameStateNames[state] << endl;
  m_state = state;
  QByteArray buffer;
  QDataStream stream(buffer, IO_WriteOnly);
  stream << state;
  sendMessage(buffer,StateChange);
}

#include <errno.h>
#include <sys/types.h>
#include <signal.h>

bool dnssdAvailable() {
  QFile f("/var/run/mdnsd.pid");
  if (!f.open(IO_ReadOnly)) return false; // no pidfile
  QString line;
  if (f.readLine(line,16)<1) return false;
  unsigned int pid = line.toUInt();
  if (pid==0) return false; // not a pid
  return (kill(pid,0)==0 || errno==EPERM);
  // signal 0 only checks if process is running, mdnsd is probably owned 
  // by 'nobody' so we will  get EPERM, if mdnsd is not running error will be ESRCH
}

Player* GameAutomaton::getAnyLocalPlayer()
{
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    if ( !((Player*)(*it))->isVirtual() )
    {
      return (Player*)(*it);
    }
  }
  return 0;
}

GameAutomaton::GameState GameAutomaton::run()
{
//   kdDebug() << "GameAutomaton::run" << endl;
  if (m_game == 0)
    return m_state;
      
  m_game->haltTimer();
  QString event = "";
  QPoint point;
  if (!m_events.empty())
  {
    QPair< QString, QPoint > pair = m_events.front();
    event = pair.first;
    point = pair.second;
    m_events.pop_front();
  }

        
//   kdDebug() << "Handling " << stateName() << " ; " << event << " ; " << point.x() << "," << point.y() << endl;
  if (event == "requestForAck")
  {
  }
  if (event == "actionNewGame")
  {
    if (m_game->actionNewGame())
    {
      state(INIT);
      return INIT;
    }
    else
    {
      return m_state;
    }
  }
  if (event == "actionOpenGame")
  {
    if (m_game->actionOpenGame())
    {
      #if KDE_IS_VERSION(3,4,0)
      if (dnssdAvailable())
        setDiscoveryInfo("_ksirk._tcp","wow");
      #endif
      int port = 20000;
      bool ok;
      port = KInputDialog::getInteger(
              i18n("KsirK - Network configuration"), 
              i18n("Please type in the port number on which to offer connections:"), 
              port, 0, 32000, 1, 10, &ok, m_game);
      offerConnections(port);
      state(WAIT_PLAYERS);
      return WAIT_PLAYERS;
    }
    else
    {
      return m_state;
    }
  }
  switch (m_state)
  {
  case INIT:
    if (event == "actionJoinNetworkGame")
    {
      joinNetworkGame();
      return m_state;
    }
    else if (currentPlayer() != 0 && isAdmin())
    {
      if  ( (event == "actionLButtonDown") && (m_game->playerPutsInitialArmy(point)) )
      {
        m_choosedToRecycleNumber = 0;
        m_game->initRecycling();
        state(WAIT_RECYCLING);
      }
    }
    break;
  case ATTACK:
    if  (event == "actionLButtonDown") 
    {
      if  ( m_game->attacker(point) ) 
        state(ATTACK2);
      else
        state(WAIT);
    }
    else if (event == "actionCancel")
    {
      m_game-> cancelAction();
      state(WAIT);
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
    break;
  case ATTACK2:
    if  (event == "actionLButtonUp") 
    {
      switch ( m_game->attacked(point) )
      {
        case 0:
          state(WAIT);
        break;
        case 1:
          m_currentPlayerPlayed = true;
          state(WAITDEFENSE);
        break;
        case 2:
          m_currentPlayerPlayed = true;
          m_game-> defense(1);
          state(FIGHT_BRING);
        break;
        case 3:
          // AI action: nothing to do.
        break;
        default:
          std::cerr << "Unknown return value from attacked" << std::endl;
          exit(1);
      } 
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case EXPLOSION_ANIMATE:
    // no more explosions
    if (!m_game->haveAnimExplodings() && isAdmin())
    {
      state(FIGHT_BRINGBACK);
    }
  break;
  case FIGHT_ANIMATE:
    // no more animated fighters
    if (isAdmin())
    {
      if (!m_game->haveAnimFighters())
      {
      //kdDebug() << ">>>>>>>>>>> myState switches from FIGHT_ANIMATE to EXPLOSION_ANIMATE" << endl;
        m_game->resolveAttack();
        state(EXPLOSION_ANIMATE);
      }
    }
  break;
  case FIGHT_BRING:
    // no more moving fighters
    if (!m_game->haveMovingFighters() && isAdmin())
    {
      //kdDebug() << ">>>>>>>>>>> myState switches from FIGHT_BRING to FIGHT_ANIMATE" << endl;
      state(FIGHT_ANIMATE);
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      sendMessage(buffer,AnimCombat);
    }
  break;
  case FIGHT_BRINGBACK:
    // no more moving fighter returning home
    if (!m_game->haveMovingFighters() && isAdmin())
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      sendMessage(buffer,TerminateAttackSequence);
    }
  break;
  case INTERLUDE:
    if  (event == "playersLooped")
    {
      m_choosedToRecycleNumber = 0;
      m_game->initRecycling();
      state(WAIT_RECYCLING);
    }
    else if  (event == "actionLButtonDown" ) 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point;
//       kdDebug() << "Sending message PlayerPutsArmy " << point << endl;
      sendMessage(buffer,PlayerPutsInitialArmy);
    }
    else if (event == "actionRButtonDown") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point;
      sendMessage(buffer,PlayerRemovesArmy);
    }
    else if (event == "actionNextPlayer") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << -1;
      sendMessage(buffer,NextPlayerRecycling);
    }
    else if (event == "actionRecycling") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      sendMessage(buffer,ActionRecycling);
    }
    else if (event == "actionRecyclingFinished") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
//       stream << currentPlayer()->name();
      m_game->clearGameActionsToolbar(false);
      PlayersArray::iterator it = playerList()->begin();
      PlayersArray::iterator it_end = playerList()->end();
      Q_UINT32 nbLocal = 0;
      for (; it != it_end; it++)
      {
        if ( !((Player*)(*it))->isVirtual() )
        {
          nbLocal++;
        }
      }
      stream << nbLocal;
      it = playerList()->begin();
      it_end = playerList()->end();
      for (; it != it_end; it++)
      {
        if ( !((Player*)(*it))->isVirtual() )
        {
          stream << ((Player*)(*it))->id();
        }
      }
      sendMessage(buffer,VoteRecyclingFinished);
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case INVADE:
    if (event == "actionInvade1")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(1);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionInvade5")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(5);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionInvade10")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(10);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionRetreat1")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(1);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionRetreat5")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(5);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionRetreat10")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(10);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionInvasionFinished")
    {
      state(WAIT);
      m_game-> invasionFinished();
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case NEWARMIES:
    if  (event == "actionLButtonDown") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point << Q_UINT32(true);
      sendMessage(buffer,PlayerPutsArmy);
    }
    else if  (event == "actionRButtonDown") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point;
      sendMessage(buffer,PlayerRemovesArmy);
    }
    else if (event == "actionNextPlayer") 
    { 
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(WAIT_RECYCLING);
      sendMessage(buffer,NextPlayerRecycling);
    }
/*      else if (event == "actionRecyclingFinished")
      {
        m_game-> displayNormalGameButtons();
        state(WAIT);
      }*/
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case SHIFT1:
    if (event == "actionCancel")
    {
      m_game-> cancelAction();
      state(WAIT);
    }
    else if (event == "actionLButtonDown")
    {
      m_game->firstCountryAt(point);
    }
    else if (event == "actionLButtonUp")
    {
      m_game->secondCountryAt(point);
      if (m_game->isMoveValid(point))
      {
        m_game->displayInvasionButtons();
        m_currentPlayerPlayed = true;
        state(SHIFT2);
      }
      else
      {
//         state(WAIT);
      }
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case SHIFT2:
    if (event == "actionInvade1")
    {
//       kdDebug() << "actionInvade1" << endl;
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(1);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionInvade5")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(5);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionInvade10")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(10);
      sendMessage(buffer,Invade);
    }
    else if (event == "actionRetreat1")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(1);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionRetreat5")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(5);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionRetreat10")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(10);
      sendMessage(buffer,Retreat);
    }
    else if (event == "actionInvasionFinished")
    {
      m_game-> shiftFinished();
      state(WAIT);
    }
    else if (event == "actionCancel")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      sendMessage(buffer,CancelShiftSource);
    }
    else
    {
      //std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case WAIT_RECYCLING:
//     kdDebug() << "event = '" << event << "'" << endl;
    if  (event == "actionLButtonDown") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point << Q_UINT32(true);
      sendMessage(buffer,PlayerPutsArmy);
    }
    else if (event == "actionRButtonDown") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << point;
      sendMessage(buffer,PlayerRemovesArmy);
    }
    else if (event == "actionNextPlayer") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << -1;
      sendMessage(buffer,NextPlayerRecycling);
    }
    else if (event == "actionRecycling") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      sendMessage(buffer,ActionRecycling);
    }
    else if (event == "actionRecyclingFinished") 
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
//       stream << currentPlayer()->name();
      m_game->clearGameActionsToolbar(false);
      PlayersArray::iterator it = playerList()->begin();
      PlayersArray::iterator it_end = playerList()->end();
      Q_UINT32 nbLocal = 0;
      for (; it != it_end; it++)
      {
        if ( !((Player*)(*it))->isVirtual() )
        {
          nbLocal++;
        }
      }
      stream << nbLocal;
      it = playerList()->begin();
      it_end = playerList()->end();
      for (; it != it_end; it++)
      {
        if ( !((Player*)(*it))->isVirtual() )
        {
          stream << (*it)->id();
        }
      }
      sendMessage(buffer,VoteRecyclingFinished);
    }
    else
    {
      if (event != "")
        std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
      else
      {
      if (allLocalPlayersComputer())
        {
          m_game->displayRecyclingButtons();
        }
      }
    }
    break;
  case WAITDEFENSE:
    if (event == "actionDefense1")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(1);
      sendMessage(buffer,ActionDefense);
    }
    else if (event == "actionDefense2")
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(2);
      sendMessage(buffer,ActionDefense);
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
  break;
  case WAIT:
    if (event == "actionNextPlayer") 
    {
      if ( currentPlayer()->isVirtual()
          || currentPlayer()->isAI()
          || m_currentPlayerPlayed  
          || (KMessageBox::questionYesNo (m_game,
                i18n("%1, you haven't played anything this turn.\nDo you really want to lose your turn ?").arg(m_currentPlayer), 
                "Really Next Player ?") == KMessageBox::Yes) )
      {
        QByteArray buffer;
        QDataStream stream(buffer, IO_WriteOnly);
        stream << (Q_UINT32)NEWARMIES;
        sendMessage(buffer,NextPlayerNormal);
      }
    }
    else if (event == "actionAttack1")
    {
      m_game->attack(1);
      state(ATTACK);
    }
    else if (event == "actionAttack2")
    {
      m_game->attack(2);
      state(ATTACK);
    }
    else if (event == "actionAttack3")
    {
      m_game->attack(3);
      state(ATTACK);
    }
    else if (event == "actionMove")
    {
//       kdDebug() << "actionMove handling" << endl;
      
      m_game->displayCancelButton();
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << QString("");
      sendMessage(buffer,FirstCountry);
      QByteArray buffer2;
      QDataStream stream2(buffer2, IO_WriteOnly);
      stream2 << QString("");
      sendMessage(buffer2,SecondCountry);
      state(SHIFT1);
    }
    else
    {
//        if (event != "")
//          std::cerr << "Unhandled event " << event << " during handling of " << stateName() << std::endl;
    }
    // other case : state dosn't change
    break;
  case WAIT_PLAYERS:
    break;
  case GAME_OVER:
    break;
  default:
    std::cerr << "Unhandled state: " << stateName() << ". Event was: " << event << std::endl; 
    exit(1); // @todo handle this error
  }
  m_game->initTimer();
  return m_state;
}

void GameAutomaton::event(const std::string& event, const QPoint& point)
{
  QString ev = event;
  m_events.push_back(qMakePair(ev, point));
}

/** returns the name of the current state */
const std::string GameAutomaton::stateName() const
{
  if ((unsigned int)(m_state) > sizeof(GameStateNames))
  {
    std::ostringstream oss;
    oss << "Invalid stored state id: " << m_state;
    kdError() << oss.str() << endl;
    return oss.str();
  }
  else
  {
    return GameStateNames[m_state];
  }
}

void GameAutomaton::saveXml(std::ostream& xmlStream)
{
    xmlStream << "<gameautomaton state=\"" << GameStateNames[m_state] << "\" />" << std::endl;
}

const QString& GameAutomaton::skin() const
{
  return m_skin.value();
}

void GameAutomaton::skin(const QString& newSkin)
{
  m_skin = newSkin;
}

// Called when a player input is received from the KGame object 
// this is e-.g. a mous event
// This is obviously the central function in the game as all player
// moves, whether network or local, end up here. So do something 
// sensible here. 
bool GameAutomaton::playerInput(QDataStream &msg, KPlayer* player)
{
//   kdDebug() << "GameAutomaton: Player input" << endl;
  // Convert the player to the right class
  Player* p = static_cast<Player*>(player);

  if (p->isVirtual())
  {
//     kdDebug() << "Network player: nothing to do" << endl;
    return false;
  }
  QString action;
  QPoint point;
  msg >> action >> point;

//   kdDebug() << " ======================================================="<<endl;
//   kdDebug()  << "Player " << p->name() << " id=" << player->id() 
//     << " uid=" << player->userId() << " : " << action << " at " << point << endl;

  if (action == "actionLButtonDown")
    m_game->slotLeftButtonDown( point );
  else if (action == "actionLButtonUp")
    m_game->slotLeftButtonUp( point );
  else if (action == "actionRButtonDown")
    m_game->slotRightButtonDown( point );
/*  else if (action == "actionRButtonUp")
    m_game->slotRightButtonUp( point );*/
  else if (action == "actionAttack1")
    m_game->slotAttack1();
  else if (action == "actionAttack2")
    m_game->slotAttack2();
  else if (action == "actionAttack3")
    m_game->slotAttack3();
  else if (action == "actionMove")
    m_game->slotMove();
  else if (action == "slotRecyclingFinished")
    m_game->slotRecyclingFinished();
  else if (action == "actionInvade10")
    m_game->slotInvade10();
  else if (action == "actionInvade5")
    m_game->slotInvade5();
  else if (action == "actionInvade1")
    m_game->slotInvade1();
  else if (action == "actionInvasionFinished")
    m_game->slotInvasionFinished();
  else if (action == "slotDefense1")
    m_game->slotDefense1();
  else if (action == "slotDefense2")
    m_game->slotDefense2();
  else if (action == "actionNextPlayer")
    m_game->slotNextPlayer();
  else if (action == "requestForAck")
    acknowledge(p, point.x());
  return false;
}

void GameAutomaton::acknowledge(Player* p, unsigned int ack)
{
  if (p->isVirtual())
  {
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
    stream << p->id() << ack;
    sendMessage(buffer,Acknowledge);
  }
  else
  {
    p->acknowledge(ack);
  }
}

// Create an IO device for the player. We could create any
// device here, e.g. mouse, keyboard, computer player. In the
// demo game here we have only the mouse player available.
void GameAutomaton::createIO(KPlayer *player,KGameIO::IOMode io)
{
//   kdDebug() << "createIO for " << player->name() << endl;
  // Error check
  if (!player) return;

  if (io&KGameIO::MouseIO)
  {
    // Create new game mouse input
    KGameMouseIO *input;
    // We want the player to work over mouse
    // in our canvas view
    input=new KGameMouseIO(m_game);

    // Connect mouse input to a function to process the actual input
    connect(input,SIGNAL(signalMouseEvent(KGameIO *,QDataStream &,QMouseEvent *,bool *)),
            m_game->frame(),SLOT(slotMouseInput(KGameIO *,QDataStream &,QMouseEvent *,bool *)));

    // Add the device to the player
    player->addGameIO(input);
  }
  else if (io&AIPLAYERIO)
  {
    if (dynamic_cast<AIPlayer*>(player) != 0)
    {
      /*AIPlayerIO* input =*/ new AIPlayerIO(dynamic_cast<AIPlayer*>(player));
    }
    else
    {
      kdError() << "Can create an AIPlayerIO only for AI players: " << io << endl;
    }
  }
  else
  {
    kdError() << "Cannot create the requested IO device " << io << endl;
  }
}

// Find out who will be the next player
// Note: The default behaviour as we have it here is done automatically
//        by the lib, too. So if all players player one after the other
//        this functions is NOT needed at all.
KPlayer * GameAutomaton::nextPlayer(KPlayer *last,bool /*exclusive*/)
{
//   kdDebug() << "GameAutomaton::nextPlayer " << last->name() << endl;
//   m_game->setCurrentPlayerToNext();
  // If a last player is given switch the player
  
  // Should be enough if the admin sets the turn (unclear)
  if (isAdmin())
  {
    currentPlayer()->setTurn(true,true);
//     last->setTurn(true,true);
//     kdDebug() << "nextPlayer::Setting turn to " << last->name() << ", " << last->id() << "("<<last->userId()<<")"<<endl;
  }
  
  // Notify the world that whose turn it is. The main window uses
  // this to show a little message
//   emit signalNextPlayer(next->userId().value(),next,this);
  
  // Return the next player
  return dynamic_cast<KPlayer *>(currentPlayer());
}

QDataStream& operator>>(QDataStream& s, GameAutomaton::GameState& state)
{
  int istate;
  s >> istate;
  state = GameAutomaton::GameState(istate);
  return s;
};


bool GameAutomaton::setupPlayersNumberAndSkin(bool& networkGame, int& port, uint& newPlayersNumber)
{
  std::map< QString, QString > nations = m_game->nationsList();
  if (nations.size() < 2)
  {
    QString mes = "";
    mes.sprintf( i18n("Error - 2 nations minimum. Got %d."), nations.size());
    KMessageBox::error(m_game, mes, i18n("Fatal Error!"));
    exit(1);
  }
  // Number of m_players
  QString skinName = m_skin;
  while ((newPlayersNumber < 2) || (newPlayersNumber > nations.size()))
  {
    bool ok;
    NewGameDialogImpl(ok, newPlayersNumber, nations.size(), skinName, networkGame, m_useGoals, m_game, "NewGameDialogImpl").exec();
//     kdDebug() << "Got " << ok << " ; " << newPlayersNumber << " ; " << skinName << endl;
    if (!ok)
    {
      return false;
    }
  }
  setMinPlayers(newPlayersNumber);
  setMaxPlayers(newPlayersNumber);
  m_nbPlayers = newPlayersNumber;
//   kdDebug() << "min and max players number set to " << newPlayersNumber << endl;
  
//   kdDebug() << "Got skin name: " << skinName << endl;
/*  if (skinName != m_skin)
  {*/
//     kdDebug() << "Changing skin" << endl;
    m_skin = skinName;
//   }
  
  port = 20000;
  if (networkGame)
  {
    
    KDialogBase* dialog = new KDialogBase( m_game, "PortNet", true, i18n("Port and Net players configuration"),
                                          KDialogBase::Ok, KDialogBase::Ok, true );
    
    QVBox *page = dialog->makeVBoxMainWidget();
    QVGroupBox* mRemoteGroup=new QVGroupBox(i18n("Number of network players"), page);
    QSpinBox* spinBox = new QSpinBox( 1, newPlayersNumber, 1, mRemoteGroup );
    KLineEdit* edit = new KLineEdit( mRemoteGroup,"NbNetPlayers" );
    edit->setText(QString::number(port));
    
    dialog->exec();
    
    m_networkPlayersNumber = spinBox->value();
    port = edit->text().toInt();
//     kdDebug() << "There will be " << m_networkPlayersNumber << " network players." << endl;
    #if KDE_IS_VERSION(3,4,0)
    if (dnssdAvailable())
      setDiscoveryInfo("_ksirk._tcp","wow");
    #endif
  }  
  return true;
}

void GameAutomaton::setGoalFor(Player* player)
{
  unsigned int max = m_goals.size();
  unsigned int goalId = Dice::roll(max);
  std::set< Goal* >::iterator it = m_goals.begin();
  for (unsigned int i = 1 ; i < goalId; it++,i++) {}
  Goal* goal = (*it);
//   kdDebug() << "Goal for " << player->name() << " is of type " << goal->type() << endl;
  if (goal->type() == Goal::GoalPlayer)
  {
    Player* target = 0;
    while (target==0 || target->id() == player->id())
    {
      unsigned int max = playerList()->count();
      unsigned int playerNum = Dice::roll(max);
//       kdDebug() << "Choosed player num " << playerNum << " on " << max << endl;
      PlayersArray::iterator itp = playerList()->begin();
      PlayersArray::iterator itp_end = playerList()->end();
      unsigned int j = 1;
      for (; j < playerNum; j++,itp++) {}
      target = static_cast< Player* >(*itp);
    }
    if (target != 0)
    {
//       kdDebug() << "Target choosed for " << player->name() << ": " << target->name() << endl;
      goal->players().insert(target->id());
    }
    else
    {
//       kdDebug() << "No target choosed for " << player->name() << endl;
    }
  }
  /// @note hack to avoid too easy countries goal when there is only two players
  else if (goal->type() == Goal::Countries 
    && playerList()->count() == 2)
  {
    goal->nbCountries(int(goal->nbCountries()*1.5));
  }
  QByteArray buffer;
  QDataStream stream(buffer, IO_WriteOnly);
  stream << player->id();
  stream << (*goal);
  sendMessage(buffer,GoalForIs);
  delete goal;
  m_goals.erase(it);
}

bool GameAutomaton::joinNetworkGame()
{
  // Set default network parameter
  QString host = "localhost";
  int port = 20000;
  
  
  KDialogBase* dialog = new KDialogBase( m_game, "JoinNet", true, i18n("Join network game configuration"),
                                         KDialogBase::Ok|KDialogBase::Cancel, KDialogBase::Ok, true );
  
  QVBox *page = dialog->makeVBoxMainWidget();
  QVGroupBox* mRemoteGroup=new QVGroupBox(i18n("Network configuration"), page);
  KLineEdit* hostEdit = new KLineEdit( mRemoteGroup,"ServerHost" );
  hostEdit->setText(host);
  KLineEdit* portEdit = new KLineEdit( mRemoteGroup,"ServerPort" );
  portEdit->setText(QString::number(port));
  
  QDialog::DialogCode valid = QDialog::DialogCode(dialog->exec());
  
  if (valid == QDialog::Rejected)
  {
    return false;
  }
  host = hostEdit->text();
  port = portEdit->text().toInt();
  
  if (messageServer() != 0)
  {
    QObject::disconnect(messageServer(),SIGNAL(connectionLost(KMessageIO *)),
          this,SLOT(slotConnectionToClientBroken(KMessageIO *)));
  }
  bool status = connectToServer(host, port);
  connect(messageServer(),SIGNAL(connectionLost(KMessageIO *)),
          this,SLOT(slotConnectionToClientBroken(KMessageIO *)));
  return status;
}

KPlayer * GameAutomaton::createPlayer(int rtti, 
                                    int io, 
                                    bool isVirtual)    
{
//   kdDebug() << "GameAutomaton::createPlayer(" << rtti << ", " << io<< ", " << isVirtual << ")" << endl;
  if (rtti == 1)
  {
    Player* p = new Player("", 0, 0);
    p->setVirtual(isVirtual);
    if (!isVirtual)
    {
//       kdDebug() << "Calling player createIO" << endl;
      createIO(p,KGameIO::IOMode(KGameIO::MouseIO));
    }
    return (KPlayer*) p;
  }
  else if (rtti == 2)
  {
    AIPlayer* aip = new AIColsonPlayer("", 0, 0,  *playerList(), m_game->theWorld(),
                                   this);
    aip->setVirtual(isVirtual);
    if (!isVirtual)
    {
//       kdDebug() << "Calling player createIO" << endl;
      createIO(aip, KGameIO::IOMode(AIPLAYERIO));
    }
    return (KPlayer*) aip;
  }
  else 
  {
    kdError() << "No rtti given... creating a Player" << endl;
    Player* p = new Player("", 0, 0);
    p->setVirtual(isVirtual);
    if (!isVirtual)
    {
//       kdDebug() << "Calling player createIO" << endl;
      createIO(p,KGameIO::IOMode(KGameIO::MouseIO));
    }
    return (KPlayer*) p;
  }
}

/**
  * @return A pointer to the given's named player ; 0 if there is no such player
  */
Player* GameAutomaton::playerNamed(const QString& playerName)
{
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    if ( (*it)-> name() ==  playerName)
    {
      return static_cast<Player*>(*it);
    }
  }
  kdError() << "GameAutomaton::playerNamed: there is no player named " 
      << playerName << endl;
  return 0;
}

Player* GameAutomaton::currentPlayer() 
{
  if (m_game && m_currentPlayer!="")
  {
    return playerNamed(m_currentPlayer);
  }
  else return 0;
}

void GameAutomaton::currentPlayer(Player* player) 
{
//   kdDebug() << "GameAutomaton::currentPlayer" << endl;
  if (player)
  {
//     kdDebug() << " name = " << player->name() << endl;
    m_currentPlayer = player->name();
    m_currentPlayerPlayed = false;
    if (isAdmin())
    {   
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << m_currentPlayer;
      sendMessage(buffer,PlayerChange);
      player->setTurn(true,true);
    }
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
    stream << player->name();
    sendMessage(buffer,SetBarFlagButton);
    if (!player->isVirtual())
    {
      m_game->chatWidget()->setFromPlayer(player);
    }
  }
  else 
    m_currentPlayer = "";
}

void GameAutomaton::slotPlayerJoinedGame(KPlayer* player)
{
//   kdDebug() << "slotPlayerJoinedGame currently " << playerList()->count() << " / " << maxPlayers() << endl;
  Player* p = static_cast<Player*>(player);
  
  if (isAdmin())
  {
    unsigned int nbWithNation = 0;
    unsigned int nbWithName = 0;
    PlayersArray::iterator it = playerList()->begin();
    PlayersArray::iterator it_end = playerList()->end();
    for (; it != it_end; it++)
    {
      if (p->getNation()->name() == ((Player*)(*it))->getNation()->name())
      {
        nbWithNation++;
      }
      if (p->name() == ((Player*)(*it))->name())
      {
        nbWithName++;
      }
    }
    if (nbWithName != 1)
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << Q_UINT32(player->id());
      sendMessage(buffer,ChangePlayerName);
      
      return;
    }
    else if (nbWithNation != 1)
    {
      QByteArray buffer;
      QDataStream stream(buffer, IO_WriteOnly);
      stream << player->id();
//       kdDebug() << "Sending ChangePlayerNation for player id " << player->id() << endl;
      sendMessage(buffer,ChangePlayerNation);
      
      return;
    }
    KStringVector messageParts;
    messageParts 
      << I18N_NOOP("%1 (%2) joined game ; waiting for %3 players to connect")
      << p-> name() << p->getNation()->name() 
      << QString::number(maxPlayers() - int(playerList()->count()));
    m_game->broadcastChangeItem(messageParts, ID_STATUS_MSG2);
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
//     kdDebug() << "Sending StartGame" << endl;
    sendMessage(buffer,StartGame);
    
  }
}

bool GameAutomaton::startGame()
{
  kdDebug() << "GameAutomaton::startGame nb players = " << playerList()->count() << " / " << maxPlayers() << endl;
//   kdDebug() << "  state is " << GameStateNames[m_state] << endl;
//   kdDebug() << "  saved state is " << GameStateNames[m_savedState] << endl;
  if (isAdmin() && int(playerList()->count()) == maxPlayers())
  {
    m_game->haltTimer();

    if (m_state == INIT && m_savedState == INVALID)
    {
      m_game->firstCountriesDistribution();
//       finalizePlayers();

      if (useGoals())
      {
        PlayersArray::iterator it = playerList()->begin();
        PlayersArray::iterator it_end = playerList()->end();
        for (; it != it_end; it++)
        {
          QByteArray buffer;
          QDataStream stream(buffer, IO_WriteOnly);
          stream << (*it)->id();
          sendMessage(buffer,SetGoalFor);
        }
      }
    }
    else if (m_state == WAIT_PLAYERS)
    {
      m_game->sendCountries();
      state(m_savedState);
      currentPlayer(playerNamed(m_savedPlayer));
      m_game->displayButtonsForState(m_savedState);
      m_savedPlayer = "";
      m_savedState = INVALID;
    }

    kdDebug() << "Sending message FinalizePlayers" << endl;
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
    sendMessage(buffer,FinalizePlayers);

    setGameStatus(KGame::Run);
    m_game->initTimer();
//     kdDebug() << "    true" << endl;
    return true;
  }
  else
  {
//     kdDebug() << "    false" << endl;
    return false;
  }
}

void GameAutomaton::changePlayerName(Player* player)
{
//   kdDebug() << "GameAutomaton::changePlayerName" << endl;
  
  std::map< QString, QString > nations = m_game->nationsList();
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    std::map<QString,QString>::iterator nationsIt;
    nationsIt = nations.find(((Player*)(*it))-> getNation()->name());
    if (nationsIt !=  nations.end())
    {
      nations.erase(nationsIt);
    }
  }
  //m_theWorld->communicatesWith_test();
  
// Players names
  QString mes = "";
  QString nationName;
  
  QString nomEntre = player->name();
  bool computer = player->isAI();
  
  bool found = true;
  KMessageBox::information(m_game, i18n("Please choose another name"), i18n("KsirK - Name already used !"));
  while(found)
  {
    bool emptyName = true;
    while (emptyName)
    {
      mes.sprintf(i18n("Player number %d, what's your name ?"), 0);
      bool network = false;
      QString password;
      KPlayerSetupDialog(m_game->theWorld(), 0, nomEntre, network, password, computer, nations, nationName, m_game, "KDialogSetupPlayer").exec();
//     kdDebug() << "After KPlayerSetupDialog. name: " << nomEntre << endl;
      if (nomEntre == "")
      {
        mes.sprintf( i18n("Error - Player %d, you have to choose a name."), 0);
        KMessageBox::sorry(m_game, mes, i18n("Error"));
        nomEntre.sprintf( i18n("Player%d"), 0);
      }
      else 
      {
        emptyName = false;
      }
    }
    found = false;
    PlayersArray::iterator it = playerList()->begin();
    PlayersArray::iterator it_end = playerList()->end();
    for (; it != it_end; it++)
    {
      if ( (*it)-> name() ==  nomEntre)
      {
        found = true;
        it = it_end;
      }
    }
    if (!found)
    {
//     kdDebug() << "Creating player " << nomEntre << "(computer: " << computer << "): " << nationName << endl;
      player->setName(nomEntre);
    }
  }
  QByteArray buffer;
  QDataStream stream(buffer, IO_WriteOnly);
  sendMessage(buffer,StartGame);
}

void GameAutomaton::changePlayerNation(Player* player)
{
//   kdDebug() << "GameAutomaton::changePlayerName" << endl;
  
  std::map< QString, QString > nations = m_game->nationsList();
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    std::map<QString,QString>::iterator nationsIt;
    nationsIt = nations.find(((Player*)(*it))-> getNation()->name());
    if (nationsIt !=  nations.end())
    {
      nations.erase(nationsIt);
    }
  }
  //m_theWorld->communicatesWith_test();
  
// Players names
  QString mes = "";
  QString nationName;
  
  QString nomEntre = player->name();
  bool computer = player->isAI();
  KMessageBox::information(m_game, i18n("Please choose another nation"), i18n("KsirK - Nation already used !"));
  bool network = false;
  QString password = false;
  KPlayerSetupDialog(m_game->theWorld(), 0, nomEntre, network, password, computer, nations, nationName, m_game, "KDialogSetupPlayer").exec();
  QByteArray buffer;
  QDataStream stream(buffer, IO_WriteOnly);
  stream << player->name() << nationName;
  sendMessage(buffer,SetNation);
  QByteArray buffer2;
  QDataStream stream2(buffer2, IO_WriteOnly);
  sendMessage(buffer2,StartGame);
}

Q_UINT32 GameAutomaton::idForMsg(const QString& msg)
{
  std::map<QString,Q_UINT32>::iterator it = m_msgs2ids.find(msg);
  if (it != m_msgs2ids.end())
  {
    return (*it).second;
  }
  else
  {
    Q_UINT32 id = m_msgs2ids.size();
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
    stream << msg << id;
    sendMessage(buffer,AddMsgIdPair);
    return id;
  }
}

QString& GameAutomaton::msgForId(Q_UINT32 id)
{
  return m_ids2msgs[id];
}

void GameAutomaton::slotPropertyChanged(KGamePropertyBase *prop,KGame *)
{
//   kdDebug() << "GameAutomaton::slotPropertyChanged " << prop->id() << endl;
  if (prop->id() == m_skinId)
  {
//     kdDebug() << "skin changed to: " << m_skin << endl;
    m_game->newSkin();
  }
}

void GameAutomaton::slotClientJoinedGame(Q_UINT32 clientid, KGame* /*me*/)
{
  if (isAdmin() && clientid!=gameId())
  {
    QByteArray buffernbp;
    QDataStream streamnbp(buffernbp, IO_WriteOnly);
    streamnbp << m_nbPlayers;
    sendMessage(buffernbp,NbPlayers,clientid);
    
    QByteArray buffer;
    QDataStream stream(buffer, IO_WriteOnly);
    if (m_game->waitedPlayers().empty())
    {
      sendMessage(buffer,SetupOnePlayer,clientid);
    }
    else
    {
      stream << Q_UINT32(m_game->waitedPlayers().size());
      std::vector<PlayerMatrix>::iterator it, it_end;
      it = m_game->waitedPlayers().begin(); it_end = m_game->waitedPlayers().end();
      for (; it != it_end; it++)
      {
        stream << (*it);
      }
      sendMessage(buffer,SetupWaitedPlayer,clientid);
    }
  }
}

void GameAutomaton::slotConnectionToServerBroken()
{
  m_game->haltTimer();
  if (m_state != GAME_OVER)
  {
    if (KMessageBox::questionYesNoCancel(m_game,
                                  i18n("KsirK - Lost connection to server !\nWhat do you want to do ?"),
                                  i18n("Starting a new game or exit."), 
                                  i18n("New Game"), i18n("Exit"), i18n("Do nothing")) == KMessageBox::Yes)
    {
      if (!m_game->actionNewGame())
        exit(0);
    }
    else
    {
      exit(0);
    }
  }
  else
  {
    m_game->haltTimer();
  }
}
  
void GameAutomaton::slotConnectionToClientBroken(KMessageIO *)
{
  m_game->haltTimer();
  if (m_state != GAME_OVER)
  {
    KMessageBox::information(m_game, 
                            i18n("Lost connection to a client.\nFor the moment, you can only save the game and start a new one or quit.\nThis will be improved in a future version."), 
                            i18n("KsirK - Lost connection to client !"));
    switch ( KMessageBox::warningYesNo( m_game, i18n("Do want to save your game?")) ) 
    {
    case KMessageBox::Yes :
      m_game->slotSaveGame();
      break;
    case KMessageBox::No :;
    default: ;
    }
    if (!m_game->actionNewGame())
      exit(1);
  }
  else
  {
    m_game->haltTimer();
  }
}

void GameAutomaton::slotNetworkData(int msgid, const QByteArray &buffer, Q_UINT32 receiver, Q_UINT32 sender)
{
  if (m_game == 0)
  {
    exit(0);
  }

  if (msgid < CountryOwner || msgid> NbPlayers)
  {
    return;
  }
//   kdDebug() << "GameAutomaton::slotNetworkData("<<KsirkMessagesIdsNames[msgid-(KGameMessage::IdUser+1)]<<", " << receiver << ", " << sender << ")" << endl;
  QDataStream stream(buffer, IO_ReadOnly);
  QString countryName, playerName, nationName;
  QPoint point;
  Q_UINT32 removable;
  Q_UINT32 theState;
  Q_UINT32 nbArmies;
  Q_UINT32 newState;
  Q_UINT32 statusBarId, messagePartsNb, messagePartsCounter, logStatus;
  Q_UINT32 propId, prop2Id;
  Q_UINT32 playerId;
  Q_UINT32 nbVotes = 0;
  Q_UINT32 explosing;
  KStringVector messageParts;
  QString messagePart;
  Player* player;
  PlayerMatrix waitedPlayerDef;
  Q_UINT32 nbWaitedPlayers;
  Q_UINT32 waitedPlayerId;
  QString waitedPlayerPassword;
  Q_UINT32 nbCountries;
  Country* country;
  Q_UINT32 msgId;
  Q_UINT32 ack;
  QString msg;
  Goal goal;
  QString playersNames;
  QPixmap pm;
  if (currentPlayer() != 0)
  {
    pm = QPixmap(*currentPlayer()->getFlag()->image(0));
  }

  QByteArray sendBuffer;
  QDataStream sendStream(sendBuffer, IO_WriteOnly);
  switch (msgid)
  {
  case CountryOwner:
    stream >> countryName >> playerName;
    kdDebug() << "CountryOwner for " << m_game->theWorld()->countryNamed(countryName) << " " << countryName << " and " << playerName << endl;
    m_game->theWorld()->countryNamed(countryName)-> owner(playerNamed(playerName));
    break;
  case PlayerPutsInitialArmy:
    stream >> point;
    if (m_game->playerPutsInitialArmy(point) && isAdmin())
    {
      event("playersLooped", point);
    }
    break;
  case PlayerPutsArmy:
    stream >> point >> removable;
    if (m_game->playerPutsArmy(point, removable) && isAdmin())
    {
      event("playersLooped", point);
    }
    break;
  case StateChange:
    stream >> theState;
//     kdDebug() << "Got new state id: " << theState << 
//         " ("<<GameStateNames[theState]<<")" << endl;
    m_state = GameState(theState);
    break;
  case PlayerChange:
    stream >> playerName;
    m_currentPlayer = playerName;
    m_currentPlayerPlayed = false;
    break;
  case RegisterCountry:
    stream >> countryName >> propId >> prop2Id;
    m_nbArmiesIdsNamesCountriesMap.insert(std::make_pair(int(propId),countryName));
    m_namesNbArmiesIdsCountriesMap.insert(std::make_pair(countryName,int(propId)));;
    m_nbAddedArmiesIdsNamesCountriesMap.insert(std::make_pair(int(prop2Id),countryName));
    m_namesNbAddedArmiesIdsCountriesMap.insert(std::make_pair(countryName,int(prop2Id)));
    break;
  case PlayerAvailArmies:
    if (sender == gameId()) break;
    stream >> playerName >> nbArmies;
    playerNamed(playerName)->setNbAvailArmies((unsigned int)nbArmies, false);
    break;
  case KGameWinAvailArmies:
    if (sender == gameId()) break;
    stream >> nbArmies;
    m_game->availArmies(nbArmies);
    break;
  case ChangeItem:
    if (sender == gameId()) break;
    stream >> statusBarId >> logStatus >> messagePartsNb >> msgId;
    messageParts << m_ids2msgs[msgId];
    kdDebug() << "Got ChangeItem on " << statusBarId << " ; nb= " << messagePartsNb << endl;
    for (messagePartsCounter = 1; messagePartsCounter < messagePartsNb; messagePartsCounter++)
    {
      int elemType;
      stream >> elemType;
      if (elemType == KStringVector::Text)
      {
        stream >> messagePart;
        messageParts << messagePart;
//       kdDebug() << " message part: " << messagePart << endl;
      }
      else if (elemType == KStringVector::Pixmap)
      {
        stream >> pm;
        messageParts << pm;
      }
    }
    m_game->changeItem(messageParts,statusBarId, logStatus);
    break;
  case DisplayRecyclingButtons:
    m_game->displayRecyclingButtons();
    break;
  case DisplayNormalGameButtons:
    m_game->displayNormalGameButtons();
    break;
  case ActionRecycling:
    if (isAdmin())
    {
      m_game->actionRecycling();
      state(NEWARMIES);
    }
    break;
  case ClearGameActionsToolbar:
    if (sender == gameId()) break;
    m_game->clearGameActionsToolbar(false);
    break;
  case DisplayDefenseButtons:
    stream >> playerName;
    if (!playerNamed(playerName)->isVirtual())
      m_game->displayDefenseButtons();
    break;
  case ActionDefense:
    stream >> nbArmies;
    if (!currentPlayer()->isVirtual())
      m_game->defense((unsigned int)nbArmies);
    break;
  case FirstCountry:
    stream >> countryName;
    m_game->firstCountry(m_game->theWorld()->countryNamed(countryName));
    break;
  case SecondCountry:
    stream >> countryName;
    m_game->secondCountry(m_game->theWorld()->countryNamed(countryName));
    break;
  case InitCombatMovement:
    m_game->initCombatMovement(m_game->firstCountry(), m_game->secondCountry());
    break;
  case AnimCombat:
    m_game->animCombat();
    break;
  case TerminateAttackSequence:
    {
      if (m_game->terminateAttackSequence() && isAdmin())
      {
        state(INVADE);
      }
      else if (isAdmin())
      {
        state(WAIT);
      }
    }
    break;
  case DisplayInvasionButtons:
    m_game->displayInvasionButtons();
    break;
  case DecrNbArmies:
    stream >> countryName >> nbArmies;
    m_game->theWorld()->countryNamed(countryName)->decrNbArmies(nbArmies);
  case DisplayNextPlayerButton:
    m_game->displayNextPlayerButton();
    break;
  case Invade:
    stream >> nbArmies;
    if (m_game-> invade(nbArmies))
      m_game-> incrNbMovedArmies(nbArmies);
    break;
  case Retreat:
    stream >> nbArmies;
    if (m_game-> retreat(nbArmies))
      m_game-> decrNbMovedArmies(nbArmies);
    break;
  case  NextPlayerNormal:
    if (isAdmin())
    {
      stream >> newState;
      if (m_game->nextPlayerNormal())
      {
        if (newState > 0)
          state(GameState(newState));
      }
    }
    break;
  case  NextPlayerRecycling:
    if (isAdmin())
    {
      stream >> newState;
      if (m_game->nextPlayerRecycling())
      {
        m_choosedToRecycleNumber = 0;
        if (newState > 0)
          state(GameState(newState));
      }
    }
    break;
  case ShowArmiesToPlace:
    messageParts << pm << I18N_NOOP("%1 : %2 armies to place") << (currentPlayer()-> name()) 
      << QString::number(currentPlayer()-> getNbAvailArmies());
    m_game->broadcastChangeItem(messageParts, ID_STATUS_MSG2);
    break;
  case PlayerRemovesArmy:
    stream >> point;
    m_game->playerRemovesArmy(point);
    break;
  case VoteRecyclingFinished:
    if (isAdmin())
    {
//       stream >> playerName;
      messageParts << I18N_NOOP("%1 choose to end recycling ; there is now %2 players who want so");
      stream >> nbVotes;
      stream >> playerId;
      playersNames = ((Player*)(findPlayer(playerId)))->name();
      if (m_choosedToRecycle.find(playerId) == m_choosedToRecycle.end())
      {
        m_choosedToRecycle.insert(playerId);
      }
      else
      {
        nbVotes--;
      }

      for (Q_UINT32 i = 1 ; i < nbVotes; i++)
      {
        stream >> playerId;
        playersNames += QString(", ") + ((Player*)(findPlayer(playerId)))->name();
        if (m_choosedToRecycle.find(playerId) == m_choosedToRecycle.end())
        {
          m_choosedToRecycle.insert(playerId);
        }
        else
        {
          nbVotes--;
        }
      }
//       kdDebug() << "VoteRecyclingFinished nb before: " << m_choosedToRecycleNumber << endl;
      m_choosedToRecycleNumber+=nbVotes;
//       kdDebug() << "VoteRecyclingFinished nb after : " << m_choosedToRecycleNumber << endl;
      messageParts << playersNames << QString::number(m_choosedToRecycleNumber);
      m_game->broadcastChangeItem(messageParts, ID_NO_STATUS_MSG);
      if (m_choosedToRecycleNumber == playerList()->count())
      {
        m_game->actionRecyclingFinished();
        m_choosedToRecycle.clear();
      }
    }
    break;
  case CancelShiftSource:
    m_game-> cancelShiftSource();
    if (isAdmin())
    {
      state(SHIFT1);
    }
    break;
  case ChangePlayerNation:
    stream >> playerId;
//     kdDebug() << "Got ChangePlayerNation for player id " << playerId << endl;
    player = (Player*)(findPlayer(playerId));
//     kdDebug() << "Found player id " << player;
    if (player && !player->isVirtual())
    {
      changePlayerNation(player);
    }
    break;
  case ChangePlayerName:
    stream >> playerId;
    player = (Player*)(findPlayer(playerId));
    if (player && !player->isVirtual())
    {
      changePlayerName(player);
    }
    break;
  case StartGame:
    if (isAdmin())
      startGame();
    break;
  case SetNation:
    stream >> playerName >> nationName;
    playerNamed(playerName)->setNation(nationName);
    break;
  case SetBarFlagButton:
    stream >> playerName;
//     kdDebug() << "Calling setBarFlagButton for player " << playerNamed(playerName) << " named " << playerName << endl;
    m_game->setBarFlagButton(playerNamed(playerName));
    break;
  case FinishMoves:
    m_game->finishMoves();
    break;
  case AnimExplosion:
    stream >> explosing;
    if (explosing != 0 && explosing != 1 && explosing != 2)
      KMessageBox::information(m_game, i18n("Problem : no one destroyed"), i18n("Ksirk - Error !"));
    else
      m_game->animExplosion(explosing);
    break;
  case SetupOnePlayer:
    if (receiver == gameId())
    {
      m_game->setupOnePlayer();
    }
    break;
  case SetupWaitedPlayer:
    stream >> nbWaitedPlayers;
    for (Q_UINT32 i = 0; i < nbWaitedPlayers; i++)
    {
      stream >> waitedPlayerDef;
      m_game->waitedPlayers().push_back(waitedPlayerDef);
    }
    m_game->setupOneWaitedPlayer();
    break;
  case ValidateWaitedPlayerPassword:
    if (isAdmin())
    {
      stream >> waitedPlayerId >> waitedPlayerPassword;
//       kdDebug() << "Validating password for waited player " << waitedPlayerId << " : " << m_game->waitedPlayers()[waitedPlayerId].password << " / " << waitedPlayerPassword << endl;
      if (m_game->waitedPlayers()[waitedPlayerId].password == waitedPlayerPassword)
      {
        sendStream << waitedPlayerId;
        sendMessage(sendBuffer,ValidPassword, sender);
      }
      else
      {
        sendMessage(sendBuffer,InvalidPassword, sender);
      }
    }
    break;
  case ValidPassword:
    if (receiver == gameId())
    {
      stream >> waitedPlayerId;
      m_game->createWaitedPlayer(waitedPlayerId);
    }
    break;
  case InvalidPassword:
    if (receiver == gameId())
    {
      KMessageBox::information(m_game, i18n("You entered an invalid password.\nPlease try again."), i18n("KsirK - Invalid password !"));
      m_game->setupOneWaitedPlayer();
    }
    break;
  case SetupCountries:
    stream >> nbCountries;
    for (unsigned int i = 0; i < nbCountries; i++)
    {
      stream >> countryName;
      country = m_game->theWorld()->countryNamed(countryName);
//       kdDebug() << "Setting up country n°" << i << " on " << nbCountries << ", " << country << " named " << countryName << endl;
      if (country)
      {
        stream >> country;
      }
    }
    break;
  case AddMsgIdPair:
    stream >> msg >> msgId;
    m_msgs2ids.insert(std::make_pair(msg,msgId));
    m_ids2msgs.insert(std::make_pair(msgId,msg));
    break;
  case CheckGoal:
    if (isAdmin())
    {
//       kdDebug() << "CheckGoal: " << endl;
      stream >> playerId;
      player = static_cast< Player* >(findPlayer(playerId));
      if (player)
      {
        if (player->checkGoal())
        {
//           kdDebug() << "    goal reached for " << player->name() << endl;
//           kdDebug() << "    setting state to " << GameStateNames[GAME_OVER] << endl;
          m_game->haltTimer();
          setGameStatus(KGame::End);
          state(GAME_OVER);
          sendStream << player->id();
//           kdDebug() << "    sending Winner" << endl;
          sendMessage(sendBuffer,Winner);
        }
      }
    }
    break;
  case SetGoalFor:
    if (isAdmin())
    {
      stream >> playerId;
      setGoalFor(static_cast<Player*>(findPlayer(playerId)));
    }
    break;
  case GoalForIs:
//     kdDebug() << "GoalForIs: " << endl;
    stream >> playerId;
//     kdDebug() << "  player id: " << playerId << endl;
    stream >> goal;
    static_cast<Player*>(findPlayer(playerId))->goal(new Goal(goal));
    break;
  case FinalizePlayers:
      kdDebug() << "Got message FinalizePlayers" << endl;
      finalizePlayers();
    break;
  case Winner:
    QObject::disconnect(messageServer(),SIGNAL(connectionLost(KMessageIO *)),
                        this,SLOT(slotConnectionToClientBroken(KMessageIO *)));
    stream >> playerId;
    m_game->winner(static_cast<Player*>(findPlayer(playerId)));
    break;
  case NbPlayers:
    stream >> m_nbPlayers;
    break;
  case Acknowledge:
      kdDebug() << "Got message Acknowledge" << endl;
      stream >> playerId;
      stream >> ack;
      acknowledge(static_cast<Player*>(findPlayer(playerId)), ack);
    break;
  default: ;
  }
}

void GameAutomaton::finalizePlayers()
{
  kdDebug() << "GameAutomaton::finalizePlayers" << endl;
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    static_cast<Player*>(*it)-> finalize();
  }
}

/** @return true if all players are played by computer ; false otherwise */
bool GameAutomaton::allComputerPlayers()
{
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    if ( ! static_cast<Player*>(*it)-> isAI() )
    {
      return false;
    }
  }
  return true;
}

bool GameAutomaton::allLocalPlayersComputer()
{
  PlayersArray::iterator it = playerList()->begin();
  PlayersArray::iterator it_end = playerList()->end();
  for (; it != it_end; it++)
  {
    if ( ( ! static_cast<Player*>(*it)-> isVirtual() ) &&  ( ! static_cast<Player*>(*it)-> isAI() ) )
    {
      return false;
    }
  }
  return true;
}


}; // closing namespace GameLogic
}; // closing namespace Ksirk

#include "gameautomaton.moc"
