//
// C++ Interface: gameautomaton
//
// Description:
//
//
// Author: Gaël de Chalendar <kleag@free.fr>, (C) 2004
//
// Copyright: See COPYING file that comes with this distribution
//
//
#ifndef KSIRK_GAMELOGIC_GAMEAUTOMATON_H
#define KSIRK_GAMELOGIC_GAMEAUTOMATON_H

#include "eventslistproperty.h"

#include <kgame/kgame.h>
#include <kgame/kplayer.h>
#include <kgame/kgameio.h>
#include <kgame/kgamepropertyarray.h>
#include <kgame/kgameproperty.h>
#include <kgame/kmessageio.h>

#include <qpoint.h>
#include <qstring.h>
#include <qstringlist.h>
#include <map>
#include <set>

namespace Ksirk {
class KGameWindow;
namespace GameLogic {

class Player;
class Country;
class Goal;

enum KsirkMessagesIds {
    CountryOwner = KGameMessage::IdUser+1, // 257
    PlayerPutsArmy, // 258
    StateChange, // 259
    PlayerChange, // 260
    RegisterCountry, // 261
    PlayerAvailArmies, // 262
    KGameWinAvailArmies, // 263
    ChangeItem, // 264
    DisplayRecyclingButtons, // 265
    DisplayNormalGameButtons, // 266
    ActionRecycling, // 267
    ClearGameActionsToolbar, // 268
    DisplayDefenseButtons, // 269
    ActionDefense, // 270
    FirstCountry, // 271
    SecondCountry, // 272
    InitCombatMovement, // 273
    AnimCombat, // 274
    DisplayInvasionButtons, // 275
    TerminateAttackSequence, // 276
    DecrNbArmies, // 277
    DisplayNextPlayerButton, // 278
    Invade, // 279
    Retreat, // 280
    NextPlayerNormal, // 281
    NextPlayerRecycling, // 282
    ShowArmiesToPlace, // 283
    PlayerPutsInitialArmy, // 284
    PlayerRemovesArmy, //285
    VoteRecyclingFinished, // 286
    CancelShiftSource, // 287
    ChangePlayerNation, // 288
    ChangePlayerName, // 289
    StartGame, // 290
    SetNation, // 291
    SetBarFlagButton, // 292
    FinishMoves, // 293
    AnimExplosion, // 294
    SetupOnePlayer, // 295
    SetupWaitedPlayer, // 296
    ValidateWaitedPlayerPassword, // 297
    ValidPassword, // 298
    InvalidPassword, // 299
    SetupCountries, // 300
    AddMsgIdPair, // 301
    CheckGoal, // 302
    SetGoalFor, // 303
    GoalForIs, // 304
    Winner, // 305
    NbPlayers, // 306
    FinalizePlayers, // 307
    Acknowledge, // 308
};

/** Messages formats:
  * 
  * CountryOwner: country name (QString), owner (QString) 
  * 
  */


/**
    * @author Gaël de Chalendar
*/
class GameAutomaton : public KGame
{
Q_OBJECT
    
public:
    /**
      * The State enum enumerates all the possible states of the game. The
      * behavior of various method will vary in function of it
      */
    enum GameState {
      INIT, 
      INTERLUDE, // Waiting state after initial distribution of armies at game beginning
      NEWARMIES, 
      WAIT, // Basic state waiting a player action
      WAIT_RECYCLING, 
      ATTACK,
      ATTACK2, 
      INVADE, 
      SHIFT1, 
      SHIFT2, 
      FIGHT_BRING,
      FIGHT_ANIMATE, 
      FIGHT_BRINGBACK, 
      WAITDEFENSE, 
      EXPLOSION_ANIMATE, 
      WAIT_PLAYERS, 
      GAME_OVER, // Game finished
      INVALID
    };

    static const GameAutomaton& single();

    static GameAutomaton& changeable();

    static void init(KGameWindow* gw);

    GameState state() const;
    void state(GameState state);
    
    GameState run();
    
    void event(const std::string& event, const QPoint& point = QPoint());
    
    /** returns the name of the current state */
    const std::string stateName() const;
    
    void saveXml(std::ostream& xmlStream);

    const QString& skin() const;
    
    void skin(const QString& newSkin);
    
    /**
    * Chooses the next player to make a move by setting setTurn(true,true)
    * in the current player object. If a last player is given then the new
    * player will be choosen so that it is not the last player, i.e. the
    * players a swapped. If no last player is given then the current player
    * is reactiveted
    * @param last - the last player who made a move
    * @param unused
    * @return the player whose turn it will be
    */
  KPlayer* nextPlayer(KPlayer *last,bool /*exclusive*/);
  
  inline unsigned int networkPlayersNumber() {return m_networkPlayersNumber;}
  
  inline Ksirk::KGameWindow* game() {return m_game;}
  
  virtual KPlayer * createPlayer(   int   rtti, 
                                              int   io, 
                                              bool   isvirtual
                                );  
  
  /**
   * @return A pointer to the given's named player ; 0 if there is no such player
   */
  Player* playerNamed(const QString& playerName);
    
  Player* currentPlayer();
  void currentPlayer(Player* player);
  
  inline bool currentPlayerPlayed() {return m_currentPlayerPlayed;}
  inline void currentPlayerPlayed(bool val) {m_currentPlayerPlayed = val;}

  inline void savedPlayer(const QString& player) {m_savedPlayer = player;}
  inline void savedState(GameState state) {m_savedState = state;}
  
  inline std::map<Q_UINT32,QString>& ids2msgs() {return m_ids2msgs;}
  inline std::map<QString,Q_UINT32>& msgs2ids() {return m_msgs2ids;}

  Q_UINT32 idForMsg(const QString& msg);
  QString& msgForId(Q_UINT32 id) ;
    
  inline const std::set< Goal* >& goals() const {return m_goals;}
  inline std::set< Goal* >& goals() {return m_goals;}
  inline bool useGoals() {return m_useGoals;}
  
  inline Q_UINT32 nbPlayers() const {return m_nbPlayers;}
  inline void nbPlayers(Q_UINT32 nb) {m_nbPlayers = nb;}
  
  /**
   * Ask the user how much players there will be in the game and what skin to 
   * use. Called during new game initialization.
   */
  bool setupPlayersNumberAndSkin(bool& networkGame, int& port,
                                 uint& newPlayersNumber);
  
    /**
     * Create an IO device like Mouse or Keyboard for the given player
     * and plug it into the player
     * @param player - the player who should get the IO device
     * @param io - the IO code for which a device needs to be created
     */
  void createIO(KPlayer *player,KGameIO::IOMode io);
  
  /** @return true if all players are played by computer ; false otherwise */
  bool allComputerPlayers();
  
  /** @return true if all local players are played by computer ; false otherwise */
  bool allLocalPlayersComputer();
    
  /**
   * Gets a local player.
   * @return Returns a local player... Any of them. Or 0 if there is no local
   * player.
   */
  Player* getAnyLocalPlayer();

public slots:
  /**
    * This slot connects to a main signal of the KGame. It will be called
    * whenever a KGame property changes its value. This slot can then
    * used to build up the event distribution of your game, i.e. you react
    * here to all events generated by the game (NewGame, ...)
    * Which property is changed you best find out via the property id.
    * For example: if (prop-id() == myvariable.id()) ...
    *
    * @param prop - the property 
    * @param game - the game object (unused)
    */
  void slotPropertyChanged(KGamePropertyBase *prop,KGame * game);
  
  void slotPlayerJoinedGame(KPlayer* player);
  
  void slotNetworkData(int msgid, const QByteArray &buffer, Q_UINT32 receiver, Q_UINT32 sender);
  
  inline std::map< int, QString >& nbArmiesIdsNamesCountriesMap() {return m_nbArmiesIdsNamesCountriesMap;}
  inline std::map< QString, int >& namesNbArmiesIdsCountriesMap() {return m_namesNbArmiesIdsCountriesMap;}
  inline std::map< int, QString >& nbAddedArmiesIdsNamesCountriesMap() {return m_nbAddedArmiesIdsNamesCountriesMap;}
  inline std::map< QString, int >& namesNbAddedArmiesIdsCountriesMap() {return m_namesNbAddedArmiesIdsCountriesMap;}
  
  void slotClientJoinedGame(Q_UINT32 clientid, KGame* me);
    
  void slotConnectionToServerBroken();
  
  void slotConnectionToClientBroken(KMessageIO *);
  
protected:
    friend class KGameWindow;
    
    /**
     * Main method of the KGame which has to be overwritten in our
     * class. It will receive all input a player makes, either via
     * network, via mouse or even a computer player.
     * @param msg - the datastream message containing the information
     * @param player - the player who did the input
     */
    bool playerInput(QDataStream &msg,KPlayer *player);

    
  void setGoalFor(Player* player);
  
private:
  GameAutomaton();
    
  GameAutomaton(const GameAutomaton& /*ga*/) : KGame() {};

  virtual ~GameAutomaton();

  bool joinNetworkGame();
    
  void changePlayerName(Player* player);
  
  void changePlayerNation(Player* player);
  
  bool startGame();

  void finalizePlayers();
  
  void acknowledge(Player* player, unsigned int ack);

    static GameAutomaton* m_singleton ;
    
  GameState m_state;
//   KGameProperty< GameState > m_state;
//     int m_stateId;
  
  KGameWindow *m_game;
  
  EventsListProperty m_events;
  
  static const char* GameStateNames[] ;
  
  static const char* KsirkMessagesIdsNames[] ;
  
  KGamePropertyQString m_skin;
  
  unsigned int m_networkPlayersNumber;
  
  int m_skinId;

  /** @brief The name of the current player */
  QString m_currentPlayer;

  /** @brief false until the current player has played something */

  bool m_currentPlayerPlayed;
//   int m_currentPlayerId;
  
  std::map< int, QString > m_nbArmiesIdsNamesCountriesMap;
  std::map< QString, int > m_namesNbArmiesIdsCountriesMap;
  std::map< int, QString > m_nbAddedArmiesIdsNamesCountriesMap;
  std::map< QString, int > m_namesNbAddedArmiesIdsCountriesMap;

  Q_UINT32 m_choosedToRecycleNumber;
  
  QString m_savedPlayer;
  GameState m_savedState;
  
  std::map<Q_UINT32,QString> m_ids2msgs;
  std::map<QString,Q_UINT32> m_msgs2ids;
  
  std::set< Goal* > m_goals;
  bool m_useGoals;
  
  Q_UINT32 m_nbPlayers;

  std::set<int> m_choosedToRecycle;

};

QDataStream& operator>>(QDataStream& s, GameAutomaton::GameState& state);
  
} // closing namespace GameLogic
} // closing namespace Ksirk

#endif // KSIRK_GAMELOGIC_GAMEAUTOMATON_H
