//
// C++ Implementation: goal
//
// Description: 
//
//
// Author: Gaël de Chalendar (aka Kleag) <kleag@free.fr>, (C) 2005
//
// Copyright: See COPYING file that comes with this distribution
//
//
#include "goal.h"
#include "player.h"
#include "onu.h"
#include "kgamewin.h"

#include <kdebug.h>
#include <klocale.h>
#include <kmessagebox.h>

namespace Ksirk {

namespace GameLogic {

Goal::Goal() :
  m_type(NoGoal),
  m_description(""),
  m_nbCountries(0),
  m_nbArmiesByCountry(0),
  m_continents(),
  m_players(),
  m_player(0)
{
}

Goal::Goal(const Goal& goal)
{
  m_type = goal.m_type;
  m_description = goal.m_description;
  m_nbCountries = goal.m_nbCountries;
  m_nbArmiesByCountry = goal.m_nbArmiesByCountry;
  m_continents = goal.m_continents;
  m_players = goal.m_players;
  m_player = goal.m_player;
}

Goal::~Goal()
{
}

bool Goal::checkFor(const GameLogic::Player* player) const 
{
  switch (type())
  {
  case Goal::GoalPlayer :
    kdDebug() << "Goal reached checked elsewhere for Player type" << endl;
    return false;
    break;
  case Goal::Countries:
    return checkCountriesFor(player);
    break;
  case Goal::Continents:
    return checkContinentsFor(player);
    break;
  default:
    return (player->getNbCountries() >= GameAutomaton::changeable().game()->theWorld()->getCountries().size());
  }
}

bool Goal::checkCountriesFor(const GameLogic::Player* player) const
{
  kdDebug() << "Goal::checkCountriesFor " << player->name() << endl;
  if (player->getNbCountries() >= m_nbCountries)
  {
    uint nbCountriesOk = 0;
    for (unsigned int i = 0; i < player->countries().size(); i++)
    {
      if ((player->countries().at(i))->nbArmies() >= m_nbArmiesByCountry)
      {
        nbCountriesOk++;
      }
    }
    return (nbCountriesOk >= m_nbCountries);
  }
  else 
  {
    return false;
  }
}

bool Goal::checkContinentsFor(const GameLogic::Player* player) const
{
  kdDebug() << "Goal::checkContinentsFor " << player->name() << endl;
  std::set<unsigned int>::const_iterator it, it_end;
  it = m_continents.begin(); it_end = m_continents.end();
  for (; it != it_end; it++)
  {
    kdDebug() << "Should be owned continent id: " << *it<< endl;
    if ( GameAutomaton::changeable().game()->theWorld()->continentWithId(*it) == 0
       || GameAutomaton::changeable().game()->theWorld()->continentWithId(*it)->owner() != player)
    {
      return false;
    }
  }
  bool otherFound = ( m_continents.find(0) == m_continents.end() );
  if (!otherFound)
  {
    QValueVector<Continent*> continents = GameAutomaton::changeable().game()->theWorld()->getContinents();
    for (unsigned int j = 0; j < continents.size(); j++)
    {
      if ( ( m_continents.find((continents.at(j))->id()) == m_continents.end() )
           && ( continents.at(j)->owner() == player) )
      {
        otherFound = true;
        break;
      }
    }
  }
  return otherFound;
}

QString Goal::message(int displayType)
{
  QString mes;

  std::set<unsigned int>::const_iterator it, it_end, it_next;
  if (displayType & GoalDesc)
  {
    mes = i18n(m_description);
    if (m_player == 0)
    {
      mes = mes.arg("");
    }
    else
    {
      mes = mes.arg(m_player->name());
    }
    kdDebug() << "Goal type='" << m_type << "' mes = '" << mes << "'" << endl;
    
    std::set<unsigned int>::const_iterator it, it_end;
    switch (m_type)
    {
    case Goal::GoalPlayer :
      if (!m_players.empty())
      {
        kdDebug() << "  player num='" << (*m_players.begin()) << endl;
        kdDebug() << "  this is player='" << GameAutomaton::changeable().findPlayer(*m_players.begin()) << endl;
        if (GameAutomaton::changeable().findPlayer(*m_players.begin())==0)
        {
          mes = mes.arg("?");
        }
        else
        {
          kdDebug() << "  its name is='" << GameAutomaton::changeable().findPlayer(*m_players.begin())->name() << endl;
          mes = mes.arg(GameAutomaton::changeable().findPlayer(*m_players.begin())->name());
        }
        mes = mes.arg(m_nbCountries);
      }
      else
      {
        mes = mes.arg(i18n("Error : no player to destroy"));
      }
      break;
    case Goal::Countries:
      kdDebug() << "  arg1 = '" << m_nbCountries << "'" << endl;
      mes = mes.arg(m_nbCountries);
      if (m_nbArmiesByCountry > 0)
      {
        kdDebug() << "  arg2 = '" << m_nbArmiesByCountry << "'" << endl;
        mes = mes.arg(m_nbArmiesByCountry);
      }
      break;
    case Goal::Continents:
      it = m_continents.begin(); it_end = m_continents.end();
      for (; it != it_end; it++)
      {
        if (*it != 0)
        {
          kdDebug() << "  arg = '" << GameAutomaton::changeable().game()->theWorld()->continentWithId(*it)->name() << "'" << endl;
          mes = mes.arg(i18n(GameAutomaton::changeable().game()->theWorld()->continentWithId(*it)->name()));
          
        }
      }
      break;
    default:;
    }
  }
  if (displayType & GoalAdvance)
  {
    int diff = 0;
    switch (m_type)
    {
    case Goal::GoalPlayer :
      if (GameAutomaton::changeable().findPlayer(*m_players.begin()) != 0)
      {
        mes += i18n("\n%1 is still alive...").arg(GameAutomaton::changeable().findPlayer(*m_players.begin())->name());
      }
      else
      {
        mes += QString(i18n("\nYou now have to conquer %1")).arg(m_nbCountries);
      }
    break;
    case Goal::Countries:
      diff  = m_nbCountries-m_player->countries().size();
      if (diff > 0)
      {
        mes += QString(i18n("\n%1, you have still %2 countries to conquer...")).arg(m_player->name()).arg(diff);
      }
      else
      {
        int nbOk = 0;
        for (int i = 0; i < m_player->countries().size();i++)
        {
          if (m_player->countries().at(i)->nbArmies() >= m_nbArmiesByCountry)
          {
            nbOk++;
          }
        }
        mes += QString(i18n("\n%1, you have enough countries but you still have to put more than %2 armies on %3 of them...")).arg(m_player->name()).arg(m_nbArmiesByCountry).arg(m_nbCountries-nbOk);
        
      }
      break;
    case Goal::Continents:
      mes += QString(i18n("\n%1, you still have to conquer ")).arg(m_player->name());
      it = m_continents.begin(); it_end = m_continents.end();
      if (*it != 0)
      {
        Continent* continent = const_cast<Continent*>(GameAutomaton::changeable().game()->theWorld()->continentWithId(*it));
        int nb = continent->getMembers().size() - continent->countriesOwnedBy(m_player).size();
        mes += QString(i18n("%1 countries in %2")).arg(nb).arg(continent->name());
      }
      it++;
      while (it != it_end)
      {
        it_next = it;
        it_next++;
        QString joint;
        if (it_next==it_end)
        {
          joint = " and ";
        }
        else
        {
          joint = ", ";
        }
        if (*it != 0)
        {
          Continent* continent = const_cast<Continent*>(GameAutomaton::changeable().game()->theWorld()->continentWithId(*it));
          int nb = continent->getMembers().size() - continent->countriesOwnedBy(m_player).size();
          mes += joint + QString(i18n("%1 in %2")).arg(nb).arg(continent->name());
        }
        it++;
      }
      mes += ".";
      break;
    default:;
    }
  }
  return mes;
}
  
void Goal::show(int displayType)
{
  KMessageBox::information(
                            GameAutomaton::changeable().game(), 
                            message(displayType), 
                            i18n("KsirK - Goal Display"));
}

QDataStream& operator<<(QDataStream& stream, const Goal& goal)
{
  stream << Q_UINT32(goal.type());
  if (goal.player() != 0)
  {
    stream << Q_UINT32(goal.player()->id());
  }
  else
  {
    stream << Q_UINT32(0);
  }
  stream << goal.description();
  std::set< unsigned int >::iterator it, it_end;
  switch (goal.type())
  {
  case Goal::GoalPlayer :
    stream << Q_UINT32(goal.players().size());
    it = goal.players().begin(); it_end = goal.players().end();
    for (; it != it_end; it++)
    {
      stream << Q_UINT32(*it);
    }
    stream << Q_UINT32(goal.nbCountries());
    break;
  case Goal::Countries:
    stream << Q_UINT32(goal.nbCountries());
    stream << Q_UINT32(goal.nbArmiesByCountry());
    break;
  case Goal::Continents:
    stream << Q_UINT32(goal.continents().size());
    it = goal.continents().begin(); it_end = goal.continents().end();
    for (; it != it_end; it++)
    {
      stream << Q_UINT32(*it);
    }
    break;
  default:;
  }
  return stream;
}

QDataStream& operator>>(QDataStream& stream, Goal& goal)
{
  Q_UINT32 type;
  QString description;
  Q_UINT32 nb, nbp;
  Q_UINT32 id, ownerId;
  stream >> type;
  stream >> ownerId;
  goal.player(static_cast<Player*>(GameAutomaton::changeable().findPlayer(ownerId)));
  goal.type(Goal::GoalType(type));
  stream >> description;
  goal.description(description);
  switch (type)
  {
  case Goal::GoalPlayer :
    goal.players().clear();
    stream >> nbp;
    for (Q_UINT32 i = 0; i < nbp; i++)
    {
      stream >> id;
      goal.players().insert(id);
    }
    stream >> nb;
    goal.nbCountries(nb);
    break;
  case Goal::Countries:
    stream >> nb;
    goal.nbCountries(nb);
    stream >> nb;
    goal.nbArmiesByCountry(nb);
    break;
  case Goal::Continents:
    stream >> nb;
    goal.continents().clear();
    for (Q_UINT32 i = 0; i < nb; i++)
    {
      stream >> id;
      goal.continents().insert(id);
    }
    break;
  default:;
  }
  return stream;
}

void Goal::saveXml(std::ostream& xmlStream) const
{
  
  xmlStream << "<goal player=\"";
  if (m_player != 0)
  {
    xmlStream << m_player->name();
  }
  xmlStream << "\" type=\"" << m_type << "\" description=\"" << m_description;
  xmlStream << "\" nbCountries=\"" << m_nbCountries << "\" nbArmiesByCountry=\"" << m_nbArmiesByCountry << "\">\n";
  xmlStream << "<continents>\n";
  std::set< unsigned int >::iterator itc, itc_end;
  itc = continents().begin(); itc_end = continents().end();
  for (; itc != itc_end; itc++)
  {
    QString name = (*itc==0)?"":GameAutomaton::changeable().game()->theWorld()->continentWithId(*itc)->name();
    xmlStream << "<continent name=\"" << name << "\"/>\n";
  }
  xmlStream << "</continents>\n";
  xmlStream << "<players>\n";
  std::set< unsigned int >::iterator itp, itp_end;
  itp = players().begin(); itp_end = players().end();
  for (; itp != itp_end; itp++)
  {
    xmlStream << "<player name=\"" << GameAutomaton::changeable().findPlayer(*itp)->name() << "\"/>\n";
  }
  xmlStream << "</players>\n";
  xmlStream << "</goal>\n";
  
    
}

}

}
