/***************************************************************************
                          onu.h  -  description
                             -------------------
    begin                : Wed Jul 18 2001
    copyright            : (C) 2001 by Gaël de Chalendar
    email                : Gael.de.Chalendar@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#define KDE_NO_COMPAT

#ifndef ONU_H
#define ONU_H

#include <qptrlist.h>
#include <qptrvector.h>
#include <qvaluevector.h>
#include <qdatastream.h>

#include "country.h"
#include "continent.h"
#include "nationality.h"

#include <iostream>
#include <vector>

namespace Ksirk
{

namespace GameLogic
{

class Country;

/**
  * Class ONU (Organisation des Nations Unies = UNO : United Nations Organization)
  * is simply the list of the countries. Its implementation file contains the
  * data definining each country.
  */
class ONU
{
public:
  ONU(const QString& configFileName);
  
  ONU() {}
  
  virtual ~ONU() {}
  
  /** {@
    * Accessors
    */
  const QString& skin() const {return m_skin;}
  const QString& name() const {return m_name;}
  const QString& description() const {return m_description;}
  const QString& mapFileName() const;
  const QPixmap& map() const {return m_map;}
  const QPixmap& snapshot() const {return m_snapshot;}
  unsigned int width() const;
  unsigned int height() const;
  
  /** @} */
  
/**
    * This method returns a pointer to the country that contains the point (x,y).
    *  f there is no country at (x,y), the functions returns 0.
    */
  Country* countryAt(int x, int y);

  /**
    * Calls its reset method for each country
    */
  void reset();

  /**
    * Return the countries list
    */
  std::vector<Country*>& getCountries();
  
  Country* getCountryAt(unsigned int i);

  /**
    * Returns the nationalities list
    */
  std::vector<Nationality*>& getNationalities();

  /** Read property of std::vector<Continent*> continents. */
  std::vector<Continent*>& getContinents();
  const std::vector<Continent*>& getContinents() const;

  const Continent* continentWithId(const unsigned int) const;
  
  /** Returns the list of countries neighbours of the parameter country that 
    * belongs to the argument player.
    */
    std::vector<Country*> neighboursBelongingTo(const Country& country, const Player* player);

  /** Returns the list of countries neighbours of the parameter country that 
    * does not belong to the argument player.
    */
    std::vector<Country*> neighboursNotBelongingTo(const Country& country, const Player* player);
  /** Returns the country named "name" ; 0 in case there is no such country */
  Country* countryNamed(const QString& name);
  Country* countryWithId(Q_UINT32 id);
  
  /** tests the neighbouring of all countries */
  void communicatesWith_test();
  /** @return the number of countries in the world */
  unsigned int getNbCountries() const;

  void saveXml(std::ostream& xmlStream);
  
  /** Returns the nation named "name" ; 0 in case there is no such nation */
  Nationality* nationNamed(const QString& name);
  
  void sendCountries(QDataStream& stream);
  
  Continent* continentNamed(const QString& name);
    
private:
  struct FontDesc
  {
    QString family;
    uint size;
    QFont::Weight weight;
    bool italic;
    QString foregroundColor;
    QString backgroundColor;
  };
  
  /**
    * The name of the XML file containing the world's definition
    */
  QString m_configFileName;
  
  /**
    * The displayable name of the skin
    */
  QString m_name;
  
  /**
    * The displayable long description of the skin
    */
  QString m_description;
  
  /**
    * The map used by this skin, decorated with countries names
    */
  QPixmap m_map;
  
  /**
    * A snaphsot of a running game with this skin
    */
  QPixmap m_snapshot;
  
  /**
    * The name of the image file containing the world's map
    */
  QString m_mapFileName;
  
  /**
    * The width and height of the map file (will be used as canvas size)
    */
  unsigned int m_width;
  unsigned int m_height;
  
  /**
    * The list of countries
    */
  std::vector<Country*> countries;

  /**
    * The list of nationalities
    */
  std::vector<Nationality*> nationalities;

  /**
    * The continents of the world
    */
  std::vector<Continent*> m_continents;

  /**
    * This image stores the mask that defines the countries of the world.
    * The blue RGB component value of each pixel gives the index of the
    * country in the countries list.
    */
  QImage countriesMask;
  
  /**
    * The path to the skin ; relative to the ksirk data dir ; loaded from the XML file
    */
  QString m_skin;
  
  FontDesc m_font;
  
  /**
   * Build the map from it's stored image and the countries names
   */
  void buildMap();
};

}
}
#endif // ONU_H

