/***************************************************************************
                          animsprite.h  -  description
                             -------------------
    begin                : Wed Jul 18 2001
    copyright            : (C) 2001-2002 by Gaël de Chalendar
    email                : Gael.de.Chalendar@free.fr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
#define KDE_NO_COMPAT


#ifndef ANIMSPRITE_H
#define ANIMSPRITE_H

#include "global.h"

#include <qimage.h>
#include <qcanvas.h>

namespace Ksirk
{

class BackGnd;
namespace GameLogic
{
  class Country;
}

/**
 * The AnimSprite objects are the animated images used for all individual
 * moving objects in KsirK. It is based on QCanvasSprite that handle
 * all sprite specific behavior (sequence of images...). The individual
 * images are taken from an image that stores all the versions of the sprite.
 * A sprite also have a direction (that allows to select the images looking
 * toward left or right) and a destination point allowing to handle sprite
 * movement.
 *
 */
class AnimSprite : public QCanvasSprite
{
public:
  /**
  * When used for fight display, the sprites can be attacker or
  * defendant. In the other cases, they are nothing particular (NONE)
  * This state is (1.0) only used for explosion animation
  */
  enum State {NONE, ATTACKER, DEFENDANT};

  enum TDir {state, right, left, up, down, N, S, E, O, NO, SO, SE, NE};

  /**
    * This constructor allows to create a new @ref AnimSprite whose images are
    * taken from the given file name with the given number of frames and
    * number of look directions
    */
  AnimSprite(QCanvasPixmapArray* pma, const QString &imgPath, BackGnd* aBackGnd,
              unsigned int nbFrames, unsigned int nbDirs, unsigned int visibility = 100);

  virtual ~AnimSprite() {}
  /**
    * Change the active frame to the next one in the list. Use the first one
    * if the current frame was the last one
    *
    */
  void nextFrame();

  /**
    * Moves the sprite by one step towards its destinationPoint.
    */
  void moveIt();

  /**
    * Moves the sprite by one step towards the point destPoint
    */
  void moveIt(const QPoint& destPoint);

  /**
    * Return true if the coordinates of the sprite are equal to those of its
    * destination point ; false elsewhere
    */
  bool atDestination() const;

  /**
    * Hides and shows the sprite. causes it to be repainted
    */
  void repaint();

  /**
    * Return true if the current frame is the last one. False elsewhere.
    */
  bool isLastFrame() const;

  /**
    * Accessors to some variables
    */
  void setPosition(const QPoint &);
  const QPoint& getPosition() const;
  void setDestination(GameLogic::Country *);
  GameLogic::Country *getDestination();
  void setDestinationPoint(const QPoint &);
  const QPoint& getDestinationPoint() const;
  bool isAttacker() const;
  void setAttacker();
  bool isDefendant() const;
  void setDefendant();
  bool isNone() const;
  void setNone();
  inline BackGnd* getBackGnd() {return backGnd;}

  /**
    * Bit to bit comparison
    */
  int operator ==(AnimSprite Arg1) const;

  /**
    * executes setLook towards left or right according to the relative
    * abscissa of the current position point and the destination point
    */
  void turnTowardDestination();

  /**
    * Change the images sequence of the sprite.
    */
  void changeSequence(const QString &imgPath, unsigned int newNbFrames, unsigned int nbDirs);

  /** Turn the sprite towards left */
  void setLookLeft(void );
  /** Turn the sprite towards right */
  void setLookRight();
  /** Test if the sprites looks to left */
  bool looksToLeft() const;
  /** Test if the sprites looks to right */
  bool looksToRight() const;

  /** Write property of bool approachDestByRight. */
  void setApproachDestByRight( const bool& _newVal);
  /** Read property of bool approachDestByRight. */
  bool getApproachDestByRight() const;
  /** Write property of bool approachDestByLeft . */
  void setApproachDestByLeft ( const bool& _newVal);
  /** Read property of bool approachDestByLeft . */
  bool getApproachDestByLeft () const;

  /** Write property of bool approachDestByTop. */
  void setApproachDestByTop( const bool& _newVal);
  /** Read property of bool approachDestByTop. */
  bool getApproachDestByTop() const;
  /** Write property of bool approachDestByBottom . */
  void setApproachDestByBottom ( const bool& _newVal);
  /** Read property of bool approachDestByBottom . */
  bool getApproachDestByBottom () const;

  /** Return the maximum value for x for this sprite by looking to its including background. Necessary for directed approaches. */
  const int getMaxX() const;
  /** Return the maximum value for y for this sprite by looking to its including background. Necessary for directed approaches. */
  const int getMaxY() const;

  /** returns the current state of the sprite */
  State getState() const;

  /** Return true if the state of the game is the argument; false otherwise */
  bool isMyState(State state) const;

  /** sets the new state of the game */
  void setState(State newState);

  /**
    * This function chooses the approach mode of a sprite towards its destination:
    * if the distance between the origin and the destination is higher than half
    * the size of the map and if the origin and destination countries comunicate,
    * then the sprite should choose an approach by left or right, through the
    * edge of the map.
    * This protected method will be called by three public functions specialized
    * using as source point, respectivly, the infantryman point, the cavalryman
    * point and the cannon point.
    */
  virtual void setupTravel(GameLogic::Country* src, GameLogic::Country* dest, 
    const QPoint& srcPoint, const QPoint *destPoint);

  /**
    * This virtual function chooses the approach mode of a sprite towards its
    * destination. It will be overloaded by subclasses:
    * if the distance between the origin and the destination is higher than half
    * the size of the map and if the origin and destination countries comunicate,
    * then the sprite should choose an approach by left or right, through the
    * edge of the map.
    */
  virtual void setupTravel(GameLogic::Country* src, GameLogic::Country* dest, 
    const QPoint* dpi=0);

  virtual void saveXml(std::ostream& xmlStream);
  
private:

  /**
  * Direction of the look of the sprite (left or right) ;
  * Allows to select the good image sequence
  */
  TDir look;

  /**
    * The bitmap that stores all the versions of the sprite
    */
  QImage bitmapInfo;

  /**
    * The number of versions of the sprite
    */
  unsigned int nbVersions;

  /**
    * The background onto which the sprite will be displayed
    */
  BackGnd *backGnd;

  /**
    * For a sprite moving from one country to another, the destination one ;
    * NULL otherwise.
    */
  GameLogic::Country *destination;

  /**
    * The coordinates of the destination (gun point or flag point for example)
    */
  QPoint destinationPoint;

  /**
    * the number of images in the sprite and the current image number
    */
  unsigned int frames, actFrame;

  /**
    * The attacking state of the sprite
    */
  State myState;

  /**
    * Position informations needed to load graphics from the pool
    */
  int spriteHeight, spriteWidth;

  /** If this member is true, the sprite should approach its destination by
    * the left. So, if it is at the right side of its dest, it will continue
    * towards right up to the right side of the world. There, it will jump at
    * the left side and continue directly towards its destination.
    * When this member is set to true, the sprite should set
    * approachDestByRight to false. If both are false, the sprite will go
    * directly towards its destination.
    */
  bool approachDestByLeft ;

  /** If this member is true, the sprite should approach its destination by
    * the right. So, if it is at the left side of its dest, it will continue
    * towards left up to the left side of the world. There, it will jump at
    * the right side and continue directly towards its destination.
    * When this member is set to true, the sprite should set
    * approachDestByLeft to false. If both are false, the sprite will go
    * directly towards its destination.
    */
  bool approachDestByRight;

  /** If this member is true, the sprite should approach its destination by
    * the top. So, if it is under its dest, it will continue
    * towards the bottom down to the bottom side of the world. There, it will
    *  jump at the top side and continue directly towards its destination.
    * When this member is set to true, the sprite should set
    * approachDestByBottom to false. If both are false, the sprite will go
    * directly towards its destination.
    */
  bool approachDestByTop ;

  /** If this member is true, the sprite should approach its destination by
    * the bottom. So, if it is upper its dest, it will continue
    * towards the top up to the top side of the world. There, it will jump at
    * the bottom side and continue directly towards its destination.
    * When this member is set to true, the sprite should set
    * approachDestByTop to false. If both are false, the sprite will go
    * directly towards its destination.
    */
  bool approachDestByBottom;

  /**
    * The position of the sprite ; made from the x and y members of the
    * QCanvasSprite
    */
  QPoint position;

  /**
    * updates the sequence of images used by the underlying QCanvasSprite with
    * the ones taken from the image found at imgPath. It is function of the
    * direction of the look and the geometry of the sprite
    */
  void sequenceConstruction();

  /**
    * changes the direction of my look and calls sequenceConstruction()
    */
  void setLook(TDir);
};

} // closing namespace Ksirk

#endif // ANIMSPRITE_H
