/**
  * The base class for KsirK application windows. It sets up the main
  * window and reads the config file as well as providing a menubar, toolbar
  * and statusbar. An instance of KsirKView creates your center view, which is
  * connected to the window's Doc object.
  * KsirKApp reimplements the methods that KMainWindow provides for main window
  * handling and supports full session management as well as using KActions.
  * @see KMainWindow
  * @see KApplication
  * @see KConfig
  *
  * @author Gaël de Chalendar (inspiration : Source Framework Automatically
  *  Generated by KDevelop, (c) The KDevelop Team.)
  * @version $Id: kgamewin.h 199 2006-12-08 22:39:42Z kleag $
  */
#define KDE_NO_COMPAT


#ifndef KGAMEWIN_H
#define KGAMEWIN_H

#include "global.h"
#include "decoratedgameframe.h"
#include "GameLogic/gameautomaton.h"
#include "GameLogic/player.h"
#include "GameLogic/country.h"
#include "Sprites/animspriteslist.h"

#include <stdlib.h>
#include <time.h>
#include <string>
#include <set>
#include <vector>

// include files for Qt
#include <qpalette.h>
#include <qpoint.h>
#include <qbitmap.h>

// include files for KDE
#include <kapplication.h>
#include <kmainwindow.h>
#include <kaccel.h>
#include <kaction.h>
#include <kstatusbar.h>
#include <kstandarddirs.h>
#include <ktoolbar.h>
#include <ktoolbarbutton.h>
#include <kdockwidget.h>
#include <kdialogbase.h>

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

class KTextEdit;
class KGameChat;


namespace Ksirk
{
namespace GameLogic
{
  class ONU;
  class KStringVector;
}

// forward declaration of the KsirK classes
class DecoratedGameFrame;

/**
 *
 */
class KGameWindow: public KDockMainWindow
{
  Q_OBJECT

public:
    
  /**
    * Create the window and initializes its members
    */
  KGameWindow(QWidget* parent=0, const char* name=0);
  
  /**
    * Deletes the background and the pool
    */
  ~KGameWindow();
    
    
  DecoratedGameFrame* frame() {return m_frame;}
    
  /**
    * Ask all the sprites to repaint
    */
  void paint();
  
  /**
    * Returns the country where inside which is the argument point. 0 f none.
    */
  GameLogic::Country* clickIn(const QPoint &Point);

  /**
    * Loads a new skin.
    */
  void newSkin(const QString& onuDefinitionFileName = "");
    
  void initTimer();
  
  void haltTimer();
    
/************** METHODS BELOW ARE DEFINED IN gestionSprites.cpp **************/
  /**
    * Prepares the sprites to be moved : removes the nb necessary sprites from
    * source, creates the moving sprites and gives them their destination, etc
    */
  bool initArmiesMovement(unsigned int nb, GameLogic::Country* src, GameLogic::Country* dest);
    
  /**
    * The 3 methods below manage the movement of, respectively, the cannons,
    * the cavalrymen and the infantrymen. It ask them to move by a certain amount
    * and, eventually, manages their arrival at destination
    */
  void moveCannons();
  void moveCavalrymen();
  void moveInfantrymen();
  
  /**
    * Initializes the sprites that will fight for the attacker and the
    * defender. Prepares them for moving
    */
  void initCombatMovement(GameLogic::Country *attacker, GameLogic::Country *defender);
    
  /**
    * Handle the moving of the fighters towards destination
    */
  void moveMovingCombatants();

  /**
    * Handle the arrival of a fighter to destination
    */
  void combatantArrival(AnimSpritesList< CannonSprite >::iterator& combatantIt);

  /**
    * Prepare the fighting animation : replace the sprites sequence by the
    * sequence of explosion
    */
  void animCombat();

  /**
    * Replaces the animated fighters by the simple cannon image
    */
  void stopCombat();

  /**
    * Replaces the sequence of destroyed cannon(s) by the explosion sequence
    */
  void animExplosion(int);

  /**
    * When all the explosion sequence has been shown for all explosing
    * devices, removes these sprites
    */
  void stopExplosion();

  /**
    * Set up the eventual survivor(s) to come back home
    */
  void initCombatBringBack(GameLogic::Country *, GameLogic::Country *);

  /**
    * Tests if there is currently moving armies (infantrymen, cavalrymen,
    * cannons or fighters)
    * @return true if there is moving armies; false otherwise
    */
  bool haveMovingArmies() const;

/************** END OF METHODS DEFINED IN gestionSprites.cpp *****************/

  /**
    * called every 50ms, it causes the scrolling of the world if necessary the
    * movement of the moving sprites. After it call the paint method to do the
    * animation of the sprites and it asks the pool to update
    */
  void evenementTimer();

  /**
    * Prepares the players for the game with human interaction. Return true
    * if successful or false if failure or cancel
    */
  bool setupPlayers();
  bool setupOnePlayer();
  bool setupOneWaitedPlayer();
  bool createWaitedPlayer(Q_UINT32 waitedPlayerId);
    
/**
    * Do the distribution of armies for all players in function of their
    * number of countries
    */
  void distributeArmies();

  /**
    * Computes the number of armies to be distributed to p at the beginning of
    * the turn, function of the number of countries he owns
    * TODO : this method should be a method of Player
    */
  int nbNewArmies(GameLogic::Player *p);

  /**
    * Changes the owner of the attacked country if its number of armies
    * becomes negative or null. Handle the end of the game of the
    * winning conditions are fullfilled.
    */
  bool attackEnd();

  /**
    * Computes the results of the fight
    */
  void resolveAttack();

  /**
    * Display and removes various toolbar buttons in function of the state
    * of the game.
    */
  void displayOpenGameButton();
  void displayNextPlayerButton();
  void displayRecyclingButtons();
  void displayNormalGameButtons();
  void displayDefenseButtons();
  void displayInvasionButtons();
  void displayCancelButton();

  void clearGameActionsToolbar(bool send = true);

  /**
    * Updates the flag in the statusbar with the one of the current player
    */
  void setBarFlagButton(const GameLogic::Player* player = 0);

  /**
    * @brief Shortcut for "statusBar()-> changeItem(text, id)"
    */
  void changeItem ( const QString& text, int id = ID_NO_STATUS_MSG, bool log = true);
  
  /**
   * @brief Receives pixmaps and strings to internationalize and display in a 
   * collection with stream API.
   * 
   * When encountering a serie of strings, the first one is the pattern and 
   * those following are the fillers. The end of the collection or a pixmap
   * signals that all fillers have been given.
   * @note No check is made (e.g. on the number of fillers)
   *
   * @param strings @b IN/OUT <I>KsirK::GameLogic::KStringVector&</I> 
   * the collection holding strings to display.
   * @param id @b IN <I>int</I> 
   * The id of the status bar where to display the internationalized message.
   */
  void changeItem ( Ksirk::GameLogic::KStringVector& strings, int id = ID_NO_STATUS_MSG, bool log = true);
  
  /**
   * @brief Receives strings to internationalize and display in a collection 
   * with stream API and broadcasts the message to all clients.
   * @param strings @b IN/OUT <I>KsirK::GameLogic::KStringVector&</I> 
   * the collection holding strings to display.
   * @param id @b IN <I>int</I> 
   * The id of the status bar where to display the internationalized message.
   */
  void broadcastChangeItem ( Ksirk::GameLogic::KStringVector& strings, int id = ID_NO_STATUS_MSG, bool log = true );
  
/**
    * @brief Reconnect the mouse events signals to their slots to allow human players
    * to play
    */
  void reconnectMouse();

  /**
    * Disconnects the mouse events signals from their slots to avoid human
    * player actions when it is the turn of the AI
    */
  void disconnectMouse();

  /** returns the current state of the game */
  GameLogic::GameAutomaton::GameState getState() const;

  /** Return true if the state of the game is the argument; false otherwise */
  bool isMyState(GameLogic::GameAutomaton::GameState state) const;
  
  /** @brief sets the current player to be the one pointed by the argument. Makes associated actions:
    * Changes the flag in the status bar,...
    * @return Returns true in case of success; false otherwise.
    */
//   bool setCurrentPlayer(const GameLogic::Player* player);
    
  /** sets the current player to be the first one. Makes associated actions. */
  int setCurrentPlayerToFirst();
  
  /** sets the current player to be the next one in the list. Makes the associated actions :
  Changes the flag in the status bar,...
  Returns 0 in case of success; non-zero otherwise. For example, it returns 1 the current player was the last one */
  int setCurrentPlayerToNext(bool restartRunningAIs = true);
  
  bool playerPutsArmy(const QPoint& point, bool removable);
  bool playerPutsInitialArmy(const QPoint& point);
  bool playerRemovesArmy(const QPoint& point);
  bool attacker(const QPoint& point);
  unsigned int attacked(const QPoint& point);
  void firstCountryAt(const QPoint& point);
  void secondCountryAt(const QPoint& point);

  /**
    * returns a pointer to the country that is currently attacked
    * @return the attacked country ; 0 if none
    */
  const GameLogic::Country* getAttackedCountry() const;

  /**
    * @brief setups window for recycling
    */
  void initRecycling();

  /**
  returns a pointer to the currently attacking  country
  @return the attacking country ; 0 if none
  */
  const GameLogic::Country* getAttackingCountry() const;

  bool haveMovingFighters() const;
  bool haveAnimFighters() const;
  bool haveAnimExplodings() const;
  
  bool terminateAttackSequence();

  bool nextPlayerRecycling();
  
  /** 
    * @return if true next state will be NEWARMIES else it will be WAIT
    */
  bool nextPlayerNormal();
  
  void attack(unsigned int nb);
  void defense(unsigned int nb);
  void cancelAction();
  void cancelShiftSource();
  bool actionNewGame();
  bool actionOpenGame();
  void actionRecycling();
  void postActionRecycling();
  void actionRecyclingFinished();
  
  bool isMoveValid(const QPoint& point);
  
  /**
    * @brief Accessor to the world
    * @return A pointer to the world
    */
  GameLogic::ONU* theWorld();

  /**
    * @brief Adds a player
    */
  void addPlayer(const QString& playerName, 
        unsigned int nbAvailArmies, 
        unsigned int nbCountries, 
        const QString& nationName, 
                 bool isAI,
                 const QString& password = "",
                 unsigned int nbAttack = 0,
                 unsigned int nbDefense = 0);

  /** 
    * accessors and manipulators of the number of armies moved during an invasion 
    * or an end of turn move 
    */
  int nbMovedArmies();
  void incrNbMovedArmies(unsigned int nb = 1);
  void decrNbMovedArmies(unsigned int nb = 1);
  
  /** 
    * causes the move to/from the first country stored from/to the second one
    * during an invasion or an end of turn move 
    */
  bool invade(unsigned int nb = 1);
  bool retreat(unsigned int nb = 1);
  
  /**
    * causes the end of an invasion or an end of turn move
    */
  void invasionFinished();
  void shiftFinished();
  
  /** @return true if the given player is the last one ; false otherwise */
  bool isLastPlayer(const GameLogic::Player& player);
    
  void firstCountriesDistribution();
  
  inline BackGnd* backGnd() {return m_backGnd;}
  
  inline void availArmies(unsigned int nb) {m_nbAvailArmies = nb;}
  
  inline void firstCountry(GameLogic::Country* country) {m_firstCountry = country;}
  
  inline void secondCountry(GameLogic::Country* country) {m_secondCountry = country;}
  
  inline GameLogic::Country* firstCountry() {return m_firstCountry;}
  
  inline GameLogic::Country* secondCountry() {return m_secondCountry;}
  
  void finishMoves();
    
  inline std::vector<GameLogic::PlayerMatrix>& waitedPlayers() {return m_waitedPlayers;}
  
  void sendCountries();
  
  void displayButtonsForState(GameLogic::GameAutomaton::GameState state);
    
  void winner(const GameLogic::Player* player);
    
  std::map< QString, QString > nationsList();
  
  inline KGameChat* chatWidget() {return m_chatDlg;}

protected:

  /**
    * Returns the current player
    */
  GameLogic::Player* currentPlayer();

  /**
    * Add the main toolbar buttons
    */
  void initActions();

  /**
    * Prepares the status bar
    */
  void initStatusBar();

  /**
    * creates and display the main frame with the background
    */
  void initView();

  /**
    * Prepares the sprites lists
    */
  void initSprites();

  /**
    * Add a button to the toolbar
    */
  void addAButton(const QString& fileName, int id, const char* slot, const QString& txt, const  KShortcut& shortcut, bool isTemp = false, const std::string& toolBarName = "gameActionsToolBar");

  /**
    * Reimplementation of the inherited function : starts the timer.
    */
  void enterEvent(QEvent* ev);

  /**
    * Reimplementation of the inherited function : stops the timer.
    */
  void leaveEvent(QEvent* ev);

  void clear();
  
  /**
    * Reimplementation of the inherited function called when a window close event arise
    */
  bool queryClose();
  bool queryExit();
    
  /** Shows some explanations on how to start playing */
  void explain();

public slots:
  /**
   * The slots associated to the keys
   * Will initiate different actions depending on the game state
   */
  void slotKey1();
  void slotKey2();
  void slotKey3();
  
  
  /**
    * The slots associated to the buttons
    */
  void slotNewGame();
  void slotJoinNetworkGame();
  void slotOpenGame();
  void slotSaveGame();
  void slotRecycling();
  void slotRecyclingFinished();
  void slotNextPlayer();
  void slotAttack1();
  void slotAttack2();
  void slotAttack3();
  void slotDefense1();
  void slotDefense2();
  void slotInvade1();
  void slotInvade5();
  void slotInvade10();
  void slotInvasionFinished();
  void slotRetreat1();
  void slotRetreat5();
  void slotRetreat10();
  void slotMove();
  void slotCancel();
  void slotDumpGameInformations();
  void slotScrollUp();
  void slotScrollDown();
  void slotScrollLeft();
  void slotScrollRight();
  void slotFinishMoves();

  /**
    * The standard slotShowAboutApplication slot
    */
  void slotShowAboutApplication();

  /**
    * Connected to the frame timer, it manages the behavior of the game in
    * function of the value of the state.
    *
    * Fundamental method !!!
    */
  void slotTimerEvent();

  /**
    * Connected to the frame mouse buttons signals, they manages the reaction
    * of the game to user interaction inside its main widget in function of
    * the state of the game.
    *
    * Fundamental methods !!!
    */
  void slotLeftButtonDown(const QPoint&);
  void slotLeftButtonUp(const QPoint&);
  void slotRightButtonDown(const QPoint&);

  void slotShowGoal();
  
  /** 
    * Connected to the chat widget returnPressed signal. Allows to take back
    * the focus.
    */
  void slotChatMessage();

private slots:
  void optionsConfigure();

private:

  /**
   * The main widget were is displayed the map
   */
  KDockWidget* m_mainDock;
  
  /**
   * The widget initialy docked at bottom where is displayed the events history
   */
  KDockWidget* m_bottomDock;
  
  /**
    * NKD = Numberof Killed Defenders
    * NKA = Numberof Killed Attackers
    * These numbers are set up by resolveAttack() and are used to compute
    * in various methods
    * TODO : this solution is ugly. Change that.
    */
  int NKD, NKA;

  /**
    * the countries
    */
  GameLogic::ONU* m_theWorld;

  /**
    * The main canvas
    */
  QCanvas* m_canvas;

  /**
    * The background
    */
  BackGnd* m_backGnd;

  /**
    * The lists of currently moving animated sprites
    */
  AnimSpritesList< CannonSprite > m_movingCannons;
  AnimSpritesList< CavalrySprite > m_movingCavalrymen;
  AnimSpritesList< InfantrySprite > m_movingInfantrymen;

  /**
    * The moving fighters (represented by cannon sprites)
    */
  AnimSpritesList< CannonSprite > m_movingFighters;

  /**
    * The fighting fighters (represented by firing cannons)
    */
  AnimSpritesList< CannonSprite > m_animFighters;

  /**
    * The explosiong fighters (explosing cannons)
    */
  AnimSpritesList< CannonSprite > m_animExplodings;

  /**
    * The static fighters (stopped cannons)
    */
  AnimSpritesList< CannonSprite > m_fighters;

  /**
    * The number of armies to move/moving/moved from one country to another
    */
  int m_nbMovedArmies;

  /**
    * The 2 countries involved in a fight or in a armies move
    */
  GameLogic::Country* m_firstCountry;
  GameLogic::Country* m_secondCountry;

  /**
    *
    */
//    QDialog *dialog;

  /**
    * The frame of the game, its visual component ; the main widget
    */
  DecoratedGameFrame* m_frame;

  /**
    * a shortcut to KGlobal::dirs
    */
  KStandardDirs* m_dirs;

  /**
    * the configuration object of the application
    */
  KConfig* m_config;

  /**
    * This button is used to display the flag of the currently active player
    * in the status bar
    */
  KToolBarButton* m_barFlagButton;
    
  KAccel m_accels;
  
  std::set< QString > m_temporaryAccelerators;
    
  /** @brief Used during countries distribution to handle network lags on the player member */
  unsigned int m_nbAvailArmies; 
  
  std::vector<GameLogic::PlayerMatrix> m_waitedPlayers;

  /**
   * The prompter where the game events and chat between network players are 
   * displayed
   */
  KGameChat *m_chatDlg;

  enum DiceColor {Blue,Red};
  std::map< DiceColor, std::vector<QPixmap> > m_dices;
  
private: // Private methods
  void attack(GameLogic::Country& attacker, GameLogic::Country& defender, unsigned int nb);
  void moveArmies(GameLogic::Country& src, GameLogic::Country& dest, unsigned int nb);
  void countriesDistribution();
  void saveXml(std::ostream& xmlStream);
  void loadDices();

};

} // closing namespace Ksirk

#endif // KGAMEWINDOW_H

