/*
 * warp - A displacement map sequence plugin for blender
 * Copyright (C) 2001-2006 Calle Laakkonen
 *
 * File        : warp.c
 * Description : Displacement map plugin
 * Author(s)   : Calle Laakkonen
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "plugin.h"

/* ******************** GLOBAL VARIABLES ***************** */

char name[24]= "Warp";	

/* structure for buttons, 
 *  butcode      name           default  min  max  0
 */

VarStruct varstr[]= {
    {LABEL,     "Input: 2 strips", 0.0, 0.0, 0.0, ""},
    {NUMSLI|FLO,"Warp X",		0.3,	-1.0,	1.0, "Horizontal displacement"},
    {NUMSLI|FLO,"Warp Y",	0.3,	-1.0,	1.0, "Vertical displacement"},
    {TOG|INT,   "Color", 0.0,    0.0,    1.0, "Use red/green values to control x and y displacement"},
    {TOG|INT,   "Wrap", 0.0,    0.0,    1.0, "Wrap around the edges"},
    {TOG|INT,   "Animated", 0.0,    0.0,    1.0, "Multiply displacement values by IPO value"}
};

/* The cast struct is for input in the main doit function
   Varstr and Cast must have the same variables in the same order */ 

typedef struct Cast {
    int dummy;			/* because of the 'label' button */
    float warpx;
    float warpy;
    int usecolor;
    int wrap;
    int useipo;
} Cast;


/* cfra: the current frame */

float cfra;

/* ******************** Fixed functions ***************** */

int plugin_seq_getversion(void) 
{
    return B_PLUGIN_VERSION;
}

void plugin_but_changed(int but) 
{
}

void plugin_init()
{
}

/* Get the displacement vector for coordinates x,y */
static inline void getDisplacement(int color, const ImBuf *buf, int x,int y, float *dx, float *dy) {
    unsigned char *pixel = (unsigned char*)(buf->rect + buf->x * y + x);
    if(color) {
        *dx = (pixel[0]-127)/255.0;
        *dy = (pixel[1]-127)/255.0;
    } else {
        *dx = *dy = (pixel[0] + pixel[1] + pixel[2]) / (3 * 255.0);
    }
}

/* Copy a pixel from source image to target image. If source pixel is outside */
/* the image, it is either wrapped around or the edge pixel is used. */
static inline void copyPixel(int wrap, const ImBuf *src, int srcx, int srcy, ImBuf *targ, int targx, int targy)
{
    if(wrap) {
        if(srcx < 0)
            srcx = src->x - (srcx % src->x);
        if(srcx >= src->x)
            srcx = srcx % src->x;

        if(srcy < 0)
            srcy = src->y - (srcy % src->y);
        if(srcy >= src->y)
            srcy = srcy % src->y;
    } else {
        if(srcx < 0) srcx = 0;
        else if(srcx >= src->x) srcx = src->x-1;

        if(srcy < 0) srcy = 0;
        else if(srcy >= src->y) srcy = src->y-1;
    }

    *(targ->rect + targ->x*targy + targx) = *(src->rect + src->x*srcy + srcx);
}

void plugin_seq_doit(Cast *cast, float facf0, float facf1, int x, int y, ImBuf *ibuf1, ImBuf *ibuf2, ImBuf *out, ImBuf *use)
{
    float warpx = cast->warpx * x;
    float warpy = cast->warpy * y;
    int X,Y;

    if(ibuf1==NULL || ibuf2==NULL || out==NULL)
        return;

    if(cast->useipo) {
        warpx *= facf0;
        warpy *= facf0;
    }

    for(Y=0;Y<y;Y++) {
        for(X=0;X<x;X++) {
            float dx,dy;
            getDisplacement(cast->usecolor,ibuf2,X,Y,&dx,&dy);
            copyPixel(cast->wrap, ibuf1, X-(dx*warpx), Y-(dy*warpy),
                    out, X, Y);
        }
    }
}

void plugin_getinfo(PluginInfo *info)
{
    info->name= name;
    info->nvars= sizeof(varstr)/sizeof(VarStruct);
    info->cfra= &cfra;

    info->varstr= varstr;

    info->init= plugin_init;
    info->seq_doit= (SeqDoit) plugin_seq_doit;
    info->callback= plugin_but_changed;
}

