/* 
 * focal blur plugin by onk, 8.99
 *
 * Sat Oct  9 17:57:18 MEST 1999
 *
 * copy this code to $BLENDERDIR/plugins/
 * and go "./bmake zblur" 
 * 
 * send improvements, comments & beer to: onk@gmx.net
 * Thanks to:	* PhaseIV for his initial unsharp depth code
 * 		* and of course to Meister Ton himself
 *
 * TODO: 
 * 
 * - field support - doesn't work yet (?) 
 * - edge detection for zbuffer antialiasing ?
 * - ALPHA support (for overlays)
 *
 * changes: 
 * - gamma/sigma correction
 * - ABGR->RGBA
 * 
 * ---> this code is subject to the Gnu Public License. No warranties, no charges. 
 */

#include <math.h>
#include <limits.h>
#include <stdio.h>
#include "plugin.h"

/* -------------------------------------------------
 * defines, protos */

/* image.h */

#define warn zb_warn
#define MINGAMMA 0.05
#define MAXGAMMA 2.0
#define MAXSIGMA 20.0
#define IMG_COMPLEX Complex
#define IMG_FLOAT float
#define IMG_BYTE unsigned char
#define sigmo(s, x, c) (tanh(c * (x - s)))


typedef struct {
	IMG_FLOAT re;
	IMG_FLOAT im;
} Complex;


typedef enum { I_RGB, I_RGBA, I_GRAY, I_LUT, I_FLOAT, I_FLOAT3, I_COMPLEX } IMGTYPE;

typedef struct {
	int x, y;
	int size, el_size;
	IMGTYPE type;
	unsigned char *data;
} Image;

/* end image.h */

typedef struct {			/* blur mask struct */
	int size;
	float fac;
	float *val;
} Mask;

typedef Mask* Maskarray;

/* don't change these */
#define NMASKS_SHIFT 2			
#define NMASKS 64

/* temp: */

/* -------------------------------------------------
 * globals */

float g_cfra;
char g_name[24]= "ZBlur v1d";	

/* structure for buttons, 
 *  butcode      name           default  min  max  0
 */

VarStruct varstr[] = {
	{	LABEL,			"Input: 1 strip", 0.0, 0.0, 0.0, ""},
	{	NUMSLI|FLO,		"Blursize ",	5.0,	2.0,	30.0, "Maximum blur radius"},
	{	NUMSLI|FLO,		"Zmin ",		0.0, 	0.0, 	1.0, "Zmin value"},
	{	NUMSLI|FLO,		"Focus ",	0.8, 	0.0, 	1.0, "Focus value"},
	{	TOG|INT,		"Autofocus",	1.0,	0.0,	1.0, "Autofocus" }, 
	{	TOG|INT,		"Ipofocus",	0.0,	0.0,	1.0, "IpoCurve controlled focus" },
	{	TOG|INT,		"Show zfront",	0.0,	0.0,	1.0, "Show foreground zbuffer" },
	{	TOG|INT,		"Show zback",	0.0,	0.0,	1.0, "Show background zbuffer" },
	{	NUMSLI|FLO,		"Gamma ",	1.0,	MINGAMMA,	MAXGAMMA, "Background gamma correction"},
	{	NUMSLI|FLO,		"Sigma ",	0.0,	0.0,	MAXSIGMA, "Foreground sigma correction"}
};

typedef struct Cast {
	int dummy;			/* because of the 'label' button */
	float rad;
	float zmin;
	float zfocus;
	float autofocus; 
	float ipofocus; 
	float zfront;
	float zback;
	float gamma;
	float sigma;
} Cast;

void plugin_seq_doit(Cast *, float, float, int, int, ImBuf *, ImBuf *, ImBuf *, ImBuf *);

/* ******************** Fixed functions ***************** */

int plugin_seq_getversion(void) 
{
	return B_PLUGIN_VERSION;
}

void plugin_but_changed(int but) 
{
}

void plugin_init()
{
}

void plugin_getinfo(PluginInfo *info)
{
	info->name= g_name;
	info->nvars= sizeof(varstr)/sizeof(VarStruct);
	info->cfra= &g_cfra;

	info->varstr= varstr;

	info->init= plugin_init;
	info->seq_doit= (SeqDoit) plugin_seq_doit;
	info->callback= plugin_but_changed;
}

/* ---------------------------------------------------------------------
 * Subroutines
 */

void lout(char *s)
{
	printf("%s\n", s);
}

void warn(char *s)
{
	char c[80];
	sprintf(c, "Warning: %.70s", s);
	lout(c);
}

char *g_working = "|/-\\";
short g_proglast = 0;

void progress(int i, int n)
{

	printf("  [%c]   %3d%%\015", g_working[g_proglast], 100*i / n);
	g_proglast += 1;
	g_proglast &= 3;
}

/* --------------------------------------------------------------------- */
/* image routines (imageops.c) */

unsigned int g_imagemem = 0;

Image *alloc_img(int x, int y, IMGTYPE type)
{
	Image *ret;
	int size, typesize;

	switch (type)
	{
	case I_RGB:
		typesize = 3;
		break;
	case I_RGBA:
		typesize = 4;
		break;
	case I_GRAY:
		typesize = 1;
		break;
	case I_FLOAT:
		typesize = sizeof(IMG_FLOAT);
		break;
	case I_FLOAT3:
		typesize = 3 * sizeof(IMG_FLOAT);
		break;
	case I_COMPLEX:
		typesize = sizeof(IMG_COMPLEX);
		break;
	default:
		return 0;
	}

	size = x * y;

	
	ret = (Image *) malloc(sizeof(Image) + size*typesize);
	if (ret)
	{
		g_imagemem += sizeof(Image) + size*typesize;
		ret->x = x;
		ret->y = y;
		ret->size = size;
		ret->el_size = typesize;
		ret->type = type;
		ret->data = (unsigned char *) (ret + 1);
		size *= typesize;
		memset(ret->data, 0, size);
	}

	return ret;
}

int free_img(Image *img)
{
	g_imagemem -= sizeof(Image) + img->size*img->el_size;
	free(img);
	return 1;
}

void g2rgb(IMG_BYTE *g, IMG_BYTE *rgb, int len)
{
	while (len--)
	{
		*rgb++ = *g;
		*rgb++ = *g;
		*rgb++ = *g;
		g++;
	}
}

/* --------------------------------------------------------------------- */
/* mask routines */

Mask *alloc_mask(int size)
{
	Mask *m;
	int memsize;

	memsize = (sizeof(Mask) + (2 * size +1) * (2 * size +1) * sizeof(float));
	g_imagemem += memsize;

	m = (Mask*) malloc(memsize);
	m->size = size;
	m->val = (float *) (m + 1);

	return m;
}

void free_mask(Mask *m)
{
	int memsize;

	memsize = 2 * m->size + 1;
	memsize *= memsize * sizeof(float);
	memsize += sizeof(Mask);
	
	g_imagemem -= memsize;
	free(m);
}

/* normalize mask to 1 */

void norm_mask(Mask *m)
{
	float fac;
	int size;
	float *v;

	fac = m->fac;
	size = (2 * m->size +1)*(2 * m->size +1);

	v = m->val;
	while(size--)
	{
		*v++ *= fac;
	}
	m->fac = 1.0;
}

/* sinc removed */

float zigma(float x)
{
	return 1/(1+pow(x, 8));
}
			
/* filters a grayvalue image with a gaussian IIR filter with blur radius "rad" 
 * For large blurs, it's more efficient to call the routine several times
 * instead of using big blur radii.
 * The original image is changed */


void gauss_blur(Image *img, float rad)
{
	Image *new;
	unsigned char *src, *dest;
	int r, n, m;
	int x, y;
	int i;
	int step, bigstep;

	register float sum, val;
	float gval;
	float *gausstab, *v;

	r = (1.5 * rad + 1.5);
	n = 2 * r + 1;
	/* ugly : */
	if ((img->x <= n) || (img->y <= n))
	{
		warn("not blurred - image too small");
		return;
	}
	gausstab = (float *) malloc(n * sizeof(float));
	if (!gausstab)
	{
		warn("image not blurred - could not malloc");
		return;
	}
	sum = 0.0;
	v = gausstab;
	for (x = -r; x <= r; x++)
	{

		val = exp(-4*(float ) (x*x)/ (float) (r*r));
		sum += val;
		*v++ = val;
	}

	i = n;
	v = gausstab;
	while (i--)
	{
		*v++ /= sum;
	}

	new = alloc_img(img->x, img->y, I_GRAY);
	if (!new)
	{
		warn("image not blurred - could not malloc");
		return;
	}

	/* horizontal */

	step = (n - 1);

	for (y = 0; y < img->y; y++)
	{
		src = (unsigned char *)img->data + (y * img->x);
		dest = (unsigned char *)new->data + (y * img->x);

		for (x = r; x > 0 ; x--)
		{
			m = n - x;
			gval = 0.0;
			sum = 0.0;
			v = gausstab + x;
			for (i = 0; i < m; i++)
			{
				val = *v++;
				sum += val;
				gval += val * (*src++);
			}
			*dest++ = gval / sum;
			src -= m;
		}

		for (x = 0; x <= (img->x - n); x++)
		{
			gval = 0.0;
			v = gausstab;

			for (i = 0; i < n; i++)
			{
				val = *v++;
				gval += val * (*src++);
			}
			*dest++ = gval;
			src -= step;
		}	

		for (x = 1; x <= r ; x++)
		{
			m = n - x;
			gval = 0.0;
			sum = 0.0;
			v = gausstab;
			for (i = 0; i < m; i++)
			{
				val = *v++;
				sum += val;
				gval += val * (*src++);
			}
			*dest++ = gval / sum;
			src -= (m - 1);
		}
	}

	/* vertikal */

	step = img->x;
	bigstep = (n - 1) * step;
	for (x = 0; x < step  ; x++)
	{
		src = new->data + x;
		dest = img->data + x;

		for (y = r; y > 0; y--)
		{
			m = n - y;
			gval = 0.0;
			sum = 0.0;
			v = gausstab + y;
			for (i = 0; i < m; i++)
			{
				val = *v++;
				sum += val;
				gval += val * src[0];
				src += step;
			}
			dest[0] = gval / sum;
			src -= m * step;
			dest+= step;
		}
		for (y = 0; y <= (img->y - n); y++)
		{
			gval = 0.0;
			v = gausstab;
			for (i = 0; i < n; i++)
			{
				val = *v++;
				gval += val * src[0];
				src += step;
			}
			dest[0] = gval;
			dest += step;
			src -= bigstep;
		}
		for (y = 1; y <= r ; y++)
		{
			m = n - y;
			gval = 0.0;
			sum = 0.0;
			v = gausstab;
			for (i = 0; i < m; i++)
			{
				val = *v++;
				sum += val;
				gval += val * src[0];
				src += step;
			}
			dest[0] = gval / sum;
			dest += step;
			src -= (m - 1) * step;
		}
	}
	free(gausstab);
	free_img(new);
}


Mask *gauss_mask(float rad)
{
	int r;
	int ix, iy;
	Mask *m;
	float sum, val, *v;

	
	r = (1.0 * rad + 1.0);
	m = alloc_mask(r);
	v = m->val;
	sum = 0.0;
	for (iy = -r; iy <= r; iy++)
	{
		for (ix = -r; ix <= r; ix++)
		{
			val = zigma((float) (ix*ix + iy*iy)/(rad*rad));
/*			val = exp(-(float) (ix*ix + iy*iy)/(rad * rad)); */
			sum += val;
			*v++ = val;
		}
	}
	m->fac = 1.0 / sum;
	norm_mask(m);
	return m;
}

/* generates #num masks with the maximal blur radius 'rad' 
 * */
Maskarray *init_masks(int num, float rad)
{
	int i;
	float r, step;
	Maskarray *maskarray;

	maskarray = (Maskarray*) malloc(num * sizeof (Maskarray));
	step = rad / num;
	r = 0.1;
	for (i = 0; i < num; i++)
	{
		maskarray[i] = gauss_mask(r);
		r += step;
	}
	return maskarray;
}

/* gamma correction, z-weighted */
void zgamma(Image *img, Image *zbuf, float gamma)
{
	int i;
	unsigned char *zptr, *iptr;
	float f, g;


	gamma = 1.0 - gamma;

	zptr = zbuf->data;
	iptr = img->data;
	i = img->size;
	while (i--)
	{
		g = 1.0 - gamma * (float) (*zptr++) / 255.0;
		f = *iptr / 255.0 ;
		f = pow(f,g);
		*iptr++ = 255.0 * f;
		f = *iptr / 255.0 ;
		f = pow(f,g);
		*iptr++ = 255.0 * f;
		f = *iptr / 255.0 ;
		f = pow(f,g);
		*iptr++ = 255.0 * f;
	}
}

/* lights warp function (like gamma correction only after value s)
 * -1.0 < c < 3.0 : warp parameter 
 *  0.0 < x < 1.0 : intensity value
 *  0.0 < s < 1.0 : warp start
 */

float hiwarp(float c, float x, float s)
{
	float a, b, e;
	float y, z;
	
	if ((x <= s) || (s == 1.0))
		return x;

	e = 1.0 / (1.0 - s);

	x -= s;
	a = c - 2.0;
	b = - (1.0 + a + c);
	y = e * x - 1.0;
	z = y*y;
	y = 1 + a*z; z *= z; y += b*z; z *= z; y+= c*z;		/* y = 1 + a * z + b*z^2 + c*z^3 */
	e *= -0.5; z = e*x*x;
	e = -0.25 / e;
	z += s + x + e*y;
	return z;
}
	

/* no longer needed 
void zlight(Image *img, Image *zbuf, float light)
{
	int i;
	unsigned char *zptr, *iptr;
	float f, g;

	zptr = zbuf->data;
	iptr = img->data;
	i = img->size;
	while (i--)
	{
		g = 1.0 - (float) (*zptr++) / 255.0;
		f = *iptr / 255.0 ;
		f = hiwarp(light, f, g);
		*iptr++ = 255.0 * f;
		f = *iptr / 255.0 ;
		f = hiwarp(light, f, g);
		*iptr++ = 255.0 * f;
		f = *iptr / 255.0 ;
		f = hiwarp(light, f, g);
		*iptr++ = 255.0 * f;
	}

}

*/



Image *zblur2(Image *src, Image *zbuf, float radius, float sigma)
{
	Image *dest, *fdest;
	Maskarray *mar;
	float *mval;
	float rval, gval, bval;
	float fac;		
	Image *norm;
	unsigned char *sptr, *dptr, *zptr;	/* source, dest */
	float *nptr, *fptr; 			/* norm image, float-rgb-image */
	int zval;
	int size;
	int row;
	int mrow;
	int x, y;
	int sx, sy, ex, ey;
	int mx, my;
	Mask *m;

	/* test vars */
	float f, g;


	if(src->type != I_RGB)
		return 0;

	dest = alloc_img(src->x, src->y, I_RGB);
	norm = alloc_img(src->x, src->y, I_FLOAT);
	fdest = alloc_img(src->x, src->y, I_FLOAT3);

	
	if (!norm || !fdest || !dest)
	{
		perror("zblur2");
		return 0;
	}
	row = src->x * src->el_size;
	sptr = src->data;
	dptr = dest->data;

	mar = init_masks(NMASKS, radius);

	for (y = 0; y < src->y; y++)
	{
		
		for (x = 0; x < src->x; x++)
		{
			zptr = zbuf->data + (y * src->x + x);
			zval = *zptr;
			m = mar[zval >> NMASKS_SHIFT];
			size = m->size;



			fptr = ((float *) fdest->data) + (y * src->x + x) * 3;
			sptr = (unsigned char *) src->data + ((y * src->x + x) * 3);
			nptr = ((float *) norm->data) + (y * src->x + x);

			rval = (float) sptr[0];
			gval = (float) sptr[1];
			bval = (float) sptr[2];

			if (size == 0)
			{
				fptr[0] = rval;
				fptr[1] = gval;
				fptr[2] = bval;
				*nptr += 1.0;
				nptr++;
				continue;
			} 

			/* dealing with the edges */
			ex = src->x - x;
			ey = src->y - y;

			sx = (x < size) ? x : size;
			sy = (y < size) ? y : size;
			ex = (ex <= size) ? ex - 1 : size;
			ey = (ey <= size) ? ey - 1 : size;

			mrow = (size << 1) + 1;

			mval = m->val + (size - sy) * mrow + size;


			fptr -= sy * src->x * 3;
			nptr -= sy * src->x;
			/* test gamma */
					g = sigma * (float) zval / 255.0;
			if (g > 0.001)
			{
#define sig(x) (sigmo(0.3, x, g))
					f = sig(0);
					fac = 255.0 /(sig(1) - f);

					rval = (sig(rval/255.0) - f) * fac;

					gval = (sig(gval/255.0) - f) * fac;

					bval = (sig(bval/255.0) - f) * fac;
/*
#undef sig(x)
*/
			}

			/* end test */

			for (my = -sy; my <= ey; my++)
			{
				for (mx = -sx; mx <= ex; mx++)
				{
					fac = mval[mx];
					nptr[mx] += fac;
					fptr[3 * mx] += fac * rval;
					fptr[3 * mx + 1] += fac * gval;
					fptr[3 * mx + 2] += fac * bval;
				}
				fptr += row;
				nptr += src->x;
				mval += mrow;
			}
		}
		progress(y, src->y);
		fflush(0);
	}
	printf("\n");

	/* now renorm and convert the image back */
	x = dest->size;
	dptr = dest->data;
	nptr = (float *) norm->data;
	fptr = (float *) fdest->data;
	while (x--)
	{
		fac = *nptr++;
		if (fac != 0.0)
		{
			*dptr++ = *fptr++ / fac;
			*dptr++ = *fptr++ / fac;
			*dptr++ = *fptr++ / fac;
		}
		else
		{	
			*dptr++ = 0;
			*dptr++ = 0;
			*dptr++ = 0;
			fptr +=3;
		}
	}
	free_img(norm); free_img(fdest);
	for (x= 0; x < NMASKS; x++)
	{
		free_mask(mar[x]);
	}
	free(mar);
	return dest;
}

Image *zblur(Image *src, Image *zbuf, float radius)
{
	Image *dest;
	Maskarray *mar;
	float *mval;				/* mask value pointer */
	float rval, gval, bval;			
	float norm, fac;			
	int tmp;
	unsigned char *sptr, *dptr, *zptr;
	int zval;
	int size;
	int row;
	int mrow;
	int x, y;
	int i;
	int sx, sy, ex, ey;
	int mx, my;
	Mask *m;

	if(src->type != I_RGB)
		return 0;

	dest = alloc_img(src->x, src->y, I_RGB);
	row = src->x * src->el_size;

	mar = init_masks(NMASKS, radius);

	for (y = 0; y < src->y  ; y++)
	{
		for (x = 0; x < src->x; x++)
		{
			dptr = (unsigned char *) dest->data + ((y * src->x + x) * src->el_size);
			zptr = ((unsigned char *) zbuf->data) + (y * src->x + x);
			zval = *zptr;
			sptr = (unsigned char *) src->data + ((y *src->x + x )* src->el_size);

			m = mar[zval >> NMASKS_SHIFT];

			size = m->size;

			if (zval == 0)
			{
				dptr[0] = sptr[0];
				dptr[1] = sptr[1];
				dptr[2] = sptr[2];
				continue;
			}

			ex = src->x - x;
			ey = src->y - y;

			sx = (x < size) ? x : size;
			sy = (y < size) ? y : size;
			ex = (ex <= size) ? ex - 1: size;
			ey = (ey <= size) ? ey - 1: size;

			sptr -= sy *src->x * src->el_size;
			zptr -= sy * src->x;
			mrow = (size << 1) + 1;
			mval = m->val + (size - sy) * mrow + size;

			norm = rval = gval = bval = 0.0;
	
			for (my = -sy; my <= ey; my++)
			{
				for (mx = -sx; mx <= ex; mx++)
				{
					tmp = 3 * mx;
					fac = mval[mx] * (float) zptr[mx] /255.0 ;
					norm += fac;
					rval += fac * sptr[tmp];
					gval += fac * sptr[tmp + 1];
					bval += fac * sptr[tmp + 2];
				}
				mval += mrow;
				sptr += row;
				zptr += src->x;
			}
			dptr[0] = rval / norm;
			dptr[1] = gval / norm;
			dptr[2] = bval / norm;
		}
		progress(y, src->y);
		fflush(0);
	}
	printf("\n");
	for (i= 0; i < NMASKS; i++)
	{
		free_mask(mar[i]);
	}
	free(mar);
	return dest;
}

/* returns the z value of a center area (2 * AF_SIZE wide) of the image */

#define AF_SIZE 2
int autofocus(ImBuf *img)
{
	int offset;
	int i, j;
	int *zptr;
	double sum;
	int row;

	offset = (img->x  * (img->y >> 1)) + (img->x >>1) ;
	offset -= AF_SIZE * img->x;

	zptr = img->zbuf + offset;
	
	row = img->x;
	sum = 0.0;
	for (i = -AF_SIZE; i <= AF_SIZE; i++)
	{
		for (j = -AF_SIZE; j <= AF_SIZE; j++)
		{
			sum += zptr[j];
		}
		zptr += row;
	}
	sum /= (2 * AF_SIZE + 1) * (2 * AF_SIZE + 1);
	return sum;
}

/* this splits the z-buffer into 2 gray-images (background, foreground)
 * which are used for the weighted blur */

void zsplit(ImBuf *img, Image *fg, Image *bg, int zfocus, int zmax, int zmin)
{
	int *zptr;
	IMG_BYTE *p, *q;
	int i;
	float fdist;
	float fgnorm, bgnorm;

	zptr = img->zbuf;

	i = img->x * img->y;
	p = fg->data;
	q = bg->data;
	bgnorm = 255.0 / ((float) zmax - (float) zmin);
	fgnorm = 255.0 / ((float) zfocus - (float) zmin);
	while(i--)
	{
		
		fdist = (float) (*zptr++);
		if (fdist < zmin)
		{
			fdist = zmin;
		}
			
		fdist -= zfocus;

		if (fdist < 0)
		{
			*p = (IMG_BYTE) (-fdist * fgnorm);
			*q = 0;
		}
		else
		{
			*q = (IMG_BYTE) (fdist * bgnorm);
			*p = 0;
		}
		p++, q++;
	}
}



/* ---------------------------------------------------------------------
 *
 * Conversion ImBuf <-> Image
 *
 * */

void imbuf2img(ImBuf *src, Image *dest)
{
	char *from;
	IMG_BYTE *to;
	int i;

	if ((src->x != dest->x) || (src->y != dest->y))
	{
		warn("wrong resolution");
		return;
	}

	from = (char *) src->rect;
	to = (IMG_BYTE *) dest->data;
	i = src->x * src->y;
	while(i--)
	{
		*to++ = from[0];
		*to++ = from[1];
		*to++ = from[2];
		from += 4;
	}
}


void img2imbuf(Image *src, ImBuf *dest)
{
	IMG_BYTE *from;
	char *to;
	int i;

	if ((src->x != dest->x) || (src->y != dest->y))
	{
		warn("wrong resolution");
		return;
	}

	from = (IMG_BYTE *) src->data;
	to = (char *) dest->rect;
	i = src->x * src->y;
	while(i--)
	{
		*to++ = from[0];
		*to++ = from[1];
		*to++ = from[2];
		*to++ = 255;
		from += 3;
	}
}


/* ---------------------------------------------------------------------
 * here it is done
 * 
 */

void buferror(ImBuf *out)
{
	int i;
	char *dest;

	dest = (char *) out->rect;
	i = out->x * out->y;
	while (i--)
	{
		dest[0] = 255;
		dest[1] = 0;
		dest[2] = 0;
		dest[3] = 0;
		dest += 4;
	}
}


void plugin_seq_doit(Cast *cast, float facf0, float facf1, int x, int y, ImBuf *in1, ImBuf *in2, ImBuf *out, ImBuf *use)
{
	Image *orig, *work, *zfront, *zback;
	int zfocus;
	int zmin;


	if (!in1->zbuf)
	{
		buferror(out);
		lout("Error: input has no z-buffer");
		return;
	}

	zfront = alloc_img(x, y, I_GRAY);
	zback = alloc_img(x, y, I_GRAY);
	orig = alloc_img(x, y, I_RGB);

	imbuf2img(in1, orig);

	zmin = (cast->zmin * (float) UINT_MAX + (float) INT_MIN) ;

	if (cast->autofocus != 0.0)
	{
		zfocus = autofocus(in1);
		cast->zfocus = ((float ) zfocus - INT_MIN) / (float) UINT_MAX;
		printf("autofocus: %f\n", cast->zfocus);
	}
	else if (cast->ipofocus != 0.0)
	{
		zfocus = (facf0 * (float) UINT_MAX + (float) INT_MIN);
	}
	else 
	{
		zfocus = (cast->zfocus * (float) UINT_MAX + (float) INT_MIN);
	}

	if (zmin >= zfocus)
	{
		buferror(out);
		lout("Error: Zmin too big!");
		free_img(orig); free_img(zfront); free_img(zback);
		return;
	}

	/* split up z buffer into 2 gray images	*/
	zsplit(in1, zfront, zback, zfocus, INT_MAX, zmin);	
	lout("blurring background zbuffer");
	gauss_blur(zback, 1.0);
	lout("blurring foreground zbuffer");
	gauss_blur(zfront, cast->rad);

	if (cast->zfront != 0.0)
	{
		g2rgb(zfront->data, orig->data, zfront->size);		/* convert to rgb */
		img2imbuf(orig, out);					/* copy to out-ImBuf */
		free_img(orig); free_img(zfront); free_img(zback);
		return;
	}

	if (cast->zback != 0.0)
	{
		g2rgb(zback->data, orig->data, zback->size);
		img2imbuf(orig, out);
		free_img(orig); free_img(zfront); free_img(zback);
		return;
	}

	lout("blurring background");
	work = zblur(orig, zback, cast->rad);
	free_img(orig);
	lout("blurring foreground");
	orig = zblur2(work, zfront, cast->rad, cast->sigma);
	if (cast->gamma != 1.0)
	{
		zgamma(orig, zback, cast->gamma);
/*		zlight(orig, zfront, -1.0); */
	}

	img2imbuf(orig, out);
	free_img(orig);
	free_img(work); free_img(zfront); free_img(zback);
	if (g_imagemem > 0)
	{
		warn("some memory not deallocated");
	}
	lout("done");
}


