#ifndef _3DS_H
#define _3DS_H

//>------ Primary Chunk, at the beginning of each file
#define PRIMARY		0x4D4D

//>------ Main Chunks
#define OBJECTINFO	0x3D3D                // This gives the version of the mesh and is found right before the material and object information
#define VERSION		0x0002                // This gives the version of the .3ds file
#define EDITKEYFRAME	0xB000                // This is the header for all of the key frame info

//>------ sub defines of OBJECTINFO
#define MATERIAL		0xAFFF                // This stored the texture info
#define OBJECT		0x4000                // This stores the faces, vertices, etc...

//>------ sub defines of MATERIAL
#define MATNAME		0xA000                // This holds the material name
#define MATDIFFUSE	0xA020                // This holds the color of the object/material
#define MATMAP		0xA200                // This is a header for a new material
#define MATMAPFILE	0xA300                // This holds the file name of the texture

#define OBJECT_MESH	0x4100                // This lets us know that we are reading a new object

//>------ sub defines of OBJECT_MESH
#define OBJECT_VERTICES	0x4110          // The objects vertices
#define OBJECT_FACES		0x4120          // The objects faces
#define OBJECT_MATERIAL	0x4130          // This is found if the object has a material, either texture map or color
#define OBJECT_UV			0x4140          // The UV texture coordinates

#include <vector>
#include <math.h>
#include "textureloader.h"

using namespace std;
typedef unsigned char BYTE;

class Load3ds
{
public:
	// This is our 3D point structure.  This will be used to store the vertices of our model. 
	struct CVector3 
	{
		float x, y, z;
		CVector3() {}
		CVector3 ( float xx, float yy, float zz )
		{
			x=xx; y=yy; z=zz;
		}
	};
	
	// This is our 2D point structure to store the UV coordinates.
	struct CVector2 
	{
		float x, y;
	};
	
	// This holds the 3ds chunk info
	struct tChunk
	{
		unsigned short int ID;                  // The chunk's ID       
		unsigned int length;                    // The length of the chunk
		unsigned int bytesRead;             // The amount of bytes read within that chunk
	};
	
	// This is our face structure.  This is is used for indexing into the vertex 
	// and texture coordinate arrays.  From this information we know which vertices
	// from our vertex array go to which face, along with the correct texture coordinates.
	struct tFace
	{
		int vertIndex[3];           // indicies for the verts that make up this triangle
		int coordIndex[3];          // indicies for the tex coords to texture this face
	};
	
	// This holds the information for a material.  It may be a texture map of a color.
	// Some of these are not used, but I left them because you will want to eventually
	// read in the UV tile ratio and the UV tile offset for some models.
	struct tMaterialInfo
	{
		char  strName[255];         // The texture name
		char  strFile[255];         // The texture file name (If this is set it's a texture map)
		BYTE  color[3];             // The color of the object (R, G, B)
		float uTile;                // u tiling of texture  (Currently not used)
		float vTile;                // v tiling of texture  (Currently not used)
		float uOffset;              // u offset of texture  (Currently not used)
		float vOffset;              // v offset of texture  (Currently not used)
	} ;
	
	// This holds all the information for our model/scene. 
	struct t3DObject
	{
		int  numOfVerts;		// The number of verts in the model 
		int  numOfFaces;		// The number of faces in the model
		int  numTexVertex;		// The number of texture coordinates
		int  materialID;			// The texture ID to use, which is the index into our texture array
		bool bHasTexture;		// This is TRUE if there is a texture map for this object
		char  strFile[255];         // The texture file name (If this is set it's a texture map)
		char strName[255];		// The name of the object
		CVector3  *pVerts;		// The object's vertices
		CVector3  *pNormals;	// The object's normals
		CVector2  *pTexVerts;	// The texture's UV coordinates
		tFace *pFaces;			// The faces information of the object
	};
	
	// This holds our model information. 
	struct t3DModel 
	{    // This reads the objects vertices
		int numOfObjects;                   // The number of objects in the model    
		int numOfMaterials;                 // The number of materials for the model
		float scale;				// The scale factor for the model
		float center_x , center_y , center_z;
		vector<tMaterialInfo> pMaterials;   // The list of material information (Textures and colors)
		vector<t3DObject> pObject;          // The object list for our model
	};
	
	Load3ds();
	~Load3ds();
	
	// This is the function that you call to load the 3DS
	int Import3DS ( TextureLoader* TextureLoaderClass , t3DModel *pModel, string strFileName );
	
	//This computes the correct scale and center coordinates
	void ScaleAndCenter ( t3DModel *pModel );
	
	// This reads in a string and saves it in the char array passed in
	int GetString(char *);
	
	// This reads the next chunk
	void ReadChunk(tChunk *);
	
	// This reads the next large chunk
	void ProcessNextChunk(t3DModel *pModel, tChunk *);
	
	// This reads the object chunks
	void ProcessNextObjectChunk(t3DModel *pModel, t3DObject *pObject, tChunk *);
	
	// This reads the material chunks
	void ProcessNextMaterialChunk(t3DModel *pModel, tChunk *);
	
	// This reads the RGB value for the object's color
	void ReadColorChunk(tMaterialInfo *pMaterial, tChunk *pChunk);
	
	// This reads the objects vertices
	void ReadVertices(t3DObject *pObject, tChunk *);
	
	// This reads the objects face information
	void ReadVertexIndices(t3DObject *pObject, tChunk *);
	
	// This reads the texture coodinates of the object
	void ReadUVCoordinates(t3DObject *pObject, tChunk *);
	
	// This reads in the material name assigned to the object and sets the materialID
	void ReadObjectMaterial(t3DModel *pModel, t3DObject *pObject, tChunk *pPreviousChunk);
	
	// This computes the vertex normals for the object (used for lighting)
	void ComputeNormals(t3DModel *pModel);
	
	CVector3 Normalize(CVector3 vNormal);
	friend CVector3 operator- ( CVector3 v1 , CVector3 v2 );
	friend CVector3 operator^ ( CVector3 v1 , CVector3 v2 );
		
	// This frees memory and closes the file
	void Release ( t3DModel *pModel );
	unsigned int ID;
	FILE *m_FilePointer;
	// These are used through the loading process to hold the chunk information
	tChunk *m_CurrentChunk;
  	tChunk *m_TempChunk;
};

#endif
