/***************************************************************************
 *   Copyright (C) 2004 by Colossus (Giuseppe Torelli)                     *
 *   gt67@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "glwidget.h"
#include <iostream>

GLWidget::GLWidget( TextureLoader *m_textureLoaderClass , GlWindow *window, QWidget *parent, const char* name )  : QGLWidget( parent, name )
{
	TextureClassPointer = m_textureLoaderClass;
	GlWindowClass = window;
	setMouseTracking ( TRUE );
	setFocusPolicy (QWidget::StrongFocus);
	setAcceptDrops ( TRUE );
	ChoosedModel = 0;
	LeftMousePressed = RightMousePressed = false;
	Default_AngleX = AngleX = -90.0f;
	Default_AngleZ = AngleZ = Default_AngleY = Default_Tx = Default_Ty = Tx = Ty = 0.0;
	Default_Depth = Depth = -100.0;
	IpX = IpY = IpZ = 0.0;
}

GLWidget::~GLWidget()
{
}

void GLWidget::paintGL()
{
	glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
	glMatrixMode( GL_MODELVIEW );
	glLoadIdentity();
	SetCamera();
	for ( std::map< GLuint , string >::iterator i= GlWindowClass->DisplayList.begin(); i != GlWindowClass->DisplayList.end(); i++)
	{
		unsigned int index = i->first;
		glPushMatrix();
			if ( mode == GL_SELECT ) glPushName ( index );
			glTranslatef ( Model [ index-1 ].x , Model [ index-1 ].y  , Model [ index-1 ].z );
			glRotatef ( Model [ index-1 ].xrot , 1.0f , 0.0f , 0.0f );
			glRotatef ( Model [ index-1 ].yrot , 0.0f , 1.0f , 0.0f );
			glRotatef ( Model [ index-1 ].zrot , 0.0f , 0.0f , 1.0f );
			glScalef ( Model [ index-1 ].sx , Model [ index-1 ].sy  , Model [ index-1 ].sz );
			glCallList ( index );
			if ( mode == GL_SELECT ) glPopName ();
		glPopMatrix();
	}
}

void GLWidget::initializeGL()
{
	glClearColor( 0.4 , 0.4 , 0.4 , 1.0 );
	glMatrixMode ( GL_MODELVIEW );
	glLoadIdentity();
	glEnable (GL_RESCALE_NORMAL);
	glDepthFunc (GL_LEQUAL);
	glEnable (GL_COLOR_MATERIAL);
	glEnable (GL_DEPTH_TEST);
	
	GLfloat LightPosition[]= { 0.0f, 0.0f, 1.0 , 0.0f };
	GLfloat LightAmbient[]= { 0.5f, 0.5f, 0.5f, 1.0f };
	GLfloat LightDiffuse[]= { 1.0f, 1.0f, 1.0f, 1.0f };
	
	glLightfv(GL_LIGHT0, GL_AMBIENT, LightAmbient);
  	glLightfv(GL_LIGHT0, GL_DIFFUSE, LightDiffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, LightPosition);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHTING);
	mode = GL_RENDER;
}

void GLWidget::resizeGL( int w, int h )
{
	glViewport( 0, 0, (GLint)w, (GLint)h );
	glMatrixMode( GL_PROJECTION );
	glLoadIdentity();
	gluPerspective( 45.0f,(GLfloat)w/(GLfloat)h, 1.0 , 8000.0 );
}

void GLWidget::SetCamera()
{
	glTranslatef ( Tx, Ty, Depth );
	glRotatef ( AngleX, 1, 0, 0 );
	glRotatef ( AngleZ, 0, 0, 1 );
}

void GLWidget::mouseMoveEvent ( QMouseEvent *event )
{
	if (LeftMousePressed)
	{
		AngleX += (event->pos().y() - mouseY) / 2;
		AngleY += (event->pos().x() - mouseX) / 2;
		AngleZ += (event->pos().x() - mouseX) / 2;
		if (AngleX > 360) AngleX -= 360;
		if (AngleX < 0) AngleX += 360;
		//if (AngleY > 360) AngleY -= 360;
		//if (AngleY < 0) AngleY += 360;
		if (AngleZ > 360) AngleZ -= 360;
		if (AngleZ < 0) AngleZ += 360;
		updateGL();
	}
	else if (RightMousePressed)
	{
		Tx += (event->pos().x() - mouseX) / 2;
      		Ty += (event->pos().y() - mouseY) / 2;
		updateGL();
	}
	mouseX = event->pos().x();
	mouseY = event->pos().y();
}

GLuint GLWidget::GetObjectHitByMouse()
{		
	GLuint buffer[128];
	GLuint * ptr, names, selected;
	GLint viewport[4];
	int numHits;
	unsigned j;
	float min, z1, z2;
	bool closest;
	// create the selection buffer with a slot for each visible object
	glSelectBuffer (128, buffer);
	glRenderMode (GL_SELECT);
	// save the old projection matrix and load in the picking matrix
	glMatrixMode (GL_PROJECTION);
	glPushMatrix ();
		glLoadIdentity ();
		glGetIntegerv (GL_VIEWPORT, viewport);
		gluPickMatrix (mouseX, viewport[3] - mouseY - 1, 3.0, 3.0, viewport);
		gluPerspective ( 45.0f, (viewport[2]+0.0 ) / viewport[3], 1.0 , 8000.0 );
		// save the modelview matrix
		glMatrixMode (GL_MODELVIEW);
		glPushMatrix ();
			// draw each object with a unique name 
			glInitNames ();
			glPushName (0);
			mode = GL_SELECT;
			updateGL();
		glPopMatrix ();
		glMatrixMode (GL_PROJECTION);
	glPopMatrix ();

	// iterate through the hits and find the closest hit location
	numHits = glRenderMode (GL_RENDER);
	ptr = buffer;
	min = 3.40282347e+38F;
	selected = 0;
	for (int i = 0; i < numHits; i ++)
	{
		names = * ptr; ptr ++;
		z1 = (float)*ptr/0x7fffffff;  ptr ++;
		z2 = (float)*ptr/0x7fffffff;  ptr ++;
		closest = (z1 < min);
		if (closest) min = z1;
		for (j = 0; j < names; j ++)
		{
			if (closest) selected = * ptr;
			ptr ++;
		}
	}
	// if we didn't hit anything, return 0, otherwise return the object id
	if (selected == 0 ) return (GLuint) 0;
	else return selected;
}

void GLWidget::mouseReleaseEvent( QMouseEvent *mr )
{
	if ( mr->button() == LeftButton)
	{
		LeftMousePressed = false;
		ChoosedModel = GetObjectHitByMouse();
		if ( ChoosedModel == 0 ) return;
		if ( GlWindowClass->del == true )
		{
			DeleteModel ( ChoosedModel );
			return;
		}
		if ( GlWindowClass->Insert_Flag )
		{
			GlWindowClass->Insert_Flag = false;
			return;
		}
		UpdateItemEditorDialog ( ChoosedModel - 1);
	}
	else if ( mr->button() == RightButton ) RightMousePressed = false;
}

void GLWidget::mousePressEvent ( QMouseEvent *mp )
{
	if ( mp->button() == LeftButton ) LeftMousePressed = true;
		else if ( mp->button() == RightButton ) RightMousePressed = true;
	if ( GlWindowClass->Insert_Flag == true && LeftMousePressed == true )
	{
		ChoosedModel = GetObjectHitByMouse();
		if ( ChoosedModel == 0 )
		{
			GlWindowClass->statusBar()->message ("You didn't click any model !");
			GlWindowClass->Insert_Flag = false;
			return;
		}
		GLint viewport[4];
		GLdouble ModelViewMatrix[16], ProjMatrix[16];
		GLint realY;  //  OpenGL y coordinate position  
		glGetIntegerv ( GL_VIEWPORT, viewport );
		glGetDoublev ( GL_MODELVIEW_MATRIX, ModelViewMatrix );
		glGetDoublev ( GL_PROJECTION_MATRIX, ProjMatrix );
		realY = viewport[3] - (GLint) mouseY - 1;
		float depth;
		glReadPixels ( mouseX , realY, 1 , 1 , GL_DEPTH_COMPONENT, GL_FLOAT, &depth );
		gluUnProject ( (GLdouble) mouseX, (GLdouble) realY, depth , ModelViewMatrix, ProjMatrix, viewport, &IpX, &IpY, &IpZ); 
		GlWindowClass->statusBar()->message ( QString ("%1 %2, %3, %4").arg ( "Insert point of new models at x,y,z:" ).arg (IpX).arg (IpY).arg (IpZ) );
	}
}

void GLWidget::keyPressEvent( QKeyEvent *k )
{
	if ( k->key() == Qt::Key_Escape && GlWindowClass->ItemDialog->isShown() ) GlWindowClass->ItemDialog->close();
	if ( k->key() == Qt::Key_W ) Depth += 20.0f;
	if ( k->key() == Qt::Key_S ) Depth -= 20.0f;
	if ( k->key() == Qt::Key_Space ) 
	{
		AngleX = Default_AngleX;
		AngleY = Default_AngleY;
		AngleZ = Default_AngleZ;
		Tx = Default_Tx;
		Ty = Default_Ty;
		Depth = Default_Depth;
	}
	updateGL();
}

void GLWidget::wheelEvent ( QWheelEvent *we )
{
	if ( we->delta() > 0) Depth += 5.0f;
		else Depth -= 5.0f;
	updateGL();
}

void GLWidget::DeleteModel ( unsigned int ChoosedModel )
{
	glDeleteLists ( ChoosedModel , 1 );
	for ( std::multimap< GLuint , string >::iterator i = GlWindowClass->TextureNamesMap.begin(); i != GlWindowClass->TextureNamesMap.end(); i++ )
	{
		if ( ChoosedModel == i->first )
		{
			GlWindowClass->SameTextureListCnt [ i->second ] --;
			if ( GlWindowClass->SameTextureListCnt [ i->second ] == 0 )
			{
				glDeleteTextures ( 1 , &TextureClassPointer->TextureMap [i->second  ]);
				TextureClassPointer->TextureMap.erase ( i->second );
			}
		}
	}
	GlWindowClass->TextureNamesMap.erase ( ChoosedModel );
	if (GlWindowClass->ItemDialog->Add_Flag)
	{
		GlWindowClass->statusBar()->message ( QString("%1 %2 %3 %4 %5").arg ("Model:").arg ( GlWindowClass->DisplayList[ ChoosedModel ].c_str() ).arg ("with id:").arg ( ChoosedModel ).arg ("successfully deleted.") );
		GlWindowClass->ItemDialog->UndoClass->AddAction ( deleteModel , GlWindowClass->DisplayList [ ChoosedModel ] , ChoosedModel );
	}
	GlWindowClass->DisplayList.erase ( ChoosedModel );
	//Model [ ChoosedModel - 1 ] = F7dVector ( 0 , IpX , IpY , IpZ , 0.0 , 0.0 , 0.0, 1.0 , 1.0 , 1.0 );
	Models_Deleted.push_back ( ChoosedModel - 1 );
	GlWindowClass->IsModified = true;
	if ( GlWindowClass->DisplayList.size() == 0 )
	{	
		GlWindowClass->IsModified = false;
		GlWindowClass->deleteModelAction->setEnabled ( false );
		GlWindowClass->saveSceneAction->setEnabled ( false );
		GlWindowClass->setInsertPointAction->setEnabled ( false );
	}
	GlWindowClass->del = false;
	updateGL();
	return;
}

void GLWidget::UpdateItemEditorDialog ( unsigned int choosedmodel )
{
	GlWindowClass->ItemDialog->texture_listbox->clear();
	if ( GlWindowClass->SceneFileName != "") GlWindowClass->ItemDialog->filename_edit->setText ( GlWindowClass->SceneFileName );
	GlWindowClass->ItemDialog->ChoosedModel = choosedmodel;
	GlWindowClass->ItemDialog->model_label->setText ( GlWindowClass->DisplayList[ choosedmodel + 1].c_str() );
	for ( std::multimap < GLuint , string >::iterator t= GlWindowClass->TextureNamesMap.begin(); t != GlWindowClass->TextureNamesMap.end(); t++)
	{
		if ( t->first == choosedmodel && GlWindowClass->ItemDialog->texture_listbox->findItem ( t->second.c_str() , Qt::ExactMatch ) == 0 ) GlWindowClass->ItemDialog->texture_listbox->insertItem ( t->second.c_str() , -1 );
	}
	GlWindowClass->ItemDialog->id_label->setText ( QString("%1").arg (choosedmodel ) );
	GlWindowClass->ItemDialog->Xpos_label->setText ( QString("%1").arg ( Model[choosedmodel ].x ) );
	GlWindowClass->ItemDialog->Ypos_label->setText ( QString("%1").arg ( Model[choosedmodel ].y ) );
	GlWindowClass->ItemDialog->Zpos_label->setText ( QString("%1").arg ( Model[choosedmodel ].z ) );
			
	GlWindowClass->ItemDialog->Xrot_label->setText ( QString("%1").arg ( Model[choosedmodel ].xrot ) );
	GlWindowClass->ItemDialog->Yrot->setText ( QString("%1").arg ( Model[choosedmodel ].yrot ) );
	GlWindowClass->ItemDialog->Zrot->setText ( QString("%1").arg ( Model[choosedmodel ].zrot ) );
	if ( Model [ choosedmodel ].checked )
	{
		GlWindowClass->ItemDialog->Scale->setChecked ( TRUE );
		GlWindowClass->ItemDialog->scaleX->setValue ( (int) (Model [ choosedmodel ].sx*112.0f) );
		GlWindowClass->ItemDialog->scaleY->setValue ( (int) (Model [ choosedmodel ].sy*112.0f) );
		GlWindowClass->ItemDialog->scaleZ->setValue ( (int) (Model [ choosedmodel ].sz*112.0f) );
	}
		else GlWindowClass->ItemDialog->Scale->setChecked ( FALSE );
	//Update the bottom model_string
	GlWindowClass->ItemDialog->UpdateModelString ();
	//GlWindowClass->ItemDialog->move ( QApplication::desktop()->width() - 470, (QApplication::desktop()->height() - 250) ) ;
	GlWindowClass->ItemDialog->show();
}

