/***************************************************************************
 *   Copyright (C) 2004 by Colossus (Giuseppe Torelli)                     *
 *   gt67@users.sourceforge.net                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include "glwindow.h"
#include "glwidget.h"
#include "undo.h"

#include "xpm/sceneopen.xpm"
#include "xpm/scenenew.xpm"
#include "xpm/quit.xpm"
#include "xpm/scenesave.xpm"
#include "xpm/modelimport.xpm"
#include "xpm/deletemodel.xpm"
#include "xpm/setcamera.xpm"
#include "xpm/setinspoint.xpm"
#include "xpm/about.xpm"
#include "xpm/undo.xpm"
#include "xpm/redo.xpm"

GlWindow::GlWindow() : QMainWindow ( 0, "Cpsed", WDestructiveClose ) , m_textureLoaderClass (new TextureLoader() )
{
	IsModified = false;
	del = false;
	Insert_Flag = false;
	Insert_in_Dummy_Flag = true;
	response = 0;
	SceneFileName = "";
	statusBar()->setSizeGripEnabled (false );
	statusBar()->setMaximumHeight (15);
	statusBar()->message ( "Camera: Zoom 20x W/S, Zoom 5x mouse wheel, left button pressed rotate, right button pressed move, SPACE to reset view" );
		
	//Create the GLWidget constructor
	c = new GLWidget ( m_textureLoaderClass , this , this, "glwidget" );
	setGeometry ( (QApplication::desktop()->width() - 768)/2, (QApplication::desktop()->height() - 400)/2 , 768 , 400);
	setCentralWidget( c );
	//Create the ItemDialogClass constructor
	ItemDialog = new ItemDialogClass ( c , this );
	
	//Create the app buttons
	QToolBar * SceneTools = new QToolBar ( this, "scene tools" );
	SceneTools->setLabel( "Scene Tools" );
	
	sceneNewAction = new QToolButton( QPixmap (scenenew),"Create a new scene", QString::null , this, SLOT( New_Scene() ) , SceneTools , "new" );
		
	openSceneAction = new QToolButton( QPixmap (sceneopen),"Open a scene file",  QString::null , this, SLOT( Open_Scene() ) , SceneTools , "open" );
			
	importModelAction = new QToolButton( QPixmap (modelimport),"Import a 3d model", QString::null , this, SLOT( Import_Object() ) , SceneTools , "import" );
	
	saveSceneAction = new QToolButton( QPixmap (scenesave),"Save a scene", QString::null , this, SLOT( Save_Scene() ) , SceneTools, "save" );
	saveSceneAction->setEnabled( false ) ;
				
	undoAction = new QToolButton( QPixmap (undo), "Undo", QString::null , this, SLOT( Undo_Action() ) , SceneTools , "undo" );
	undoAction->setEnabled ( false );
	
	redoAction = new QToolButton( QPixmap (redo), "Redo", QString::null , this, SLOT( Redo_Action() ) , SceneTools , "redo" );
	redoAction->setEnabled ( false );

	deleteModelAction = new QToolButton( QPixmap (deletemodel),"Delete a model from the scene", QString::null , this, SLOT( Delete_Object() ) , SceneTools , "delete" );
	deleteModelAction->setEnabled (false);
			
	setInsertPointAction = new QToolButton( QPixmap (setinspoint), "Choose a new models insert point", QString::null , this, SLOT( InsertPoint() ) , SceneTools , "setinspoint" );
	setInsertPointAction->setEnabled ( false );
	
	setCameraAction = new QToolButton( QPixmap (setcamera),"Choose the default camera viewpoint", QString::null , this, SLOT( Set_Default_ViewPoint() ) ,  SceneTools , "setcamera" );
			
	aboutAction = new QToolButton( QPixmap (about), "About Cpsed" , QString::null , this, SLOT( About() ) , SceneTools , "About" );
	quitAction = new QToolButton( QPixmap (quit) , "Quit", QString::null , this, SLOT( close() ) , SceneTools , "quit" );
}

GlWindow::~GlWindow()
{
	delete ItemDialog;
	delete m_textureLoaderClass;
}

void GlWindow::Undo_Action()
{
	ItemDialog->UndoClass->UndoAction();
}

void GlWindow::Redo_Action()
{
	ItemDialog->UndoClass->RedoAction();
}

void GlWindow::New_Scene()
{
	if (DisplayList.size() == 0) return;
	if ( IsModified) response = QMessageBox::warning ( this , "Warning !" , "This operation is NOT reversible.\nDo you really want to delete ALL the models from the scene ?" , QMessageBox::Yes , QMessageBox::No );
	if ( response == QMessageBox::No) return;
	IsModified = false;
	DeleteEverything();
	this->setCaption ( "Cpsed 0.4: [unnamed]" );
	statusBar()->message ( "Camera: Zoom 20x W/S, Zoom 5x mouse wheel, left button pressed rotate, right button pressed move, SPACE to reset view" );
	c->updateGL();
}

void GlWindow::Import_Object()
{
	if (ItemDialog->isShown() ) ItemDialog->close();
	QString filename = QFileDialog::getOpenFileName( NULL , "All supported formats *.dff *.DFF *.3ds *.3DS" , this , "Import" , "Import a 3d model" );
	if (filename == NULL) return;
	Insert_in_Dummy_Flag = false;
	response = LoadModelFile ( filename );
	Insert_in_Dummy_Flag = true;
	if ( response > 0 )
	{
		Add_Model_Coordinates ( 0 , c->IpX , c->IpY , c->IpZ , 0.0 , 0.0 , 0.0, 1.0 , 1.0 , 1.0 );
		if ( ItemDialog->Add_Flag )
		{
			ItemDialog->UndoClass->AddAction ( unLoadModel , DisplayList [ Index_Display_List ] , Index_Display_List );
		}
	}
	else return;
}

void GlWindow::Add_Model_Coordinates ( bool _checked , GLfloat _x, GLfloat _y, GLfloat _z, GLfloat _xrot, GLfloat _yrot, GLfloat _zrot, GLfloat _sx, GLfloat _sy, GLfloat _sz )
{	
	if ( IsModified == false ) IsModified = true;
	if ( c->Models_Deleted.size() != 0 )
	{
		vector <int>::iterator it = c->Models_Deleted.begin();
		c->Model [ *it ] = CVector7 ( _checked , _x , _y , _z , _xrot , _yrot , _zrot, _sx , _sy , _sz );
		c->Models_Deleted.erase ( it );
	}
		else c->Model.push_back ( CVector7 ( _checked , _x , _y , _z , _xrot , _yrot , _zrot, _sx , _sy , _sz ) );
}

int GlWindow::LoadModelFile ( QString dummyfilename )
{
	int result;
	Copy_of_filename = dummyfilename.latin1(); //calling latin1() here fixes the compilation error on gcc 3.3.3 and 3.3.2
	if ( dummyfilename.right( 3 ) == "3ds" || dummyfilename.right( 3 ) == "3DS" )
	{
		result = my3ds.Import3DS ( m_textureLoaderClass , &g_3DModel , (char *) dummyfilename.latin1() );
		if ( result == 0 )
		{
			QMessageBox::critical ( this, "Cpsed" , QString ("%1%2").arg ("An error occurred while opening file ").arg (dummyfilename) );
			return 0;
		}
		if ( result == -1 )
		{
			QMessageBox::critical ( this, "Cpsed" , "This is not a valid 3ds file !");
			return -1;
		}
		Index_Display_List = Generate_3DS_Display_List ( g_3DModel );
		DisplayList[ Index_Display_List ] = Copy_of_filename;
		my3ds.Release ( &g_3DModel );
	}
	//Ok, the file is .dff renderware format, load it
	else
	{
		result = mydff.ImportDFF ( m_textureLoaderClass , (char *) dummyfilename.latin1() );
		if ( result == 0 )
		{
			QMessageBox::critical ( this, "Cpsed" , QString ("%1%2").arg ("An error occurred while opening file ").arg (dummyfilename) );
			return 0;
		}
		Index_Display_List = Generate_DFF_Display_List ();
		DisplayList[ Index_Display_List ] = Copy_of_filename;
		mydff.Release ();
	}
	if ( Insert_in_Dummy_Flag ) dummy_id_vector.push_back ( Index_Display_List );
	if ( deleteModelAction->isEnabled() == false ) deleteModelAction->setEnabled( true ) ;
	if ( saveSceneAction->isEnabled() == false ) saveSceneAction->setEnabled( true ) ;
	if ( setInsertPointAction->isEnabled() == false ) setInsertPointAction->setEnabled ( true ) ;
	return result;
}

void GlWindow::Open_Scene()
{
	QFile FileHandle;
	int dummy;
	char modelfilename[255];
	int checked;
	float x,y,z,xrot,yrot,zrot, sx = 0.0 , sy = 0.0 , sz = 0.0;
	if ( ItemDialog->Add_Flag )
	{
		SceneFileName = QFileDialog::getOpenFileName ( NULL , "scene files (*.cps)" , this , "Open" , "Open a scene descriptor file" );
		if ( SceneFileName == NULL ) return;
	}
	FileHandle.setName ( SceneFileName );
	if ( ! FileHandle.open ( IO_ReadOnly ) )
	{
		QMessageBox::critical ( this, "Cpsed" , QString ("%1%2").arg ("An error occurred while opening file: ").arg (SceneFileName) );
		return;
	}
	QTextStream ts (&FileHandle);
	sscanf ( ts.readLine().latin1() , "%f,%f,%f,%f,%f,%f", &c->Tx ,&c->Ty,&c->Depth,&c->AngleX,&c->AngleY,&c->AngleZ );
	while ( ! ts.atEnd() )
	{
		sscanf ( ts.readLine().latin1() , "%d,%[^,],%f,%f,%f,%f,%f,%f,%d,%f,%f,%f" , &dummy , modelfilename , &x , &y , &z , &xrot , &yrot , &zrot , &checked , &sx ,&sy , &sz );
		if ( ! checked ) sx = 1.0f , sy = 1.0f , sz = 1.0f;
		response = LoadModelFile ( modelfilename );
		if ( response > 0 ) Add_Model_Coordinates ( checked , x , y , z , xrot , yrot , zrot , sx , sy , sz );
	}
	FileHandle.close ();
	if ( deleteModelAction->isEnabled() == false ) deleteModelAction->setEnabled ( true );
	if ( saveSceneAction->isEnabled() == false ) saveSceneAction->setEnabled ( true );
	this->setCaption ( "Cpsed 0.4: " + SceneFileName );
	if ( ItemDialog->Add_Flag ) ItemDialog->UndoClass->AddAction ( deleteScene , dummy_id_vector , SceneFileName.latin1() );
	dummy_id_vector.clear();
	Insert_in_Dummy_Flag = true;
}

void GlWindow::Delete_Object()
{
	if (del == true)
	{
		del = false;
		statusBar()->message ("Aborted.");
		return;
	}
	if (ItemDialog->isShown() ) ItemDialog->close();
	statusBar()->message ("Click once on the model you want to delete. Click Delete again to abort.");
	del = true;
}

void GlWindow::InsertPoint()
{
	Insert_Flag = true;
	statusBar()->message ("Please click on a model, the next imported one will be positioned close to this.");
	return;
}

void GlWindow::Save_Scene()
{
	bool Different_Path = false;
	string Path_with_ModelName;
	if ( QMessageBox::question ( this , "Cpsed" , QString ("%1%2 %3").arg ("Do you want to choose another models path instead of\n\"").arg ( m_textureLoaderClass->RemoveMeshFileNameFromPath (Copy_of_filename).c_str() ).arg ("\" ?") , QMessageBox::Yes , QMessageBox::No) == QMessageBox::Yes )
	{
		Different_Path = true;
		Different_Path_Name = QFileDialog::getExistingDirectory ( NULL, this, "get existing directory" ,  "Choose a new models path", true );
		if ( Different_Path_Name == NULL)
		{
			statusBar()->message ("You didn't choose any new path, no scene has been saved !");
			return;
		}
		
	}
	QString filename = QFileDialog::getSaveFileName( NULL , "scene files (*.cps)" , this , "Save" , "Save the current scene" );
	if (filename == NULL) return;
	if ( filename.contains (".cps",1) == 0 ) filename = filename + ".cps";
	FileHandle.setName ( filename);
	if ( FileHandle.exists() && QMessageBox::warning ( this , "Warning ! " , QString ("%1%2%3").arg ("\"").arg (filename).arg ("\" already exists.\nDo you want to overwrite it ?") , QMessageBox::Yes , QMessageBox::No) == QMessageBox::No) return;
	if ( ! FileHandle.open ( IO_WriteOnly ) )
	{
		QMessageBox::critical ( this, "Cpsed" , "An error occurred while trying to save the file.\nThe scene hasn't been saved !");
		return;
	}
        QTextStream ts ( &FileHandle );
	//Let's save camera coordinates first
	ts << c->Tx << "," << c->Ty << "," << c->Depth << "," << c->AngleX << "," << c->AngleY << "," << c->AngleZ << "\n";
    	for ( std::map< unsigned int , string >::iterator i= DisplayList.begin(); i != DisplayList.end(); i++)
	{
		unsigned int index = i->first;
		if (Different_Path)
		{
			int pos = i->second.rfind ("/")+1;
			Path_with_ModelName = Different_Path_Name.latin1() + i->second.assign (i->second , pos , ( i->second.size() - pos) );
		}
		else Path_with_ModelName = i->second;
		ts << index << ","  << Path_with_ModelName.c_str() << "," << c->Model[ index-1 ].x << "," << c->Model[ index-1 ].y << "," << c->Model[ index-1 ].z << "," << c->Model[ index-1 ].xrot << "," << c->Model[ index-1 ].yrot << "," << c->Model[ index-1 ].zrot;
		if ( c->Model [ index - 1 ].checked ) ts << ",1," << c->Model[ index-1 ].sx << "," << c->Model[ index-1 ].sy << "," << c->Model[ index-1 ].sz << "\n";
		else ts << ",0\n";
	}
	FileHandle.close();
	this->setCaption ( "Cpsed 0.4: " + filename );
	statusBar()->message ("Scene has been saved successfully.");
	IsModified = false;
}

void GlWindow::closeEvent( QCloseEvent *ce )
{
	if ( IsModified )
	{
		response = QMessageBox::warning ( this , "Scene is unsaved !" , "You didn't save the current scene.\nWould you like to do it ?" , QMessageBox::Yes , QMessageBox::No , QMessageBox::Cancel );
		if ( response == QMessageBox::Yes) Save_Scene();
		else if ( response == QMessageBox::Cancel ) return;
	}
	DeleteEverything();
	ce->accept();
	close();
}

void GlWindow::Set_Default_ViewPoint()
{
	c->Default_AngleX = c->AngleX;
	c->Default_AngleZ = c->AngleZ;
	c->Default_Tx = c->Tx;
	c->Default_Ty = c->Ty;
	c->Default_Depth = c->Depth;
	statusBar()->message ( QString ("%1 %2, %3, %4, %5, %6").arg ( "New camera default viewpoint at x,y,z, angleX, angleZ:" ).arg (c->Tx).arg (c->Ty).arg (c->Depth).arg (c->AngleX).arg (c->AngleZ) );
}

void GlWindow::About()
{
	QMessageBox::about ( this, "Cpsed 0.4", "\nCpsed is an OpenGL 3D scene editor.\nIt imports .dff RenderWare and .3ds\nAutoDesk models and positions,\nrotates and scales each of them.\n\nThanks to Steve M. of\nhttp://gtaforums.com for\nhelping me with dff format.\n\n(c) 2004, Colossus (Giuseppe Torelli)\ngt67@users.sourceforge.net\n" );
}

GLuint GlWindow::Generate_3DS_Display_List ( Load3ds::t3DModel g_3DModel )
{
	GLuint dummy = glGenLists(1);
	if ( dummy !=0 )
	{
		glNewList ( dummy , GL_COMPILE);
		for (int i = 0; i < g_3DModel.numOfObjects; i++)
		{
        		// Make sure we have valid objects just in case.
        		if (g_3DModel.pObject.size() <= 0) break;
	        	// Get the current object that we are displaying
        		Load3ds::t3DObject *pObject = &g_3DModel.pObject[i];
			// Check to see if this object has a texture map, if so bind the texture to it.
			if ( pObject->bHasTexture )
			{
				// Turn on texture mapping and turn off color
				glEnable(GL_TEXTURE_2D);
				// Reset the color to normal again
				glColor3ub(255, 255, 255);
				// Bind the texture map to the object by it's materialID
				glBindTexture ( GL_TEXTURE_2D ,  m_textureLoaderClass->TextureMap [pObject->strFile] );
				//Associate the texture filename with the displaylist id given by glNewList
				TextureNamesMap.insert ( pair < GLuint , string > ( dummy , pObject->strFile ) );
				SameTextureListCnt [ pObject->strFile ]++;
        		} else{
					// Turn off texture mapping and turn on color
					glDisable(GL_TEXTURE_2D);
					glDisable(GL_BLEND);
					// Reset the color to normal again
					glColor3ub(255, 255, 255);
				}
			// This determines if we are in wireframe or normal mode
			glBegin( GL_TRIANGLES );                    // Begin drawing with our selected mode (triangles or lines)
			// Go through all of the faces (polygons) of the object and draw them
			for(int j = 0; j < pObject->numOfFaces; j++)
			{
				// Go through each corner of the triangle and draw it.
				for(int whichVertex = 0; whichVertex < 3; whichVertex++)
				{
					// Get the index for each point of the face
					int index = pObject->pFaces[j].vertIndex[whichVertex];
    					// Give OpenGL the normal for this vertex.
					glNormal3f(pObject->pNormals[ index ].x, pObject->pNormals[ index ].y, pObject->pNormals[ index ].z);
					// If the object has a texture associated with it, give it a texture coordinate.
					if(pObject->bHasTexture)
					{
			                	// Make sure there was a UVW map applied to the object or else it won't have tex coords.
						if(pObject->pTexVerts)
						{
        						glTexCoord2f(pObject->pTexVerts[ index ].x, pObject->pTexVerts[ index ].y);
    						}
					} else
						{
							// Make sure there is a valid material/color assigned to this object.
    							// You should always at least assign a material color to an object, 
							// but just in case we want to check the size of the material list.
							// if the size is at least one, and the material ID != -1,
							// then we have a valid material.
    							if(g_3DModel.pMaterials.size() && pObject->materialID >= 0)
    							{
        							// Get and set the color that the object is, since it must not have a texture
        							BYTE *pColor = g_3DModel.pMaterials[pObject->materialID].color;
								// Assign the current color to this model
								glColor3ub ( pColor[0], pColor[1], pColor[2] );
							}
						}
					// Draw in the current vertex of the object (Corner of current face)
                	    		glVertex3f ( pObject->pVerts[ index ].x, pObject->pVerts[ index ].y, pObject->pVerts[ index ].z);
                		}
			}
		glEnd();
		}
		glEndList ();
	}
	return dummy;
}

GLuint GlWindow::Generate_DFF_Display_List ()
{
	uint spl,i;
	GLuint dummy = glGenLists(1);
	if ( dummy != 0 )
	{
		glNewList ( dummy , GL_COMPILE );
		TDFFAtomic *a = mydff.clump->Atomic;
		while ( a )
		{
			TDFFFrame &frm = mydff.clump->FrameList.Data.Frame[ a->Data.FrameNum ];
			// We don't need to draw the damaged parts
			if ( ! strstr ( frm.Name , "_vlo" ) && ! strstr ( frm.Name , "_dam" ) )
			{
				glPushMatrix();
					glMultMatrixf ( frm.LTM.v );
					TDFFGeometry &geom = mydff.clump->GeometryList.Geometry[ a->Data.GeometryNum ];
					for (spl = 0; spl < geom.MaterialSplit.Header.SplitCount; spl++)
					{
						TDFFSplit &split = geom.MaterialSplit.Split[spl];
						TDFFMaterial &mat = geom.MaterialList.Material[split.MaterialIndex];
						if ( strlen ( mat.Texture.Name ) > 0 )
						{
							glDisable (GL_BLEND);
							glEnable (GL_TEXTURE_2D);
							glBindTexture ( GL_TEXTURE_2D , m_textureLoaderClass->TextureMap [mat.Texture.Name] );
							TextureNamesMap.insert ( pair < GLuint , string > ( dummy , mat.Texture.Name ) );
							SameTextureListCnt [ mat.Texture.Name ]++;
						}
						else 
						{
							glDisable (GL_TEXTURE_2D);
							glEnable(GL_BLEND);
							glBlendFunc( GL_DST_COLOR , GL_ZERO );
						}
						if (geom.MaterialSplit.Header.Data == 0)
							glBegin ( GL_TRIANGLES );
							else glBegin ( GL_TRIANGLE_STRIP );
						for (i = 0; i < split.FaceCount; i++)
						{
							if ( geom.Data.Normal ) glNormal3fv (geom.Data.Normal[split.Index[i]].v );
							if ( geom.Data.Color ) glColor3ub(geom.Data.Color[split.Index[i]].r, geom.Data.Color[split.Index[i]].g, geom.Data.Color[split.Index[i]].b );
							if ( geom.Data.UV ) glTexCoord2fv ( geom.Data.UV[split.Index[i]].v );
							glVertex3fv ( geom.Data.Vertex[split.Index[i]].v );
						}
						glEnd();
					}
				glPopMatrix();
			}
			a = a->next;
		}
		glEndList ();
	}
	return dummy;
}

void GlWindow::DeleteEverything()
{
	c->Model.clear();
	SceneFileName = "";
	for ( std::map< unsigned int , string >::iterator i= DisplayList.begin(); i != DisplayList.end(); i++)
	{
		glDeleteLists ( i->first , 1 );
	}
	for ( std::map< string , GLuint >::iterator i= m_textureLoaderClass->TextureMap.begin(); i != m_textureLoaderClass->TextureMap.end(); i++)
	{
		glDeleteTextures ( 1 ,  &i->second );
	}
	DisplayList.clear();
	SameTextureListCnt.clear();
	m_textureLoaderClass->TextureMap.clear();
	TextureNamesMap.clear();
	dummy_id_vector.clear();
	deleteModelAction->setEnabled( false );
	saveSceneAction->setEnabled( false );
	undoAction->setEnabled( false );
	redoAction->setEnabled( false );
	setInsertPointAction->setEnabled ( false ) ;
	//Did I forget anything else ?
}

