/***********************************************************************
//
//	TextureLoader class
//	By Colossus (Giuseppe Torelli) on 07 August 2004:
//
//	This class loads a TGA / JPG / BMP file
//	and sets the OpenGL appropriate parameters
//	
//	Credit for the image loaders go to the respective authors.
//	I just assembled them all in one class for convenience
//	This texture loader class changes the case of the texture file names when not found
***********************************************************************
*/

#include "textureloader.h"

TextureLoader::TextureLoader() 
{
}

TextureLoader::~TextureLoader() 
{
}
	
int TextureLoader::CreateTexture ( string Path , string strFileName , unsigned int textureID )
{
	string Copy_of_strFileName = strFileName;
	if ( strFileName.find (".BMP") != string::npos || strFileName.find (".bmp") !=string::npos )
	{
		if ( LoadBMP ( Path , strFileName , &texture ) == -1 )		//File not found ?
		{
			printf ("File not found: %s, try changing case\n" , strFileName.c_str() );
			Path = RemoveMeshFileNameFromPath ( Path );
			//Check the case of the string and return the other one
			strFileName = ChangeCase ( strFileName );
			//Ok, now it should load !
			if ( LoadBMP ( Path , strFileName , &texture ) == -1 )
			{
				printf ( "File not found: %s\n" , strFileName.c_str() );
				return -1;
			}
		}
	}
	else if ( strFileName.find (".TGA") != string::npos || strFileName.find (".tga") !=string::npos )
	{
		if ( LoadTGA ( Path , strFileName , &texture ) == -1 )
		{
			printf ("File not found: %s, try changing case\n" , strFileName.c_str() );
			Path = RemoveMeshFileNameFromPath ( Path );
			strFileName = ChangeCase ( strFileName );
			if ( LoadTGA ( Path , strFileName , &texture ) == -1 )
			{
				printf ( "File not found: %s\n" , strFileName.c_str()  );
				return -1;
			}
		}
	}
	
	else if ( strFileName.find (".jpg") != string::npos || strFileName.find (".jpg") !=string::npos )
	{
		if ( LoadJPG ( Path , strFileName , &texture ) == -1 )
		{
			printf ("File not found: %s, try changing case\n" , strFileName.c_str() );
			Path = RemoveMeshFileNameFromPath ( Path );
			strFileName = ChangeCase ( strFileName );
			if ( LoadJPG ( Path , strFileName , &texture ) == -1)
			{
				printf ( "File not found: %s" , strFileName.c_str() );
				return -1;
			}
		}
	}
	
	else if ( strFileName.find (".gif") != string::npos || strFileName.find (".gif") !=string::npos )
	{
		printf ("Sorry ! GIF textures not yet supported.\n");
		return -1;
		/*
		if ( LoadGIF ( Path , strFileName , &texture ) == -1 )
		{
			printf ("File not found: %s, try changing case\n" , strFileName);
			Path = RemoveMeshFileNameFromPath ( Path );
			strFileName = ChangeCase ( strFileName );
			if ( LoadGIF ( Path , strFileName , &texture ) == -1)
			{
				printf ( "File not found: %s" , strFileName );
				return -1;
			}
		}
		*/
	}
	else
	{
		printf ("\nNo extension found, trying TGA\n");
		if ( LoadTGA ( Path , strFileName +  ".tga" , &texture ) == -1 )
		{
			printf ("File not found: %s, try changing case\n" , strFileName.c_str() );
			Path = RemoveMeshFileNameFromPath ( Path );
			strFileName = ChangeCase ( strFileName +  ".tga" );
			if ( LoadTGA ( Path , strFileName , &texture ) == -1 )
			{
				printf ( "File not found: %s\n" , strFileName.c_str() );
				return -1;
			}
		}
	}
	// Generate a texture with the associative texture ID stored in the array
	glGenTextures ( 1, &textureID );
	// Bind the texture to the texture arrays index and set the texture parameters
	glBindTexture ( GL_TEXTURE_2D , textureID );
	SetOpenGLTexParam ( &texture );
	
	//Let's insert the texture filename into the map so we load it once
	TextureMap [ Copy_of_strFileName ] = textureID;
	return 0;
}

void TextureLoader::SetOpenGLTexParam ( glTexture *texture )
{
	//Build different resolutions of the texture to look it better at a far distance
	gluBuild2DMipmaps ( GL_TEXTURE_2D, texture->texFormat , texture->width, texture->height, texture->texFormat , GL_UNSIGNED_BYTE , texture->imageData );
	glTexParameteri (GL_TEXTURE_2D,GL_TEXTURE_MAG_FILTER,GL_LINEAR);    
	glTexParameteri (GL_TEXTURE_2D,GL_TEXTURE_MIN_FILTER,GL_LINEAR_MIPMAP_NEAREST);
	glTexEnvf (GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	
	//Release the allocated memory for the texture pixels since OpenGL stored it internally
	if (texture->imageData) delete [] texture->imageData;
}

string TextureLoader::ChangeCase ( string s2 )
{
	if ( s2.find ("a") and s2.find ("z" ) ) s2 = toUpper ( s2 );
	else s2 = toLower ( s2 );
	return s2;
}

string TextureLoader::toUpper ( string s3 )
{
	for(unsigned int i(0); i < s3.size(); i++)
		s3[i]= (char) toupper (s3[ i ]);
	return s3;
}

string TextureLoader::toLower ( string  s4)
{
	for(unsigned int i(0); i < s4.size(); i++)
		s4[i]= (char) tolower (s4[ i ]);
	return s4;
}

string TextureLoader::RemoveMeshFileNameFromPath ( string strFileName )
{
	string Path_Only;
	int pos = strFileName.rfind ("/");
	Path_Only = strFileName.assign (strFileName , 0 , pos+1 );
	return Path_Only;
}

int TextureLoader::LoadBMP ( string Path , string FileName, glTexture *texture )
{
	//I do not know the author, please write me if you believe to be it
	FILE *file;
	unsigned short int bfType;
	long int bfOffBits;
	short int biPlanes;
	short int BitCount;
	long int biSizeImage;
	int i;
	unsigned char temp;
	string FileToOpen = Path + FileName;
    	printf ("LoadBMP: %s\n" , FileToOpen.c_str() );
	if ( (file = fopen(FileToOpen.c_str() , "rb")) == NULL ) return -1;
	if ( !fread(&bfType, sizeof(short int), 1, file) )
	{
		printf("Error reading file!\n");
		return 0;
	}
	/* check if file is a bitmap */
	if (bfType != 19778)
	{
		printf("Not a Bitmap-File!\n");
		return 0;
	}        
	/* get the file size */
	/* skip file size and reserved fields of bitmap file header */
	fseek ( file, 8, SEEK_CUR);
	/* get the position of the actual bitmap data */
	if ( ! fread ( &bfOffBits, sizeof(long int), 1, file) )
	{
		printf("Error reading file!\n");
		return 0;
	}
	/* skip size of bitmap info header */
	fseek( file, 4, SEEK_CUR);
	fread ( &texture->width, sizeof(int) , 1, file);
	fread ( &texture->height, sizeof(int) , 1, file);
	fread ( &biPlanes, sizeof(short int), 1, file);
	if (biPlanes != 1)
	{
		printf("Error: number of Planes not 1!\n");
		return 0;
	}
	/* get the number of bits per pixel */
	if ( ! fread(&BitCount, sizeof(short int), 1, file) )
	{
		printf("Error reading file!\n");
		return 0;
	}
	if (BitCount != 24)
	{
		printf("Bits per Pixel not 24\n");
		return 0;
	}
	/* calculate the size of the image in bytes */
	biSizeImage = texture->width * texture->height * 3;
	printf ("Width: %d\tHeight: %d\tBpp: %d\tSize: %ld\n" , texture->width , texture->height , BitCount , biSizeImage);
	texture->imageData = new unsigned char[biSizeImage];
	/* seek to the actual data */
	fseek (file, bfOffBits, SEEK_SET);
	if ( ! fread (texture->imageData, biSizeImage, 1, file) )
	{
		printf("Error loading file!\n");
		return 0;
	}
	/* swap red and blue (bgr -> rgb) */
	for (i = 0; i < biSizeImage; i += 3)
	{
		temp = texture->imageData[i];
		texture->imageData[i] = texture->imageData[i + 2];
		texture->imageData[i + 2] = temp;
	}

	fclose( file);
	texture->texFormat = GL_RGB;
	return 0;
}

int TextureLoader::LoadTGA ( string Path , string FileName, glTexture *tgaFile ) 
{
	//Author:        DigiBen     digiben@gametutorials.com
	//The comments here belong to DigiBen
	
	int width = 0, height = 0;          // The dimensions of the image
	byte length = 0;                    // The length in bytes to the pixels
	byte imageType = 0;                 // The image type (RLE, RGB, Alpha...)
	byte bits = 0;                      // The bits per pixel for the image (16, 24, 32)
	FILE *pFile = NULL;                 // The file pointer
	int channels = 0;                   // The channels of the image (3 = RGA : 4 = RGBA)
	int stride = 0;                     // The stride (channels * width)
	int i = 0;                          // A counter
		
	// This function loads in a TARGA (.TGA) file and returns its data to be
	// used as a texture or what have you.  This currently loads in a 16, 24
	// and 32-bit targa file, along with RLE compressed files.  Eventually you
	// will want to do more error checking to make it more robust.  This is
	// also a perfect start to go into a modular class for an engine.
	// Basically, how it works is, you read in the header information, then
	// move your file pointer to the pixel data.  Before reading in the pixel
	// data, we check to see the if it's an RLE compressed image.  This is because
	// we will handle it different.  If it isn't compressed, then we need another
	// check to see if we need to convert it from 16-bit to 24 bit.  24-bit and
	// 32-bit textures are very similar, so there's no need to do anything special.
	// We do, however, read in an extra bit for each color.
	
    	// Open a file pointer to the targa file and check if it was found and opened 
       	string FileToOpen = Path + FileName;
   	printf ("LoadTGA: %s\n" , FileToOpen.c_str() );
    	if((pFile = fopen(FileToOpen.c_str() , "rb")) == NULL) return -1;
		
	// Allocate the structure that will hold our eventual image data (must free it!)
	// Read in the length in bytes from the header to the pixel data
	fread(&length, sizeof(byte), 1, pFile);
	
	// Jump over one byte
	fseek(pFile,1,SEEK_CUR); 
	
	// Read in the imageType (RLE, RGB, etc...)
	fread(&imageType, sizeof(byte), 1, pFile);
	
	// Skip past general information we don't care about
	fseek(pFile, 9, SEEK_CUR); 
	
	// Read the width, height and bits per pixel (16, 24 or 32)
	fread(&width,  sizeof(WORD), 1, pFile);
	fread(&height, sizeof(WORD), 1, pFile);
	fread(&bits,   sizeof(byte), 1, pFile);
	printf ("Width: %d\tHeight: %d\tBpp: %d\n" , width , height , bits );
	// Now we move the file pointer to the pixel data
	fseek(pFile, length + 1, SEEK_CUR); 
	
	// Check if the image is RLE compressed or not
	if(imageType != TGA_RLE)
	{
		// Check if the image is a 24 or 32-bit image
		if(bits == 24 || bits == 32)
		{
		// Calculate the channels (3 or 4) - (use bits >> 3 for more speed).
		// Next, we calculate the stride and allocate enough memory for the pixels.
			channels = bits / 8;
			stride = channels * width;
			tgaFile->imageData = new unsigned char[stride * height];

			// Load in all the pixel data line by line
			for(int y = 0; y < height; y++)
			{
				// Store a pointer to the current line of pixels
				unsigned char *pLine = &(tgaFile->imageData[stride * y]);

				// Read in the current line of pixels
				fread(pLine, stride, 1, pFile);
			
				// Go through all of the pixels and swap the B and R values since TGA
				// files are stored as BGR instead of RGB (or use GL_BGR_EXT verses GL_RGB)
				for(i = 0; i < stride; i += channels)
				{
				int temp     = pLine[i];
				pLine[i]     = pLine[i + 2];
				pLine[i + 2] = temp;
				}
			}
		}
		// Check if the image is a 16 bit image (RGB stored in 1 unsigned short)
		else if(bits == 16)
		{
		unsigned short pixels = 0;
		int r=0, g=0, b=0;
	
		// Since we convert 16-bit images to 24 bit, we hardcode the channels to 3.
		// We then calculate the stride and allocate memory for the pixels.
		channels = 3;
		stride = channels * width;
		tgaFile->imageData = new unsigned char[stride * height];
	
		// Load in all the pixel data pixel by pixel
		for(int i = 0; i < width*height; i++)
		{
			// Read in the current pixel
			fread(&pixels, sizeof(unsigned short), 1, pFile);
			
			// To convert a 16-bit pixel into an R, G, B, we need to
			// do some masking and such to isolate each color value.
			// 0x1f = 11111 in binary, so since 5 bits are reserved in
			// each unsigned short for the R, G and B, we bit shift and mask
			// to find each value.  We then bit shift up by 3 to get the full color.
			b = (pixels & 0x1f) << 3;
			g = ((pixels >> 5) & 0x1f) << 3;
			r = ((pixels >> 10) & 0x1f) << 3;
			
			// This essentially assigns the color to our array and swaps the
			// B and R values at the same time.
			tgaFile->imageData[i * 3 + 0] = r;
			tgaFile->imageData[i * 3 + 1] = g;
			tgaFile->imageData[i * 3 + 2] = b;
		}
		}   
		// Else return a NULL for a bad or unsupported pixel format
		else
		return -1;
	}
	// Else, it must be Run-Length Encoded (RLE)
	else
	{
		// First, let me explain real quickly what RLE is.  
		// For further information, check out Paul Bourke's intro article at: 
		// http://astronomy.swin.edu.au/~pbourke/dataformats/rle/
		// 
		// Anyway, we know that RLE is a basic type compression.  It takes
		// colors that are next to each other and then shrinks that info down
		// into the color and a integer that tells how much of that color is used.
		// For instance:
		// aaaaabbcccccccc would turn into a5b2c8
		// Well, that's fine and dandy and all, but how is it down with RGB colors?
		// Simple, you read in an color count (rleID), and if that number is less than 128,
		// it does NOT have any optimization for those colors, so we just read the next
		// pixels normally.  Say, the color count was 28, we read in 28 colors like normal.
		// If the color count is over 128, that means that the next color is optimized and
		// we want to read in the same pixel color for a count of (colorCount - 127).
		// It's 127 because we add 1 to the color count, as you'll notice in the code.
	
		// Create some variables to hold the rleID, current colors read, channels, & stride.
		byte rleID = 0;
		int colorsRead = 0;
		channels = bits / 8;
		stride = channels * width;
	
		// Next we want to allocate the memory for the pixels and create an array,
		// depending on the channel count, to read in for each pixel.
		tgaFile->imageData = new unsigned char[stride * height];
		byte *pColors = new byte [channels];
	
		// Load in all the pixel data
		while(i < width*height)
		{
		// Read in the current color count + 1
		fread(&rleID, sizeof(byte), 1, pFile);
		
		// Check if we don't have an encoded string of colors
		if(rleID < 128)
		{
			// Increase the count by 1
			rleID++;
	
			// Go through and read all the unique colors found
			while(rleID)
			{
			// Read in the current color
			fread(pColors, sizeof(byte) * channels, 1, pFile);
	
			// Store the current pixel in our image array
			tgaFile->imageData[colorsRead + 0] = pColors[2];
			tgaFile->imageData[colorsRead + 1] = pColors[1];
			tgaFile->imageData[colorsRead + 2] = pColors[0];
	
			// If we have a 4 channel 32-bit image, assign one more for the alpha
			if(bits == 32)
				tgaFile->imageData[colorsRead + 3] = pColors[3];
	
			// Increase the current pixels read, decrease the amount
			// of pixels left, and increase the starting index for the next pixel.
			i++;
			rleID--;
			colorsRead += channels;
			}
		}
		// Else, let's read in a string of the same character
		else
		{
			// Minus the 128 ID + 1 (127) to get the color count that needs to be read
			rleID -= 127;
	
			// Read in the current color, which is the same for a while
			fread(pColors, sizeof(byte) * channels, 1, pFile);
	
			// Go and read as many pixels as are the same
			while(rleID)
			{
			// Assign the current pixel to the current index in our pixel array
			tgaFile->imageData[colorsRead + 0] = pColors[2];
			tgaFile->imageData[colorsRead + 1] = pColors[1];
			tgaFile->imageData[colorsRead + 2] = pColors[0];
	
			// If we have a 4 channel 32-bit image, assign one more for the alpha
			if(bits == 32) tgaFile->imageData[colorsRead + 3] = pColors[3];
			// Increase the current pixels read, decrease the amount
			// of pixels left, and increase the starting index for the next pixel.
			i++;
			rleID--;
			colorsRead += channels;
			}
		}
		}
	}
	
	// Close the file pointer that opened the file
	fclose(pFile);
	
	// Fill in our tImageTGA structure to pass back
	//pImageData->channels = channels;
	tgaFile->width	= width;
	tgaFile->height	= height;
	if( channels == 4) tgaFile->texFormat = GL_RGBA;
	else tgaFile->texFormat = GL_RGB;
	return 1;
}

int TextureLoader::LoadJPG ( string Path , string FileName, glTexture *texture) 
{
	//I do not know the author, please write me if you believe to be it
	struct jpeg_decompress_struct cinfo;
	struct libjpeg_my_error_mgr jerr;
	FILE	*infile;
	JSAMPARRAY scanline;
	int row_stride; // row width

	string FileToOpen = Path + FileName;
    	printf ("LoadJPG: %s\n" , FileToOpen.c_str() );
	infile = fopen( FileToOpen.c_str() , "rb");
	if( !infile) return -1;

	// allocate and initialize JPEG decompression object
	// We set up the normal JPEG error routines, then override error_exit.
	cinfo.err = jpeg_std_error(&jerr.pub);
	// Establish the setjmp return context for my_error_exit to use.
	if( setjmp(jerr.setjmp_buffer) )
	{
		// it failed, so cleanup
		jpeg_destroy_decompress(&cinfo);
		fclose(infile);
		return false;
	}
	// initialize the JPEG decompression object.
	jpeg_create_decompress(&cinfo);

	// data source
	jpeg_stdio_src(&cinfo, infile);

	// read header
	jpeg_read_header(&cinfo, true);

	// start decompressor
	jpeg_start_decompress(&cinfo);

	// scanline size in bytes
	row_stride = cinfo.output_width * cinfo.output_components;

	// allocate our data pointer
	texture->imageData = new unsigned char[ row_stride * cinfo.output_height ];
	printf ("Width: %d\tHeight: %d\n" , cinfo.output_width , cinfo.output_height );
	// Make a one-row-high sample array that will
	// go away when done with image
	scanline = (*cinfo.mem->alloc_sarray)
	((j_common_ptr) &cinfo, JPOOL_IMAGE, row_stride, 1);

	// read scanlines
	unsigned char* p = texture->imageData;
	while (cinfo.output_scanline < cinfo.output_height)
	{
		// jpeg_read_scanlines expects an array of
		// pointers to scanlines.
		// Here the array is only one element long
		jpeg_read_scanlines(&cinfo, scanline, 1);

		memmove( p, scanline[0], row_stride );
		p += row_stride;
	}

	// finish decompression
	jpeg_finish_decompress(&cinfo);

	// init texture fields
	texture->width	 = cinfo.output_width;
	texture->height = cinfo.output_height;
	texture->texFormat	= GL_RGB;

	// release JPEG decompression object
	jpeg_destroy_decompress(&cinfo);

	if( jerr.pub.num_warnings != 0) fprintf( stderr,"JPEG Loader: warning: detected corruption on decompress." );
	fclose(infile);
	return 1;
}

