// Source file for evolvotron
// Copyright (C) 2002,2003 Tim Day
/*
This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
*/

/*! \file
  \brief Implementation of class MutatableImageComputerFarm.
*/

#include "mutatable_image_computer_farm.h"
#include <algorithm>

/*! Creates the specified number of threads and store pointers to them.
 */
MutatableImageComputerFarm::MutatableImageComputerFarm(uint n_threads)
{
  _done_position=_done.end();

  for (uint i=0;i<n_threads;i++)
    {
      // The computer's constructor includes a start()
      _computers.push_back(new MutatableImageComputer(this));
    }
}

/*! Destructor kills off all compute threads and frees their resources.  
  NB The MutatableImageComputer destructor signals the thread to stop and waits for it.
 */
MutatableImageComputerFarm::~MutatableImageComputerFarm()
{
  std::clog << "Compute farm shut down begun...\n";

  // Kill all the computers
  for (std::vector<MutatableImageComputer*>::iterator it=_computers.begin();it!=_computers.end();it++)
    {
      delete (*it);
    }

  // Clear all the tasks in queues
  _mutex.lock();

  for (TodoQueue::iterator it=_todo.begin();it!=_todo.end();it++)
    {
      delete (*it);
    }
  _todo.clear();

  for (DoneQueueByDisplay::iterator it0=_done.begin();it0!=_done.end();it0++)
    {
      for (DoneQueue::iterator it1=(*it0).second.begin();it1!=(*it0).second.end();it1++)
	{
	  delete (*it1);
	}
      (*it0).second.clear();
    }
  _done.clear();

  _mutex.unlock();

  std::clog << "...completed compute farm shut down\n";
}

//! Predicate function to test whether a task has been aborted
static const bool predicate_aborted(const MutatableImageComputerTask*const t) 
{
  return t->aborted();
}

void MutatableImageComputerFarm::fasttrack_aborted()
{
  _mutex.lock();
  
  // \todo: Inefficient starting search again each time.  Some problem with erase otherwise though, but might have been task abort mem leak.
  TodoQueue::iterator it;
  while (
	 (
	  it=std::find_if(_todo.begin(),_todo.end(),predicate_aborted)
	  )
	 !=
	 _todo.end()
	 )
    {
      _done[(*it)->display()].insert(*it);
      _todo.erase(it);
    }
  
  _mutex.unlock();
}

void MutatableImageComputerFarm::push_todo(MutatableImageComputerTask* task)
{
  _mutex.lock();
  _todo.insert(task);
  
  // Check if any of the computers are executing lower priority tasks and if so defer least important one.
  for (std::vector<MutatableImageComputer*>::iterator it=_computers.begin();it!=_computers.end();it++)
    {
      ((*it)->defer_if_less_important_than(task->priority()));
    }

  _mutex.unlock();
}

MutatableImageComputerTask*const MutatableImageComputerFarm::pop_todo()
{
  MutatableImageComputerTask* ret=0;
  _mutex.lock();
  
  TodoQueue::iterator it=_todo.begin();
  if (it!=_todo.end())
    {
      ret=(*it);
      _todo.erase(it);
    }

  _mutex.unlock();
  return ret;
}

void MutatableImageComputerFarm::push_done(MutatableImageComputerTask* task)
{
  _mutex.lock();
  _done[task->display()].insert(task);
  _mutex.unlock();
}

MutatableImageComputerTask* MutatableImageComputerFarm::pop_done()
{
  MutatableImageComputerTask* ret=0;
  _mutex.lock();
  
  if (_done_position==_done.end())
    {
      _done_position=_done.begin();
    }

  if (_done_position!=_done.end())
    {
      DoneQueue& q=(*_done_position).second;
      DoneQueue::iterator it=q.begin();
      if (it!=q.end())
	{
	  ret=(*it);
	  q.erase(it);
	}

      if (q.empty())
	{
	  DoneQueueByDisplay::iterator advanced_done_position=_done_position;
	  advanced_done_position++;	  
	  _done.erase(_done_position);
	  _done_position=advanced_done_position;
	}
      else
	{
	  _done_position++;
	}
    }

  _mutex.unlock();
  return ret;
}

void MutatableImageComputerFarm::abort_all()
{
  _mutex.lock();

  for (TodoQueue::iterator it=_todo.begin();it!=_todo.end();it++)
    {
      (*it)->abort();
      delete (*it);
      _todo.erase(it);
    }

  for (std::vector<MutatableImageComputer*>::iterator it=_computers.begin();it!=_computers.end();it++)
    {
      (*it)->abort();
    }

  for (DoneQueueByDisplay::iterator it0=_done.begin();it0!=_done.end();it0++)
    {
      DoneQueue& q=(*it0).second;
      for (DoneQueue::iterator it1=q.begin();it1!=q.end();it1++)
	{
	  (*it1)->abort();
	  delete (*it1);
	  q.erase(it1);
	}
    }
  _mutex.unlock();  
}

void MutatableImageComputerFarm::abort_for(const MutatableImageDisplay* disp)
{
  _mutex.lock();

  for (TodoQueue::iterator it=_todo.begin();it!=_todo.end();it++)
    {
      if ((*it)->display()==disp)
	{
	  (*it)->abort();
	  delete (*it);
	  _todo.erase(it);
	}
    }

  for (std::vector<MutatableImageComputer*>::iterator it=_computers.begin();it!=_computers.end();it++)
    {      
	(*it)->abort_for(disp);
    }

  
  DoneQueueByDisplay::iterator it0=_done.find(disp);
  if (it0!=_done.end())
    {
      DoneQueue& q=(*it0).second;

      //! \todo It would be pretty odd if display didn't match the queue bin: change to assert
      for (DoneQueue::iterator it1=q.begin();it1!=q.end();it1++)
	{
	  if ((*it1)->display()==disp)
	    {
	      (*it1)->abort();
	      delete (*it1);
	      q.erase(it1);
	    }
	}
    }
  
  _mutex.unlock();  
}

std::ostream& MutatableImageComputerFarm::write_info(std::ostream& out) const
{
  uint active_computers=0;

  for (std::vector<MutatableImageComputer*>::const_iterator it=_computers.begin();it!=_computers.end();it++)
    {      
      if ((*it)->active())
	{
	  active_computers++;
	}
    }

  //! /todo _done.size() no longer returns number of tasks to be delivered to, but number of displays with pending deliveries.  Fix ?
  out << "[" << _todo.size() << "/" << active_computers << "/" << _done.size() << "]";
  return out;
}

const uint MutatableImageComputerFarm::tasks() const
{
  uint ret=0;
  
  for (std::vector<MutatableImageComputer*>::const_iterator it=_computers.begin();it!=_computers.end();it++)
    {      
      if ((*it)->active())
	{
	  ret++;
	}
    }

  _mutex.lock();

  ret+=_todo.size();

  for (DoneQueueByDisplay::const_iterator it=_done.begin();it!=_done.end();it++)
    ret+=(*it).second.size();

  _mutex.unlock();

  return ret;
}
