/*
 *  FillStyles.cpp
 *  Transform Examples
 *
 *  Created by smackay on Fri May 09 2003.
 *  Copyright (c) 2001-2004 Flagstone Software Ltd. All rights reserved.
 *
 *  This code is distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 *  EXPRESS OR IMPLIED, AND Flagstone HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *  PURPOSE, AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 */

#ifndef __FillStyles_h__
#define __FillStyles_h__ 1

/*
 * This example shows how the different fill styles can be used to fill a shape.
 * A shape can be filled with solid colour, an image or a colour gradient. If an 
 * image is smaller than the shape it may be tiled to fill the area or clipped
 * where colour around the edge of the image is used to fill the remaining area.
 * Gradients may be linear - changing in one dimension or radial - changing in two
 * dimensions.
 *
 * To run this example, type the following on a command line:
 *
 * examples --example FillStyles --image image-file [--resultDir path]
 *
 * where
 *
 *    image-file is the path to a file containing a JPEG image. The image should
 *    be 180 pixels wide and 135 pixels high to illustrate how images are placed
 *    and tiled within the shape they are filling, however any size of image will 
 *    work. NOTE the size of the rectanlge used to display the shape is fixed so
 *    images larger than 180 x 135 will not be partially hidden.
 *
 *    resultDir is the directory where the Flash file generated by the example 
 *    is written to.
 */
class FillStyles
{
public:
    FillStyles(map< string, string>& options)
    {
        FSMovie movie;

        int width = 3600;
        int height = 2700;

        string imageFile = stringOption(options, "image");

        /*
        * A row of five rectangles will be displayed - one for each fill style.
        * The additional space is ued to provide padding between the rectangles 
        * and to the edge of the screen.
        */
        movie.setFrameSize(FSBounds(0, 0, (width+800)*5 , height+800));
        movie.setFrameRate(1.0f);
        movie.add(new FSSetBackgroundColor(FSColorTable::lightblue()));

        /*
        * Define the rectangle that is filled with the different fill styles.
        */
        FSBounds bounds(-width/2, -height/2, width/2, height/2);

        FSShape rectangle;

        rectangle.add(new FSShapeStyle(1, 1, 0, -width/2, -height/2));
        rectangle.add(new FSLine(width, 0));
        rectangle.add(new FSLine(0, height));
        rectangle.add(new FSLine(-width, 0));
        rectangle.add(new FSLine(0, -height));

        /*
        * The following variables are reused for each fill style to simplify the code.
        * In Transform an object assumes ownership for any added to it, so the shape
        * and style arrays can be safely reused in the FSDefineShape constructor. A 
        * shallow copy of the object is made. The FSDefineShape object assumes 
        * ownership of the objects they contain and no explicit memory management 
        * is required - the objects will be deleted when the shape is deleted.
        */

        int identifier = 0;
        
        FSShape shape;
        
        FSVector<FSLineStyle*> lineStyles(1);
        FSVector<FSFillStyle*> fillStyles(1);

        /*************************************************
        * Create a rectangle filled with a solid colour.
        *************************************************/
        shape = rectangle;
        identifier = movie.newIdentifier();
        
        lineStyles[0] = new FSSolidLine(20, FSColorTable::black());
        fillStyles[0] = new FSSolidFill(FSColorTable::red());

        movie.add(new FSDefineShape(identifier, bounds, fillStyles, lineStyles, shape));
        movie.add(new FSPlaceObject(identifier, 1, 2000, 1600));

        /******************************************
        * Create a rectangle tiled with an image.
        ******************************************/
        shape = rectangle;
        identifier = movie.newIdentifier();

        lineStyles[0] = new FSSolidLine(20, FSColorTable::black());

        try 
        {
            size_t size = 0;
            byte* bytes = dataFromFile(imageFile.c_str(), size);

            FSDefineJPEGImage* image = new FSDefineJPEGImage(movie.newIdentifier(), bytes, size);

            /*
            * Scale the loaded image to half its original size so it will tile four times inside the
            * rectangle. When an image is loaded its width and height default to twips rather than
            * pixels. An image 300 x 200 pixels will be displayed as 300 x 200 twips (15 x 10 pixels).
            * Scaling the image by 20 (20 twips = 1 pixel) would restore it to its original size. Here
            * we only scale the image to half its original size so variable sized images passed on the 
            * command line when the example is run are more easily seen.
            *
            * The coordinate transform relocates the top left corner of the image to the top left
            * corner of the rectangle in which it is displayed.
            */
            FSCoordTransform transform = FSCoordTransform(-width/2, -height/2, 10.0, 10.0);

            fillStyles[0] = new FSBitmapFill(FSFillStyle::TiledBitmap, image->getIdentifier(), transform);

            movie.add(image);
            movie.add(new FSJPEGEncodingTable()); // The image will still be displayed with an empty table
            movie.add(new FSDefineShape(identifier, bounds, fillStyles, lineStyles, shape));
            movie.add(new FSPlaceObject(identifier, 2, 6000, 1600));
        }
        catch (FSFileOpenException e)
        {
            cerr << e.what();
        }
        catch (FSAccessException e)
        {
            cerr << e.what();
        }

        /**************************************************
        * Create a rectangle filled with a clipped image.
        **************************************************/
        shape = rectangle;
        identifier = movie.newIdentifier();

        lineStyles[0] = new FSSolidLine(20, FSColorTable::black());

        try
        {
            size_t size = 0;
            byte* bytes = dataFromFile(imageFile.c_str(), size);

            FSDefineJPEGImage* image = new FSDefineJPEGImage(movie.newIdentifier(), bytes, size);

            /*
            * Scale the loaded image to half its original size so variable sized images are 
            * more easily seen. The coordinate transformation, taking into account the scaling
            * factor will display the image centered in the middle of the rectangle.
            */
            FSCoordTransform transform = FSCoordTransform(-(image->getWidth()*10)/2, -(image->getHeight()*10)/2, 10.0, 10.0);

            fillStyles[0] = new FSBitmapFill(FSFillStyle::ClippedBitmap, image->getIdentifier(), transform);

            movie.add(image);
            movie.add(new FSJPEGEncodingTable()); // The image will still be displayed with an empty table
            movie.add(new FSDefineShape(identifier, bounds, fillStyles, lineStyles, shape));
            movie.add(new FSPlaceObject(identifier, 3, 10000, 1600));
        }
        catch (FSFileOpenException e)
        {
            cerr << e.what();
        }
        catch (FSAccessException e)
        {
            cerr << e.what();
        }

        /*********************************************************
        * Create a rectangle filled with linear gradient colour.
        *********************************************************/
        shape = rectangle;
        identifier = movie.newIdentifier();

        /*
        * The gradient square must be mapped to the rectangle being filled. The square
        * measures 32768 x 32768 twips. The coordinates range from -16384, -16384 at the
        * bottom left corner to (16384, 16384) at the top right corner. Mapping takes place
        * in three steps:
        *
        * 1. The coordinate system of the gradient square is translated to match the coordinate
        * system of the shape in which the gradient is displayed. In this example, both the
        * gradient square and the rectangle in which it will be displayed are both centred at
        * (0,0) - see the FSBounds object created above. For this example the full range of the
        * gradient will be displayed so no translation is required. The second example that
        * displays a radial gradient illustrates how the translation can be calculated to change
        * the portion of the gradient being displayed.
        *
        * 2. The gradient square is scaled so that the full range of the gradient will be
        * displayed inside the rectangle. If a smaller scaling factor is set then the
        * portion of the gradient being displayed will change. The second example shows 
        * how the scaling factor affects the gradient.
        *
        * 3. The gradient can be rotated to change the direction of the colour change. If a
        * rotation is not applied then the gradient changes in the positive x axis - front left
        * to right on the Flash Player's screen.
        */

        // To display the full gradient, calculate the ratio of the shape's width to the width
        // of the gradient square. The gradient will be rotated so the width is used rather than
        // calculating the length of the diagonal.
        
        float scale = bounds.getWidth() / 32768.0f;

        // Since the centre of the rectangle and the gradient square are both at (0,0) no
        // additional translation is required.

        int translateX = 0;
        int translateY = 0;

        // The order of composition is important. If the scale transform was the first argument
        // then the translation coordinates would also be scaled.

        // The scaling is performed in the x and y direction as the gradient is rotated in the
        // the following step.
        
        FSCoordTransform transform = FSCoordTransform(translateX, translateY, scale, scale);

        // Apply a rotation so the gradient changes across the diagonal of the rectangle.
        
        transform.rotate(45);

        /*
        * The array of FSGradient objects defines how the colour changes across the gradient
        * square. The ratio defines the proportion across the square: 0 is the left side and
        * 255 is the right side.
        */
        FSVector<FSGradient> gradients(2);

        gradients[0] = FSGradient(0, FSColorTable::white());
        gradients[1] = FSGradient(255, FSColorTable::black());

        lineStyles[0] = new FSSolidLine(20, FSColorTable::black());
        fillStyles[0] = new FSGradientFill(FSFillStyle::LinearGradient, transform, gradients);

        movie.add(new FSDefineShape(identifier, bounds, fillStyles, lineStyles, shape));
        movie.add(new FSPlaceObject(identifier, 4, 14000, 1600));

        /*********************************************************
        * Create a rectangle filled with radial gradient colour.
        *********************************************************/
        shape = rectangle;
        identifier = movie.newIdentifier();

        // Resize the gradients array so the gradient will contain 4 colours.
        
        gradients.resize(4);

        /* The gradient will be centred in the bottom left corner of the rectangle
        * since the full spectrum of the radial gradient will be displayed the
        * gradient square need only be scaled by half the amount compared to the
        * linear gradient above.
        */
        float scaleX = (bounds.getWidth() / 32768.0f) * 2.0f;
        float scaleY = (bounds.getHeight() / 32768.0f) * 2.0f;

        /* Here the translation is calculated so the centre is expressed as a percentage
        * of the width of the rectangle. The general form of the calculation is:
        *
        * translateX = bounds.getMinX() + bounds.getWidth() * 0.0;
        * translateY = bounds.getMinY() + bounds.getHeight() * 0.25;
        */

        translateX = bounds.getMinX();
        translateY = bounds.getMinY();

        transform = FSCoordTransform(translateX, translateY, scaleX, scaleY);
        
        gradients[0] = FSGradient(15, FSColorTable::red());
        gradients[1] = FSGradient(63, FSColorTable::orange());
        gradients[2] = FSGradient(127, FSColorTable::yellow());
        gradients[3] = FSGradient(255, FSColorTable::green());

        lineStyles[0] = new FSSolidLine(1, FSColorTable::black());
        fillStyles[0] = new FSGradientFill(FSFillStyle::RadialGradient, transform, gradients);

        movie.add(new FSDefineShape(identifier, bounds, fillStyles, lineStyles, shape));
        movie.add(new FSPlaceObject(identifier, 5, 18000, 1600));

        movie.add(new FSShowFrame());

        saveMovie(movie, stringOption(options, "resultDir"), "FillStyles.swf");
    }
};

#endif
