/*
 *  BasicButtons.hpp
 *  Utilities Examples
 *
 *  Created by smackay on Tue Jun 02 2004.
 *  Copyright (c) 2001-2004 Flagstone Software Ltd. All rights reserved.
 *
 *  This code is distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 *  EXPRESS OR IMPLIED, AND Flagstone HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *  PURPOSE, AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 */

#ifndef __BasicButtons_h__
#define __BasicButtons_h__ 1

/*
 * This example shows how to create buttons using the FSDefineButton and FSDefineButton2
 * classes. Each button displays a web page in a browser when clicked. The second button
 * also plays a sound when clicked.
 *
 * To run this example, type the following on a command line:
 *
 *     examples --example BasicButtons
 *         --font font-file
 *         --sound sound-file
 *         --label string
 *        [--resultDir path]
 *
 * where
 *
 *     font-file is the path to a file containing a font definition used to 
 *     render the text for label on the button.
 *
 *     sound-file is the path to a file containing a WAVE or MP3 format sound
 *     which is played when a button is clicked.
 *
 *     string the label that will be displayed on the button - enclose in quotes 
 *     if the label contains spaces.
 *
 *     resultDir is the directory where the Flash file generated by the example 
 *     is written to.
 */
class BasicButtons
{
public:
	BasicButtons(map< string, string>& options)
	{
		string fontFile = stringOption(options, "font");
		string soundFile = stringOption(options, "sound");
		string buttonLabel = stringOption(options, "label");
	    
		FSTextConstructor* textGenerator = TextConstructor();
		FSSoundConstructor* soundGenerator = SoundConstructor();
		FSShapeConstructor* path = ShapeConstructor();
        FSMovie movie;

		int status = TransformUtil::OK;

		if ((status = textGenerator->setFontFromFile(fontFile.c_str())) != TransformUtil::OK)
		{
			switch (status)
			{
				case TransformUtil::FileNotFound:
					cout << "Could not find font file" << endl; break;
				case TransformUtil::ReadError:
					cout << "Could not read font file" << endl; break;
				case TransformUtil::FormatError:
					cout << "Could not read font file" << endl; break;
			}
		}

		if ((status = soundGenerator->setSoundFromFile(soundFile.c_str())) != TransformUtil::OK)
		{
			switch (status)
			{
				case TransformUtil::FileNotFound:
					cout << "Could not find sound file" << endl; break;
				case TransformUtil::ReadError:
					cout << "Could not read sound file" << endl; break;
				case TransformUtil::FormatError:
					cout << "Could not read sound file" << endl; break;
			}
		}

		if (status == TransformUtil::OK)
		{
			int layer = 0;       // Starting layer for objects in the display list.
			int fontSize = 280;  // Font size in twips 1 point = 20 twips

			/*
			* Define margins added to define the width of the BasicButtons
			* relative to the size of the label and the screen width
			* relative to the width of the button.
			*/
			int buttonMargin = 200;
			int screenMargin = 400;

			/*
			* Define colours used to fill the BasicButtons in each of its states.
			*/
			FSColor lineColor = FSColorTable::black();
			FSColor shadowColor = FSColorTable::gray();
			FSColor upColor = FSColorTable::red();
			FSColor overColor = FSColorTable::orange();
			FSColor downColor = FSColorTable::orange();

			/*
			* Initialize the FSTextConstructor object used to generate the text and font 
			* definitions. The size and colour can be changed at any point - the values
			* are specified in the FSDefineText[2] objects not in the font definition.
			*/
			FSColor black = FSColorTable::black();

			textGenerator->setIdentifier(movie.newIdentifier());
			textGenerator->setSize(fontSize);
			textGenerator->setColor(&black);

			/*
			* Define the arrays used to hold the objects added to the button definitions. 
			* When an array is passed to an object in Transform a shallow copy is made an
			* the object assumes ownership of all the objects in an array so it can be reused
			* safely.
			*/
			FSVector<FSButton>records;
			FSVector<FSActionObject*>actions;
			FSVector<FSButtonEvent>events;

			/*
			* Define coordinate and color transforms applied to the basic button shape to 
			* construct the BasicButtons. Here the transform is used to display a shape
			* slightly offset when the button is clicked giving a pseudo 3D effect.
			*/
			FSCoordTransform recess(60, 60);
	        
			/*
			* Define the shapes used to create the BasicButtons.
			*
			* The size of the shapes is calculated relative to the bounding rectangle
			* enclosing an arbitrary long label. The ascent of the text is used rather
			* than the full height of the bounding rectangle as this gives better
			* placement of the text relative to the button edges.
			*/
			FSDefineShape3* label = textGenerator->defineShape(movie.newIdentifier(), buttonLabel.c_str());

			// define a 10 pixel margin around the text.
	        
			int buttonWidth = label->getBounds().getWidth() + buttonMargin ;
			int buttonHeight = -label->getBounds().getMinY() + buttonMargin;

			int lineWidth = 20;
			int cornerRadius = 100;

			/*
			* The button will cast as shadow when it in the up state. Recessing the button
			* by changing the location of the shapes when the button is clicked allows a
			* very simple animation to be performed. More complex animations can be created
			* by using movie clips.
			*/
			int xShadow = 60;
			int yShadow = 60;

			/*
			* Used to hold the width and height of the bounding rectangle that
			* encloses the button labels.
			*/
			int width = 0;
			int height = 0;

			FSSolidLine* shadowLine = new FSSolidLine(lineWidth, shadowColor);
			FSSolidFill* shadowFill = new FSSolidFill(shadowColor);

			path->addLineStyle(shadowLine);
			path->addFillStyle(shadowFill);
			path->rect(0, 0, buttonWidth, buttonHeight, cornerRadius);
	        
			FSDefineShape3* shadow = path->defineShape(movie.newIdentifier());
	            
			FSSolidLine* upLine = new FSSolidLine(lineWidth, lineColor);
			FSSolidFill* upFill = new FSSolidFill(upColor);

			path->setLineStyle(0, upLine);
			path->setFillStyle(0, upFill);
			path->rect(0, 0, buttonWidth, buttonHeight, cornerRadius);
	        
			FSDefineShape3* upShape = path->defineShape(movie.newIdentifier());

			FSSolidLine* overLine = new FSSolidLine(lineWidth, lineColor);
			FSSolidFill* overFill = new FSSolidFill(overColor);

			path->setLineStyle(0, overLine);
			path->setFillStyle(0, overFill);
			path->rect(0, 0, buttonWidth, buttonHeight, cornerRadius);
	        
			FSDefineShape3* overShape = path->defineShape(movie.newIdentifier());

			delete shadowLine;
			delete shadowFill;
			delete upLine;
			delete upFill;
			delete overLine;
			delete overFill;

			/**************************
			*      First Button
			**************************/

			/*
			* This button is created using the FSDefineButton class which only defines actions
			* when the button is clicked. The FSDefineButton2 class supports a more complex model that
			* allows actions to be performed for a wide range of different button events.
			*/
			records.resize(9);
			actions.resize(1);
			events.resize(1);

			/*
			* Define the label. For the first button the label text is traced out as  a shape.
			* In the second half of this example the text object is used directly.
			*/
			FSDefineShape3* firstLabel = textGenerator->defineShape(movie.newIdentifier(), buttonLabel.c_str());

			/*
			* Realign the labels so they are displayed in the centre of the button. The ascent of the
			* text is used rather than the full height of the bounding rectangle.
			*/
			width = firstLabel->getBounds().getWidth();
			height = -firstLabel->getBounds().getMinY();

			/*
			* Create the FSButtonRecord objects that define the button's appearance. More than
			* one button record can be defined for a given state allowing for example button
			* definitions to be reused with only the label changing. The layer number controls
			* the order in which the shapes are displayed.
			*
			* Use one of the button shapes to define the active area where the button will respond
			* to events generated by the mouse.
			*/
			records[0] = FSButton(FSButton::Active, upShape->getIdentifier(), ++layer);

			/*
			* Define the appearance of the button in the up state. A shape giving a drop 
			* shadow effect is placed in the background and the label is in the foreground.
			*/
			records[1] = FSButton(FSButton::Up, shadow->getIdentifier(), ++layer, recess);
			records[2] = FSButton(FSButton::Up, upShape->getIdentifier(), ++layer);
			records[3] = FSButton(FSButton::Up, firstLabel->getIdentifier(), ++layer, 
				FSCoordTransform(-width/2, height/2));

			/*
			* The button appears the same when the mouse is over the active area. Only the 
			* colour changes to highlight the button. The drop shadow must still be drawn.     
			*/
			records[4] = FSButton(FSButton::Over, shadow->getIdentifier(), ++layer, recess);
			records[5] = FSButton(FSButton::Over, overShape->getIdentifier(), ++layer);
			records[6] = FSButton(FSButton::Over, firstLabel->getIdentifier(), ++layer, 
				FSCoordTransform(-width/2, height/2));

			/*
			* The button appears is recessed when it is clicked so the shape for the drop
			* shadow is not required.
			*/
			records[7] = FSButton(FSButton::Down, overShape->getIdentifier(), ++layer, recess);
			records[8] = FSButton(FSButton::Down, firstLabel->getIdentifier(), ++layer, 
				FSCoordTransform(-width/2 + xShadow, height/2 + yShadow));

			/*
			* When the button is clicked a web page will be loaded into the web browser. 
			* The browser must be running for this to work, the Flash Player won't launch 
			* the browser by itself.
			*/
			actions[0] = new FSGetUrl("http://www.flagstonesoftware.com", "");

			FSDefineButton* firstButton = new FSDefineButton(movie.newIdentifier(), records, actions);


			/**************************
			*     Second Button
			**************************/

			/*
			* The second button is created using the FSDefineButton2 class which provides a richer
			* model for responding to moue events. It is similar in appearance to the first except
			* that the label is created using FSDefineText2 objects rather than tracing out the
			* text as a shape.
			*/
			records.resize(9);
			actions.resize(1);
			events.resize(1);
	        
			/*
			* The FSDefineText2 object contains an FSCoordTransform which is used here to change 
			* the placement of the text so the label appears in the middle of the button.
			*/
			FSDefineText2* secondLabel = textGenerator->defineText(movie.newIdentifier(), buttonLabel.c_str());

			width = secondLabel->getBounds().getWidth();
			height = - secondLabel->getBounds().getMinY();

			/*
			* Add the button records that define the button's appearance
			*/
			records[0] = FSButton(FSButton::Active, upShape->getIdentifier(), ++layer);

			records[1] = FSButton(FSButton::Up, shadow->getIdentifier(), ++layer, recess);
			records[2] = FSButton(FSButton::Up, upShape->getIdentifier(), ++layer);
			records[3] = FSButton(FSButton::Up, secondLabel->getIdentifier(), ++layer, 
				FSCoordTransform(-width/2, height/2));

			records[4] = FSButton(FSButton::Over, shadow->getIdentifier(), ++layer, recess);
			records[5] = FSButton(FSButton::Over, overShape->getIdentifier(), ++layer);
			records[6] = FSButton(FSButton::Over, secondLabel->getIdentifier(), ++layer, 
				FSCoordTransform(-width/2, height/2));

			records[7] = FSButton(FSButton::Down, overShape->getIdentifier(), ++layer, recess);
			records[8] = FSButton(FSButton::Down, secondLabel->getIdentifier(), ++layer,
				FSCoordTransform(-width/2 + xShadow, height/2 + yShadow));

			/*
			* FSDefineButton2 contains a series of FSButtonEvent objects that define the actions
			* performed in response to different mouse events. As in the previous example a web
			* page will be loaded into the web browser when the button is clicked.
			*/   
			actions[0] = new FSGetUrl("http://www.flagstonesoftware.com", "");
	        
			events[0] = FSButtonEvent(FSButtonEvent::Release, actions);

			FSDefineButton2* secondButton = new FSDefineButton2(movie.newIdentifier(), 
				FSDefineButton2::Push, records, events);

			/*
			* Define a sound that plays when the button is clicked. The sound file contains a
			* stereo sound with 16-bit samples. The sound is compressed to the ADPCM format
			* with each sample taking 5 bits.
			*/
			FSDefineSound* sound = soundGenerator->defineSound(movie.newIdentifier());
			FSButtonSound* buttonSound = new FSButtonSound(secondButton->getIdentifier(), 
				FSButtonEvent::Release, FSSound(sound->getIdentifier(), 0));

			/***************************************************
			*     Put all the objects together in a movie
			***************************************************/

			/*
			 * Make the screen tall enough for a column of five BasicButtons.
			 */
			int screenWidth = buttonWidth + 2*screenMargin;
			int rowHeight = buttonHeight + screenMargin;
	            
			movie.setFrameSize(FSBounds(0, 0, screenWidth, rowHeight*2));
			movie.setFrameRate(1.0f);
			movie.add(new FSSetBackgroundColor(FSColorTable::lightblue()));

			// Add the font definition for the text labels
			movie.add(textGenerator->defineFont());

			// Add the shapes used to define the BasicButtons
			movie.add(shadow);
			movie.add(upShape);
			movie.add(overShape);

			// Add the first button and show it in the first row
			movie.add(firstLabel);
			movie.add(firstButton);

			movie.add(new FSPlaceObject2(firstButton->getIdentifier(), ++layer, 
				screenWidth/2, rowHeight/2));

			// Add the second button and show it in the second row
			movie.add(secondLabel);
			movie.add(secondButton);

			movie.add(sound);
			movie.add(buttonSound);

			movie.add(new FSPlaceObject2(secondButton->getIdentifier(), ++layer, 
				screenWidth/2, rowHeight + rowHeight/2));

			movie.add(new FSShowFrame());
	        
			saveMovie(movie, stringOption(options, "resultDir"), "BasicButtons.swf");
		}
		delete textGenerator;
		delete soundGenerator;
		delete path;
	}
};

#endif