/*
 *  StreamingSounds.cpp
 *  Transform Examples
 *
 *  Created by smackay on Tue Jun 02 2004.
 *  Copyright (c) 2001-2004 Flagstone Software Ltd. All rights reserved.
 *
 *  This code is distributed on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
 *  EXPRESS OR IMPLIED, AND Flagstone HEREBY DISCLAIMS ALL SUCH WARRANTIES, INCLUDING
 *  WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
 *  PURPOSE, AND NONINFRINGEMENT OF THIRD PARTY RIGHTS.
 */

#ifndef __StreamingSounds_h__
#define __StreamingSounds_h__ 1

/*
 * This example shows how the FSSoundConstructor class can be used to generate
 * the objects used to play streaming sounds in a movie.
 *
 * To run this example, type the following on a command line:
 *
 *     examples --example StreamingSounds --file image-file [--resultDir path]
 *
 * where
 *
 *     sound-file is the path to a file containing a WAVE or MP3 format sound.
 *
 *     resultDir is the directory where the Flash file generated by the example 
 *     is written to.
 *
 */
class StreamingSounds
{
public:
	StreamingSounds(map< string, string>& options)
	{
		string soundFile = stringOption(options, "file");

		FSSoundConstructor* soundGenerator = SoundConstructor();
		FSMovie movie;

		int status = TransformUtil::OK;

		if ((status = soundGenerator->setSoundFromFile(soundFile.c_str())) != TransformUtil::OK)
		{
			switch (status)
			{
				case TransformUtil::FileNotFound:
					cout << "Could not find sound file" << endl; break;
				case TransformUtil::ReadError:
					cout << "Could not read sound file" << endl; break;
				case TransformUtil::FormatError:
					cout << "Could not read sound file" << endl; break;
			}
		}

		if (status == TransformUtil::OK) 
		{
			float framesPerSecond = 12.0f;

			/*
			* Calculate the time it takes to play the sound and the number of frames 
			* this represents.
			*/
			float duration = ((float) soundGenerator->getSamplesPerChannel()) / 
				((float) soundGenerator->getSampleRate());
	            
			int numberOfFrames = (int) (duration * framesPerSecond);
	        
			/*
			* Calculate the number of decoded sound samples played for each frame and
			* the size, in bytes, of each block compressed sound data.
			*/
			int samplesPerBlock = soundGenerator->getSampleRate() / (int) framesPerSecond;
			int numberOfBlocks = soundGenerator->getSamplesPerChannel() / samplesPerBlock;

			/* 
			* Add all the objects together to create the movie.
			*/
			movie.setFrameSize(FSBounds(0, 0, 8000, 4000));
			movie.setFrameRate(framesPerSecond);
			movie.add(new FSSetBackgroundColor(FSColorTable::lightblue()));

			/* 
			* An FSSoundStreamHeader2 object defines the attributes of the streaming sound.
			*/
			movie.add(soundGenerator->streamHeader(samplesPerBlock));

			/* 
			* Add a streaming block for each frame so the sound is played as each frame 
			* is displayed.
			*/
			for (int i=0; i<numberOfBlocks; i++)
			{
				movie.add(soundGenerator->streamBlock(i, samplesPerBlock));
				movie.add(new FSShowFrame());
			}
	        
			saveMovie(movie, stringOption(options, "resultDir"), "StreamingSounds.swf");
		}
		delete soundGenerator;
	}
};

#endif