/***************************************************************************
                          kcorrview.cpp  -  description
                             -------------------
    begin                : Sun Feb 18 2001
    copyright            : (C) 2001 by M. Ritscher
    email                : unreachable@gmx.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "kcorrview.h"
#include "kcorrsetupdlg.h"
#include "correlationmeter.h"
#include "meterlabel.h"
#include "correlationscope.h"
#include "vumeter.h"
#include "waveview.h"

#include <kconfig.h>
#include <kapplication.h>
#include <klocale.h>

#include <qcolor.h>
#include <qlayout.h>
#include <qstring.h>
#include <qwhatsthis.h>

#include <kmessagebox.h>

KCorrView::KCorrView(QWidget *parent, const char *name ) : QWidget(parent,name)
{
	_deviceOpen = false;
	setMinimumSize(262, 342);
	setSizePolicy( QSizePolicy( QSizePolicy::Minimum, QSizePolicy::Minimum ) );
	
	QBoxLayout* vBox= new QVBoxLayout(this);
	
	timer = 0;
	ssystem = ssystem_asound09;
	capture_handle = NULL;

	// for Debugging
	generateSignal();

/* reading the configuration if stored */
	KConfig* conf = kapp->config();
	conf->setGroup("Device_Settings");


	QString tempSSystem = conf->readEntry("Sound_System");
	if(tempSSystem.isEmpty() || tempSSystem == "asound09"){
		ssystem = ssystem_asound09;
	}else if(tempSSystem == "OSS"){
		ssystem = ssystem_asound09;
		qWarning("OSS not implemented any longer - switching to ALSA!!");
	}else{
		qDebug("Setup - Constructor: No other sound system implemented");
	}
	
  sDevice = conf->readEntry("DSP_Device");
  if (sDevice.isEmpty()){
		if(ssystem == ssystem_asound09){
			sDevice = "plughw:0,0";
		}else{
			qDebug("Setup - Constructor: No other sound system implemented");
		}
	}

  uSampleRate = conf->readUnsignedNumEntry("SamplingRate");
  if (uSampleRate == 0)
    uSampleRate = 22050;

  uIntervall = conf->readUnsignedNumEntry("Interval");
  if (uIntervall == 0)
    uIntervall = 10;

  uTrace = conf->readUnsignedNumEntry("Trace_Length");
  if (uTrace == 0)
    uTrace = 100;

  uAverage = conf->readUnsignedNumEntry("CorrAverageTime");
  if (uAverage == 0)
    uAverage = 100;


	/*setting up scope*/
	
	QColor* tmpColour;
	
  cScope = new CorrelationScope(this, "scope");
  vBox->addWidget(cScope);
	QWhatsThis* what = new QWhatsThis(this);
	what->add(cScope,i18n("The scope shows the correlation between \
												the left and the right audio channel \
												with the aid of Lissajus figures."));
	

	conf->setGroup("Scope_Colours");
	
	if(conf->hasKey("RayColour"))
		cScope->setRayColour(conf->readColorEntry("RayColour",(QColor*) tmpColour));
	else
		cScope->setRayColour(QColor(46,211,5));

	if(conf->hasKey("GridColour"))
		cScope->setGridColour(conf->readColorEntry("GridColour",(QColor*) tmpColour));
	else
	  cScope->setGridColour(QColor(128,128,128));

  if(conf->hasKey("BackGroundColour"))
		cScope->setBackgroundColour(conf->readColorEntry("BackGroundColour",(QColor*) tmpColour));
	else
		cScope->setBackgroundColour(QColor(0,0,0));	

	
	/* setting up correlation meter */
	
	cMeter = new CorrelationMeter(this,"cmeter");
	vBox->addWidget(cMeter);
	cMeter->setFixedHeight(20);
	what->add(cMeter,i18n(" The Meter displays the correlation between \
												 the left and the right audio channel \
												 1 stands for total correlation."));
	
	/* retrieving colours for correlation meter */
	conf->setGroup("Corr_Meter_Colours");

	if(conf->hasKey("PointerColour"))
		cMeter->setPointerPenColour(conf->readColorEntry("PointerColour",(QColor*) tmpColour));
	else
		cMeter->setPointerPenColour(QColor(255, 0, 0));
	
	if(conf->hasKey("GridColour"))
		cMeter->setGridPenColour(conf->readColorEntry("GridColour",(QColor*) tmpColour));
	else
		cMeter->setGridPenColour(QColor(128, 128, 128));

  if(conf->hasKey("BackGroundColour"))
		cMeter->setBackgroundColour(conf->readColorEntry("BackGroundColour",(QColor*) tmpColour));
	else
		cMeter->setBackgroundColour(QColor(255,254,231 ));

	//Creating the labeling scale for cMeter
  cMeterScale = new MeterLabel(this,"scale");
	vBox->addWidget(cMeterScale);
	
	//setting up VU Meter
	cVUMeter = new VUMeter(this,"vumeter");
	vBox->addWidget(cVUMeter);
	what->add(cVUMeter,i18n(" The VU Meter displays the level \
												   of the sampled audio input in dB"));
	
	conf->setGroup("VU_Meter");
	unsigned int d;
  if(conf->hasKey("PeakDelay")){
	  d = conf->readUnsignedNumEntry("PeakDelay");
	}else{
		d = 500;
	}
	( d>uIntervall)? d=d/uIntervall : d=0;
	cVUMeter->setDelay(d);

	// WaveView - Easter Egg
	conf->setGroup("App_Settings");
	QString tempWave = conf->readEntry("WaveView");
	if(tempWave == "true"){
		cWaveView = new WaveView(this,"waveview");
		vBox->addWidget(cWaveView);
		setMinimumSize(262, 542);
	}else{
		cWaveView = NULL;
	}

	//connecting signals
	connect(cMeter,SIGNAL(rbPressed(QPoint)), this, SLOT(rbPopup(QPoint)));
	connect(cScope,SIGNAL(rbPressed(QPoint)), this, SLOT(rbPopup(QPoint)));
	connect(cMeterScale,SIGNAL(rbPressed(QPoint)), this, SLOT(rbPopup(QPoint)));
	connect(cVUMeter,SIGNAL(rbPressed(QPoint)), this, SLOT(rbPopup(QPoint)));

}


KCorrView::~KCorrView(){
	stop();
}

/** Opens a Dialog which lets the user enter the Options */
void KCorrView::showSetupDlg(){

// KMessageBox::information(this,"here will be shown the setup dialog");
	setupDlg = new KCorrSetupDlg(this, "setup");

	SSampleState mSState;

  mSState.uSSystem = ssystem;
	mSState.sDevice = sDevice;
	mSState.uSampleRate = uSampleRate;
	mSState.uIntervall = uIntervall;
	mSState.uTrace = uTrace;
	mSState.uAverage = uAverage;

	setupDlg->setSampleState(mSState);
	
	SColourState mCState;
	
	mCState.scopeBgColour = cScope->getBackgroundColour();
	mCState.scopeRayColour = cScope->getRayColour();
	mCState.scopeGridColour = cScope->getGridColour();
	mCState.meterPointerColour = cMeter->getRayColour();
	mCState.meterBgColour = cMeter->getBackGroundColour();
	mCState.meterGridColour = cMeter->getGridColour();
		
	setupDlg->setColourState(mCState);

	connect(setupDlg, SIGNAL(sampleDeviceChoice(SSampleState)),
						  this, SLOT(slotDeviceSettings(SSampleState)));
						
	connect(setupDlg, SIGNAL(colourChoice(SColourState)),
						  this, SLOT(slotColourSettings(SColourState)));

	connect(setupDlg, SIGNAL(VUMeterChoice(SVUMeterState)),
						  this, SLOT(slotVUSettings(SVUMeterState)));
						
						
	setupDlg->show();

}


void KCorrView::closeEvent(QCloseEvent* qcloseevent) {
//
  qcloseevent->accept();
}


/** updates sample device settings */
void KCorrView::slotDeviceSettings(SSampleState state){

	stop();
	
	uSampleRate = state.uSampleRate;

	// if the user selected a different
	// device or sound system
	if((state.sDevice.compare(sDevice))!=0 || 	ssystem != state.uSSystem){
		ssystem = state.uSSystem;
		sDevice = state.sDevice;
	}

	uTrace = state.uTrace;
	cScope->setTrace((uTrace * uSampleRate) / 1000);
	
	uAverage = state.uAverage;
  cMeter->setAverage((uAverage * uSampleRate) / 1000);
	
	setIntervall(state.uIntervall);
	start();
		
	/* saving adjusted configuration to rc file*/
	
	KConfig* conf = kapp->config();
	conf->setGroup("Device_Settings");

	conf->writeEntry("Sound_System",(ssystem == ssystem_asound09)?"asound09":"Other");
	conf->writeEntry("DSP_Device",sDevice);
	conf->writeEntry("SamplingRate",uSampleRate);
  conf->writeEntry("Interval", uIntervall);
	conf->writeEntry("Trace_Length", uTrace);
  conf->writeEntry("CorrAverageTime", uAverage);
  conf->sync();
}

/** sets for- and background colours for scope and meter */
void KCorrView::slotColourSettings(SColourState state){

	cScope->setRayColour(state.scopeRayColour);
  cScope->setGridColour(state.scopeGridColour);
  cScope->setBackgroundColour(state.scopeBgColour);	

 	cMeter->setPointerPenColour(state.meterPointerColour);
	cMeter->setGridPenColour(state.meterGridColour);
	cMeter->setBackgroundColour(state.meterBgColour);


		/* saving adjusted configuration to rc file*/
	
	KConfig* conf = kapp->config();
	conf->setGroup("Scope_Colours");
	conf->writeEntry("RayColour",state.scopeRayColour);
  conf->writeEntry("BackGroundColour", state.scopeBgColour);
  conf->writeEntry("GridColour", state.scopeGridColour);
	
	conf->setGroup("Corr_Meter_Colours");
	conf->writeEntry("PointerColour",state.meterPointerColour);
  conf->writeEntry("BackGroundColour", state.meterBgColour);
	conf->writeEntry("GridColour", state.meterGridColour);

  conf->sync();
}

/** sets settings for VU Meter */
void KCorrView::slotVUSettings(SVUMeterState state)
{
	cVUMeter->setDelay(state.peakDelay/uIntervall);
}


/** Opening and setting up DSP device  */
bool KCorrView::openDevice(){

	int rc = 0;
	
	if(ssystem == ssystem_asound09){
		rc = openALSADevice();
		//rc = openNewALSADev();
	}else{
		// older versions of kcorrmeter used OSS but 
		// that was broken - contact me, if you need the
		// code
		qDebug(" no other sound system implmented " ); 
	}
  if (!rc) {
		cScope->setTrace((uTrace * uSampleRate) / 1000);
		cMeter->setAverage((uAverage * uSampleRate) / 1000);
		uCaptureCount = (uSampleRate << 2 ) * uIntervall / 1000;
		_deviceOpen = true;
	}

//	if(uCaptureCount < 400) uCaptureCount = 400;
//	qDebug("captureCount set to: %i", uCaptureCount);

	return _deviceOpen;
}

/** Shutting down DSP device */
void KCorrView::closeDevice(){

	if(ssystem == ssystem_asound09){
		if(capture_handle != NULL){
			snd_pcm_close (capture_handle);
			capture_handle = NULL;
			_deviceOpen = false;
		}
	}else{
		// older versions of kcorrmeter used OSS but 
		// that was broken - contact me, if you need the
		// code
		qDebug(" no other sound system implmented " ); 
	}
}

/** No descriptions */
void KCorrView::start(){

	if(openDevice()){
		emit started(true);
	}else{
		emit started(false);
		return;
	}
	timer = startTimer(uIntervall);
}
/** No descriptions */
void KCorrView::stop(){
	emit started(false);
	if(timer){
		killTimer(timer);
		timer = 0;
	}	
	closeDevice();
}


/** adjusting the timer interall */
void KCorrView::setIntervall(unsigned int i){

	/** set new interval for timer */
  uIntervall = i;

  /** if running, reset timer */
	if(timer){
	  killTimer(timer);
	  timer = startTimer(uIntervall);
	}
}

/** Retrieving new sample and refreshing screen */
void KCorrView::timerEvent(QTimerEvent* tevt){

  if (isVisible()) {
		static int iDataStart = 0;
		if(ssystem == ssystem_asound09){
			/****************************************
			 *    ALSA
			 ****************************************/
			static int counter = 0;
			
//			unsigned int capCount = uCaptureCount << 2;
			long rc = snd_pcm_readi (capture_handle, sBuf, uCaptureCount);
			if(rc < 0){
				qWarning("<<< buffer overrun | counter: %i  return_code: %i => %s >>>", counter, (int)rc, snd_strerror (rc) );
//				counter=0;
				if(rc == -32)
					snd_pcm_prepare(capture_handle);
			}else{
				if((unsigned int)rc > uCaptureCount){
//					qDebug("not as many captures (%i) as specified (%i)", (unsigned int)rc, uCaptureCount);
				}
			}
			/*
			counter++;
			if((counter % 10) == 0){
				std::cout << "length: " << rc << std::endl;
			}*/

//			long rc = 30;
//			buffer_size= 65536;
//			qDebug("buff_size: %i", buffer_size);

			if(rc > 0){
				int iDataEnd = (iDataStart + rc) % buffer_size;
				if(cVUMeter){
					cVUMeter->newData(sBuf, iDataStart, iDataEnd, buffer_size);
				}
				if(cScope){
					cScope->newData(sBuf, iDataStart, iDataEnd, buffer_size);
				} 
				if(cWaveView){
					cWaveView->newData(sBuf, iDataStart, iDataEnd, buffer_size);
				} 
				if(cMeter){
						cMeter->newData(sBuf, iDataStart, iDataEnd, buffer_size);
//					cMeter->newData(sBuf, 0, rc, 127);
				}
//				iDataStart = (iDataEnd + 1) % buffer_size;
			}
			
		}else{
			qDebug("timer Event - no other sound system implemented");
		}
	}
}

/** overiding mouse events */
void KCorrView::mousePressEvent(QMouseEvent *e){

	emit rbPressed(e->globalPos());
}

/** chaining for children - doesn't like that but how to make it better?? */
void KCorrView::rbPopup(QPoint p){

	emit rbPressed(p);
}

/** stops timer and closes device  */
void KCorrView::halt(){

	stop();
}

/** opens device and starts timer */
void KCorrView::resume(){

	start();
}

/** Opens the capturing device ALSA specific */
int KCorrView::openALSADevice(){

	int rc = 0;
	if( _deviceOpen ) return rc;

	snd_pcm_hw_params_t *hw_params;
	//snd_pcm_access_t access = ;
	snd_pcm_stream_t stream_capture = SND_PCM_STREAM_CAPTURE;

	rc = snd_pcm_open (&capture_handle, sDevice.latin1(), stream_capture, 0);
//	rc = snd_pcm_open(&pcm_handle, sDevice.latin1(), stream_capture, 0);
	if ( rc < 0){
		qFatal("cannot open audio device ( %s ): %s", sDevice.latin1(), snd_strerror (rc));
		return rc;
	}else{
		qDebug("Success, opening device %s", sDevice.latin1());
	}
/*	if ((rc = snd_pcm_hw_params_malloc (&hw_params)) < 0) {
		std::cerr << "cannot allocate hardware parameter structure" <<  snd_strerror (rc);
		return rc;
	} */

	snd_pcm_hw_params_alloca (&hw_params);
	
	if ((rc = snd_pcm_hw_params_any (capture_handle, hw_params)) < 0) {
		qFatal("cannot initialize hardware parameter structure: %s", snd_strerror (rc));
		return rc;
	}
	
	if ((rc = snd_pcm_hw_params_set_access (capture_handle, hw_params, SND_PCM_ACCESS_RW_INTERLEAVED)) < 0) {
		qFatal("cannot set access type: %s", snd_strerror (rc));
		return rc;
	}
	if ((rc = snd_pcm_hw_params_set_format (capture_handle, hw_params, SND_PCM_FORMAT_U8)) < 0) {
		qFatal("cannot set sample format %s", snd_strerror (rc));
		return rc;
	}
	
	if ((rc = snd_pcm_hw_params_set_channels(capture_handle, hw_params, 2))  < 0) {
		qFatal("Could not set your hardware to stereo: %s", snd_strerror(rc));
		return rc;
	}

	unsigned int rrate = uSampleRate;
	if((rc = snd_pcm_hw_params_set_rate_near (capture_handle, hw_params, &rrate, 0)) < 0){
		qWarning("Failed to set sample rate (%i):  %s", uSampleRate, snd_strerror (rc));
	}
	if(rrate !=  uSampleRate) {
		qWarning("Sample rate %i is'nt supported by your hardware. Using %i Hz instaed.",uSampleRate, rrate);
		uSampleRate = rrate; //remember used sample frequency for calculating number of capture points
	}
	qWarning("snd_pcm_hw_params_set_rate_near returned with: %i ", rc);


	if ((rc = snd_pcm_hw_params_set_channels (capture_handle, hw_params, 2)) < 0) {
		qFatal("cannot set channel count - %s", snd_strerror (rc));
		return rc;
	}
	if ((rc = snd_pcm_hw_params (capture_handle, hw_params)) < 0) {
		qFatal("cannot set parameters - %s ", snd_strerror (rc));
		return rc;
	}
//	snd_pcm_hw_params_free (hw_params);
	if ((rc = snd_pcm_prepare (capture_handle)) < 0) {
		qFatal( "cannot prepare audio interface for use: %s ", snd_strerror (rc));
		return rc;
	}
	buffer_size = 65536;

  qWarning( "successfully set up ALSA 0.9 device" );

	qDebug("open Alsa Device: %i ",rc);
  return rc;
}

void KCorrView::generateSignal()
{
//this function is solely for debugging purposes

//  frequence genrating =========================
// 	float r,l;
// 
// 	int s[62];
// 	float rad = 0.0;
// 	for (int x = 0; x < 62; x++){
// 		s[x] = (int) ( (sin( rad) + 1.0) * 128.0 );
// 		rad = rad + 0.1;
// 		if(s[x] > 255) qDebug("x: %i, s[x]: %i",x,s[x]);
// 	}
// 
// 
// 	for (unsigned int p=0; p <= 32768; p++){
// //		r = (sinf( (float)p) + 1.0) * 127.0;
// //	circle
// //		l = (cosf( (float)p + 180) + 1.0) * 127.0;
// 
// 		int index = p % 62;
// 		sBuf[p*2]   =  s[index];
// 		sBuf[p*2+1] = 255 - s[index];
// 	if(sBuf[p*2] > 255) qDebug("rr: %i -> %i",p*2,sBuf[p*2]);
// 	if(sBuf[p*2+1] > 255 || sBuf[p*2+1] < 0) qDebug( "l: %i -> %i == %i",p*2+1,sBuf[p*2+1], s[index]);
// 
// 	}
// 
// 	for( int i = 0; i< 62; i++){
// 		int index = i % 62;
// 		iBuf[i*2] = s[index] - 127;
// 		iBuf[i*2+1] = 0 - (s[index] - 127);
// 
// 		if(abs(iBuf[i*2])   > 127) qDebug( "r: %i -> %i",i*2, iBuf[i*2]);
// 		if(abs(iBuf[i*2+1]) > 127) qDebug( "l: %i -> %i == %i",i*2+1,iBuf[i*2+1], s[index]);
// 	}
//  frequence genrating ========================= END ==

}
