// LiVES - SDL playback engine
// (c) G. Finch 2003 - 2006 <salsaman@xs4all.nl>
// released under the GNU GPL 2.0 or higher
// see file COPYING or www.gnu.org for details

#include <SDL.h>
#include <gtk/gtk.h>
#include <gmodule.h>
#include <string.h>
#include <stdlib.h>

// all playback modules need to implement these functions


// these can be called at startup
G_MODULE_EXPORT const gchar *
g_module_check_init(GModule *GM);

G_MODULE_EXPORT const gchar *
version (void);

// set the palette used (currently only RGB24 is supported)
G_MODULE_EXPORT gboolean 
set_palette (const gchar *p_name);
#define DEFAULT_PALETTE "RGB24"


// this will be called after the palette is set
G_MODULE_EXPORT guint 
get_capabilities (void);

G_MODULE_EXPORT void 
set_keyfun (void (*fn)(gboolean, guint16, guint16));

// these are called during playback

// ready the screen to play on
G_MODULE_EXPORT gboolean 
init_screen (int width, int height, gboolean fullscreen);

// display one frame, adding effects if you like,
// and resizing it to screen size if possible
// if return_data is non-NULL, you should either fill it with the effected,
// unresized data or set it back to NULL if you can't
G_MODULE_EXPORT gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data);

// free the frame if necessary, pixel_data will be freed by the caller after
G_MODULE_EXPORT void 
free_frame (void);


// fn to send keycodes
G_MODULE_EXPORT gboolean 
send_keycodes (void);


// destroy the screen, return mouse to original posn., allow the GUI to take over
G_MODULE_EXPORT void 
exit_screen (gint16 mouse_x, gint16 mouse_y);



// this is called at the end
G_MODULE_EXPORT void
g_module_unload(GModule *GM);


typedef struct {
  gchar name[32];
  guint bits;
} _palette;

static _palette *palette;

static gchar plugin_version[64];
static gchar error[256];
static void (*key_fn) (gboolean down, guint16 unicode, guint16 keymod);
static gboolean (*render_fn)(int hsize, int vsize, void *pixel_data, void *return_data);
gboolean 
render_frame_rgb (int hsize, int vsize, void *pixel_data, void *return_data);
gboolean 
render_frame_uyvy (int hsize, int vsize, void *pixel_data, void *return_data);
gboolean 
render_frame_unknown (int hsize, int vsize, void *pixel_data, void *return_data);
#define MOD_NEEDS_TRANSLATION 1<<15

/////////////////////////////////////////////
// SDL specific stuff


static  SDL_Surface *screen;
static  SDL_Surface *RGBimage;
static  SDL_Overlay *overlay;
static  int ov_hsize;
static  int ov_vsize;
static  SDL_Rect *rect;
static  SDL_Event event;
static  SDLMod mod;




//////////////////////////////////////////////


const gchar *
g_module_check_init(GModule *GM) {
  if (getenv ("HAVE_SDL")==NULL&&g_find_program_in_path ("sdl-config")==NULL) {
    g_snprintf (error,256,"\n\nUnable to find sdl-config in your path.\nPlease make sure you have SDL installed correctly to use this plugin.\nYou can override this with 'export HAVE_SDL=1'\n");
    return error;
  }

  key_fn=NULL;
  render_fn=&render_frame_unknown;

  RGBimage=NULL;
  overlay=NULL;
  ov_vsize=ov_hsize=0;

  palette=(_palette *)g_malloc (sizeof(_palette));
  palette->bits=0;

  rect=(SDL_Rect *)g_malloc (sizeof(SDL_Rect));

  return NULL;
}



const gchar *
version (void) {
  g_snprintf (plugin_version,64,"LiVES SDL playback engine version 1.0");
  return plugin_version;
}


void 
set_keyfun (void (*fn)(gboolean, guint16, guint16)) {
  key_fn=fn;
}



guint 
get_capabilities (void) {
  // this function will be called after set_palette
  // returns a bitmap
  // bit 0 - 1=can resize to screen size
  // bit 1 - 1=can return return_data
  // bit 2 - 1=uses fixed framerates // TODO - get a list
  // bit 3 - 1=no display (for streaming plugins)
  // bits 4-64 unused 
  if (!palette->bits) {
    set_palette (DEFAULT_PALETTE);
  }

  if (!strncmp (palette->name,"RGB",3)) {
    return 0;
  }
  return 1;
}


gboolean 
set_palette (const gchar *p_name) {
  if (!strcmp (p_name,"RGB24")) {
    g_snprintf (palette->name,32,"RGB24");
    palette->bits=24;
    render_fn=&render_frame_rgb;
    return TRUE;
  }
  if (!strcmp (p_name,"UYVY")) {
    g_snprintf (palette->name,32,"UYVY");
    palette->bits=24;  // hardware bits
    render_fn=&render_frame_uyvy;
    return TRUE;
  }
  // invalid palette
  return FALSE;
}




gboolean 
init_screen (int width, int height, gboolean fullscreen) {
  if (!palette->bits) {
    set_palette (DEFAULT_PALETTE);
  }

  setenv ("SDL_VIDEO_YUV_DIRECT", "1", 1);
  setenv ("SDL_VIDEO_HWACCEL", "1", 1);

  if((SDL_Init (SDL_INIT_VIDEO)==-1)) {
    g_print ("SDL player : Could not initialize SDL: %s.\n", SDL_GetError());
    return FALSE;
  }

  SDL_ShowCursor (FALSE);
  screen = SDL_SetVideoMode (width, height, palette->bits, SDL_HWSURFACE | SDL_ASYNCBLIT | SDL_DOUBLEBUF | (fullscreen==TRUE?SDL_FULLSCREEN:0) | SDL_HWACCEL | SDL_ANYFORMAT);
  if ( screen == NULL ) {
    g_print ("SDL player : Couldn't set %dx%dx24 video mode: %s\n", width, height,
	     SDL_GetError());
    // do we need SDL_ShowCursor/SDL_quit here ?
    return FALSE;
  }

  /* Enable Unicode translation */
  SDL_EnableUNICODE ( 1 );

  // if palette is RGB, create RGB surface the same size as the screen
  if (!strncmp (palette->name,"RGB24",5)) {
    //RGBimage=SDL_CreateRGBSurface (SDL_HWSURFACE, width, height, 24, 0x0000FF, 0x00FF00, 0xFF0000, 0);
    return TRUE;
  }
  if (!strcmp (palette->name,"UYVY")) {
    rect->x=rect->y=0;
    rect->h=height;
    rect->w=width;
    return TRUE;
  }
  return FALSE;
}


gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data) {
  // call the function which was set in set_palette
  return render_fn (hsize,vsize,pixel_data,return_data);
}

gboolean 
render_frame_rgb (int hsize, int vsize, void *pixel_data, void *return_data) {
  // hsize and vsize are in pixels (n-byte)
  RGBimage = SDL_CreateRGBSurfaceFrom(pixel_data,
				      hsize, vsize, 24,
				      hsize*4,
				      0x0000FF, 0x00FF00, 0xFF0000, 0x00);
  if (RGBimage == NULL)
    {
      fprintf(stderr,"Can't create: %s\n", SDL_GetError());
      return FALSE;
    }
  SDL_BlitSurface(RGBimage, NULL, screen, NULL);
  SDL_FreeSurface(RGBimage);
  SDL_UpdateRect(screen, 0, 0, 0, 0);
  return TRUE;
}

gboolean 
render_frame_uyvy (int hsize, int vsize, void *pixel_data, void *return_data) {
  if ((ov_hsize!=hsize||ov_vsize!=vsize)&&(overlay!=NULL)) {
    SDL_FreeYUVOverlay(overlay);
    overlay=NULL;
  }
  if (overlay==NULL) {
    overlay=SDL_CreateYUVOverlay (hsize, vsize, SDL_UYVY_OVERLAY, screen);
    ov_hsize=hsize;
    ov_vsize=vsize;
  }
  SDL_LockYUVOverlay (overlay);
  memcpy (overlay->pixels[0],pixel_data,hsize*vsize*2);
  SDL_UnlockYUVOverlay (overlay);
  SDL_DisplayYUVOverlay (overlay,rect);
  return TRUE;
}


gboolean 
render_frame_unknown (int hsize, int vsize, void *pixel_data, void *return_data) {
  return FALSE;
}

void 
free_frame (void) {
  return;
}


void 
exit_screen (gint16 mouse_x, gint16 mouse_y) {
  if (!strncmp (palette->name,"RGB24",5)) {
    if (RGBimage!=NULL) {
      SDL_FreeSurface (RGBimage);
      RGBimage=NULL;
    }
  }
  else if (!strcmp (palette->name,"UYVY")) {
    if (overlay!=NULL) {
      SDL_FreeYUVOverlay(overlay);
      overlay=NULL;
    }
  }
  if (mouse_x>=0&&mouse_y>=0) {
    SDL_ShowCursor (TRUE);
    SDL_WarpMouse ((guint16)mouse_x, (guint16)mouse_y);
  }
  SDL_Quit();
}


void 
g_module_unload(GModule *GM) {
  g_free (palette);
  g_free (rect);
}




gboolean  
send_keycodes (void) {
  // poll for keyboard events, pass them back to the caller
  guint16 mod_mask,scancode=0;

  if (key_fn==NULL) return FALSE;

  while(SDL_PollEvent (&event)) {
    mod_mask=0;
    if (event.type==SDL_KEYDOWN||event.type==SDL_KEYUP) {
      mod=event.key.keysym.mod;

      if (mod&KMOD_CTRL) {
	mod_mask|=GDK_CONTROL_MASK;
      }
      if (mod&KMOD_ALT) {
	mod_mask|=GDK_MOD1_MASK;
      }
      if (event.type==SDL_KEYDOWN) {
	if (!mod_mask) {
	  scancode=event.key.keysym.unicode;
	}
	if (!scancode) {
	  scancode=(guint16)event.key.keysym.scancode;
	  mod_mask|=MOD_NEEDS_TRANSLATION;
	}
	key_fn (TRUE,scancode,mod_mask);
      }
      
      else {
	// key up - no unicode :-(
	key_fn (FALSE,(guint16)event.key.keysym.scancode,(mod_mask|MOD_NEEDS_TRANSLATION));
      }
    }
  }
  return TRUE;
}

