// LiVES - vjack playback engine
// (c) G. Finch 2006 <salsaman@xs4all.nl>
// released under the GNU GPL 2.0 or higher
// see file COPYING or www.gnu.org for details

#include <jack/jack.h>
#include <jack/video.h>
#include <jack/ringbuffer.h>

#include <gtk/gtk.h>
#include <gmodule.h>
#include <string.h>
#include <stdlib.h>

// all playback modules need to implement these functions


// these can be called at startup
G_MODULE_EXPORT const gchar *
g_module_check_init(GModule *GM);

G_MODULE_EXPORT const gchar *
version (void);

G_MODULE_EXPORT gchar *
get_fps_list (void);

// set the palette used (currently only RGBA32 is supported)
G_MODULE_EXPORT gboolean 
set_palette (const gchar *p_name);
#define DEFAULT_PALETTE "RGBA32"


// this will be called after the palette is set
G_MODULE_EXPORT guint 
get_capabilities (void);

G_MODULE_EXPORT void 
set_keyfun (void (*fn)(gboolean, guint16, guint16));

// these are called during playback

// ready the screen to play on
G_MODULE_EXPORT gboolean 
init_screen (int width, int height, gboolean fullscreen);

// display one frame, adding effects if you like,
// and resizing it to screen size if possible
// if return_data is non-NULL, you should either fill it with the effected,
// unresized data or set it back to NULL if you can't
G_MODULE_EXPORT gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data);

// free the frame if necessary, pixel_data will be freed by the caller after
G_MODULE_EXPORT void 
free_frame (void);


// fn to send keycodes
G_MODULE_EXPORT gboolean 
send_keycodes (void);


// destroy the screen, return mouse to original posn., allow the GUI to take over
G_MODULE_EXPORT void 
exit_screen (gint16 mouse_x, gint16 mouse_y);


// this is called at the end
G_MODULE_EXPORT void
g_module_unload(GModule *GM);


typedef struct {
  gchar name[32];
  guint bits;
} _palette;

static _palette *palette;

static gchar plugin_version[64];
static void (*key_fn) (gboolean down, guint16 unicode, guint16 keymod);
//gboolean render_frame_rgb (int hsize, int vsize, void *pixel_data, void *return_data);

void 
set_keyfun (void (*fn)(gboolean, guint16, guint16)) {
}


gboolean  
send_keycodes (void) {
  return TRUE;
}







/////////////////////////////////////////////

static jack_ringbuffer_t *rb;
static size_t frame_size;
static jack_port_t *output_port;
static jack_client_t *client;

int server_process (jack_nframes_t nframes, void *arg) {
  // jack calls this
  uint8_t* in;

  if (rb==NULL) return 0;

  in = jack_port_get_buffer (output_port, nframes);
  while ( jack_ringbuffer_read_space(rb) >= frame_size) {
    jack_ringbuffer_read (rb, (void *) in, frame_size);
  }
  return 0;
}


//////////////////////////////////////////////
gchar *get_fps_list(void) {
  gdouble fps=(gdouble)jack_get_sample_rate(client);
  gchar *fps_list=g_strdup_printf("%.8f",fps);
  return fps_list;
}



const gchar *
g_module_check_init(GModule *GM) {
  const char *client_name="LiVES";
  jack_options_t options = JackNullOption;
  jack_status_t status;
  const char *server_name=NULL; // set to "video0" if running with audio on "default"

  key_fn=NULL;

  palette=(_palette *)g_malloc (sizeof(_palette));
  palette->bits=0;

  // connect to vjack
  client = jack_client_open (client_name, options, &status, server_name);
  if (client == NULL) {
    fprintf (stderr, "jack_client_open() failed, "
	     "status = 0x%2.0x\n", status);
    if (status & JackServerFailed) {
      fprintf (stderr, "Unable to connect to JACK server\n");
    }
    return "unable to connect";
  }
  if (status & JackServerStarted) {
    fprintf (stderr, "JACK server started\n");
  }
  if (status & JackNameNotUnique) {
    client_name = jack_get_client_name(client);
    fprintf (stderr, "unique name `%s' assigned\n", client_name);
  }

  fprintf (stderr,"engine sample rate: %" PRIu32 "\n",
	   jack_get_sample_rate (client));

  output_port = jack_port_register (client, 
				    "video_out",
				    JACK_DEFAULT_VIDEO_TYPE,
				    JackPortIsOutput, 
				    0);
  
  if (output_port==NULL) {
    fprintf(stderr, "no more JACK ports available\n");
    return "no jack ports available";
  }

  rb = NULL; 

  // set process callback and start
  jack_set_process_callback (client, server_process, NULL);  

  // and start the processing
  if (jack_activate (client)) {
    fprintf (stderr, "cannot activate client\n");
    return FALSE;
  }


  return NULL;
}



const gchar *
version (void) {
  g_snprintf (plugin_version,64,"LiVES vjack output 1.0");
  return plugin_version;
}



guint 
get_capabilities (void) {
  // this function will be called after set_palette
  // returns a bitmap
  // bit 0 - 1=can resize to screen size
  // bit 1 - 1=can return return_data
  // bit 2 - 1=fixed fps
  // bit 3 - 1=no local display
  // bits 4-64 unused 
  set_palette (DEFAULT_PALETTE);

  return 12;
}


gboolean 
set_palette (const gchar *p_name) {
  if (!strcmp (p_name,"RGBA32")) {
    g_snprintf (palette->name,32,"RGBA32");
    palette->bits=32;
    return TRUE;
  }
  // invalid palette
  return FALSE;
}





gboolean 
init_screen (int width, int height, gboolean fullscreen) {
  jack_video_set_width_and_height(client,output_port,width,height);
  frame_size=width*height*4;
  rb = jack_ringbuffer_create (2* frame_size);
  return TRUE;
}



gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data) {
  // hsize and vsize are in pixels (n-byte)

  jack_ringbuffer_write (rb, pixel_data, frame_size);
  return TRUE;
}


void 
free_frame (void) {
  return;
}


void 
exit_screen (gint16 mouse_x, gint16 mouse_y) {
  if (jack_deactivate (client)) {
    fprintf (stderr, "cannot deactivate client\n");
  }

  if (rb!=NULL) jack_ringbuffer_free(rb);
  rb=NULL;

}


void 
g_module_unload(GModule *GM) {
  // disconnect from vjack
  jack_client_close (client);
  fprintf(stderr, "jack port unregistered\n");
}




