// LiVES - yuv4mpeg stream engine
// (c) G. Finch 2004 <salsaman@xs4all.nl>
// released under the GNU GPL 2.0 or higher
// see file COPYING or www.gnu.org for details

#include <sys/types.h>

#include <gtk/gtk.h>
#include <gmodule.h>
#include <string.h>
#include <stdlib.h>
#include <mjpegtools/yuv4mpeg.h>
// all playback modules need to implement these functions


// these can be called at startup
G_MODULE_EXPORT const gchar *
g_module_check_init(GModule *GM);

G_MODULE_EXPORT const gchar *
version (void);

G_MODULE_EXPORT gboolean 
set_palette (const gchar *p_name);
#define DEFAULT_PALETTE "UYVY"


// this will be called after the palette is set
G_MODULE_EXPORT guint 
get_capabilities (void);

G_MODULE_EXPORT void 
set_keyfun (void (*fn)(gboolean, guint16, guint16));

// these are called during playback

// ready the screen to play on
G_MODULE_EXPORT gboolean 
init_screen (int width, int height, gboolean fullscreen);

// display one frame, adding effects if you like,
// and resizing it to screen size if possible
// if return_data is non-NULL, you should either fill it with the effected,
// unresized data or set it back to NULL if you can't
G_MODULE_EXPORT gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data);

// free the frame if necessary, pixel_data will be freed by the caller after
G_MODULE_EXPORT void 
free_frame (void);



// destroy the screen, return mouse to original posn., allow the GUI to take over
G_MODULE_EXPORT void 
exit_screen (gint16 mouse_x, gint16 mouse_y);



// this is called at the end
G_MODULE_EXPORT void
g_module_unload(GModule *GM);


// extra functions for fixed fps
G_MODULE_EXPORT gchar *
get_fps_list (void);

G_MODULE_EXPORT gboolean 
set_fps (gdouble fps);



typedef struct {
  gchar name[32];
  guint bits;
} _palette;

static _palette *palette;

static gchar plugin_version[64];
static gboolean (*render_fn)(int hsize, int vsize, void *pixel_data, void *return_data);
gboolean 
render_frame_yuv420 (int hsize, int vsize, void *pixel_data, void *return_data);
gboolean 
render_frame_unknown (int hsize, int vsize, void *pixel_data, void *return_data);
#define MOD_NEEDS_TRANSLATION 1<<15


static gint ov_vsize,ov_hsize;



// yuv4mpeg specific stuff


typedef struct {
  y4m_stream_info_t streaminfo;
  y4m_frame_info_t frameinfo;
  y4m_ratio_t sar;
  y4m_ratio_t dar;
  int fd;
  int hsize;
  int vsize;
  y4m_ratio_t fps;
} yuv4m_t;


static yuv4m_t *yuv4mpeg;

yuv4m_t *yuv4mpeg_alloc (void)
{
  yuv4m_t *yuv4mpeg = (yuv4m_t *) g_try_malloc (sizeof(yuv4m_t));
  if(!yuv4mpeg) return NULL;
  yuv4mpeg->sar = y4m_sar_UNKNOWN;
  //yuv4mpeg->dar = y4m_dar_4_3;
  y4m_init_stream_info (&(yuv4mpeg->streaminfo));
  y4m_init_frame_info (&(yuv4mpeg->frameinfo));
  return yuv4mpeg;
}

//////////////////////////////////////////////


const gchar *
g_module_check_init(GModule *GM) {
  render_fn=&render_frame_unknown;
  ov_vsize=ov_hsize=0;

  palette=(_palette *)g_malloc (sizeof(_palette));
  palette->bits=0;

  yuv4mpeg=yuv4mpeg_alloc();

  return NULL;
}



const gchar *
version (void) {
  g_snprintf (plugin_version,64,"LiVES yuv4mpeg playback engine version 1.0");
  return plugin_version;
}


void 
set_keyfun (void (*fn)(gboolean, guint16, guint16)) {
}



guint 
get_capabilities (void) {
  // this function will be called after set_palette
  // returns a bitmap
  // bit 0 - 1=can resize to screen size
  // bit 1 - 1=can return return_data
  // bit 2 - 1=uses fixed framerates // TODO - get a list
  // bit 3 - 1=no display (for streaming plugins)
  // bits 4-64 unused 
  if (!palette->bits) {
    set_palette (DEFAULT_PALETTE);
  }
  return 13; // 8 + 4 + 1 (no_display, can_resize, fixed fps)
}


gboolean 
set_palette (const gchar *p_name) {
  if (!yuv4mpeg) return FALSE;
  if (!strcmp (p_name,"YUV420")) {
    g_snprintf (palette->name,32,"YUV420");
    palette->bits=24;  // must be non-zero, non important 
    render_fn=&render_frame_yuv420;
    return TRUE;
  }
  // invalid palette
  return FALSE;
}


gchar * get_fps_list (void) {
  return "24|24000:1001|25|";
}


gboolean set_fps (gdouble in_fps) {
  if (in_fps>23.9999&&in_fps<24.0001) {
    yuv4mpeg->fps=y4m_fps_FILM;
    return TRUE;
  }
  // TODO - etc...
  return FALSE;
}

gboolean 
init_screen (int width, int height, gboolean fullscreen) {
  if (!palette->bits) {
    set_palette (DEFAULT_PALETTE);
  }

  y4m_si_set_framerate(&(yuv4mpeg->streaminfo),yuv4mpeg->fps);
  y4m_si_set_interlace(&(yuv4mpeg->streaminfo), Y4M_ILACE_NONE);
  
  //y4m_log_stream_info(LOG_INFO, "lives-yuv4mpeg", &(yuv4mpeg->streaminfo));
  return TRUE;
}


gboolean 
render_frame (int hsize, int vsize, void *pixel_data, void *return_data) {
  // call the function which was set in set_palette
  return render_fn (hsize,vsize,pixel_data,return_data);
}

gboolean 
render_frame_yuv420 (int hsize, int vsize, void *pixel_data, void *return_data) {
  int i;
  guchar *planes[3];
  uint8_t *pixels=(guchar *)pixel_data;

  planes[0]=&(pixels[0]);
  planes[1]=&(pixels[hsize*vsize]);
  planes[2]=&(pixels[hsize*vsize*5/4]);

  if ((ov_hsize!=hsize||ov_vsize!=vsize)) {
    //start new stream
    y4m_si_set_width(&(yuv4mpeg->streaminfo), hsize);
    y4m_si_set_height(&(yuv4mpeg->streaminfo), vsize);
    
    y4m_si_set_sampleaspect(&(yuv4mpeg->streaminfo), yuv4mpeg->sar);
    
    i = y4m_write_stream_header(1, &(yuv4mpeg->streaminfo));

    if (i != Y4M_OK) return FALSE;

    ov_hsize=hsize;
    ov_vsize=vsize;
  }

  i = y4m_write_frame(1, &(yuv4mpeg->streaminfo),
  		&(yuv4mpeg->frameinfo), (uint8_t **)&planes[0]);
  if (i != Y4M_OK) return FALSE;

  return TRUE;
}


gboolean 
render_frame_unknown (int hsize, int vsize, void *pixel_data, void *return_data) {
  return FALSE;
}

void 
free_frame (void) {
  return;
}


void 
exit_screen (gint16 mouse_x, gint16 mouse_y) {
  y4m_fini_stream_info(&(yuv4mpeg->streaminfo));
  y4m_fini_frame_info(&(yuv4mpeg->frameinfo));
}


void 
g_module_unload(GModule *GM) {
  if (yuv4mpeg!=NULL) {
    g_free (yuv4mpeg);
  }
}




gboolean  
send_keycodes (void) {
  return TRUE;
}

