// simple_blend.c
// weed plugin
// (c) G. Finch (salsaman) 2005
//
// released under the GNU GPL 2.0 or higher
// see file COPYING or www.gnu.org for details

#include "weed.h"

///////////////////////////////////////////////////////////////////

static int num_versions=2; // number of different weed api versions supported
static int api_versions[]={110,100}; // array of weed api versions supported in plugin, in order of preference (most preferred first)

// api functions
weed_leaf_set_f weed_leaf_set;
weed_leaf_get_f weed_leaf_get;
weed_plant_new_f weed_plant_new;
weed_plant_list_leaves_f weed_plant_list_leaves;
weed_leaf_num_elements_f weed_leaf_num_elements;
weed_leaf_element_size_f weed_leaf_element_size;
weed_leaf_seed_type_f weed_leaf_seed_type;
weed_leaf_get_flags_f weed_leaf_get_flags;

weed_malloc_f weed_malloc;
weed_free_f weed_free;
weed_memcpy_f weed_memcpy;
weed_memset_f weed_memset;


static int package_version=1; // version of this package

//////////////////////////////////////////////////////////////////

#include "weed-utils.c" // optional
#include "weed-plugin-utils.c" // optional

/////////////////////////////////////////////////////////////


static void paint_pixel(unsigned char *dst, int dof, unsigned char *src, int sof, double alpha) {
  double invalpha;
  dst[dof]=dst[dof]*((invalpha=1.-alpha))+src[sof]*alpha;
  dst[dof+1]=dst[dof+1]*invalpha+src[sof+1]*alpha;
  dst[dof+2]=dst[dof+2]*invalpha+src[sof+2]*alpha;
}


int compositor_process (weed_plant_t *inst, weed_timecode_t timecode) {
  int error;
  weed_plant_t **in_channels=NULL;
  int num_in_channels=0;
  weed_plant_t *out_channel=weed_get_plantptr_value(inst,"out_channels",&error);

  unsigned char *src;
  unsigned char *dst=weed_get_voidptr_value(out_channel,"pixel_data",&error),*dst2;

  int owidth=weed_get_int_value(out_channel,"width",&error),owidth3=owidth*3;
  int oheight=weed_get_int_value(out_channel,"height",&error);

  int irowstride;
  int orowstride=weed_get_int_value(out_channel,"rowstrides",&error);
  //int palette=weed_get_int_value(out_channel,"current_palette",&error);

  weed_plant_t **in_params;

  int numscalex,numscaley,numoffsx,numoffsy,numalpha;
  int *bgcol;
  double *offsx,*offsy,*scalex,*scaley,*alpha;

  double myoffsx,myoffsy,myscalex,myscaley,myalpha;

  int width,height,sx,sy;

  unsigned char *end;

  register int x,y,z;

  if (weed_plant_has_leaf(inst,"in_channels")) {
    num_in_channels=weed_leaf_num_elements(inst,"in_channels");
    in_channels=weed_get_plantptr_array(inst,"in_channels",&error);
  }

  in_params=weed_get_plantptr_array(inst,"in_parameters",&error);

  numoffsx=weed_leaf_num_elements(in_params[0],"value");
  offsx=weed_get_double_array(in_params[0],"value",&error);

  numoffsy=weed_leaf_num_elements(in_params[1],"value");
  offsy=weed_get_double_array(in_params[1],"value",&error);

  numscalex=weed_leaf_num_elements(in_params[2],"value");
  scalex=weed_get_double_array(in_params[2],"value",&error);

  numscaley=weed_leaf_num_elements(in_params[3],"value");
  scaley=weed_get_double_array(in_params[3],"value",&error);

  numalpha=weed_leaf_num_elements(in_params[4],"value");
  alpha=weed_get_double_array(in_params[4],"value",&error);

  bgcol=weed_get_int_array(in_params[5],"value",&error);

  // set out frame to bgcol

  end=dst+oheight*orowstride;
  for (dst2=dst;dst2<end;dst2+=orowstride) {
    for (x=0;x<owidth3;x+=3) {
      dst2[x]=bgcol[0];
      dst2[x+1]=bgcol[1];
      dst2[x+2]=bgcol[2];
    }
  }

  weed_free(bgcol);

  // add overlays in reverse order

  for (z=num_in_channels-1;z>=0;z--) {
    // check if host disabled this channel : this is allowed as we have set "max_repeats"
    if (weed_plant_has_leaf(in_channels[z],"disabled")&&weed_get_boolean_value(in_channels[z],"disabled",&error)==WEED_TRUE) continue;

    if (z<numoffsx) myoffsx=(int)(offsx[z]*(double)owidth);
    else myoffsx=0;
    if (z<numoffsy) myoffsy=(int)(offsy[z]*(double)oheight);
    else myoffsy=0;
    if (z<numscalex) myscalex=scalex[z];
    else myscalex=1.;
    if (z<numscaley) myscaley=scaley[z];
    else myscaley=1.;
    if (z<numalpha) myalpha=alpha[z];
    else myalpha=1.;

    width=weed_get_int_value(in_channels[z],"width",&error);
    height=weed_get_int_value(in_channels[z],"height",&error);

    src=weed_get_voidptr_value(in_channels[z],"pixel_data",&error);
    irowstride=weed_get_int_value(in_channels[z],"rowstrides",&error);

    for (y=myoffsy;y<oheight&&y<myoffsy+(int)(height*myscaley);y++) {
      for (x=myoffsx;x<owidth&&x<myoffsx+(int)(width*myscalex);x++) {
	sx=(int)((double)(x-myoffsx)/myscalex);
	sy=(int)((double)(y-myoffsy)/myscaley);
	//printf("vals %d %d %d %d\n",x,y,sx,sy);
	if (sx<width&&sy<height) {
	  paint_pixel(dst,y*orowstride+x*3,src,sy*irowstride+sx*3,myalpha);
	}
      }
    }
  }

  weed_free(offsx);
  weed_free(offsy);
  weed_free(scalex);
  weed_free(scaley);
  weed_free(alpha);

  if (num_in_channels>0) weed_free(in_channels);
  return WEED_NO_ERROR;
}



weed_plant_t *weed_setup (weed_bootstrap_f weed_boot) {
  weed_plant_t *plugin_info=weed_plugin_info_init(weed_boot,num_versions,api_versions);
  if (plugin_info!=NULL) {
    int palette_list[]={WEED_PALETTE_RGB24,WEED_PALETTE_END};
    weed_plant_t *in_chantmpls[]={weed_channel_template_init("in channel 0",WEED_CHANNEL_SIZE_CAN_VARY,palette_list),NULL};
    weed_plant_t *out_chantmpls[]={weed_channel_template_init("out channel 0",WEED_CHANNEL_SIZE_CAN_VARY,palette_list),NULL};

    weed_plant_t *in_params[]={weed_float_init("xoffs","_X offset",0.,0.,1.),weed_float_init("yoffs","_Y offset",0.,0.,1.),weed_float_init("scalex","Scale _width",1.,0.,1.),weed_float_init("scaley","Scale _height",1.,0.,1.),weed_float_init("alpha","_Alpha",1.0,0.0,1.0),weed_colRGBi_init("bgcol","_Background color",0,0,0),NULL};

    weed_plant_t *filter_class=weed_filter_class_init("compositor","salsaman",1,0,NULL,&compositor_process,NULL,in_chantmpls,out_chantmpls,in_params,NULL);

    weed_plant_t *gui=weed_filter_class_get_gui(filter_class);

    // define RFX layout
    char *rfx_strings[]={"layout|p0|p1|","layout|p2|p3|","layout|p4|","layout|hseparator|","layout|p5|","special|framedraw|multrect|0|1|2|3|4|"};

    int api_used=weed_get_api_version(plugin_info);
    // set 0 to infinite repeats
    weed_set_int_value(in_chantmpls[0],"max_repeats",0);
    weed_set_boolean_value(in_chantmpls[0],"optional",WEED_TRUE);
    
    // this is necessary for the host
    if (api_used==100) {
      weed_set_int_value(in_params[0],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS);
      weed_set_int_value(in_params[1],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS);
      weed_set_int_value(in_params[2],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS);
      weed_set_int_value(in_params[3],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS);
      weed_set_int_value(in_params[4],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS);
    }
    else if (api_used==110) {
      // use WEED_PARAMETER_ELEMENT_PER_CHANNEL from spec 110
      weed_set_int_value(in_params[0],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS|WEED_PARAMETER_ELEMENT_PER_CHANNEL);
      weed_set_int_value(in_params[1],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS|WEED_PARAMETER_ELEMENT_PER_CHANNEL);
      weed_set_int_value(in_params[2],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS|WEED_PARAMETER_ELEMENT_PER_CHANNEL);
      weed_set_int_value(in_params[3],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS|WEED_PARAMETER_ELEMENT_PER_CHANNEL);
      weed_set_int_value(in_params[4],"flags",WEED_PARAMETER_VARIABLE_ELEMENTS|WEED_PARAMETER_ELEMENT_PER_CHANNEL);
    }

    // set default value for elements added by the host
    weed_set_double_value(in_params[0],"new_default",0.);
    weed_set_double_value(in_params[1],"new_default",0.);
    weed_set_double_value(in_params[2],"new_default",1.);
    weed_set_double_value(in_params[3],"new_default",1.);
    weed_set_double_value(in_params[4],"new_default",1.);

    // set RFX layout
    weed_set_string_value(gui,"layout_scheme","RFX");
    weed_set_string_value(gui,"rfx_delim","|");
    weed_set_string_array(gui,"rfx_strings",6,rfx_strings);

    weed_plugin_info_add_filter_class (plugin_info,filter_class);

    weed_set_int_value(plugin_info,"version",package_version);

  }

  return plugin_info;
}
