/* videojack test plugin for Weed
   authors: Salsaman (G. Finch) <salsaman@xs4all.nl>

// released under the GNU GPL 2 or higher
// see file COPYING or www.gnu.org for details

 (c) 2005, project authors
*/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <jack/jack.h>
#include <jack/video.h>
#include <jack/ringbuffer.h>


#include "weed.h"

///////////////////////////////////////////////////////////////////

static int num_versions=1; // number of different weed api versions supported
static int api_versions[]={100}; // array of weed api versions supported in plugin, in order of preference (most preferred first)

// api functions
weed_leaf_set_f weed_leaf_set;
weed_leaf_get_f weed_leaf_get;
weed_plant_new_f weed_plant_new;
weed_plant_list_leaves_f weed_plant_list_leaves;
weed_leaf_num_elements_f weed_leaf_num_elements;
weed_leaf_element_size_f weed_leaf_element_size;
weed_leaf_seed_type_f weed_leaf_seed_type;
weed_leaf_get_flags_f weed_leaf_get_flags;

weed_malloc_f weed_malloc;
weed_free_f weed_free;
weed_memcpy_f weed_memcpy;
weed_memset_f weed_memset;


static int package_version=1; // version of this package

//////////////////////////////////////////////////////////////////

#include "weed-utils.c" // optional
#include "weed-plugin-utils.c" // optional

/////////////////////////////////////////////////////////////


static int instances;

typedef struct {
  jack_port_t *input_port;
  jack_ringbuffer_t *rb;
  jack_client_t *client;
#define SMOOTH
#ifdef SMOOTH
  unsigned char *bgbuf;
#endif
} sdata;



int server_process (jack_nframes_t nframes, void *arg)
{
  // this is called by jack when a frame is received
  sdata *sd=(sdata *)arg;
  unsigned int width=jack_video_get_width(sd->client,sd->input_port);
  unsigned int height=jack_video_get_height(sd->client,sd->input_port);
  unsigned int frame_size;
  uint8_t* in = jack_port_get_buffer (sd->input_port, nframes);

  frame_size = width * height * 4;

  if (frame_size==0||in==NULL) return 0;

  // enough space for one more frame
  if (jack_ringbuffer_write_space(sd->rb) >= frame_size) {
    jack_ringbuffer_write (sd->rb, (void *) in, frame_size);
  }
  else {
    //fprintf (stderr, "drop frame\n");
  }
  
  return 0;
}




/* declare our init function */
int vjack_rcv_init(weed_plant_t *inst) {
  weed_plant_t *out_channel;
  int error;
  unsigned int out_frame_size;
  const char *assigned_client_name;
  char *server_name,*conffile;
  jack_options_t options = JackServerName;
  jack_status_t status;
  weed_plant_t **in_params;
  sdata *sd;
  char *client_name="Weed-receiver";
  unsigned int out_height,out_width;
  char com[512];
  double jack_sample_rate;

  sd=weed_malloc(sizeof(sdata));
  if(sd==NULL) return WEED_ERROR_MEMORY_ALLOCATION;

  in_params=weed_get_plantptr_array(inst,"in_parameters",&error);
  server_name=weed_get_string_value(in_params[0],"value",&error);
  conffile=weed_get_string_value(in_params[1],"value",&error);
  weed_free(in_params);

  instances++;

  out_channel=weed_get_plantptr_value(inst,"out_channels",&error);

  out_height=weed_get_int_value(out_channel,"height",&error);
  out_width=weed_get_int_value(out_channel,"width",&error);

  out_frame_size=out_width*out_height*4;

#ifdef SMOOTH
  sd->bgbuf=weed_malloc(out_frame_size);
  if(sd->bgbuf==NULL) {
    weed_free(sd);
    return WEED_ERROR_MEMORY_ALLOCATION;
  }
  weed_memset(sd->bgbuf,0,out_frame_size);
#endif

  // use user defined rc file
  //  system("/bin/mv -f ~/.jackdrc ~/.jackdrc._bak>/dev/null 2>&1");
  //snprintf(com,512,"/bin/ln -s %s ~/.jackdrc>/dev/null 2>&1",conffile);
  //system(com);

  sd->client = jack_client_open (client_name, options, &status, server_name);
  weed_free(server_name);
  weed_free(conffile);

  //  system("/bin/mv -f ~/.jackdrc._bak ~/.jackdrc>/dev/null 2>&1");

  if (sd->client==NULL) {
    fprintf (stderr, "jack_client_open() failed, "
	     "status = 0x%2.0x\n", status);
    if (status & JackServerFailed) {
      fprintf (stderr, "Unable to connect to JACK server\n");
    }
#ifdef SMOOTH
    weed_free(sd->bgbuf);
#endif
    weed_free(sd);
    return WEED_ERROR_INIT_ERROR;
  }
  if (status & JackServerStarted) {
    fprintf (stderr, "JACK server started\n");
  }
  if (status & JackNameNotUnique) {
    assigned_client_name = jack_get_client_name(sd->client);
    fprintf (stderr, "unique name `%s' assigned\n", assigned_client_name);
  }
  
  fprintf (stderr,"engine sample rate: %" PRIu32 "\n",
	   jack_get_sample_rate (sd->client));



  sd->input_port = jack_port_register (sd->client, 
				   "video_in",
				   JACK_DEFAULT_VIDEO_TYPE,
				   JackPortIsInput, 
				   0);
  
  if (sd->input_port==NULL) 
    {
      fprintf(stderr, "no more JACK ports available\n");
#ifdef SMOOTH
      weed_free(sd->bgbuf);
#endif
      weed_free(sd);
      return WEED_ERROR_INIT_ERROR;
    }
  
  // communication structure between process and display thread
  sd->rb = jack_ringbuffer_create (2*out_width*out_height*4);
  
  // set process callback and start
  
  jack_set_process_callback (sd->client, server_process, sd);  
  
  if (jack_activate (sd->client)) 
    {
      fprintf (stderr, "cannot activate client");
#ifdef SMOOTH
      weed_free(sd->bgbuf);
#endif
      weed_free(sd);
      return WEED_ERROR_INIT_ERROR;
    }
  
  //jack_on_shutdown (client, jack_shutdown, 0);
  weed_set_voidptr_value(inst,"plugin_internal",sd);

  jack_sample_rate=jack_get_sample_rate(sd->client);

  weed_set_double_value(inst,"target_fps",jack_sample_rate); // set reasonable value
  return WEED_NO_ERROR;
}


int vjack_rcv_process (weed_plant_t *inst, weed_timecode_t timestamp) {
  // TODO check for server shutdown
  int error;
  unsigned int frame_size;
  weed_plant_t *out_channel=weed_get_plantptr_value(inst,"out_channels",&error);
  unsigned char *dst=weed_get_voidptr_value(out_channel,"pixel_data",&error);
  int out_width=weed_get_int_value(out_channel,"width",&error);
  int out_height=weed_get_int_value(out_channel,"height",&error);
  int wrote=0;
  sdata *sd=(sdata *)weed_get_voidptr_value(inst,"plugin_internal",&error);

  unsigned int in_width=jack_video_get_width(sd->client,sd->input_port);
  unsigned int in_height=jack_video_get_height(sd->client,sd->input_port);

  // TODO - resize
  if (in_width!=out_width||in_height!=out_height) return WEED_NO_ERROR;

  frame_size = in_width*in_height*4;

  while ( jack_ringbuffer_read_space(sd->rb) >= frame_size) {
    jack_ringbuffer_read (sd->rb, (char *)dst, frame_size);
    wrote=1;
  }

#ifdef SMOOTH
  if (!wrote) weed_memcpy(dst,sd->bgbuf,frame_size);
  else weed_memcpy(sd->bgbuf,dst,frame_size);
#endif

  return WEED_NO_ERROR;
}


int vjack_rcv_deinit (weed_plant_t *inst) {
  int error;
  sdata *sd=(sdata *)weed_get_voidptr_value(inst,"plugin_internal",&error);

  jack_deactivate (sd->client);
  jack_client_close (sd->client);
  if (--instances<0) instances=0;

#ifdef SMOOTH
  weed_free(sd->bgbuf);
#endif
  weed_free(sd);
  return WEED_NO_ERROR; // success
}





weed_plant_t *weed_setup (weed_bootstrap_f weed_boot) {
  weed_plant_t *plugin_info=weed_plugin_info_init(weed_boot,num_versions,api_versions);
  if (plugin_info!=NULL) {
    int palette_list[]={WEED_PALETTE_RGBA32,WEED_PALETTE_END};

    weed_plant_t *out_chantmpls[]={weed_channel_template_init("out channel 0",0,palette_list),NULL};
    
    weed_plant_t *in_params[]={weed_text_init("servername","_Server name","default"),weed_text_init("conffile","_Config file","~/.jackdrc.vjack"),NULL};

    weed_plant_t *filter_class=weed_filter_class_init("vjack_rcv","martin/salsaman",1,0,&vjack_rcv_init,&vjack_rcv_process,&vjack_rcv_deinit,NULL,out_chantmpls,in_params,NULL);

    weed_plant_t *gui=weed_parameter_template_get_gui(in_params[0]);
    weed_set_int_value(gui,"maxchars",32);

    gui=weed_parameter_template_get_gui(in_params[1]);
    weed_set_int_value(gui,"maxchars",128);

    weed_plugin_info_add_filter_class (plugin_info,filter_class);
    
    weed_set_int_value(plugin_info,"version",package_version);
  }
  return plugin_info;
}



