/* WEED is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public
   License as published by the Free Software Foundation; either
   version 2.1 of the License, or (at your option) any later version.

   Weed is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License for more details.

   You should have received a copy of the GNU Lesser General Public
   License along with this source code; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA


   Weed is developed by:

   Gabriel "Salsaman" Finch - http://lives.sourceforge.net

   mainly based on LiViDO, which is developed by:


   Niels Elburg - http://veejay.sf.net

   Gabriel "Salsaman" Finch - http://lives.sourceforge.net

   Denis "Jaromil" Rojo - http://freej.dyne.org

   Tom Schouten - http://zwizwa.fartit.com

   Andraz Tori - http://cvs.cinelerra.org

   reviewed with suggestions and contributions from:

   Silvano "Kysucix" Galliani - http://freej.dyne.org

   Kentaro Fukuchi - http://megaui.net/fukuchi

   Jun Iio - http://www.malib.net

   Carlo Prelz - http://www2.fluido.as:8080/

*/

/* (C) Gabriel "Salsaman" Finch, 2005 */

#ifndef __WEED_H__
#define __WEED_H__

#ifdef __cplusplus
extern "C"
{
#endif /* __cplusplus */

  /* for size_t */
#include <sys/types.h>

  /* API version * 110 */
#define WEED_API_VERSION 110
#define WEED_API_VERSION_100
#define WEED_API_VERSION_110

#define WEED_TRUE 1
#define WEED_FALSE 0

/* Palette types */
/* RGB palettes */
#define WEED_PALETTE_END 0
#define WEED_PALETTE_RGB888 1
#define WEED_PALETTE_RGB24 1
#define WEED_PALETTE_BGR888 2
#define WEED_PALETTE_BGR24 2
#define WEED_PALETTE_RGBA8888 3
#define WEED_PALETTE_RGBA32 3
#define WEED_PALETTE_ARGB8888 4
#define WEED_PALETTE_ARGB32 4
#define WEED_PALETTE_RGBFLOAT 5
#define WEED_PALETTE_ARGBFLOAT  6
#define WEED_PALETTE_BGRA8888 7
#define WEED_PALETTE_BGRA32 7

/* YUV palettes */
#define WEED_PALETTE_YUV422P 513
#define WEED_PALETTE_YV16 513
#define WEED_PALETTE_YUV420P 514
#define WEED_PALETTE_YV12 514
#define WEED_PALETTE_YVU420P 515
#define WEED_PALETTE_I420 515
#define WEED_PALETTE_YUV444P 516
#define WEED_PALETTE_YUVA4444P 517
#define WEED_PALETTE_YUV4444P 517
#define WEED_PALETTE_YUYV8888 518
#define WEED_PALETTE_UYVY8888 519
#define WEED_PALETTE_YUV411 520
#define WEED_PALETTE_YUV888 521
#define WEED_PALETTE_YUVA8888 522

/* Alpha palettes */
#define WEED_PALETTE_A1 1025
#define WEED_PALETTE_A8 1026
#define WEED_PALETTE_AFLOAT 1027

/* Parameter hints */
#define WEED_HINT_UNSPECIFIED     0
#define WEED_HINT_INTEGER         1
#define WEED_HINT_FLOAT           2
#define WEED_HINT_TEXT            3
#define WEED_HINT_SWITCH          4
#define WEED_HINT_COLOR           5

/* Colorspaces for Color parameters */
#define WEED_COLORSPACE_RGB   1
#define WEED_COLORSPACE_RGBA  2
#define WEED_COLORSPACE_HSV   3

/* Filter flags */
#define WEED_FILTER_NON_REALTIME    (1<<0)
#define WEED_FILTER_IS_CONVERTOR    (1<<1)

/* Channel flags */
#define WEED_CHANNEL_REINIT_ON_SIZE_CHANGE    (1<<0)
#define WEED_CHANNEL_REINIT_ON_PALETTE_CHANGE (1<<1)
#define WEED_CHANNEL_CAN_DO_INPLACE           (1<<2)
#define WEED_CHANNEL_SIZE_CAN_VARY            (1<<3)
#define WEED_CHANNEL_PALETTE_CAN_VARY         (1<<4)

/* API version 110 */
#define WEED_CHANNEL_FOLLOWS_OUTPUT           (1<<5)

/* Parameter flags */
#define WEED_PARAMETER_REINIT_ON_VALUE_CHANGE (1<<0)
#define WEED_PARAMETER_VARIABLE_ELEMENTS      (1<<1)

/* API version 110 */
#define WEED_PARAMETER_ELEMENT_PER_CHANNEL    (1<<2)

/* Caller Types */
#define WEED_CALLER_HOST 0
#define WEED_CALLER_PLUGIN 1

/* Leaf flags */
#define WEED_LEAF_READONLY_PLUGIN (1<<0)
#define WEED_LEAF_READONLY_HOST (1<<1)

/* YUV sampling types */
#define WEED_YUV_SAMPLING_NONE   0
#define WEED_YUV_SAMPLING_SMPTE  1
#define WEED_YUV_SAMPLING_JPEG   2
#define WEED_YUV_SAMPLING_MPEG2  3
#define WEED_YUV_SAMPLING_DVPAL  4
#define WEED_YUV_SAMPLING_DVNTSC 5

/* Weed errors */
/* Core errors */
#define WEED_NO_ERROR 0
#define WEED_ERROR_MEMORY_ALLOCATION 1
#define WEED_ERROR_LEAF_READONLY 2
#define WEED_ERROR_NOSUCH_ELEMENT 3
#define WEED_ERROR_NOSUCH_LEAF 4
#define WEED_ERROR_WRONG_SEED_TYPE 5

/* Plugin errors */
#define WEED_ERROR_TOO_MANY_INSTANCES 6
#define WEED_ERROR_HARDWARE 7
#define WEED_ERROR_INIT_ERROR 8
#define WEED_ERROR_PLUGIN_INVALID 64

/* Seed types */
/* Fundamental seeds */
#define WEED_SEED_INT     1
#define WEED_SEED_DOUBLE  2
#define WEED_SEED_BOOLEAN 3
#define WEED_SEED_STRING  4
#define WEED_SEED_INT64   5

/* Pointer seeds */
#define WEED_SEED_VOIDPTR  65
#define WEED_SEED_PLANTPTR 66

/* Plant types */
#define WEED_PLANT_UNKNOWN 0 // used for host deserialisation only
#define WEED_PLANT_PLUGIN_INFO 1
#define WEED_PLANT_FILTER_CLASS 2
#define WEED_PLANT_FILTER_INSTANCE 3
#define WEED_PLANT_CHANNEL_TEMPLATE 4
#define WEED_PLANT_PARAMETER_TEMPLATE 5
#define WEED_PLANT_CHANNEL 6
#define WEED_PLANT_PARAMETER 7
#define WEED_PLANT_GUI 8
#define WEED_PLANT_HOST_INFO 255

typedef void *(*weed_malloc_f) (size_t size);
typedef void (*weed_free_f) (void *ptr);
typedef void *(*weed_memset_f) (void *s, int c, size_t n);
typedef void *(*weed_memcpy_f) (void *dest, const void *src, size_t n);

extern weed_malloc_f weed_malloc;
extern weed_free_f weed_free;
extern weed_memset_f weed_memset;
extern weed_memcpy_f weed_memcpy;

#ifndef HAVE_WEED_PLANT_T
#define HAVE_WEED_PLANT_T

typedef struct weed_leaf weed_leaf_t;
typedef struct weed_data weed_data_t;
typedef weed_leaf_t weed_plant_t;

struct weed_leaf {
  const char *key;
  int seed_type;
  int num_elements;
  weed_data_t **data;
  int flags;
  weed_leaf_t *next;
};

struct weed_data {
  size_t size;
  void *value;
};

#endif

typedef int64_t weed_timecode_t;

  /* this is fixed for ever, set in bootstrap_func */
typedef int (*weed_default_getter_f)(weed_plant_t *plant, const char *key, int idx, void *value);

  /* host and plugin functions */
typedef weed_plant_t *(*weed_plant_new_f)(int plant_type);
typedef char **(*weed_plant_list_leaves_f)(weed_plant_t *plant);
typedef int (*weed_leaf_set_f)(weed_plant_t *plant, const char *key, int seed_type, int num_elems, void *value);
typedef int (*weed_leaf_get_f)(weed_plant_t *plant, const char *key, int idx, void *value);
typedef int (*weed_leaf_num_elements_f)(weed_plant_t *plant, const char *key);
typedef size_t (*weed_leaf_element_size_f)(weed_plant_t *plant, const char *key, int idx);
typedef int (*weed_leaf_seed_type_f)(weed_plant_t *plant, const char *key);
typedef int (*weed_leaf_get_flags_f)(weed_plant_t *plant, const char *key);

  /* host only functions */
typedef void (*weed_plant_free_f)(weed_plant_t *plant);
typedef int (*weed_leaf_delete_f)(weed_plant_t *plant, const char *key);
typedef int (*weed_leaf_set_flags_f)(weed_plant_t *plant, const char *key, int flags);
typedef weed_plant_t *(*weed_bootstrap_f) (void **value, int num_versions, int *plugin_versions);

  /* plugin only functions */
typedef weed_plant_t *(*weed_setup_f)(weed_bootstrap_f weed_boot);
typedef int (*weed_init_f) (weed_plant_t *filter_instance);
typedef int (*weed_process_f) (weed_plant_t *filter_instance, weed_timecode_t timestamp);
typedef int (*weed_deinit_f) (weed_plant_t *filter_instance);

  /* special plugin functions */
typedef void (*weed_display_f)(weed_plant_t *parameter);
typedef int (*weed_interpolate_f)(weed_plant_t **in_values, weed_plant_t *out_value);

extern weed_leaf_set_f weed_leaf_set;

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif // #ifndef __WEED_H__
