// jack.c
// LiVES (lives-exe)
// (c) G. Finch 2005 - 2006
// Released under the GPL 2.0 or higher
// see file ../COPYING for licensing details

#include "main.h"
#include "audio.h"
#include "effects-weed.h"
#include "support.h"

inline void sample_silence_dS (float *dst, unsigned long nsamples) {
  memset(dst,0,nsamples*sizeof(float));
}


void sample_move_d8_d16(short *dst, unsigned char *src,
			unsigned long nsamples, float scale, int nDstChannels, int nSrcChannels) {
  register int nSrcCount, nDstCount;
  register float src_offset_f=0.f;
  register int src_offset_i=0;
  register int ccount;
  
  //TRACE("nsamples == %ld, nDstChannels == %d, nSrcChannels == %d\n", nsamples, nDstChannels, nSrcChannels);

  if(!nSrcChannels) return;

  if (scale<0.f) {
    src_offset_f=((float)(nsamples)*(-scale)-1.f);
    src_offset_i=(int)src_offset_f*nSrcChannels;
  }

  while(nsamples--) {
    nSrcCount = nSrcChannels;
    nDstCount = nDstChannels;
    ccount=0;

    /* loop until all of our destination channels are filled */
    while(nDstCount) {
      nSrcCount--;
      nDstCount--;

      *(dst++) = *(src+ccount+src_offset_i)<<8; /* copy the data over */
      ccount++;

      /* if we ran out of source channels but not destination channels */
      /* then start the src channels back where we were */
      if(!nSrcCount && nDstCount) {
	ccount=0;
	nSrcCount = nSrcChannels;
      }
    }
    
    /* advance the the position */
    src_offset_i=(int)(src_offset_f+=scale)*nSrcChannels;
  }
}



/* convert from any number of source channels to any number of destination channels */
void sample_move_d16_d16(short *dst, short *src,
			 unsigned long nsamples, float scale, int nDstChannels, int nSrcChannels, int swap_endian) {
  register int nSrcCount, nDstCount;
  register float src_offset_f=0.f;
  register int src_offset_i=0;
  register int ccount;

  if(!nSrcChannels) return;

  if (scale<0.f) {
    src_offset_f=((float)(nsamples)*(-scale)-1.f);
    src_offset_i=(int)src_offset_f*nSrcChannels;
  }

  while(nsamples--) {
    if ((nSrcCount = nSrcChannels)==(nDstCount = nDstChannels)&&!swap_endian) {
      // same number of channels
#ifdef ENABLE_OIL
      oil_memcpy(dst,src+src_offset_i,nSrcChannels*sizeof(short));
#else
      memcpy(dst,src+src_offset_i,nSrcChannels*sizeof(short));
#endif
      dst+=nDstCount;
    } 
    else {
      ccount=0;

      /* loop until all of our destination channels are filled */
      while(nDstCount) {
	nSrcCount--;
	nDstCount--;
	
	if (!swap_endian) *(dst++) = *(src+ccount+src_offset_i); /* copy the data over */
	else *(dst++)=((*(src+ccount+src_offset_i)&0x00FF)<<8)+(*(src+ccount+src_offset_i)>>8);
	ccount++;
	
	/* if we ran out of source channels but not destination channels */
	/* then start the src channels back where we were */
	if(!nSrcCount && nDstCount) {
	  ccount=0;
	  nSrcCount = nSrcChannels;
	}
      }
    }
    /* advance the the position */
    src_offset_i=(int)(src_offset_f+=scale)*nSrcChannels;
  }
}

/* convert from 16 bit to floating point */
/* channels to a buffer that will hold a single channel stream */
/* src_skip is in terms of 16bit samples */
void sample_move_d16_float (float *dst, short *src, unsigned long nsamples, unsigned long src_skip, int is_unsigned, float vol) {
  register float svol;
  float val;

#ifdef ENABLE_OIL
  double x;
  double y=0.f;
#endif

  if (vol==0.) vol=0.0000001f;
  svol=SAMPLE_MAX_16BIT/vol;

#ifdef ENABLE_OIL
  x=1./svol;
#endif

  while (nsamples--) {
    if (!is_unsigned) {
#ifdef ENABLE_OIL
      oil_scaleconv_f32_s16(&val,src,1,&y,&x);
#else
      if ((val = (float)((*src) / svol))>1.0f) val=1.0f;
      else if (val<-1.0f) val=-1.0f;
#endif
    }
    else {
#ifdef ENABLE_OIL
      oil_scaleconv_f32_u16(&val,(unsigned short *)src,1,&y,&x);
#else
      if ((val=((float)((unsigned short)(*src)) / SAMPLE_MAX_16BIT - 1.0f)*vol)>1.0f) val=1.0f;
      else if (val<-1.0f) val=-1.0f;
#endif
    }
    *(dst++)=val;
    src += src_skip;
  }

}



long sample_move_float_int(void *holding_buff, float **float_buffer, int nsamps, float scale, int chans, int asamps, int usigned, int swap_endian) {
  // convert float samples back to int
  long frames_out=0l;
  short val;
  unsigned short valu=0;
  register int i;
  register int offs=0,coffs=0;
  register float coffs_f=0.f;
  short *hbuffs=(short *)holding_buff;
  unsigned short *hbuffu=(unsigned short *)holding_buff;
  unsigned char *hbuffc=(guchar *)holding_buff;

  while ((nsamps-coffs)>0) {
    frames_out++;
    for (i=0;i<chans;i++) {
      val=(short)(*(float_buffer[i]+coffs)*SAMPLE_MAX_16BIT);
      if (usigned) {
	valu=val+SAMPLE_MAX_16BITI;
      }
      if (asamps==16) {
	if (!swap_endian) {
	  if (usigned) *(hbuffu+offs)=valu;
	  else *(hbuffs+offs)=val;
	}
	else {
	  *(hbuffc+offs)=val&0x00FF;
	  *(hbuffc+(++offs))=val&0xFF00;
	}
      }
      else {
	if (usigned) *(hbuffc+offs)=(guchar)(valu>>8);
	else *(hbuffc+offs)=(guchar)(val/256.);
      }
      offs++;
    }
    coffs=(gint)(coffs_f+=scale);
  }
  return frames_out;
}

#define RENDER_BLOCK_SIZE 1024
#define SILENCE_BLOCK_SIZE 65536

void render_audio_segment(gint from_file, gint to_file, gdouble avel, gdouble fromtime, gdouble uptotime, weed_timecode_t tc, gdouble autofade_start, gdouble autofade_end) {
  // called during multitrack rendering to create the actual audio file

  file *infile=mainw->files[from_file];
  file *outfile=mainw->files[to_file];
  gdouble dbytes,dblocksize;
  size_t tbytes,blocksize;
  guchar *in_buff;
  gint in_asamps=infile->asampsize;
  gint in_achans=infile->achans;
  gint in_arate=infile->arate;
  gint in_unsigned=infile->signed_endian&AFORM_UNSIGNED;

  gint in_bendian=infile->signed_endian&AFORM_BIG_ENDIAN;
  gint out_asamps=outfile->asampsize;
  gint out_achans=outfile->achans;
  gint out_arate=outfile->arate;
  gint out_unsigned=outfile->signed_endian&AFORM_UNSIGNED;
  gint out_bendian=outfile->signed_endian&AFORM_BIG_ENDIAN;
  short holding_buff[RENDER_BLOCK_SIZE*out_achans];
  float *float_buffer[out_achans];
  int c;
  register int i;
  size_t bytes_read;
  int in_fd,out_fd;
  gulong nframes;
  gboolean in_reverse_endian,out_reverse_endian;
  long seekstart;
  gchar *infilename,*outfilename;
  gdouble ins_pt=tc/U_SEC;
  long ins_size,cur_size;
  guchar *zero_buff;
  size_t sblocksize=SILENCE_BLOCK_SIZE;
  gint sbytes;
  gdouble time=0.;
  gdouble vol=1.;
  float out_scale;
  long frames_out;

  if (mainw->multitrack==NULL) vol=autofade_start;

  avel*=(gdouble)in_arate/(gdouble)out_arate;
  dblocksize=RENDER_BLOCK_SIZE*in_achans*in_asamps/8*ABS(avel);
  blocksize=((size_t)(dblocksize/in_achans/(in_asamps/8)))*in_achans*(in_asamps/8);

  in_buff=g_malloc(blocksize);

  dbytes=ABS(uptotime-fromtime)*in_arate*in_asamps/8*in_achans;
  tbytes=((size_t)(dbytes/in_achans/(in_asamps/8)))*in_achans*(in_asamps/8);

  outfilename=g_strdup_printf("%s/%s/audio",prefs->tmpdir,outfile->handle);
  out_fd=open(outfilename,O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR);

  if ((!out_bendian&&(G_BYTE_ORDER==G_BIG_ENDIAN))||(out_bendian&&(G_BYTE_ORDER==G_LITTLE_ENDIAN))) out_reverse_endian=TRUE;
  else out_reverse_endian=FALSE;

  if ((!in_bendian&&(G_BYTE_ORDER==G_BIG_ENDIAN))||(in_bendian&&(G_BYTE_ORDER==G_LITTLE_ENDIAN))) in_reverse_endian=TRUE;
  else in_reverse_endian=FALSE;

  seekstart=(long)(fromtime*in_arate)*in_achans*in_asamps/8;
  seekstart=((long)(seekstart/in_achans/(in_asamps/8)))*in_achans*(in_asamps/8);

  infilename=g_strdup_printf("%s/%s/audio",prefs->tmpdir,infile->handle);
  in_fd=open(infilename,O_RDONLY);
  lseek(in_fd,seekstart,SEEK_SET);

  g_free(infilename);
  g_free(outfilename);

  out_scale=in_arate/out_arate;

  cur_size=get_file_size(out_fd);

  ins_pt*=out_achans*out_arate*(out_asamps/8);
  ins_size=((long)(ins_pt/out_achans/(out_asamps/8)))*out_achans*(out_asamps/8);

  // fill to ins_pt with zeros
  sbytes=ins_size-cur_size;

  if (sbytes>0) {
    zero_buff=g_malloc0(SILENCE_BLOCK_SIZE);
    
    for (i=0;i<sbytes;i+=SILENCE_BLOCK_SIZE) {
      if (sbytes-i<SILENCE_BLOCK_SIZE) sblocksize=sbytes-i;
      write (out_fd,zero_buff,sblocksize);
    }
    
    g_free(zero_buff);
  }
  else if (sbytes<0) {
    lseek(out_fd,cur_size+sbytes,SEEK_SET);
  }

  for (i=0;i<out_achans;i++) {
    float_buffer[i]=g_malloc(RENDER_BLOCK_SIZE*sizeof(float));
  }

  for (i=0;i<tbytes;i+=blocksize) {
    if (tbytes-i<blocksize) blocksize=tbytes-i;

    if (avel<0.) lseek(in_fd,seekstart-blocksize,SEEK_SET);

    // read in a block
    bytes_read=read(in_fd,in_buff,blocksize);

    if (bytes_read==0) break;

    nframes=bytes_read/(in_asamps/8)/in_achans/ABS(avel);

    // convert to float
    if (in_asamps==8) {
      sample_move_d8_d16 (holding_buff,(guchar *)in_buff,nframes,avel,out_achans,in_achans);
    }
    else {
      sample_move_d16_d16(holding_buff,(short*)in_buff,nframes,avel,out_achans,in_achans,in_reverse_endian);
    }

    for(c=0;c<out_achans;c++) {
      sample_move_d16_float(float_buffer[c],holding_buff+c,nframes,out_achans,in_unsigned,vol);
    }

    // apply audio filter(s)
    if (mainw->multitrack!=NULL&&mainw->multitrack->avol_init_event!=NULL) {
      weed_apply_audio_instance(mainw->multitrack->avol_init_event,float_buffer,out_achans,nframes,to_file,tc);
    }

    // convert back to int

    frames_out=sample_move_float_int((void *)holding_buff,float_buffer,nframes,out_scale,out_achans,out_asamps,out_unsigned,out_reverse_endian);

    write (out_fd,holding_buff,frames_out*(out_asamps/8)*out_achans);

    if (mainw->multitrack==NULL) {
      time+=(gdouble)nframes/(gdouble)out_arate;
      vol=autofade_start+(autofade_end-autofade_start)*(time/(uptotime-fromtime));
    }
    else tc+=(gdouble)nframes/(gdouble)out_arate*U_SEC;

    if (avel<0.) {
      seekstart-=blocksize;
      lseek(in_fd,seekstart,SEEK_SET);
    }
  }

  g_free(in_buff);

  for (i=0;i<out_achans;i++) {
    g_free(float_buffer[i]);
  }

  // close files
  close (in_fd);
  close (out_fd);

}


inline void aud_fade(gint fileno, gdouble startt, gdouble endt, gdouble startv, gdouble endv) {
  render_audio_segment(fileno,fileno,1.,startt,endt,startt*U_SECL,startv,endv);
}



#ifdef ENABLE_JACK
void rec_audio_to_clip(gint fileno, gboolean is_window_grab) {
  // open audio file for writing
  file *outfile=mainw->files[fileno];
  gchar *outfilename=g_strdup_printf("%s/%s/audio",prefs->tmpdir,outfile->handle);
  gint out_bendian=outfile->signed_endian&AFORM_BIG_ENDIAN;

  mainw->aud_rec_fd=open(outfilename,O_WRONLY|O_CREAT,S_IRUSR|S_IWUSR);
  g_free(outfilename);

  mainw->jackd_read=jack_get_driver(0,FALSE);
  mainw->jackd_read->playing_file=fileno;
  mainw->jackd_read->frames_written=0;

  if ((!out_bendian&&(G_BYTE_ORDER==G_BIG_ENDIAN))||(out_bendian&&(G_BYTE_ORDER==G_LITTLE_ENDIAN))) mainw->jackd_read->reverse_endian=TRUE;
  else mainw->jackd_read->reverse_endian=FALSE;

  // start jack recording
  jack_open_device_read(mainw->jackd_read);
  jack_read_driver_activate(mainw->jackd_read);

  // in grab window mode, just return, we will call rec_audio_end on playback end
  if (is_window_grab) return;

  mainw->cancelled=CANCEL_NONE;
  mainw->cancel_type=CANCEL_SOFT;
  // show countdown/stop dialog
  d_print(_("Recording audio..."));
  do_auto_dialog(_("Recording audio"),1);
  rec_audio_end();
}



void rec_audio_end(void) {
  // recording ended

  // stop recording
  jack_close_device(mainw->jackd_read,TRUE);
  mainw->jackd_read=NULL;

  // close file
  close(mainw->aud_rec_fd);
  mainw->cancel_type=CANCEL_KILL;
}

#endif
