// colourspace.h
// LiVES
// (c) G. Finch 2004 <salsaman@xs4all.nl>
// Released under the GPL 2.0 or higher
// see file ../COPYING for licensing details

// code for palette conversions

/*
 *  Some portions
 *  Copyright (C) 2001 Matthew J. Marjanovic <maddog@mir.com>
 *
 *
 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License
 *  as published by the Free Software Foundation; either version 2
 *  of the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */
// modifications to handle packed RGB24 by G. Finch (salsaman)


#include "main.h"

inline G_GNUC_CONST int gdk_rowstride_value (int rowstride) {
  // from gdk-pixbuf.c
  /* Always align rows to 32-bit boundaries */
  return (rowstride + 3) & ~3;
}


inline int G_GNUC_CONST gdk_last_rowstride_value (int width, int nchans) {
  // from gdk pixbuf docs
  return width*(((nchans<<3)+7)>>3);
}

static void
lives_free_buffer (guchar *pixels, gpointer data)
{
  g_free (pixels);
}


/* precomputed tables */

static gint Y_R[256];
static gint Y_G[256];
static gint Y_B[256];
static gint Cb_R[256];
static gint Cb_G[256];
static gint Cb_B[256];
static gint Cr_R[256];
static gint Cr_G[256];
static gint Cr_B[256];
static gint conv_RY_inited = 0;

static gint RGB_Y[256];
static gint R_Cr[256];
static gint G_Cb[256];
static gint G_Cr[256];
static gint B_Cb[256];
static gint conv_YR_inited = 0;


static gint myround(gdouble n)
{
  if (n >= 0) 
    return (gint)(n + 0.5);
  else
    return (gint)(n - 0.5);
}



static void init_RGB_to_YCbCr_tables(void)
{
  gint i;

  /*
   * Q_Z[i] =   (coefficient * i
   *             * (Q-excursion) / (Z-excursion) * fixed-pogint-factor)
   *
   * to one of each, add the following:
   *             + (fixed-pogint-factor / 2)         --- for rounding later
   *             + (Q-offset * fixed-pogint-factor)  --- to add the offset
   *             
   */
  for (i = 0; i < 256; i++) {
    Y_R[i] = myround(0.299 * (gdouble)i 
		     * 219.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Y_G[i] = myround(0.587 * (gdouble)i 
		     * 219.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Y_B[i] = myround((0.114 * (gdouble)i 
		      * 219.0 / 255.0 * (gdouble)(1<<FP_BITS))
		     + (gdouble)(1<<(FP_BITS-1))
		     + (16.0 * (gdouble)(1<<FP_BITS)));

    Cb_R[i] = myround(-0.168736 * (gdouble)i 
		      * 224.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Cb_G[i] = myround(-0.331264 * (gdouble)i 
		      * 224.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Cb_B[i] = myround((0.500 * (gdouble)i 
		       * 224.0 / 255.0 * (gdouble)(1<<FP_BITS))
		      + (gdouble)(1<<(FP_BITS-1))
		      + (128.0 * (gdouble)(1<<FP_BITS)));

    Cr_R[i] = myround(0.500 * (gdouble)i 
		      * 224.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Cr_G[i] = myround(-0.418688 * (gdouble)i 
		      * 224.0 / 255.0 * (gdouble)(1<<FP_BITS));
    Cr_B[i] = myround((-0.081312 * (gdouble)i 
		       * 224.0 / 255.0 * (gdouble)(1<<FP_BITS))
		      + (gdouble)(1<<(FP_BITS-1))
		      + (128.0 * (gdouble)(1<<FP_BITS)));
  }
  conv_RY_inited = 1;
}




static void init_YCbCr_to_RGB_tables(void)
{
  gint i;

  gint startclamp_y=16;
  gint endclamp_y=235;

  gint startclamp_crcb=16;
  gint endclamp_crcb=240;

  /*
   * Q_Z[i] =   (coefficient * i
   *             * (Q-excursion) / (Z-excursion) * fixed-pogint-factor)
   *
   * to one of each, add the following:
   *             + (fixed-pogint-factor / 2)         --- for rounding later
   *             + (Q-offset * fixed-pogint-factor)  --- to add the offset
   *             
   */

  /* clip Y values under 16 */
    for (i = 0; i < startclamp_y; i++) {
    RGB_Y[i] = myround((1.0 * (gdouble)(16 - 16) 
		     * 255.0 / 219.0 * (gdouble)(1<<FP_BITS))
		    + (gdouble)(1<<(FP_BITS-1)));
		    }
  for (i = startclamp_y; i <= endclamp_y; i++) {
    RGB_Y[i] = myround((1.0 * (gdouble)(i - 16) 
		     * 255.0 / 219.0 * (gdouble)(1<<FP_BITS))
		    + (gdouble)(1<<(FP_BITS-1)));
  }
  /* clip Y values above 235 */
    for (i = endclamp_y+1; i < 256; i++) {
    RGB_Y[i] = myround((1.0 * (gdouble)(235 - 16) 
		     * 255.0 / 219.0 * (gdouble)(1<<FP_BITS))
		    + (gdouble)(1<<(FP_BITS-1)));
  }
  
  /* clip Cb/Cr values below 16 */	 
    for (i = 0; i < startclamp_crcb; i++) {
    R_Cr[i] = myround(1.402 * (gdouble)(-112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (gdouble)(-112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (gdouble)(-112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (gdouble)(-112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
		   }
  for (i = startclamp_crcb; i <= endclamp_crcb; i++) {
    R_Cr[i] = myround(1.402 * (gdouble)(i - 128)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (gdouble)(i - 128)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (gdouble)(i - 128)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (gdouble)(i - 128)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
  }
  /* clip Cb/Cr values above 240 */	 
    for (i = endclamp_crcb+1; i < 256; i++) {
    R_Cr[i] = myround(1.402 * (gdouble)(112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cr[i] = myround(-0.714136 * (gdouble)(112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    G_Cb[i] = myround(-0.344136 * (gdouble)(i - 128)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
    B_Cb[i] = myround(1.772 * (gdouble)(112)
		   * 255.0 / 224.0 * (gdouble)(1<<FP_BITS));
		   }
  conv_YR_inited = 1;
}



/////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////


static inline void rgb2uyvy (guchar r0, guchar g0, guchar b0, guchar r1, guchar g1, guchar b1, uyvy_frame *uyvy) {
  register short a,b,c;
  uyvy->y0 = (Y_R[r0] + Y_G[g0]+ Y_B[b0]) >> FP_BITS;
  uyvy->y1 = (Y_R[r1] + Y_G[g1]+ Y_B[b1]) >> FP_BITS;
  if ((c=(((a=((Cb_R[r0]+Cb_G[g0]+Cb_B[b0])>>FP_BITS)-128)+(b=((Cb_R[r1]+Cb_G[g1]+Cb_B[b1])>>FP_BITS)-128))-((a*b)>>8))+128)>240) c=240;
  uyvy->v0=c<16?16:c;
  if ((c=(((a=((Cr_R[r0]+Cr_G[g0]+Cr_B[b0])>>FP_BITS)-128)+(b=((Cr_R[r1]+Cr_G[g1]+Cr_B[b1])>>FP_BITS)-128))-((a*b)>>8))+128)>240) c=240;
  uyvy->u0=c<16?16:c;
}


static void uyvy2rgb (uyvy_frame *uyvy, guchar *r0, guchar *g0, guchar *b0, guchar *r1, guchar *g1, guchar *b1) {
  register int r,g,b;

  if (G_UNLIKELY(!conv_YR_inited)) init_YCbCr_to_RGB_tables();

    r = (RGB_Y[uyvy->y0] + R_Cr[uyvy->v0]) >> FP_BITS;
    g = (RGB_Y[uyvy->y0] + G_Cb[uyvy->u0]+ G_Cr[uyvy->v0]) >> FP_BITS;
    b = (RGB_Y[uyvy->y0] + B_Cb[uyvy->u0]) >> FP_BITS;

    *r0 = (r < 0) ? 0 : (r > 255) ? 255 : r ;
    *g0 = (g < 0) ? 0 : (g > 255) ? 255 : g ;
    *b0 = (b < 0) ? 0 : (b > 255) ? 255 : b ;

    r = (RGB_Y[uyvy->y1] + R_Cr[uyvy->v0]) >> FP_BITS;
    g = (RGB_Y[uyvy->y1] + G_Cb[uyvy->u0]+ G_Cr[uyvy->v0]) >> FP_BITS;
    b = (RGB_Y[uyvy->y1] + B_Cb[uyvy->u0]) >> FP_BITS;

    *r1 = (r < 0) ? 0 : (r > 255) ? 255 : r ;
    *g1 = (g < 0) ? 0 : (g > 255) ? 255 : g ;
    *b1 = (b < 0) ? 0 : (b > 255) ? 255 : b ;
}

// remaining code (c) salsaman


void convert_yuv422p_to_rgb_frame(uint8_t *src, gint hsize, gint vsize, gint orowstride, guchar *dest) {
  register int x,y;
  int hs3=hsize*3;
  int hs2=hsize/2;

  uyvy_frame uyvy;
  guchar r0=0,g0=0,b0=0,r1=0,g1=0,b1=0;
  guchar *Y,*Cb,*Cr;

  gint size=hsize*vsize;
  gint flip=0;

  Y = src;
  Cb = &src[size];
  Cr = &src[size+size/4];

  for (y=0;y<size;y+=hsize) {
    for (x=0;x<hsize;x+=2) {
      uyvy.u0 = Cb[0];
      uyvy.y0 = Y[0];
      uyvy.v0 = Cr[0];
      uyvy.y1 = Y[1];

      uyvy2rgb (&uyvy, &r0, &g0, &b0, &r1, &g1, &b1);

      dest[0]=r0;
      dest[1]=g0;
      dest[2]=b0;
      dest[3]=r1;
      dest[4]=g1;
      dest[5]=b1;

      dest+=6;
      Y += 2;
      ++Cb;
      ++Cr;
    }
    if ((flip=!flip)) {
      // a little wasteful as we read in chroma twice 
      // but we can tap off a uyvy frame to pass to SDL
      Cr-=hs2;
      Cb-=hs2;
    }
    dest+=(orowstride-hs3);
  }
}


void convert_rgb_to_uyvy_frame(guchar *rgbdata, gint hsize, gint vsize, gint rowstride) {
  // for odd sized widths, cut the rightmost pixel
  gint hs3=(int)(hsize/2)*6;

  guchar *end=rgbdata+(rowstride*vsize)-5;
  uyvy_frame* u;
  register int i;

  mainw->noswitch=TRUE;
  if (G_UNLIKELY((mainw->uyvy_width!=hsize||mainw->uyvy_height!=vsize)&&mainw->uyvy_frame!=NULL)) {
    g_free(mainw->uyvy_frame);
    mainw->uyvy_frame=NULL;
  }
  mainw->noswitch=FALSE;

  if (G_UNLIKELY(mainw->uyvy_frame==NULL)) {
    size_t uyvy_size=(size_t)((int)(hs3/6)*vsize);
    mainw->uyvy_frame=(uyvy_frame*)g_try_malloc (uyvy_size*sizeof(uyvy_frame));
    if (mainw->uyvy_frame==NULL) return;
    mainw->uyvy_width=hs3/6;
    mainw->uyvy_height=vsize;
  }

  u=mainw->uyvy_frame;

  hs3-=5;

  if (G_UNLIKELY(!conv_RY_inited)) init_RGB_to_YCbCr_tables();

  for (;rgbdata<end;rgbdata+=rowstride) {
    for (i=0;i<hs3;i+=6) {
      // convert 6 RGBRGB bytes to 4 UYVY bytes
      rgb2uyvy (rgbdata[i+2],rgbdata[i+1],rgbdata[i],rgbdata[i+5],rgbdata[i+4],rgbdata[i+3],u++);
    }
  }
}



void convert_rgb_to_yuv420_frame(guchar *rgbdata, gint hsize, gint vsize, gint rowstride) {
  // for odd sized widths, cut the rightmost pixel
  gint hs3=(int)(hsize/2)*6;

  guchar *y,*Cb,*Cr;
  gint size=hsize*vsize;
  uyvy_frame u;
  register int i,j;
  gint chroma_row=1;

  mainw->noswitch=TRUE;
  // user could have switched clips while we were not looking...
  if (mainw->reinit_effects&&mainw->yuv_frame!=NULL) {
    g_free(mainw->yuv_frame);
    mainw->yuv_frame=NULL;
  }
  mainw->noswitch=FALSE;

  if (mainw->yuv_frame==NULL) {
    mainw->yuv_frame=(guchar *)g_try_malloc (size*1.5);
    if (mainw->yuv_frame==NULL) return;
  }
  y=mainw->yuv_frame;
  Cb=&(y[size]);
  Cr=&(y[size*5/4]);

  size=rowstride*vsize;

  if (!conv_RY_inited) init_RGB_to_YCbCr_tables();

  hs3-=5;
  for (i=0;i<size;i+=rowstride) {
    for (j=0;j<hs3;j+=6) {
      // convert 6 RGBRGB bytes to 4 UYVY bytes
      rgb2uyvy (rgbdata[i+j+2],rgbdata[i+j+1],rgbdata[i+j],rgbdata[i+j+5],rgbdata[i+j+4],rgbdata[i+j+3],&u);

      // TODO - make a UYVY frame here which can be passed to SDL
      // allow multiple pb plugins

      y[0]=u.y0;
      y[1]=u.y1;

      // halve the chroma samples for (lower quality) yuv420
      if (chroma_row) {
	Cb[0]=u.u0;
	Cr[0]=u.v0;
      }
      else {
	// average two rows
	Cb[0]=(Cb[0]+u.u0)>>1;
	Cr[0]=(Cr[0]+u.v0)>>1;
      }
      Cb++;
      Cr++;
      y+=2;
    }
    if (chroma_row) {
      Cb-=hsize/2;
      Cr-=hsize/2;
    }
    chroma_row=!chroma_row;
  }
}



/////////////////////////////////////////////////////////////////
// RGB palette conversions

static void convert_swap3_frame (guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // swap 3 byte palette
  guchar *end=src+height*irowstride;

  if ((irowstride==width*3)&&(orowstride==irowstride)) {
    // quick version
#ifdef ENABLE_OIL
    oil_rgb2bgr(dest,src,width*height);
#else
    for (;src<end;src+=3) {
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
    }
#endif
  }
  else {
    register int i;
    int width3=width*3;
    orowstride-=width3;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width3;i+=3) {
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
      }
      dest+=orowstride;
    }
  }
}


static void convert_swap4_frame (guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // swap 4 byte palette
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==irowstride)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[3]; // alpha
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i+3]; // alpha
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
      }
      dest+=orowstride;
    }
  }
}


static void convert_swap3addpost_frame(guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // swap 3 bytes, add post alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*3)&&(orowstride==width*4)) {
    // quick version
    for (;src<end;src+=3) {
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
      *(dest++)=255; // alpha
    }
  }
  else {
    register int i;
    int width3=width*3;
    orowstride-=width*4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width3;i+=3) {
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
	*(dest++)=255; // alpha
      }
      dest+=orowstride;
    }
  }
}


static void convert_swap3addpre_frame(guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // swap 3 bytes, add pre alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*3)&&(orowstride==width*4)) {
    // quick version
    for (;src<end;src+=3) {
      *(dest++)=255; // alpha
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
    }
  }
  else {
    register int i;
    int width3=width*3;
    orowstride-=width*4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width3;i+=3) {
	*(dest++)=255; // alpha
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
      }
      dest+=orowstride;
    }
  }
}


static void convert_swap3postalpha_frame(guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // swap 3 bytes, leave alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==irowstride)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
      *(dest++)=src[3]; // alpha
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
	*(dest++)=src[i+3]; // alpha
      }
      dest+=orowstride;
    }
  }
}

static void convert_addpost_frame(guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // add post alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*3)&&(orowstride==width*4)) {
    // quick version
#ifdef ENABLE_OIL
    oil_rgb2rgba(dest,src,width*height);
#else
    for (;src<end;src+=3) {
      *(dest++)=src[0]; // r
      *(dest++)=src[1]; // g
      *(dest++)=src[2]; // b
      *(dest++)=255; // alpha
    }
#endif
  }
  else {
    register int i;
    int width3=width*3;
    orowstride-=width*4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width3;i+=3) {
	*(dest++)=src[i]; // r
	*(dest++)=src[i+1]; // g
	*(dest++)=src[i+2]; // b
	*(dest++)=255; // alpha
      }
      dest+=orowstride;
    }
  }
}


static void convert_addpre_frame(guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest) {
  // add pre alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*3)&&(orowstride==width*4)) {
    // quick version
    for (;src<end;src+=3) {
      *(dest++)=255; // alpha
      *(dest++)=src[0]; // r
      *(dest++)=src[1]; // g
      *(dest++)=src[2]; // b
    }
  }
  else {
    register int i;
    int width3=width*3;
    orowstride-=width*4;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width3;i+=3) {
	*(dest++)=255; // alpha
	*(dest++)=src[i]; // r
	*(dest++)=src[i+1]; // g
	*(dest++)=src[i+2]; // b
      }
      dest+=orowstride;
    }
  }
}


static void convert_swap3delpost_frame(guchar *src,int width,int height, int irowstride, int orowstride, guchar *dest) {
  // swap 3 bytes, delete post alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==width*3)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[2]; // red
      *(dest++)=src[1]; // green
      *(dest++)=src[0]; // blue
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width*3;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i+2]; // red
	*(dest++)=src[i+1]; // green
	*(dest++)=src[i]; // blue
      }
      dest+=orowstride;
    }
  }
}


static void convert_delpost_frame(guchar *src,int width,int height, int irowstride, int orowstride, guchar *dest) {
  // delete post alpha
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==width*3)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[0]; // r
      *(dest++)=src[1]; // g
      *(dest++)=src[2]; // b
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width*3;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i]; // r
	*(dest++)=src[i+1]; // g
	*(dest++)=src[i+2]; // b
      }
      dest+=orowstride;
    }
  }
}


static void convert_delpre_frame(guchar *src,int width,int height, int irowstride, int orowstride, guchar *dest) {
  // delete pre alpha
  guchar *end=src+height*irowstride;

  src++;

  if ((irowstride==width*4)&&(orowstride==width*3)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[0]; // r
      *(dest++)=src[1]; // g
      *(dest++)=src[2]; // b
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width*3;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i]; // r
	*(dest++)=src[i+1]; // g
	*(dest++)=src[i+2]; // b
      }
      dest+=orowstride;
    }
  }
}


static void convert_delpreswap3_frame(guchar *src,int width,int height, int irowstride, int orowstride, guchar *dest) {
  // delete pre alpha, swap last 3
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==width*3)) {
    // quick version
    for (;src<end;src+=4) {
      *(dest++)=src[3]; // red
      *(dest++)=src[2]; // green
      *(dest++)=src[1]; // blue
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width*3;
    for (;src<end;src+=irowstride) {
      for (i=0;i<width4;i+=4) {
	*(dest++)=src[i+3]; // red
	*(dest++)=src[i+2]; // green
	*(dest++)=src[i+1]; // blue
      }
      dest+=orowstride;
    }
  }
}

static void convert_swapprepost_frame (guchar *src, int width, int height, int irowstride, int orowstride, guchar *dest, gboolean alpha_first) {
  // swap first and last bytes in a 4 byte palette
  guchar *end=src+height*irowstride;

  if ((irowstride==width*4)&&(orowstride==irowstride)) {
    // quick version
    if (alpha_first) {
      for (;src<end;src+=4) {
	*(dest++)=src[1];
	*(dest++)=src[2];
	*(dest++)=src[3];
	*(dest++)=src[0];
      }
    }
    else {
      for (;src<end;src+=4) {
	*(dest++)=src[3];
	*(dest++)=src[0];
	*(dest++)=src[1];
	*(dest++)=src[2];
      }
    }
  }
  else {
    register int i;
    int width4=width*4;
    orowstride-=width4;
    for (;src<end;src+=irowstride) {
      if (alpha_first) {
	for (i=0;i<width4;i+=4) {
	*(dest++)=src[i+1];
	*(dest++)=src[i+2];
	*(dest++)=src[i+3];
	*(dest++)=src[i];
	}
      }
      else {
	for (i=0;i<width4;i+=4) {
	  *(dest++)=src[i+3];
	  *(dest++)=src[i];
	  *(dest++)=src[i+1];
	  *(dest++)=src[i+2];
	}
      }
      dest+=orowstride;
    }
  }
}

////////////////////////////////////////////////////////////////////////////////////////


void create_empty_pixel_data(weed_plant_t *layer) {
  // width, height are src size in pixels
  // rowstrides: - here we use the same as gdk-pixbuf: for RGB24/BGR24 we use a rowstride which produces a multiple of 32 bits
  int error;
  int palette=weed_get_int_value(layer,"current_palette",&error);
  int width=weed_get_int_value(layer,"width",&error);
  int height=weed_get_int_value(layer,"height",&error);
  int rowstride;

  guchar *pixel_data;

  switch (palette) {
  case WEED_PALETTE_RGB24:
  case WEED_PALETTE_BGR24:
  case WEED_PALETTE_YUV888:
    rowstride=width*3;
    pixel_data=(guchar *)calloc(width*height,3);
    weed_set_voidptr_value(layer,"pixel_data",pixel_data);
    weed_set_int_value(layer,"rowstrides",rowstride);
    break;
  case WEED_PALETTE_RGBA32:
  case WEED_PALETTE_BGRA32:
  case WEED_PALETTE_ARGB32:
  case WEED_PALETTE_UYVY8888:
  case WEED_PALETTE_YUYV8888:
  case WEED_PALETTE_YUVA8888:
    rowstride=width*4;
    pixel_data=(guchar *)calloc(width*height,4);
    weed_set_voidptr_value(layer,"pixel_data",pixel_data);
    weed_set_int_value(layer,"rowstrides",rowstride);
    break;
  default:
    g_printerr("Warning: asked to create empty pixel_data for palette %d !\n",palette);
  }
}


gboolean convert_layer_palette(weed_plant_t *layer, int outpl) {
  guchar *gusrc,*gudest;
  int width,height,orowstride,irowstride;
  int error,inpl;

  inpl=weed_get_int_value(layer,"current_palette",&error);
  if (inpl==outpl) return TRUE;

  width=weed_get_int_value(layer,"width",&error);
  height=weed_get_int_value(layer,"height",&error);

  switch (inpl) {
  case WEED_PALETTE_BGR24:
    gusrc=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
    irowstride=weed_get_int_value(layer,"rowstrides",&error);
    switch (outpl) {
    case WEED_PALETTE_RGBA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3addpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGB24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_BGRA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_addpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_ARGB32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3addpre_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
   default:
      g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
      return FALSE;
    }
    break;
  case WEED_PALETTE_RGBA32:
    gusrc=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
    irowstride=weed_get_int_value(layer,"rowstrides",&error);
    switch (outpl) {
    case WEED_PALETTE_BGR24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3delpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGB24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_delpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_BGRA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3postalpha_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_ARGB32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swapprepost_frame(gusrc,width,height,irowstride,orowstride,gudest,FALSE);
      break;
    default:
      g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
      return FALSE;
    }
    break;
  case WEED_PALETTE_RGB24:
    gusrc=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
    irowstride=weed_get_int_value(layer,"rowstrides",&error);
    switch (outpl) {
    case WEED_PALETTE_BGR24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGBA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_addpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_BGRA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3addpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_ARGB32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_addpre_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    default:
      g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
      return FALSE;
    }
    break;
  case WEED_PALETTE_BGRA32:
    gusrc=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
    irowstride=weed_get_int_value(layer,"rowstrides",&error);
    switch (outpl) {
    case WEED_PALETTE_BGR24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_delpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGB24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3delpost_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGBA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap3postalpha_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_ARGB32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap4_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    default:
      g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
      return FALSE;
    }
    break;
  case WEED_PALETTE_ARGB32:
    gusrc=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
    irowstride=weed_get_int_value(layer,"rowstrides",&error);
    switch (outpl) {
    case WEED_PALETTE_BGR24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_delpreswap3_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGB24:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_delpre_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    case WEED_PALETTE_RGBA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swapprepost_frame(gusrc,width,height,irowstride,orowstride,gudest,TRUE);
      break;
    case WEED_PALETTE_BGRA32:
      weed_set_int_value(layer,"current_palette",outpl);
      create_empty_pixel_data(layer);
      orowstride=weed_get_int_value(layer,"rowstrides",&error);
      gudest=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error);
      convert_swap4_frame(gusrc,width,height,irowstride,orowstride,gudest);
      break;
    default:
      g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
      return FALSE;
    }
    break;
  default:
    g_printerr("Invalid palette conversion: %d to %d not written yet !!\n",inpl,outpl);
    return FALSE;
  }
  g_free(gusrc);
  return TRUE;
}



/////////////////////////////////////////////////////////////////////////////////////


GdkPixbuf *gdk_pixbuf_new_blank(gint width, gint height, int palette) {
  GdkPixbuf *pixbuf;
  guchar *pixels;
  gint rowstride;

  switch (palette) {
  case WEED_PALETTE_RGB24:
  case WEED_PALETTE_BGR24:
    pixbuf=gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
    pixels=gdk_pixbuf_get_pixels (pixbuf);
    rowstride=gdk_pixbuf_get_rowstride(pixbuf);
    break;
  case WEED_PALETTE_RGBA32:
  case WEED_PALETTE_BGRA32:
    pixbuf=gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, width, height);
    pixels=gdk_pixbuf_get_pixels (pixbuf);
    rowstride=gdk_pixbuf_get_rowstride(pixbuf);
    break;
  case WEED_PALETTE_ARGB32:
    pixbuf=gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, width, height);
    pixels=gdk_pixbuf_get_pixels (pixbuf);
    rowstride=gdk_pixbuf_get_rowstride(pixbuf);
    break;
  default:
    return NULL;
  }
  return pixbuf;
}



static GdkPixbuf *gdk_pixbuf_cheat(GdkColorspace colorspace, gboolean has_alpha, int bits_per_sample, int width, int height, guchar *buf) {
  // we can cheat if our buffer is correctly sized
  int channels=has_alpha?4:3;
  int rowstride=gdk_rowstride_value(width*channels);
  return gdk_pixbuf_new_from_data (buf, colorspace, has_alpha, bits_per_sample, width, height, rowstride, lives_free_buffer, NULL);
}


GdkPixbuf *layer_to_pixbuf (weed_plant_t *layer) {
  int error;
  GdkPixbuf *pixbuf;
  int palette=weed_get_int_value(layer,"current_palette",&error);
  int width=weed_get_int_value(layer,"width",&error);
  int height=weed_get_int_value(layer,"height",&error);
  int irowstride=weed_get_int_value(layer,"rowstrides",&error);
  int rowstride,orowstride;
  guchar *pixel_data=(guchar *)weed_get_voidptr_value(layer,"pixel_data",&error),*pixels,*end,*orig_pixel_data=pixel_data;
  gboolean cheat=FALSE;
  gint n_channels;

  switch (palette) {
  case WEED_PALETTE_RGB24:
  case WEED_PALETTE_BGR24:
    if (irowstride==gdk_rowstride_value(width*3)) {
      pixbuf=gdk_pixbuf_cheat(GDK_COLORSPACE_RGB, FALSE, 8, width, height, pixel_data);
      cheat=TRUE;
    }
    else pixbuf=gdk_pixbuf_new (GDK_COLORSPACE_RGB, FALSE, 8, width, height);
    n_channels=3;
    break;
  case WEED_PALETTE_RGBA32:
  case WEED_PALETTE_BGRA32:
  case WEED_PALETTE_ARGB32: // TODO - change to RGBA ??
    if (irowstride==gdk_rowstride_value(width*4)) {
      pixbuf=gdk_pixbuf_cheat(GDK_COLORSPACE_RGB, TRUE, 8, width, height, pixel_data);
      cheat=TRUE;
    }
    else pixbuf=gdk_pixbuf_new (GDK_COLORSPACE_RGB, TRUE, 8, width, height);
    n_channels=4;
    break;
  default:
    return NULL;
  }
  pixels=gdk_pixbuf_get_pixels (pixbuf);
  orowstride=gdk_pixbuf_get_rowstride(pixbuf);
  
  if (irowstride>orowstride) rowstride=orowstride;
  else rowstride=irowstride;
  end=pixels+orowstride*height;

  if (!cheat) {
    gboolean done=FALSE;
    for (;pixels<end&&!done;pixels+=orowstride) {
      if (pixels+orowstride>=end) {
	orowstride=rowstride=gdk_last_rowstride_value(width,n_channels);
	done=TRUE;
      }
#ifdef ENABLE_OIL
      oil_memcpy(pixels,pixel_data,rowstride);
#else
      memcpy(pixels,pixel_data,rowstride);
#endif
      if (rowstride<orowstride) memset (pixels+rowstride,0,orowstride-rowstride);
      pixel_data+=irowstride;
    }
    g_free(orig_pixel_data);
    pixel_data=NULL;
    weed_set_voidptr_value(layer,"pixel_data",pixel_data);
  }
  return pixbuf;
}


void resize_layer (weed_plant_t *layer, int width, int height, int interp) {
  int error;
  GdkPixbuf *pixbuf,*new_pixbuf=NULL;
  int palette=weed_get_int_value(layer,"current_palette",&error);
  gint n_channels=4;
  gint nwidth=0,nheight=0,rowstride=0;

  switch (palette) {
  case WEED_PALETTE_RGB24:
  case WEED_PALETTE_BGR24:
    n_channels=3;
  case WEED_PALETTE_ARGB32: // TODO - change to RGBA ??
  case WEED_PALETTE_RGBA32:
  case WEED_PALETTE_BGRA32:
    pixbuf=layer_to_pixbuf(layer);
    new_pixbuf=gdk_pixbuf_scale_simple(pixbuf,width,height,interp);
    if (new_pixbuf!=NULL) {
      weed_set_int_value(layer,"width",(nwidth=gdk_pixbuf_get_width(new_pixbuf)));
      weed_set_int_value(layer,"height",(nheight=gdk_pixbuf_get_height(new_pixbuf)));
      weed_set_int_value(layer,"rowstrides",(rowstride=gdk_pixbuf_get_rowstride(new_pixbuf)));
    }
    g_object_unref(pixbuf);
    break;
  default:
    g_printerr("Warning: resizing unknown palette %d\n",palette);
  }

  if (new_pixbuf==NULL||(width!=weed_get_int_value(layer,"width",&error)||height!=weed_get_int_value(layer,"height",&error))) g_printerr("unable to scale layer to %d x %d for palette %d\n",width,height,palette);

  if (new_pixbuf!=NULL) {
    if (pixbuf_to_layer(layer,new_pixbuf)) {
      mainw->do_not_free=gdk_pixbuf_get_pixels(new_pixbuf);
      mainw->free_fn=lives_free_with_check;
    }
    g_object_unref(new_pixbuf);
    mainw->do_not_free=NULL;
    mainw->free_fn=free;
  }
  return;
}


gboolean pixbuf_to_layer(weed_plant_t *layer, GdkPixbuf *pixbuf) {
  // this function is needed because layers always have a memory size height*rowstride, whereas gdkpixbuf can have
  // a shorter last row

  // return TRUE if we can use the original pixbuf pixels

  int rowstride=gdk_pixbuf_get_rowstride(pixbuf);
  int width=gdk_pixbuf_get_width(pixbuf);
  int height=gdk_pixbuf_get_height(pixbuf);
  int nchannels=gdk_pixbuf_get_n_channels(pixbuf);
  void *pixel_data;
  void *in_pixel_data=gdk_pixbuf_get_pixels(pixbuf);

  if (rowstride==gdk_last_rowstride_value(width,nchannels)) {
    weed_set_voidptr_value(layer,"pixel_data",in_pixel_data);
    return TRUE;
  }

  pixel_data=calloc(height*rowstride,1);
  memcpy(pixel_data,in_pixel_data,rowstride*(height-1));
  memcpy(pixel_data+rowstride*(height-1),in_pixel_data+rowstride*(height-1),gdk_last_rowstride_value(width,nchannels));
  weed_set_voidptr_value(layer,"pixel_data",pixel_data);
  return FALSE;

}
